"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const ec2 = require("@aws-cdk/aws-ec2");
const ecr = require("@aws-cdk/aws-ecr");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const batch = require("../lib");
describe('Batch Job Definition', () => {
    let stack;
    let jobDefProps;
    beforeEach(() => {
        stack = new cdk.Stack();
        const role = new iam.Role(stack, 'job-role', {
            assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
        });
        const linuxParams = new ecs.LinuxParameters(stack, 'job-linux-params', {
            initProcessEnabled: true,
            sharedMemorySize: 1,
        });
        jobDefProps = {
            jobDefinitionName: 'test-job',
            container: {
                command: ['echo "Hello World"'],
                environment: {
                    foo: 'bar',
                },
                jobRole: role,
                gpuCount: 1,
                image: ecs.EcrImage.fromRegistry('docker/whalesay'),
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
                linuxParams,
                memoryLimitMiB: 1,
                mountPoints: new Array(),
                privileged: true,
                readOnly: true,
                ulimits: new Array(),
                user: 'root',
                vcpus: 2,
                volumes: new Array(),
            },
            nodeProps: {
                count: 2,
                mainNode: 1,
                rangeProps: new Array(),
            },
            parameters: {
                foo: 'bar',
            },
            retryAttempts: 2,
            timeout: cdk.Duration.seconds(30),
        };
    });
    test('renders the correct cloudformation properties', () => {
        // WHEN
        new batch.JobDefinition(stack, 'job-def', jobDefProps);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            JobDefinitionName: jobDefProps.jobDefinitionName,
            ContainerProperties: jobDefProps.container ? {
                Command: jobDefProps.container.command,
                Environment: [
                    {
                        Name: 'foo',
                        Value: 'bar',
                    },
                ],
                InstanceType: jobDefProps.container.instanceType ? jobDefProps.container.instanceType.toString() : '',
                LinuxParameters: {},
                Memory: jobDefProps.container.memoryLimitMiB,
                MountPoints: [],
                Privileged: jobDefProps.container.privileged,
                ReadonlyRootFilesystem: jobDefProps.container.readOnly,
                Ulimits: [],
                User: jobDefProps.container.user,
                Vcpus: jobDefProps.container.vcpus,
                Volumes: [],
            } : undefined,
            NodeProperties: jobDefProps.nodeProps ? {
                MainNode: jobDefProps.nodeProps.mainNode,
                NodeRangeProperties: [],
                NumNodes: jobDefProps.nodeProps.count,
            } : undefined,
            Parameters: {
                foo: 'bar',
            },
            RetryStrategy: {
                Attempts: jobDefProps.retryAttempts,
            },
            Timeout: {
                AttemptDurationSeconds: jobDefProps.timeout ? jobDefProps.timeout.toSeconds() : -1,
            },
            Type: 'container',
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can use an ecr image', () => {
        // WHEN
        const repo = new ecr.Repository(stack, 'image-repo');
        new batch.JobDefinition(stack, 'job-def', {
            container: {
                image: ecs.ContainerImage.fromEcrRepository(repo),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            ContainerProperties: {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    4,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::GetAtt': [
                                                    'imagerepoD116FAF0',
                                                    'Arn'
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            },
                            '.dkr.ecr.',
                            {
                                'Fn::Select': [
                                    3,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::GetAtt': [
                                                    'imagerepoD116FAF0',
                                                    'Arn'
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/',
                            {
                                Ref: 'imagerepoD116FAF0'
                            },
                            ':latest'
                        ]
                    ]
                },
                Memory: 4,
                Privileged: false,
                ReadonlyRootFilesystem: false,
                Vcpus: 1
            }
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can use a registry image', () => {
        // WHEN
        new batch.JobDefinition(stack, 'job-def', {
            container: {
                image: ecs.ContainerImage.fromRegistry('docker/whalesay'),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            ContainerProperties: {
                Image: 'docker/whalesay',
                Memory: 4,
                Privileged: false,
                ReadonlyRootFilesystem: false,
                Vcpus: 1,
            },
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can be imported from an ARN', () => {
        // WHEN
        const importedJob = batch.JobDefinition.fromJobDefinitionArn(stack, 'job-def-clone', 'arn:aws:batch:us-east-1:123456789012:job-definition/job-def-name:1');
        // THEN
        expect(importedJob.jobDefinitionName).toEqual('job-def-name:1');
        expect(importedJob.jobDefinitionArn).toEqual('arn:aws:batch:us-east-1:123456789012:job-definition/job-def-name:1');
    });
});
//# sourceMappingURL=data:application/json;base64,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