# -*- coding: utf-8 -*-
"""
Price extractor was developed mostly using PRICE_PARSING_EXAMPLES,
then evaluated on PRICE_PARSING_EXAMPLES_2, then improved to work better on
PRICE_PARSING_EXAMPLES_3, PRICE_PARSING_EXAMPLES_NO_PRICE
and PRICE_PARSING_EXAMPLES_NO_CURRENCY.

This data is collected from a random sample of pages from different domains.

PRICE_PARSING_EXAMPLES_BUGS_CAUGHT are manually added examples for the bugs
we've found in a wild; PRICE_PARSING_EXAMPLES_NEW is a list of tests for
new features. New tests should probably go these two lists.
"""
from typing import Optional, Union
from decimal import Decimal

import pytest

from price_parser import Price


class Example(Price):
    """ A Price wrapper for tests """
    def __init__(self,
                 currency_raw: Optional[str],
                 price_raw: Optional[str],
                 currency: Optional[str],
                 amount_text: Optional[str],
                 amount_float: Optional[Union[float, Decimal]],
                 decimal_separator: Optional[str] = None) -> None:
        self.currency_raw = currency_raw
        self.price_raw = price_raw
        self.decimal_separator = decimal_separator
        amount_decimal = None  # type: Optional[Decimal]
        if isinstance(amount_float, Decimal):
            amount_decimal = amount_float
        elif amount_float is not None:
            # don't use Decimal(amount_float), as this is not what
            # one usually means, because of float precision
            amount_decimal = Decimal(str(amount_float))
        super().__init__(
            amount=amount_decimal,
            currency=currency,
            amount_text=amount_text,
        )

    def __eq__(self, other):
        if not isinstance(other, Price):
            return super().__eq__(other)
        return (
            self.amount == other.amount and
            self.currency == other.currency and
            self.amount_text == other.amount_text
        )


PRICE_PARSING_EXAMPLES_BUGS_CAUGHT = [
    Example(None, 'US$:12.99',
            'US$', '12.99', 12.99),
    Example('GBP', '34.992001',
            'GBP', '34.992001', 34.992001),
    Example('GBP', '29.1583',
            'GBP', '29.1583', 29.1583),
    Example(None, '1.11000000000000009770',
            None, '1.11000000000000009770', Decimal('1.11000000000000009770')),
]


PRICE_PARSING_EXAMPLES_NEW = [
    Example(None, 'PTE 120 000 000',
            'PTE', '120 000 000', 120000000),
    Example(None, 'DEM 170 000',
            'DEM', '170 000', 170000),
    Example(None, '₤1700.',
            '₤', '1700', 1700),
    Example(None, '$ 1.144.000',
            '$', '1.144.000', 1144000),
    Example(None, 'A$190.00',
            'A$', '190.00', 190),
    Example(None, '205,68 € 205.68',
            '€', '205,68', 205.68),
    Example(None, 'AED 8000 (USD 2179)',
            'AED', '8000', 8000),
    Example(None, '13800 ₶',
            '₶', '13800', 13800),
    Example(None, '12,000원',
            '원', '12,000', 12000),
    Example(None, '3,500円',
            '円', '3,500', 3500)
]


PRICE_PARSING_EXAMPLES = [
    Example('90 728.00 руб 103 100.00 руб', '399 167.00 руб 420 176.16 руб',
            'руб', '399 167.00', 399167),
    Example('45,00 zł', '45,00 zł',
            'zł', '45,00', 45),
    Example('$', '$ 22.00',
            '$', '22.00', 22),
    Example('$3.99', '$14.99',
            '$', '14.99', 14.99),
    Example('Price:', '$11.95',
            '$', '11.95', 11.95),
    Example('$19.95', '$19.95',
            '$', '19.95', 19.95),
    Example('Price From £ 39.95', '39.95',
            '£', '39.95', 39.95),
    Example('comprar', 'R$260,00',
            'R$', '260,00', 260),
    Example('$79.00', '$79.00',
            '$', '79.00', 79),
    Example('34,50 € *', '34,50 € *',
            '€', '34,50', 34.5),
    Example('35,70 €', '35,70 €',
            '€', '35,70', 35.7),
    Example('Price', '$147.44',
            '$', '147.44', 147.44),
    Example('Up to', '$4.70',
            '$', '4.70', 4.7),
    Example('$999.99', '$924.99',
            '$', '924.99', 924.99),
    Example('€', '690,00',
            '€', '690,00', 690),
    Example('Cena', 'Běžná cena 9 800 Kč',
            'Kč', '9 800', 9800),
    Example('729,00 €', '479,00 €',
            '€', '479,00', 479),
    Example('naše cena', 'běžná cena 590 Kč',
            'Kč', '590', 590),
    Example('Statt 249,00 EUR **', '249,00 EUR',
            'EUR', '249,00', 249),
    Example('€13.95', '€13.95',
            '€', '13.95', 13.95),
    Example('€36', '€89',
            '€', '89', 89),
    Example('171900 руб.', '171900 руб.',
            'руб.', '171900', 171900),
    Example('$1.06', '$0.31',
            '$', '0.31', 0.31),
    Example('$45.49', '$45.49',
            '$', '45.49', 45.49),
    Example('$', '28',
            '$', '28', 28),
    Example('4.81 16 von 5', '4.81',
            None, '4.81', 4.81),
    Example('CHF 86.00 *', 'CHF 47.00 *',
            'CHF', '47.00', 47),
    Example('320 руб. 400 руб.', '419 руб.',
            'руб.', '419', 419),
    Example('129,99 €', '129,99 €',
            '€', '129,99', 129.99),
    Example('Cena', '1,10 €',
            '€', '1,10', 1.1),
    Example('Price:', '$559.00',
            '$', '559.00', 559),
    Example('$49.99', '$49.99',
            '$', '49.99', 49.99),
    Example('Price: $32.00', '$32.00',
            '$', '32.00', 32),
    Example('Price (high to low)', '$699,000',
            '$', '699,000', 699000),
    Example('€ 11.69', '€ 11.69',
            '€', '11.69', 11.69),
    Example('100,00 руб.', '100,00',
            'руб.', '100,00', 100),
    Example('€ 1', '17,35',
            '€', '17,35', 17.35),
    Example('$19.95', '$19.95',
            '$', '19.95', 19.95),
    Example('-', '£11.95',
            '£', '11.95', 11.95),
    Example('грн.', '5 870',
            'грн.', '5 870', 5870),
    Example('Price', '80 Kč',
            'Kč', '80', 80),
    Example('€ 640,', '€ 610,',
            '€', '610', 610),
    Example('Cena', '299 Kč',
            'Kč', '299', 299),
    Example('1128240 рублей', '1128240 рублей',
            'руб', '1128240', 1128240),
    Example('Now £9.99 $13.71 11,16 € £8.33 $13.71 9,30 €', '£9.99',
            '£', '9.99', 9.99),
    Example('Free!', 'Free!',
            None, '0', 0),
    Example('49,96€', '49,96€',
            '€', '49,96', 49.96),
    Example('SKU:', '$9.00',
            '$', '9.00', 9),
    Example('£ 8.29', '8.29',
            '£', '8.29', 8.29),
    Example('Р', '6 300 Р',
            'Р', '6 300', 6300),
    Example('99,99 EUR (-30,00%) 69,99 EUR', '99,99 EUR',
            'EUR', '99,99', 99.99),
    Example('Р', '1 890 Р',
            'Р', '1 890', 1890),
    Example('Disposable Arts', '15,95 €',
            '€', '15,95', 15.95),
    Example('zł', '16.00',
            'zł', '16.00', 16),
    Example('£ 4.99', '4.99',
            '£', '4.99', 4.99),
    Example('Price', '$119. 95',
            '$', '119. 95', 119.95),
    Example('269 Kč', '269 Kč',
            'Kč', '269', 269),
    Example('>', '13,30 €',
            '€', '13,30', 13.3),
    Example('₪ 115', '₪98',
            '₪', '98', 98),
    Example('80,00 €', '80,00 €',
            '€', '80,00', 80),
    Example('33,95 €', '29,95 €',
            '€', '29,95', 29.95),
    Example('£ 80.00', '80.00',
            '£', '80.00', 80),
    Example('EUR', '€16.95',
            '€', '16.95', 16.95),
    Example('DKK', '23,40',
            'DKK', '23,40', 23.4),
    Example('Price: $ 34.44', '34.44',
            '$', '34.44', 34.44),
    Example('Rp 31.500', 'Rp 31.500',
            'Rp', '31.500', 31500),
    Example('běžná cena 6,00 Kč', 'běžná cena 6,00 Kč',
            'Kč', '6,00', 6),
    Example('€', '19,59',
            '€', '19,59', 19.59),
    Example('Out of stock', '2 285,81 грн.',
            'грн.', '2 285,81', 2285.81),
    Example('$ 119,999', '$ 119,999',
            '$', '119,999', 119999),
    Example('1,20 €', '1,20 €',
            '€', '1,20', 1.2),
    Example('€ 1.99', '1.99',
            '€', '1.99', 1.99),
    Example('Cena 283,50 PLN', 'Cena 283,50 PLN',
            'PLN', '283,50', 283.5),
    Example('SKU:', '99.99',
            None, '99.99', 99.99),
    Example('$', '$ 979.00',
            '$', '979.00', 979),
    Example('Cena 169,00 Kč', '169,00 Kč',
            'Kč', '169,00', 169),
    Example('Р', '94,456 Р',
            'Р', '94,456', 94456),
    Example('€', '180',
            '€', '180', 180),
    Example('Savings: $0.90', '2.85',
            '$', '2.85', 2.85),
    Example('฿ 3.3900', '฿ 3.3900',
            '฿', '3.3900', 3.39),
    Example('5,90 €', '5,90 €',
            '€', '5,90', 5.90),
    Example('1 930 р.', '1 030 р.',
            'р.', '1 030', 1030),
    Example('$', '$ 22.00',
            '$', '22.00', 22),
    Example('HUF', '39900',
            'HUF', '39900', 39900),
    Example('59,00 €', '79,00 €',
            '€', '79,00', 79),
    Example('(-38,23%)', '429,00 EUR',
            'EUR', '429,00', 429),
    Example('302,44€ Χωρίς ΦΠΑ: 243,90€', '302,44€',
            '€', '302,44', 302.44),
    Example('RENAULT', '129.900',
            None, '129.900', 129900),
    Example('USD', '£54.17',
            '£', '54.17', 54.17),
    Example('$3.00', '$3.00',
            '$', '3.00', 3),
    Example('3,49 zł 1,75 zł', '3,49 zł 1,75 zł',
            'zł', '3,49', 3.49),
    Example('59,90 EUR 599,00 EUR pro Kilogramm', '59,90 EUR',
            'EUR', '59,90', 59.9),
    Example('Vanaf € 49.95', 'Vanaf € 49.95',
            '€', '49.95', 49.95),
    Example('$299.00', '$299.00',
            '$', '299.00', 299),
    Example('€', '€30,40 **',
            '€', '30,40', 30.4),
    Example('$59.99', '$59.99',
            '$', '59.99', 59.99),
    Example('EUR', '2800',
            'EUR', '2800', 2800),
    Example('225,00 € *', '225,00 € *',
            '€', '225,00', 225),
    Example('17 940,00 руб', '17 940,00 руб',
            'руб', '17 940,00', 17940),
    Example('159,50 €', '159,50 €',
            '€', '159,50', 159.5),
    Example('£18.99', '£18.99',
            '£', '18.99', 18.99),
    Example('2.999,00 EUR (-10,00%) 2.699,00 EUR', '699,00 EUR',
            'EUR', '699,00', 699),
    Example('$', '45.33',
            '$', '45.33', 45.33),
    Example('грн.', '33',
            'грн.', '33', 33),
    Example('EUR', '16,50 €',
            '€', '16,50', 16.5),
    Example('₫', '880,000 ₫',
            '₫', '880,000', 880000),
    Example('руб.', '13.50',
            'руб.', '13.50', 13.5),
    Example('336 р.', '336 р.',
            'р.', '336', 336),
    Example('....', '200',
            None, '200', 200),
    Example('Our Price:', '$860.00',
            '$', '860.00', 860),
    Example('10 €', '10 €',
            '€', '10', 10),
    Example('€', '13',
            '€', '13', 13),
    Example('60,00 DKK', '60,00 DKK',
            'DKK', '60,00', 60),
    Example('63,90 EUR', '63,90 EUR',
            'EUR', '63,90', 63.9),
    Example('$', '$ 11.99',
            '$', '11.99', 11.99),
    Example('25,00 €', '25,00 €',
            '€', '25,00', 25),
    Example('Cena', '106,40 €',
            '€', '106,40', 106.4),
    Example('EA', '$426.20/EA',
            '$', '426.20', 426.2),
    Example('zł', '5.76',
            'zł', '5.76', 5.76),
    Example('$4.50', '$4.50',
            '$', '4.50', 4.5),
    Example('51090 M Фильтроэлемент OMEGA', '2 660 руб',
            'руб', '2 660', 2660),
    Example('Р', '5 000 Р',
            'Р', '5 000', 5000),
    Example('Add to Cart', '$4.60',
            '$', '4.60', 4.6),
    Example('£14.99', '£14.99',
            '£', '14.99', 14.99),
    Example('SKU:', '$150.00',
            '$', '150.00', 150),
    Example('€', '70,31',
            '€', '70,31', 70.31),
    Example('/', '$379',
            '$', '379', 379),
    Example('€', '139,95',
            '€', '139,95', 139.95),
    Example('Add to cart', '22,90 €',
            '€', '22,90', 22.9),
    Example('94,99 zł', '94,99 zł',
            'zł', '94,99', 94.99),
    Example('Price', '89.99',
            None, '89.99', 89.99),
    Example('USD', '$4.00',
            '$', '4.00', 4),
    Example('$', 'Type of Transfer *',
            '$', None, None),
    Example('Add to Order', '$892.00',
            '$', '892.00', 892),
    Example('грн', '60800',
            'грн', '60800', 60800),
    Example('zł', '1849.00',
            'zł', '1849.00', 1849),
    Example('OK', '0,00 EUR',
            'EUR', '0,00', 0),
    Example('руб.', '5 450 руб.',
            'руб.', '5 450', 5450),
    Example('Price: $15.95/mo', 'Price: $15.95/mo',
            '$', '15.95', 15.95),
    Example('MX924 IX', '850 руб.',
            'руб.', '850', 850),
    Example('6,000.00 руб. 5,500.00 руб.', '6,000.00 руб. 5,500.00 руб.',
            'руб.', '6,000.00', 6000),
    Example('5.590,00 € *', '9.990,00 €',
            '€', '9.990,00', 9990),
    Example('1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30', '$175.00',
            '$', '175.00', 175),
    Example('Price', '$99.99',
            '$', '99.99', 99.99),
    Example('€', '119,00',
            '€', '119,00', 119),
    Example('$', '$ 11.96',
            '$', '11.96', 11.96),
    Example('€', '9,94',
            '€', '9,94', 9.94),
    Example('$69.00', '$69.00',
            '$', '69.00', 69),
    Example('AQUAFINESSE', 'AQUAFINESSE',
            None, None, None),
    Example('£55.00', '£50.00',
            '£', '50.00', 50),
    Example('$39.99', '$39.99',
            '$', '39.99', 39.99),
    Example('65 000 руб. 75 000 руб.', '75 000 руб.',
            'руб.', '75 000', 75000),
    Example('DKK/stk', '17,95',
            'DKK', '17,95', 17.95),
    Example('841,00 zł', '841,00 zł – 995,00 zł',
            'zł', '841,00', 841),
    Example('$', '$ 18.95',
            '$', '18.95', 18.95),
    Example('9.0', '434',
            None, '434', 434),
    Example('£14.50', '£21.00',
            '£', '21.00', 21),
    Example('ab', '19,90 €',
            '€', '19,90', 19.9),
    Example('30 руб', '30 руб',
            'руб', '30', 30),
    Example('€', '684,25',
            '€', '684,25', 684.25),
    Example('€ 125.00', '€ 45.00',
            '€', '45.00', 45),
    Example("ht / L'unité", '17.90',
            None, '17.90', 17.9),
    Example('$0.00', '$0.00',
            '$', '0.00', 0),
    Example('49 Kč', '49 Kč',
            'Kč', '49', 49),
    Example('Cena 685,00 Kč', '1 995,00 Kč',
            'Kč', '1 995,00', 1995),
    Example('€ 2,99', '€ 2,99',
            '€', '2,99', 2.99),
    Example('Cкидка до 10% от цены купона', '475 руб.',
            'руб.', '475', 475),
    Example('€', '11,76',
            '€', '11,76', 11.76),
    Example('$99.99', '$99.99',
            '$', '99.99', 99.99),
    Example('1\xa0298,00 €', '1\xa0298,00 €',
            '€', '1 298,00', 1298.00),
    Example('$1\xa0298,00', '$1\xa0298,00',
            '$', '1 298,00', 1298.00),
    Example('1\xa0298,00', '1\xa0298,00',
            None, '1 298,00', 1298.00),
    Example(None, '.75 €',
            '€', '.75', 0.75),
    Example('$.75', '$.75',
            '$', '.75', 0.75),
    Example('$..75', '$..75',
            '$', '.75', 0.75),
    Example('$.75,333', '$.75,333',
            '$', '.75,333', 75333),
    Example('$.750.30', '$.750.30',
            '$', '750.30', 750.30),
    Example('i', 'i',
            None, None, None),
]


PRICE_PARSING_EXAMPLES_2 = [
    Example('7,99 €', '7,99 €',
            '€', '7,99', 7.99),
    Example('2 Piece - $6.75 2 Piece - $6.75', '$6.75',
            '$', '6.75', 6.75),
    Example('£7', '£24.99',
            '£', '24.99', 24.99),
    Example('23,00 €', '23,00 €',
            '€', '23,00', 23.00),
    Example('€', 'bežná cena 12,96 €',
            '€', '12,96', 12.96),
    Example('$', '3,20 €',
            '€', '3,20', 3.2),
    Example('€ 2.109,00', '€ 2.109,00',
            '€', '2.109,00', 2109),
    Example('руб.', '32690',
            'руб.', '32690', 32690),
    Example(': 3 250 000 €', ': 3 250 000 €',
            '€', '3 250 000', 3250000),
    Example('$20.00', '$20.00',
            '$', '20.00', 20),
    Example('Цена:', '950 грн.',
            'грн.', '950', 950),
    Example('TTC', '149,00',
            None, '149,00', 149.00),
    Example('A partire da 9,08 €', '166,81 € iva inclusa',
            '€', '166,81', 166.81),
    # Example('Ref:', '4229',
    #         None, None, ),
    Example('EUR', '679.00',
            'EUR', '679.00', 679.00),
    Example('10,50 € 10,50 €', '10,50 €',
            '€', '10,50', 10.50),
    Example('2,20 € *', '2,20 € *',
            '€', '2,20', 2.2),
    Example('€', '€ 10.00 € 8.00 – € 10.00',
            '€', '10.00', 10),
    Example('Bestellen', '€16,95',
            '€', '16,95', 16.95),
    Example('$', '$ 95.00',
            '$', '95.00', 95.00),
    Example('Р', '1 400 Р',
            'Р', '1 400', 1400),
    Example('£', '£ 15.00',
            '£', '15.00', 15),
    Example('Price: $119.00', '$119.00',
            '$', '119.00', 119),
    Example('50,- Kč', '50,- Kč',
            'Kč', '50', 50),
    Example('What is this?', '$25.95',
            '$', '25.95', 25.95),
    Example('$29.99 Excluding Sales Tax in Illinois', '$29.99',
            '$', '29.99', 29.99),
    Example('ID:', '125,00 zł',
            'zł', '125,00', 125),
    Example('Offer Price: Rs 1,306.14', 'Rs 3,600.00',
            'Rs', '3,600.00', 3600),
    Example('29 RON', '29 RON',
            'RON', '29', 29),
    Example('Р', '2 690 Р',
            'Р', '2 690', 2690),
    Example('$134.96', '$134.96',
            '$', '134.96', 134.96),
    Example('362,68 € χωρίς Φ.Π.Α', '449,72 €',
            '€', '449,72', 449.72),
    Example('MSRP: $588.00', '499.80',
            '$', '499.80', 499.80),
    Example('15.00 €', '15.00 €',
            '€', '15.00', 15),
    Example('Цена: от 2 750 руб.', '2 750 руб.',
            'руб.', '2 750', 2750),
    Example('P.V.P. 4,56 €', 'P.V.P. 4,56 €',
            '€', '4,56', 4.56),
    Example('SPECIAL Add to Cart for Price', '$40.00',
            '$', '40.00', 40),
    Example('2,90 €', '2,90',
            '€', '2,90', 2.9),
    Example('Цена: 19 000 руб.', 'Цена: 19 000 руб.',
            'руб.', '19 000', 19000),
    Example('COMMANDER', '184,20 €HT',
            '€', '184,20', 184.2),
    Example('Цена: 1 849 руб.', '1 849 руб.',
            'руб.', '1 849', 1849),
    Example('€', '12,00',
            '€', '12,00', 12),
    Example('Discounted Price £ 205.99', '205.99',
            '£', '205.99', 205.99),
    Example('SKU', 'No longer available',
            None, None, None),
    Example('€14.90', '€69.90',
            '€', '69.90', 69.9),
    Example('руб.', '876 руб.',
            'руб.', '876', 876),
    Example('$79.95', '$79.95',
            '$', '79.95', 79.95),
    Example('3,990 Ft 2,990 Ft', '3,990 Ft',
            'Ft', '3,990', 3990),
    Example('Cena 298,00 Kč', '288,00 Kč',
            'Kč', '288,00', 288),
    Example('Pris NOK 899,00', '899,00',
            'NOK', '899,00', 899),
    Example('50,00 €', '50,00 €',
            '€', '50,00', 50),
    Example('59,99 €', '54,99 €',
            '€', '54,99', 54.99),
    Example('EAN', '139,00 € *',
            '€', '139,00', 139),
    Example('Now', '238.00',
            None, '238.00', 238),
    Example('(You save $15.01)', '$119.99',
            '$', '119.99', 119.99),
    Example('Item #:', '$ 99.95',
            '$', '99.95', 99.95),
    Example('UK:', '£14.99',
            '£', '14.99', 14.99),
    Example('р.', '150',
            'р.', '150', 150),
    Example('12x de R$ 44,00 sem juros', 'R$ 528,00',
            'R$', '528,00', 528),
    Example('€', '14,90',
            '€', '14,90', 14.9),
    Example('5 290 Kč s DPH', '5 290 Kč',
            'Kč', '5 290', 5290),
    Example('(Out of Stock)', '39.95',
            None, '39.95', 39.95),
    Example('₪370', '₪370',
            '₪', '370', 370),
    Example('HUF', '39000',
            'HUF', '39000', 39000),
    Example('Kč', '1 995 Kč',
            'Kč', '1 995', 1995),
    Example('C$5.95', 'C$5.95',
            'C$', '5.95', 5.95),
    Example('руб.', '22800.00',
            'руб.', '22800.00', 22800),
    Example('Cena', '359 Kč',
            'Kč', '359', 359),
    Example('Nuestro precio', '189,00 €',
            '€', '189,00', 189),
    Example('PLN', '179.62',
            'PLN', '179.62', 179.62),
    Example('€', '145,79',
            '€', '145,79', 145.79),
    Example('$', '$ 695.00',
            '$', '695.00', 695),
    Example('Р', '2 999 Р',
            'Р', '2 999', 2999),
    Example('6,50 € *', '6,50',
            '€', '6,50', 6.50),
    Example('Price: $61.00', '$61.00',
            '$', '61.00', 61.00),
    Example('Pris NOK 1 999,00', '5 799,00',
            'NOK', '5 799,00', 5799),
    Example('7,31 €', '7,31 €',
            '€', '7,31', 7.31),
    Example('53,20 €', '53,20 €',
            '€', '53,20', 53.20),
    Example('Cena', '668 Kč',
            'Kč', '668', 668),
    Example('€ 28,25', '€ 27,95',
            '€', '27,95', 27.95),
    Example('From', 'From $109.90',
            '$', '109.90', 109.9),
    Example('Unit Price: $4.75', '$4.75',
            '$', '4.75', 4.75),
    Example('Więcej', '14,90 zł',
            'zł', '14,90', 14.90),
    Example('$23.00', '$23.00',
            '$', '23.00', 23),
    Example('€ 0.00', '€ 33.00',
            '€', '33.00', 33),
    Example('Zľava 36 €', '78 €',
            '€', '78', 78),
    Example('£', '5.00',
            '£', '5.00', 5),
    Example('€', '29,90',
            '€', '29,90', 29.9),
    Example('Р', '860 Р',
            'Р', '860', 860),
    Example('60,55 €', '60,55 €',
            '€', '60,55', 60.55),
    Example('€', '129.00',
            '€', '129.00', 129),
    Example('£13.50', '£13.50',
            '£', '13.50', 13.50),
    Example('SKU:', '$39.99',
            '$', '39.99', 39.99),
    Example('Disponibilidade: Pronta entrega R$60,00 até 2x de R$30,00', 'R$60,00',
            'R$', '60,00', 60),
    Example('67,99 €', '64,59 €',
            '€', '64,59', 64.59),
    Example('€ 9,90', '€ 13,50',
            '€', '13,50', 13.50),
    Example('Р', '30 Р',
            'Р', '30', 30),
    Example('€', '€ 139.00',
            '€', '139.00', 139),
    Example('There are 163 products.', 'From 26 to 50 €',
            '€', '26', 26),
    Example('Pris NOK 1 999,00', '139,00',
            'NOK', '139,00', 139),
    Example('/sqft', '1.52',
            None, '1.52', 1.52),
    Example('$4.95', '$4.95',
            '$', '4.95', 4.95),
    Example('$38', '$12',
            '$', '12', 12),
    Example('Cena 4.10 €', '4.10',
            '€', '4.10', 4.1),
    Example('руб.', '590',
            'руб.', '590', 590),
    Example('€', '99,90',
            '€', '99,90', 99.9),
    Example('9,50 EUR', '9,50 EUR',
            'EUR', '9,50', 9.50),
    Example('$74.99', '$74.99',
            '$', '74.99', 74.99),
    Example('$', '$ 3.22',
            '$', '3.22', 3.22),
    Example('£', '3,301.00 £',
            '£', '3,301.00', 3301),
    Example('41,78 EUR', '41,78 EUR',
            'EUR', '41,78', 41.78),
    Example('44,50 EUR', '44,50 EUR',
            'EUR', '44,50', 44.50),
    Example('£4.00', '£4.00',
            '£', '4.00', 4),
    Example('Р', '7 390 Р',
            'Р', '7 390', 7390),
    Example('eMail', '34.99€',
            '€', '34.99', 34.99),
    Example('€', '12,99',
            '€', '12,99', 12.99),
    Example('RRP: £ 180.00', '149.00',
            '£', '149.00', 149),
    Example('9.00 руб.', '27.00 руб.',
            'руб.', '27.00', 27),
    Example('Price: $19.95', '$7.95',
            '$', '7.95', 7.95),
    Example('$16,500', '$16,500',
            '$', '16,500', 16500),
    Example('•', '$59.99',
            '$', '59.99', 59.99),
    Example('USD', '349.95',
            'USD', '349.95', 349.95),
    Example("35€ 99 dont 0,02 € d'éco-part", '35€ 99',
            '€', '35€99', 35.99),
    Example('5,72 €', '5,72 €',
            '€', '5,72', 5.72),
    Example('199,00 грн.', '199,00 грн.',
            'грн.', '199,00', 199),
    Example('£4.07', '£4.07',
            '£', '4.07', 4.07),
    Example('$15.00', '$15.00',
            '$', '15.00', 15),
    Example('€', '20.00',
            '€', '20.00', 20),
    Example('€', '14.50',
            '€', '14.50', 14.5),
    Example('DKK', '59,00',
            'DKK', '59,00', 59),
    Example('9 472 Ft', '9 472 Ft',
            'Ft', '9 472', 9472),
    Example('Cena', '95 Kč',
            'Kč', '95', 95),
    Example('tax excl.', '€ 0,00',
            '€', '0,00', 0),
    Example('TL', '69,50',
            'TL', '69,50', 69.5),
    Example('Price', 'Rp 169.000',
            'Rp', '169.000', 169000),
    Example('En stock', '4,50 €',
            '€', '4,50', 4.50),
    Example('$183.00', '$145.00',
            '$', '145.00', 145),
    Example('£', '11.70',
            '£', '11.70', 11.7),
    Example('25,13 €', '25,13 €',
            '€', '25,13', 25.13),
    Example('58,00 zł', '58,00 zł',
            'zł', '58,00', 58),
    Example('€', '15,55',
            '€', '15,55', 15.55),
    Example('Prezzo SVB: 19,95 €', 'Prezzo SVB: 19,95 €',
            '€', '19,95', 19.95),
    Example('In stock', '$185.00',
            '$', '185.00', 185),
    Example('18 €', '18 €',
            '€', '18', 18),
    Example('0€', '0 €',
            '€', '0', 0),
    Example('*', '9,41 €',
            '€', '9,41', 9.41),
    Example('грн.', '1 075',
            'грн.', '1 075', 1075),
    Example('More', '$22.99',
            '$', '22.99', 22.99),
    Example('Р', '9 282 Р',
            'Р', '9 282', 9282),
    Example('RM5', 'RM50',
            'RM', '50', 50),
    Example('7,60 €', '7,60 €',
            '€', '7,60', 7.6),
    Example('$', '0.00',
            '$', '0.00', 0),
    Example('. AMOSTRA .', 'R$ 6,00',
            'R$', '6,00', 6),
    Example('5 833,00 € -5% 6 140,00 € Tasse incl. IVA 4%', '5 833,00 €',
            '€', '5 833,00', 5833),
    Example('€ 26,95', '€ 26,95',
            '€', '26,95', 26.95),
    Example('35,00 €', '35,00 €',
            '€', '35,00', 35),
    Example('Р', '2 500 Р',
            'Р', '2 500', 2500),
    Example('499,00 €', '499,00 €',
            '€', '499,00', 499),
    Example('руб.', '1 120 900',
            'руб.', '1 120 900', 1120900),
    Example('125,00 125,00 €', '125,00',
            '€', '125,00', 125),
    Example('₽', '7362',
            '₽', '7362', 7362),
    Example('SKU:', '€12.99',
            '€', '12.99', 12.99),
    Example('р.', '600',
            'р.', '600', 600),
    Example('79,50 €', '29,90€',
            '€', '29,90', 29.9),
    Example('$150.00', '$150.00',
            '$', '150.00', 150),
    Example('Р', '1 987 Р',
            'Р', '1 987', 1987),
    Example('€69.00', '€69.00',
            '€', '69.00', 69),
    Example('USD', '280',
            'USD', '280', 280),
    Example('Rabatt', '6 450,00',
            None, '6 450,00', 6450),
    Example('SKU:', '$178.00',
            '$', '178.00', 178),
    Example('$', '$ 300.00',
            '$', '300.00', 300),
    Example('0€', '19,1€',
            '€', '19,1', 19.1),
    Example('R$49,90', 'R$49,90',
            'R$', '49,90', 49.9),
    Example('En stock', '790,00 € HT',
            '€', '790,00', 790),
    Example('Cena', 'Cena 14,79 €',
            '€', '14,79', 14.79),
    Example('For Sale', '$395,000',
            '$', '395,000', 395000),
    Example('€', '10.14',
            '€', '10.14', 10.14),
    Example('₫', '5.00 trên 5',
            '₫', '5.00', 5),
    Example('$ 599.00', '549.00',
            '$', '549.00', 549),
    Example('$19', '$19',
            '$', '19', 19),
    Example('$210', '$210',
            '$', '210', 210),
    Example('$', '$ 79.95',
            '$', '79.95', 79.95),
    Example('руб.', '279 000',
            'руб.', '279 000', 279000),
    Example('2 390 €', '2 390 €',
            '€', '2 390', 2390),
    Example('$', '$ 281.40 $ 265.70',
            '$', '281.40', 281.4),
    Example('3200 руб. 1500 руб.', '3200 руб. 1500 руб.',
            'руб.', '3200', 3200),
    Example('$1,299.00', '$999.00',
            '$', '999.00', 999),
    Example('15,00 Kč', '15,00 Kč',
            'Kč', '15,00', 15),
    Example('€', '€ 20.70',
            '€', '20.70', 20.7),
    Example('out of stock', '30.00',
            None, '30.00', 30),
    Example('220,00 €', '398,00 €',
            '€', '398,00', 398),
    Example('£94.50 GBP', '£94.50 GBP',
            '£', '94.50', 94.5),
    Example('5 427 Kč', '5 427 Kč',
            'Kč', '5 427', 5427),
    Example('$ 24.99', '$ 24.99',
            '$', '24.99', 24.99),
    Example('£ 2.00', '2.00',
            '£', '2.00', 2),
    Example('>', '10,00 €',
            '€', '10,00', 10),
    Example('32,50 EUR', '32,50 EUR',
            'EUR', '32,50', 32.5),
    Example('In stock', 'Rs1,599.00',
            'Rs', '1,599.00', 1599),
    Example('$', '14.95',
            '$', '14.95', 14.95),
    Example('RRP: £ 49.99', '42.90',
            '£', '42.90', 42.9),
    Example('Rs206.00 Rs164.00 SAVE 20%', 'Rs164.00',
            'Rs', '164.00', 164),
    Example('Price:', '$6,200.00',
            '$', '6,200.00', 6200),
    Example('PLN', '69.99',
            'PLN', '69.99', 69.99),
    Example('€', '48,39',
            '€', '48,39', 48.39),
    Example('1.350.000', '320.000',
            None, '320.000', 320000),
    Example('1.650,00 € *', '2.130,00 €',
            '€', '2.130,00', 2130),
    Example('£537.00', '£179.00',
            '£', '179.00', 179),
    Example('SKU:', '$39.99',
            '$', '39.99', 39.99),
    Example('SGD$4.90', 'SGD$4.90',
            'SGD', '4.90', 4.9),
    Example('SGD4.90 $', 'SGD4.90 $',
            'SGD', '4.90', 4.9),
    Example('$ SGD4.90', '$ SGD4.90',
            'SGD', '4.90', 4.9),
]


PRICE_PARSING_EXAMPLES_NO_PRICE = [
    Example('EUR', None,
            'EUR', None, None),
    Example('Kč', None,
            'Kč', None, None),
    Example('50', None,
            None, None, None),
    Example('>', None,
            None, None, None),
    Example('REI', None,
            None, None, None),
    Example('rate', None,
            None, None, None),
]


PRICE_PARSING_EXAMPLES_NO_CURRENCY = [
    Example(None, '67',
            None, '67', 67),
    Example(None, 'naša cena 67,30 €',
            '€', '67,30', 67.30),
    Example(None, 'EUR29.66 (inc VAT 20% - UK & EEC) EUR24.71 (exc VAT 20% - UK & EEC)',
            'EUR', '29.66', 29.66),
    Example(None, '17,00 €',
            '€', '17,00', 17),
    Example(None, '0,29 € + iva',
            '€', '0,29', 0.29),
    Example(None, '39,00 €',
            '€', '39,00', 39),
    Example(None, '24,00 Kč',
            'Kč', '24,00', 24),
    Example(None, 'běžná cena 7 940 Kč',
            'Kč', '7 940', 7940),
    Example(None, '$46.00',
            '$', '46.00', 46.00),
    Example(None, '$9.99 & Under',
            '$', '9.99', 9.99),
    Example(None, 'běžná cena 459,00 Kč',
            'Kč', '459,00', 459),
    Example(None, '$60.00',
            '$', '60.00', 60),
    Example(None, '0,82 €',
            '€', '0,82', 0.82),
    Example(None, '599 Kč',
            'Kč', '599', 599),
    Example(None, '€10.90',
            '€', '10.90', 10.90),
    Example(None, '299,00 EUR',
            'EUR', '299,00', 299),
    Example(None, 'naša cena 21,95 €',
            '€', '21,95', 21.95),
    Example(None, '343,05 €',
            '€', '343,05', 343.05),
    Example(None, '1 139,00 €',
            '€', '1 139,00', 1139),
    Example(None, '157,000 تومان',
            'تومان', '157,000', 157000),
    Example(None, '35.00',
            None, '35.00', 35),
    Example(None, '8.000.000 ₫',
            '₫', '8.000.000', 8000000),
    Example(None, '6790 Dinara',
            'Dinara', '6790', 6790),
    Example(None, '3.99',
            None, '3.99', 3.99),
    Example(None, '£ 165.00',
            '£', '165.00', 165.00),
    Example(None, '$844,900',
            '$', '844,900', 844900),
    Example(None, '140,00 pln',
            'pln', '140,00', 140),
    Example(None, '2,99 €',
            '€', '2,99', 2.99),
    Example(None, '145,00 lei',
            'lei', '145,00', 145),
    Example(None, '3,90 €',
            '€', '3,90', 3.90),
    Example(None, '$149.99',
            '$', '149.99', 149.99),
    Example(None, '4,30 €',
            '€', '4,30', 4.30),
    Example(None, '$36.95',
            '$', '36.95', 36.95),
    Example(None, '2 499,00 zł',
            'zł', '2 499,00', 2499),
    Example(None, '800 руб.',
            'руб.', '800', 800),
    Example(None, '89.00',
            None, '89.00', 89),
    Example(None, '3 100р.',
            'р.', '3 100', 3100),
    Example(None, '0,85 €',
            '€', '0,85', 0.85),
    Example(None, '35,95 €',
            '€', '35,95', 35.95),
    Example(None, '$0.42',
            '$', '0.42', 0.42),
    Example(None, '80,000 تومان',
            'تومان', '80,000', 80000),
    Example(None, '550,00 € *',
            '€', '550,00', 550),
    Example(None, '25,00 zł',
            'zł', '25,00', 25),
    Example(None, '17,45 EUR',
            'EUR', '17,45', 17.45),
    Example(None, '49,00 €',
            '€', '49,00', 49),
    Example(None, '169.00',
            None, '169.00', 169),
    Example(None, '8,99 €',
            '€', '8,99', 8.99),
    Example(None, '1 099 Kč',
            'Kč', '1 099', 1099),
    Example(None, '17.99',
            None, '17.99', 17.99),
    Example(None, '$274.95',
            '$', '274.95', 274.95),
    Example(None, '70,20 €',
            '€', '70,20', 70.20),
    Example(None, '289,00 zł',
            'zł', '289,00', 289),
    Example(None, '18,00 €',
            '€', '18,00', 18),
    Example(None, '12,00 €',
            '€', '12,00', 12),
    Example(None, '$19.97',
            '$', '19.97', 19.97),
    # Example(None, '(save $2.95)',
    #         '$', None, None),
    Example(None, '749,00 euro',
            'euro', '749,00', 749),
    Example(None, '$48.25',
            '$', '48.25', 48.25),
    Example(None, '5.00',
            None, '5.00', 5.00),
    Example(None, '18,00 € *',
            '€', '18,00', 18),
    Example(None, '$3.00',
            '$', '3.00', 3.00),
    Example(None, '1,85 EUR',
            'EUR', '1,85', 1.85),
    Example(None, '4.25',
            None, '4.25', 4.25),
    Example(None, '£1.20',
            '£', '1.20', 1.20),
    Example(None, '$196.50',
            '$', '196.50', 196.50),
    Example(None, 'Price: $129.00',
            '$', '129.00', 129.00),
    Example(None, '179,00 €',
            '€', '179,00', 179.00),
    Example(None, '$80.00',
            '$', '80.00', 80.00),
    Example(None, '14.50',
            None, '14.50', 14.50),
    Example(None, 'From $ 24.95',
            '$', '24.95', 24.95),
    Example(None, '$5.11',
            '$', '5.11', 5.11),
    Example(None, 'EUR 6,99',
            'EUR', '6,99', 6.99),
    Example(None, '40% OFF',
            None, None, None),
    Example(None, '29.99',
            None, '29.99', 29.99),
    Example(None, '14.00€',
            '€', '14.00', 14.00),
    Example(None, '22.00',
            None, '22.00', 22.00),
    Example(None, '$1000.00',
            '$', '1000.00', 1000.00),
    Example(None, '$12.95',
            '$', '12.95', 12.95),
    Example(None, 'běžná cena 987,20 Kč',
            'Kč', '987,20', 987.20),
    Example(None, '104,64 zł',
            'zł', '104,64', 104.64),
    Example(None, '163,80 €',
            '€', '163,80', 163.80),
    Example(None, '$89.00',
            '$', '89.00', 89.00),
    Example(None, '1 600 руб.',
            'руб.', '1 600', 1600),
    Example(None, '20,95 € *',
            '€', '20,95', 20.95),
    Example(None, '9,50 €',
            '€', '9,50', 9.50),
    Example(None, '170,00 €',
            '€', '170,00', 170),
    Example(None, '170,00€',
            '€', '170,00', 170),
    Example(None, '6.00',
            None, '6.00', 6.00),
    Example(None, '$24.00',
            '$', '24.00', 24.00),
    Example(None, '9.95',
            None, '9.95', 9.95),
    Example(None, '34.12 (40.94 inc VAT)',
            None, '34.12', 34.12),
    Example(None, 'Rp 350.000',
            'Rp', '350.000', 350000),
    Example(None, '$55.00',
            '$', '55.00', 55.00),
    Example(None, '$595.00',
            '$', '595.00', 595.00),
    Example(None, '7,00 €',
            '€', '7,00', 7),
    Example(None, '119.95',
            None, '119.95', 119.95),
    Example(None, '1.95',
            None, '1.95', 1.95),
    Example(None, '390,00 €',
            '€', '390,00', 390),
    Example(None, '3.24',
            None, '3.24', 3.24),
    Example(None, '12 590 Kč',
            'Kč', '12 590', 12590),
    Example(None, '330 Kč',
            'Kč', '330', 330),
    Example(None, '8 500 руб.',
            'руб.', '8 500', 8500),
    Example(None, '589,00 €',
            '€', '589,00', 589),
    Example(None, '1,099.99',
            None, '1,099.99', 1099.99),
    Example(None, '14 196 Р',
            'Р', '14 196', 14196),
    Example(None, '19.00',
            None, '19.00', 19.00),
    Example(None, '870 Kč',
            'Kč', '870', 870),
    Example(None, '59,00 €',
            '€', '59,00', 59),
    Example(None, 'Pris från 172 kr',
            'kr', '172', 172),
    Example(None, '1 573 Kč',
            'Kč', '1 573', 1573),
    Example(None, '$2.99',
            '$', '2.99', 2.99),
    Example(None, '13,90 €',
            '€', '13,90', 13.90),
    Example(None, '29.95',
            None, '29.95', 29.95),
    Example(None, '/',
            None, None, None),
    Example(None, '16,90 €',
            '€', '16,90', 16.90),
    Example(None, '149.95',
            None, '149.95', 149.95),
    Example(None, '8.90',
            None, '8.90', 8.90),
    Example(None, '419',
            None, '419', 419),
    Example(None, '$50.00',
            '$', '50.00', 50.00),
    Example(None, '3 291,00 €',
            '€', '3 291,00', 3291),
    Example(None, '13,00 €',
            '€', '13,00', 13),
    Example(None, 'DKK 449,00',
            'DKK', '449,00', 449),
    Example(None, '$20.00',
            '$', '20.00', 20.00),
    Example(None, '$154',
            '$', '154', 154),
    Example(None, '22.48',
            None, '22.48', 22.48),
    Example(None, '20,00 EUR',
            'EUR', '20,00', 20),
    Example(None, '73,460 €',
            '€', '73,460', 73460),
    Example(None, '850 руб',
            'руб', '850', 850),
    Example(None, '$14.99',
            '$', '14.99', 14.99),
    Example(None, '$79.95',
            '$', '79.95', 79.95),
    Example(None, '40,00 €',
            '€', '40,00', 40),
    Example(None, '149,98 €',
            '€', '149,98', 149.98),
    Example(None, '1 150 грн.',
            'грн.', '1 150', 1150),
    Example(None, '399.00',
            None, '399.00', 399.00),
    Example(None, '33,90 €',
            '€', '33,90', 33.90),
    Example(None, '79,50 €',
            '€', '79,50', 79.50),
    Example(None, '40 130',
            None, '40 130', 40130),
    Example(None, '$69.99',
            '$', '69.99', 69.99),
    Example(None, '1 090 Kč',
            'Kč', '1 090', 1090),
    Example(None, '395 Kč',
            'Kč', '395', 395),
    Example(None, '53,95 €',
            '€', '53,95', 53.95),
    Example(None, '£0.99',
            '£', '0.99', 0.99),
    Example(None, '5,60 € *',
            '€', '5,60', 5.60),
    Example(None, '29,50 zł',
            'zł', '29,50', 29.50),
    Example(None, '2 990',
            None, '2 990', 2990),
    Example(None, '0,00',
            None, '0,00', 0),
    Example(None, '$24.99 with card',
            '$', '24.99', 24.99),
    Example(None, '18,00€',
            '€', '18,00', 18),
    Example(None, '€600,00',
            '€', '600,00', 600),
    Example(None, '£25.00 (tax incl.)',
            '£', '25.00', 25),
    Example(None, '8,55 €',
            '€', '8,55', 8.55),
    Example(None, '1,422.50',
            None, '1,422.50', 1422.50),
    Example(None, '244,00 €',
            '€', '244,00', 244.00),
    Example(None, '12,90 €',
            '€', '12,90', 12.90),
    Example(None, '12 900,00 руб',
            'руб', '12 900,00', 12900),
    Example(None, '1.727 Ft',
            'Ft', '1.727', 1727),
    Example(None, '79,00 €',
            '€', '79,00', 79),
    Example(None, 'NZD $100.70',
            'NZD', '100.70', 100.70),
    Example(None, '479.00',
            None, '479.00', 479.00),
    Example(None, '$ 69.00',
            '$', '69.00', 69.00),
    Example(None, '135,00 €',
            '€', '135,00', 135.00),
    Example(None, '25.00',
            None, '25.00', 25.0),
    Example(None, '94,90 €',
            '€', '94,90', 94.90),
    Example(None, '149.99',
            None, '149.99', 149.99),
    Example(None, '44,00 €',
            '€', '44,00', 44.00),
    Example(None, '$24.99',
            '$', '24.99', 24.99),
    Example(None, '22,00 EUR',
            'EUR', '22,00', 22.00),
    Example(None, '89,90 €',
            '€', '89,90', 89.90),
    Example(None, '$24.95',
            '$', '24.95', 24.95),
    Example(None, '£ 1.99',
            '£', '1.99', 1.99),
    Example(None, '1 099,00 zł',
            'zł', '1 099,00', 1099),
    Example(None, 'běžná cena 28 270,00 Kč',
            'Kč', '28 270,00', 28270),
    Example(None, 'da € 72.00',
            '€', '72.00', 72.00),
    Example(None, '$15.95',
            '$', '15.95', 15.95),
    Example(None, 'تومان56,000',
            'تومان', '56,000', 56000),
    Example(None, '$1,695.00',
            '$', '1,695.00', 1695.00),
    Example(None, '£595.00',
            '£', '595.00', 595.00),
    Example(None, '$11.95',
            '$', '11.95', 11.95),
    Example(None, '290,00 Kč',
            'Kč', '290,00', 290),
    Example(None, '199.90 Fr.',
            'Fr.', '199.90', 199.90),
    Example(None, '197 PLN',
            'PLN', '197', 197),
    Example(None, '9.99',
            None, '9.99', 9.99),
    Example(None, '$56.00',
            '$', '56.00', 56.00),
    Example(None, '4 980 Kč',
            'Kč', '4 980', 4980),
    Example(None, '124,00 €',
            '€', '124,00', 124),
    Example(None, '$104.99',
            '$', '104.99', 104.99),
    Example(None, '39,00 €',
            '€', '39,00', 39),
    Example(None, '1 029,00 €',
            '€', '1 029,00', 1029),
    Example(None, 'Běžná cena 299,00 Kč',
            'Kč', '299,00', 299),
    Example(None, '745,00 €',
            '€', '745,00', 745),
    Example(None, '$89.00',
            '$', '89.00', 89),
    Example(None, '$29.95',
            '$', '29.95', 29.95),
    Example(None, '2.00',
            None, '2.00', 2.00),
    Example(None, '249.99',
            None, '249.99', 249.99),
    Example(None, '24.99',
            None, '24.99', 24.99),
    Example(None, '1 499 Kč',
            'Kč', '1 499', 1499),
    Example(None, '199,95 €',
            '€', '199,95', 199.95),
    Example(None, '6,00 €',
            '€', '6,00', 6),
    Example(None, '$28.49',
            '$', '28.49', 28.49),
    Example(None, '200.000 đ',
            'đ', '200.000', 200000),
    Example(None, '9,24 €',
            '€', '9,24', 9.24),
    Example(None, '48,00 €',
            '€', '48,00', 48.00),
    Example(None, 'Cena : 890 Kč',
            'Kč', '890', 890),
    Example(None, '790.00',
            None, '790.00', 790.00),
    Example(None, '17 260 руб.',
            'руб.', '17 260', 17260),
    Example(None, '227,000 تومان',
            'تومان', '227,000', 227000),
    Example(None, '295,88 €',
            '€', '295,88', 295.88),
    Example(None, '£1399',
            '£', '1399', 1399),
    Example(None, '11,33 Br',
            'Br', '11,33', 11.33),
    Example(None, '325.95',
            None, '325.95', 325.95),
    Example(None, '$19.50',
            '$', '19.50', 19.50),
    Example(None, '19,00 €',
            '€', '19,00', 19),
    Example(None, '2 999,00 €',
            '€', '2 999,00', 2999),
    Example(None, '49.95',
            None, '49.95', 49.95),
    Example(None, '99 LEI',
            'LEI', '99', 99),
    Example(None, '249 Kč',
            'Kč', '249', 249),
    Example(None, '3.79',
            None, '3.79', 3.79),
    Example(None, 'běžná cena 890 Kč',
            'Kč', '890', 890),
    Example(None, '$809,000',
            '$', '809,000', 809000),
    Example(None, '450 000 ₫',
            '₫', '450 000', 450000),
    Example(None, '30,00 €',
            '€', '30,00', 30.00),
    Example(None, '14.95',
            None, '14.95', 14.95),
    Example(None, '12.50',
            None, '12.50', 12.50),
    Example(None, '129,00 € (-15%)',
            '€', '129,00', 129.00),
    Example(None, '12,90 €',
            '€', '12,90', 12.90),
    Example(None, 'A partir de 11,70 €',
            '€', '11,70', 11.70),
    Example(None, '15.49',
            None, '15.49', 15.49),
    Example(None, '12.34 €',
            '€', '12.34', 12.34),
    Example(None, '€799,00',
            '€', '799,00', 799),
    Example(None, '230 лв.',
            'лв.', '230', 230),
    Example(None, '14.55 €',
            '€', '14.55', 14.55),
    Example(None, '133,86 LEI',
            'LEI', '133,86', 133.86),
    Example(None, '7 990,00 Kč',
            'Kč', '7 990,00', 7990),
    Example(None, '350.00',
            None, '350.00', 350.00),
    Example(None, 'Cena: 55,72 zł brutto',
            'zł', '55,72', 55.72),
    Example(None, 'O blenderach Omniblend',
            None, None, None),
    Example(None, '3,822.00',
            None, '3,822.00', 3822),
    Example(None, '0,15 €',
            '€', '0,15', 0.15),
    Example(None, '430,00 €',
            '€', '430,00', 430),
    Example(None, '$29.00',
            '$', '29.00', 29.00),
    Example(None, '39.99',
            None, '39.99', 39.99),
    Example(None, '$15.00',
            '$', '15.00', 15.00),
    Example(None, '21,00 Lei',
            'Lei', '21,00', 21.00),
    Example(None, 'naše cena 250,00 Kč',
            'Kč', '250,00', 250.00),
    Example(None, '$24.95',
            '$', '24.95', 24.95),
    Example(None, '162.18',
            None, '162.18', 162.18),
    Example(None, '39,60 EUR',
            'EUR', '39,60', 39.60),
    Example(None, '10,75 €',
            '€', '10,75', 10.75),
    Example(None, '219 руб.',
            'руб.', '219', 219),
    Example(None, '89,00 € *',
            '€', '89,00', 89.00),
    Example(None, '151,200 تومان',
            'تومان', '151,200', 151200),
    Example(None, '$159.99',
            '$', '159.99', 159.99),
    Example(None, '2.49',
            None, '2.49', 2.49),
    Example(None, '7.38',
            None, '7.38', 7.38),
    Example(None, '62,00 zł',
            'zł', '62,00', 62.00),
    Example(None, '$20.00',
            '$', '20.00', 20),
    Example(None, '$ 50.00',
            '$', '50.00', 50),
    Example(None, '34.99',
            None, '34.99', 34.99),
    Example(None, '318,00 €',
            '€', '318,00', 318),
    Example(None, '11.499,00 EUR',
            'EUR', '11.499,00', 11499),
    Example(None, '571.12坪',  # area, not currency
            None, '571.12', 571.12),
    Example(None, '€ 75.00',
            '€', '75.00', 75.00),
    Example(None, '11,90 € *',
            '€', '11,90', 11.90),
    Example(None, '€0.51',
            '€', '0.51', 0.51),
    Example(None, '6,50 €',
            '€', '6,50', 6.50),
    Example(None, '790 Kč',
            'Kč', '790', 790),
    Example(None, 'ab 2.99 €',
            '€', '2.99', 2.99),
    Example(None, '369',
            None, '369', 369),
    Example(None, '134.96',
            None, '134.96', 134.96),
    Example(None, '135 lei',
            'lei', '135', 135),
    Example(None, '2,99 € *',
            '€', '2,99', 2.99),
    Example(None, '$9.99',
            '$', '9.99', 9.99),
    Example(None, '2.950,00 €',
            '€', '2.950,00', 2950),
    Example(None, '19.99',
            None, '19.99', 19.99),
    Example(None, '49 lei',
            'lei', '49', 49),
    Example(None, '31,07 € (bez DPH)',
            '€', '31,07', 31.07),
    Example(None, '56.00',
            None, '56.00', 56.00),
    Example(None, '54.95',
            None, '54.95', 54.95),
    Example(None, '$ 80.00',
            '$', '80.00', 80.00),
    Example(None, '$39.00',
            '$', '39.00', 39.00),
    Example(None, 'Rp 221.000',
            'Rp', '221.000', 221000),
    Example(None, '35,90 EUR',
            'EUR', '35,90', 35.90),
    Example(None, '4 835,50 €',
            '€', '4 835,50', 4835.50),
    Example(None, '75,00€',
            '€', '75,00', 75),
    Example(None, '$21.95',
            '$', '21.95', 21.95),
    Example(None, '737,00',
            None, '737,00', 737),
    Example(None, '129,00 € **',
            '€', '129,00', 129),
    Example(None, '2 399 Kč',
            'Kč', '2 399', 2399),
    Example(None, '430 руб',
            'руб', '430', 430),
    Example(None, '69.95',
            None, '69.95', 69.95),
    Example(None, '$0.00',
            '$', '0.00', 0),
    Example(None, '49.56',
            None, '49.56', 49.56),
    Example(None, '0,00 EUR',
            'EUR', '0,00', 0),
    Example(None, '145,00 Kč',
            'Kč', '145,00', 145),
    Example(None, '99,00 lei',
            'lei', '99,00', 99),
    Example(None, '$750,000',
            '$', '750,000', 750000),
    Example(None, '$49.99',
            '$', '49.99', 49.99),
    Example(None, '29.00',
            None, '29.00', 29.00),
    Example(None, '$7.20',
            '$', '7.20', 7.20),
    Example(None, '69.00',
            None, '69.00', 69.00),
    Example(None, '4.47',
            None, '4.47', 4.47),
    Example(None, '39,90 € *',
            '€', '39,90', 39.90),
    Example(None, '469,00 €',
            '€', '469,00', 469),
    Example(None, '24.38',
            None, '24.38', 24.38),
    Example(None, '6,24',
            None, '6,24', 6.24),
    Example(None, '$89.00',
            '$', '89.00', 89.00),
    Example(None, '24,35 €',
            '€', '24,35', 24.35),
    Example(None, 'Pris från 805 kr',
            'kr', '805', 805),
    Example(None, '295 Kč',
            'Kč', '295', 295),
    Example(None, '175.00',
            None, '175.00', 175.00),
    Example(None, '7 990 kr',
            'kr', '7 990', 7990),
    Example(None, '14,00 €',
            '€', '14,00', 14),
    Example(None, '249 Kč',
            'Kč', '249', 249),
    Example(None, '£39.95',
            '£', '39.95', 39.95),
    Example(None, '10,75 TL',
            'TL', '10,75', 10.75),
    Example(None, '$25.00',
            '$', '25.00', 25.00),
    Example(None, '1 720,00 zł',
            'zł', '1 720,00', 1720),
    Example(None, 'běžná cena 749 Kč',
            'Kč', '749', 749),
    Example(None, '425,00 €',
            '€', '425,00', 425),
    Example(None, '59.00',
            None, '59.00', 59.00),
    Example(None, '1,120.00',
            None, '1,120.00', 1120),
    Example(None, 'a partire da 7,32 € *',
            '€', '7,32', 7.32),
    Example(None, '148.50 Inc GST',
            None, '148.50', 148.50),  # India?
    Example(None, '7.49',
            None, '7.49', 7.49),
    Example(None, '80.00',
            None, '80.00', 80.00),
    Example(None, '93 499 Kč',
            'Kč', '93 499', 93499),
    Example(None, '1.599,00 € *',
            '€', '1.599,00', 1599),
    Example(None, 'ab 3,63 EUR',
            'EUR', '3,63', 3.63),
    Example(None, '29,90 EUR',
            'EUR', '29,90', 29.90),
    Example(None, '$3.95',
            '$', '3.95', 3.95),
    Example(None, '3430 лв.',
            'лв.', '3430', 3430),
    Example(None, '724,00 €',
            '€', '724,00', 724),
    Example(None, '18,00 €',
            '€', '18,00', 18),
    Example(None, '6,75 €',
            '€', '6,75', 6.75),
    Example(None, '29,90 € *',
            '€', '29,90', 29.90),
    Example(None, '135.99',
            None, '135.99', 135.99),
    Example(None, '30,000 تومان',
            'تومان', '30,000', 30000),
    # Example(None, '191.6 KB',
    #         None, None, None),
    Example(None, '1 500 Kč',
            'Kč', '1 500', 1500),
    Example(None, '349,00 €',
            '€', '349,00', 349),
    Example(None, '$250.00',
            '$', '250.00', 250.00),
    Example(None, '44.95',
            None, '44.95', 44.95),
    Example(None, '$22.75',
            '$', '22.75', 22.75),
    Example(None, '250,00 €',
            '€', '250,00', 250),
    Example(None, '14.96 €',
            '€', '14.96', 14.96),
    Example(None, '$4,350.00',
            '$', '4,350.00', 4350),
    Example(None, '379 Kč',
            'Kč', '379', 379),
    Example(None, '19,50 EUR',
            'EUR', '19,50', 19.5),
    Example(None, '33,68 zł',
            'zł', '33,68', 33.68),
    Example(None, '6.70€',
            '€', '6.70', 6.70),
    Example(None, '$29.99',
            '$', '29.99', 29.99),
    Example(None, '6.50',
            None, '6.50', 6.50),
]


PRICE_PARSING_EXAMPLES_3 = [
    Example('R$', 'R$ 139,99 R$ 135,99',
            'R$', '139,99', 139.99),
    Example('£', '£ 34.99',
            '£', '34.99', 34.99),
    Example('Price', '$7.65',
            '$', '7.65', 7.65),
    Example('€', '75,00',
            '€', '75,00', 75),
    Example(None, '34,90 €',
            '€', '34,90', 34.90),
    Example(None, '629.95',
            None, '629.95', 629.95),
    Example('11000 руб.', '47700 руб.',
            'руб.', '47700', 47700),
    Example('$29.99 – $74.99', '$29.99',
            '$', '29.99', 29.99),
    Example(None, '174,00 €',
            '€', '174,00', 174),
    Example(None, '18,00 €',
            '€', '18,00', 18),
    Example('Price:', '$19.50',
            '$', '19.50', 19.50),
    Example('8 390 руб.', '8 390 руб.',
            'руб.', '8 390', 8390),
    Example(None, '55,00 €',
            '€', '55,00', 55),
    Example('€333.00', '€299.71',
            '€', '299.71', 299.71),
    Example(None, '384,00 €',
            '€', '384,00', 384),
    Example('From:', 'From: $14.97',
            '$', '14.97', 14.97),
    Example(None, '0,00 €',
            '€', '0,00', 0),
    Example(None, '€ 280,00',
            '€', '280,00', 280),
    Example(None, '11 450 Kč',
            'Kč', '11 450', 11450),
    Example('Price $118.15', '$118.15',
            '$', '118.15', 118.15),
    Example('€', '49.99',
            '€', '49.99', 49.99),
    Example('1,15 €', '1,15 €',
            '€', '1,15', 1.15),
    Example('17,99 €', '31,93 €',
            '€', '31,93', 31.93),
    Example(None, 'र24,401',
            'र', '24,401', 24401),
    Example(None, '$60.00',
            '$', '60.00', 60),
    Example('12,15 €', '12,15 €',
            '€', '12,15', 12.15),
    Example('£ 163.95', '163.95',
            '£', '163.95', 163.95),
    Example(None, '30,00 €',
            '€', '30,00', 30),
    Example('zł', '165,00 zł',
            'zł', '165,00', 165),
    Example('469.00zł Bez podatku: 381.30zł', '469.00zł',
            'zł', '469.00', 469),
    Example(None, '72.95',
            None, '72.95', 72.95),
    Example('Costo: $2,222.- Más IVA', 'MX$3,179.00',
            'MX$', '3,179.00', 3179),
    Example(None, 'naše cena 4 370 Kč',
            'Kč', '4 370', 4370),
    Example('€', '21,33 €',
            '€', '21,33', 21.33),
    Example(None, '49.95',
            None, '49.95', 49.95),
    Example(None, 'Před slevou 59 900 Kč',
            'Kč', '59 900', 59900),
    Example('ab', '6,78 € *',
            '€', '6,78', 6.78),
    Example(None, '442',
            None, '442', 442),
    Example('18.10 €', '16.00 €',
            '€', '16.00', 16.00),
    Example(None, '0.00',
            None, '0.00', 0.00),
    Example(None, '379,00 € *',
            '€', '379,00', 379.00),
    Example(None, '125.00',
            None, '125.00', 125.00),
    Example(None, '£ 30.84',
            '£', '30.84', 30.84),
    Example(None, '259,00 €',
            '€', '259,00', 259),
    Example('à partir de 1540 € / pers', '1540 €',
            '€', '1540', 1540),
    Example(None, '95 €',
            '€', '95', 95),
    Example(None, '53.79',
            None, '53.79', 53.79),
    Example('NT$', 'NT$ 1,160',
            'NT$', '1,160', 1160),
    Example('ACTIVE', '$69,900',
            '$', '69,900', 69900),
    Example(None, '$14.95',
            '$', '14.95', 14.95),
    Example('₹', '₹ 4649',
            '₹', '4649', 4649),
    Example('25 грн', '25 грн',
            'грн', '25', 25),
    Example('€', '16,40',
            '€', '16,40', 16.40),
    Example('PLN', '0,46',
            'PLN', '0,46', 0.46),
    Example('£', '£ 261.25',
            '£', '261.25', 261.25),
    Example(None, '$0.00',
            '$', '0.00', 0),
    Example(None, '24.95',
            None, '24.95', 24.95),
    Example('грн.', '27.00',
            'грн.', '27.00', 27.00),
    Example('New', '$189,900',
            '$', '189,900', 189900),
    Example('NA', '$269',
            '$', '269', 269),
    Example('$279', '$189',
            '$', '189', 189),
    Example(None, '160,00 zł',
            'zł', '160,00', 160),
    Example('2 069 рублей', '2 400',
            'руб', '2 400', 2400),
    Example('Sale Price: $4.59', '$4.59',
            '$', '4.59', 4.59),
    Example('Купить', '542 руб.',
            'руб.', '542', 542),
    Example(None, '$19.99',
            '$', '19.99', 19.99),
    Example('Price', '$6.45',
            '$', '6.45', 6.45),
    Example(None, '32.99',
            None, '32.99', 32.99),
    Example(None, '$86.44',
            '$', '86.44', 86.44),
    Example(None, '25.00€',
            '€', '25.00', 25.00),
    Example(None, '99,00 €',
            '€', '99,00', 99.00),
    Example(None, '103.90',
            None, '103.90', 103.90),
    Example('14,00 € *', '25,00 € *',
            '€', '25,00', 25.00),
    Example(None, '$6.49',
            '$', '6.49', 6.49),
    Example('€ 59,95', '€ 59,95',
            '€', '59,95', 59.95),
    Example(None, 'Běžná cena 75 990,00 Kč',
            'Kč', '75 990,00', 75990),
    Example('Price', 'Rp 1.550.000',
            'Rp', '1.550.000', 1550000),
    Example('грн.', '1 430',
            'грн.', '1 430', 1430),
    Example('руб. (шт)', '1 690,54 руб. (шт)',
            'руб.', '1 690,54', 1690.54),
    Example('69 TL 41.90 TL', '69 TL 41.90 TL',
            'TL', '69', 69),
    Example('ALIDAD', '960,00 €',
            '€', '960,00', 960),
    Example(None, '184,35 lei',
            'lei', '184,35', 184.35),
    Example(None, '1 505 Kč',
            'Kč', '1 505', 1505),
    Example(None, '23,00 € *',
            '€', '23,00', 23),
    Example(None, '25.97',
            None, '25.97', 25.97),
    Example(None, '58,19 €',
            '€', '58,19', 58.19),
    Example(None, '27.00 лв.',
            'лв.', '27.00', 27.00),
    Example('48,00 €', '3,85 €',
            '€', '3,85', 3.85),
    Example(None, '10,90 €',
            '€', '10,90', 10.90),
    Example('$ 879.0', '$ 879.0',
            '$', '879.0', 879.0),
    Example('EUR', '25.88',
            'EUR', '25.88', 25.88),
    Example(None, 'R$215,10',
            'R$', '215,10', 215.10),
    Example('£', '£ 12.50',
            '£', '12.50', 12.50),
    Example(None, '3 173,00 €',
            '€', '3 173,00', 3173),
    Example(None, '34,94 € *',
            '€', '34,94', 34.94),
    Example(None, 'Ops!',
            None, None, None),
    Example(None, '392. 00',
            None, '392. 00', 392),
    Example('€', '213,62',
            '€', '213,62', 213.62),
    Example('3,00 €', '3,00 €',
            '€', '3,00', 3),
    Example('£0.00', '£0.00',
            '£', '0.00', 0.00),
    Example('€', '10 990,00',
            '€', '10 990,00', 10990),
    Example(None, '€ 24,95',
            '€', '24,95', 24.95),
    Example(None, 'Not Available',
            None, None, None),
    Example(None, '$19.99',
            '$', '19.99', 19.99),
    Example('Р', '15 130 Р',
            'Р', '15 130', 15130),
    Example('$5.95', '$5.95',
            '$', '5.95', 5.95),
    Example(None, '199,99 €',
            '€', '199,99', 199.99),
    Example('Code', '£23.40',
            '£', '23.40', 23.40),
    Example('$29.99', '$29.99',
            '$', '29.99', 29.99),
    Example(None, '795',
            None, '795', 795),
    Example('Sorry, this item is currently out of stock but you can still order, we will send as soon a product arrives', '34.99',
            None, '34.99', 34.99),
    Example('Our Price: $149.95', 'Our Price: $149.95',
            '$', '149.95', 149.95),
    Example('$119.95', '$119.95',
            '$', '119.95', 119.95),
    Example(None, '339 грн',
            'грн', '339', 339),
    Example('$0.00', '$0.00',
            '$', '0.00', 0.00),
    Example('€', '79,00',
            '€', '79,00', 79.00),
    Example(None, '378.00',
            None, '378.00', 378.00),
    Example(None, 'Pure & IP BP Ph. Eur. USP ACS AR LR',
            None, None, None),
    Example(None, '$356.03',
            '$', '356.03', 356.03),
    Example('naše cena', 'běžná cena 890 Kč',
            'Kč', '890', 890),
    Example(None, '$49.99',
            '$', '49.99', 49.99),
    Example(None, '5 550 Kč',
            'Kč', '5 550', 5550),
    Example(None, '5 770 Kč',
            'Kč', '5 770', 5770),
    Example(None, 'Free!',
            None, '0', 0),
    Example('194 ₹', '199 ₹',
            '₹', '199', 199),
    Example('5€', '16,50 € *',
            '€', '16,50', 16.50),
    Example(None, '$42.95',
            '$', '42.95', 42.95),
    Example(None, '1.837, 32 €',
            '€', '1.837, 32', 1837.32),
    Example('$', '$ 791.00 $ 479.00',
            '$', '791.00', 791.00),
    Example(None, '$69.30',
            '$', '69.30', 69.30),
    Example(None, '$163,900',
            '$', '163,900', 163900),
    Example(None, '36.95',
            None, '36.95', 36.95),
    Example('Rp 235.000', 'Rp 235.000',
            'Rp', '235.000', 235000),
    Example('£', '11,13 €',
            '€', '11,13', 11.13),
    Example(None, '160,00 lei',
            'lei', '160,00', 160),
    Example('3 300 руб', '3 300 руб',
            'руб', '3 300', 3300),
    Example('Р', '4 690 Р',
            'Р', '4 690', 4690),
    Example('189,00 € *', '189,00 € *',
            '€', '189,00', 189),
    Example('€', None,
            '€', None, None),
    Example('$ 30.00', '$ 30.00',
            '$', '30.00', 30.00),
    Example('$', '$ 5.95',
            '$', '5.95', 5.95),
    Example('£62.90', '£74.00',
            '£', '74.00', 74.00),
    Example(None, '158,24 €',
            '€', '158,24', 158.24),
    Example(None, '550,00 лв',
            'лв', '550,00', 550),
    Example('7,25 € *', '7,25 € *',
            '€', '7,25', 7.25),
    Example(None, '94,000 تومان',
            'تومان', '94,000', 94000),
    Example(None, '$8.27',
            '$', '8.27', 8.27),
    Example('Đã có VAT', '12.500 ₫',
            '₫', '12.500', 12500),
    Example(None, '27.50',
            None, '27.50', 27.50),
    Example('23.90', '23.90',
            None, '23.90', 23.90),
    Example('Р', '18 000 Р',
            'Р', '18 000', 18000),
    Example(None, '48,96 €',
            '€', '48,96', 48.96),
    Example('DKK', '199 DKK',
            'DKK', '199', 199),
    Example('Price: £6.95 - £9.95', '£6.95 - £9.95',
            '£', '6.95', 6.95),
    Example(None, '599.97',
            None, '599.97', 599.97),
    Example(None, '$40.00',
            '$', '40.00', 40.00),
    Example('Cena 300,00 Kč', '100,00 Kč',
            'Kč', '100,00', 100),
    Example('18,25 €', '18,25 €',
            '€', '18,25', 18.25),
    Example(None, '29,00 €',
            '€', '29,00', 29),
    Example('€', '€ 39,95',
            '€', '39,95', 39.95),
    Example(None, '32.00',
            None, '32.00', 32.00),
    Example(None, '32.99',
            None, '32.99', 32.99),
    Example('HUF', '39000',
            'HUF', '39000', 39000),
    Example(None, '850,000 ریال',
            'ریال', '850,000', 850000),
    Example(None, '24,00 €',
            '€', '24,00', 24.00),
    Example('Versand', 'CHF 19.90',
            'CHF', '19.90', 19.90),
    Example('', '530,42 Zł',
            'Zł', '530,42', 530.42),
]


PRICE_PARSING_EXAMPLES_XFAIL = [
    # amount is picked as a price
    Example('3 Ausgaben für nur 14,85 EUR', '3 Ausgaben für nur 14,85 EUR',
            'EUR', '14,85', 14.85),
    Example(None, 'Buy Now - 2 Litre Was $120.00 Now $60.00',
            '$', '60.00', 60),
    Example('Цена: уточняйте (мин. заказ: 1 )', 'Цена: уточняйте (мин. заказ: 1 )',
            None, None, None),
    Example(None, '50 - $2.00 100 - $2.75 400 - $4.50 1,000 - $9.00 2,000 - $17.00 3,000 - $24.00 10,000 - $75.00',
            '$', '2.00', 2),

    # no detection of such single-letter currencies
    Example('R273.00', 'R273.00',
            'R', '273.00', 273),
    Example('R8,499', 'R8,499',
            'R', '8,499', 8499),
    Example('Cuneo', '61.858 L',  # Romanian New Leu
            'L', '61.858', 61858),

    # "р" / "руб" is detected as currency
    Example('>', 'См. цену в прайсе',
            None, None, None),
    Example('Купить', 'Печная труба',
            None, None, None),

    # dates
    Example(None, 'July, 2004',
            None, None, None),

    Example(None, '15.08.2017',
            None, None, None),

    # other incorrectly extracted prices
    Example('8.5', '25-09',
            None, None, None),

    # misc
    Example('of', '16.00 ft',
            None, None, None),
    # Example('7 724 134.40 114.32', '7 724 134.40 114.32',
    #         '', ''),
    # Example('中古価格（税込）： ¥20,800', '132',
    #         '¥', '132', 132),
    Example('Free Shipping on Orders $49+.', 'Free Shipping on Orders $49+.',
            '$', None, None),
]


PRICE_PARSING_DECIMAL_SEPARATOR_EXAMPLES = [
    Example(None, '1250€ 600',
            '€', '1250', 1250, "€"),
    Example(None, '1250€ 60',
            '€', '1250€60', 1250.60, "€"),
    Example(None, '1250€600',
            '€', '1250€600', 1250.600, "€"),
    Example(None, '.75 €',
            '€', '.75', 0.75, '.'),
    Example('$.75', '$.75',
            '$', '.75', 0.75, '.'),
    Example('$..75', '$..75',
            '$', '.75', 0.75, '.'),
    Example('$..75,333', '$..75,333',
            '$', '.75,333', 0.75333, '.'),
    Example('$..75,333', '$..75,333',
            '$', '.75,333', 75.333, ','),
    Example('$.750.30', '$.750.30',
            '$', '750.30', 750.30, '.')
]


@pytest.mark.parametrize(
    ["example"],
    [[e] for e in PRICE_PARSING_EXAMPLES_BUGS_CAUGHT] +
    [[e] for e in PRICE_PARSING_EXAMPLES_NEW] +
    [[e] for e in PRICE_PARSING_EXAMPLES] +
    [[e] for e in PRICE_PARSING_EXAMPLES_2] +
    [[e] for e in PRICE_PARSING_EXAMPLES_3] +
    [[e] for e in PRICE_PARSING_EXAMPLES_NO_PRICE] +
    [[e] for e in PRICE_PARSING_EXAMPLES_NO_CURRENCY] +
    [[e] for e in PRICE_PARSING_DECIMAL_SEPARATOR_EXAMPLES] +
    [pytest.param(e, marks=pytest.mark.xfail())
     for e in PRICE_PARSING_EXAMPLES_XFAIL]
)
def test_parsing(example: Example):
    parsed = Price.fromstring(example.price_raw, example.currency_raw, example.decimal_separator)
    assert parsed == example, f"Failed scenario: price={example.price_raw}, currency_hint={example.currency_raw}"


@pytest.mark.parametrize(
    "amount,amount_float",
    (
        (None, None),
        (Decimal('1.23'), 1.23),
    )
)
def test_price_amount_float(amount, amount_float):
    assert Price(amount, None, None).amount_float == amount_float


@pytest.mark.parametrize(
    "price_raw,decimal_separator,expected_result",
    (
        ("140.000", None, Decimal("140000")),
        ("140.000", ",", Decimal("140000")),
        ("140.000", ".", Decimal("140.000")),
        ("140€33", "€", Decimal("140.33")),
        ("140,000€33", "€", Decimal("140000.33")),
        ("140.000€33", "€", Decimal("140000.33")),
    )
)
def test_price_decimal_separator(price_raw, decimal_separator, expected_result):
    parsed = Price.fromstring(
        price_raw,
        decimal_separator=decimal_separator
    )
    assert parsed.amount == expected_result
