import os
import json
import bcrypt
import base64
import getpass
import time
from datetime import datetime, date, timedelta  # Added timedelta import
from cryptography.fernet import Fernet
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
from colorama import Fore, Style, init
from typing import Optional, Dict, List, Any
import shutil
from pathlib import Path
import tempfile
import sys
import site

class TaskmanBackend:
    def __init__(self):
        """Initialize TaskMan backend"""
        try:
            # Get user's Documents folder path
            if os.name == 'nt':  # Windows
                import winreg
                with winreg.OpenKey(winreg.HKEY_CURRENT_USER, r"Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders") as key:
                    documents_path = winreg.QueryValueEx(key, "Personal")[0]
            else:  # Unix-like
                documents_path = os.path.expanduser("~/Documents")
                
            # Create absolute paths in Documents
            self.data_dir = os.path.join(documents_path, "TaskmanSV_Data")
            self.users_dir = os.path.join(self.data_dir, "users")
            
            print(f"Using Documents folder paths:")
            print(f"Data directory: {self.data_dir}")
            print(f"Users directory: {self.users_dir}")
            
            # Create directories with explicit paths and handle permissions
            for directory in [self.data_dir, self.users_dir]:
                try:
                    if not os.path.exists(directory):
                        # Try to create with full permissions
                        os.makedirs(directory, mode=0o777)
                        print(f"Created directory: {directory}")
                    
                    # Verify write permissions with a test file
                    test_file = os.path.join(directory, ".write_test")
                    try:
                        with open(test_file, 'w') as f:
                            f.write("test")
                        os.remove(test_file)
                    except Exception as e:
                        raise PermissionError(f"Cannot write to directory {directory}: {e}")
                        
                except Exception as e:
                    print(f"Error with directory {directory}: {e}")
                    raise
            
            print(f"TaskMan-SV initialized successfully")
            
        except Exception as e:
            print(f"{Fore.RED}Error during initialization: {str(e)}")
            print(f"Current user: {os.getenv('USERNAME')}")
            print(f"Current working directory: {os.getcwd()}")
            print(f"Process user: {os.getlogin()}")
            raise

    def generate_key(self, password: str, salt: bytes = None) -> tuple:
        """Generate encryption key from password"""
        if salt is None:
            salt = os.urandom(16)
        kdf = PBKDF2HMAC(
            algorithm=hashes.SHA256(),
            length=32,
            salt=salt,
            iterations=100000,
        )
        key = base64.urlsafe_b64encode(kdf.derive(password.encode()))
        return key, salt

    def register_user(self, username: str, password: str) -> bool:
        """Register user with proper data structure"""
        try:
            user_file = os.path.join(self.users_dir, f"{username}.json")
            
            if os.path.exists(user_file):
                print(f"{Fore.RED}Username already exists!")
                return False
            
            # Generate salt and hash password
            salt = os.urandom(16)
            hashed = bcrypt.hashpw(password.encode(), bcrypt.gensalt())
            
            # Generate encryption key
            key, key_salt = self.generate_key(password, salt)
            
            # Create user data with proper structure
            user_data = {
                "username": username,
                "password": base64.b64encode(hashed).decode('utf-8'),
                "salt": base64.b64encode(salt).decode('utf-8'),
                "key_salt": base64.b64encode(key_salt).decode('utf-8'),
                "data": {
                    "notes": {},
                    "tasks": {},
                    "fastnotes": {}
                }
            }
            
            # Write to temporary file first
            temp_file = user_file + '.tmp'
            with open(temp_file, 'w') as f:
                json.dump(user_data, f, indent=2)
                
            # Replace the original file
            os.replace(temp_file, user_file)
            
            print(f"User registered successfully: {username}")
            return True
            
        except Exception as e:
            print(f"{Fore.RED}Error during registration: {str(e)}")
            if 'temp_file' in locals():
                try:
                    os.remove(temp_file)
                except:
                    pass
            return False

    def authenticate(self, username: str, password: str) -> bool:
        """Authenticate user and setup encryption"""
        try:
            user_file = os.path.join(self.users_dir, f"{username}.json")
            
            if not os.path.exists(user_file):
                return False

            with open(user_file, 'r') as f:
                user_data = json.loads(f.read())
            
            stored_hash = base64.b64decode(user_data['password'].encode('utf-8'))
            
            if not bcrypt.checkpw(password.encode(), stored_hash):
                return False

            # Setup encryption
            key_salt = base64.b64decode(user_data['key_salt'].encode('utf-8'))
            self.key, _ = self.generate_key(password, key_salt)
            self.fernet = Fernet(self.key)
            self.user = username
            self.user_file = user_file
            
            # Ensure data structure exists
            if 'data' not in user_data:
                user_data['data'] = {}
            if 'notes' not in user_data['data']:
                user_data['data']['notes'] = {}
            if 'tasks' not in user_data['data']:
                user_data['data']['tasks'] = {}
            if 'fastnotes' not in user_data['data']:
                user_data['data']['fastnotes'] = {}
            
            # Save the initialized structure
            with open(user_file, 'w') as f:
                json.dump(user_data, f, indent=2)
            
            print(f"User authenticated: {username}")
            print(f"Available data types: {list(user_data['data'].keys())}")
            return True
            
        except Exception as e:
            print(f"{Fore.RED}Error during authentication: {str(e)}")
            return False

    def save_data(self, data: dict, data_type: str) -> bool:
        """Save data with proper structure"""
        if not hasattr(self, 'user') or not hasattr(self, 'fernet'):
            print(f"Debug: Missing user or fernet attributes")
            return False

        try:
            user_file = os.path.join(self.users_dir, f"{self.user}.json")
            print(f"\nDebug: Saving {data_type} to {user_file}")
            
            # Read current data
            with open(user_file, 'r') as f:
                user_data = json.loads(f.read())
            
            print(f"Debug: Current user data keys: {list(user_data.keys())}")
            
            # Ensure data structure exists
            if 'data' not in user_data:
                print(f"Debug: Creating new data structure")
                user_data['data'] = {}
            
            # Encrypt the data
            print(f"Debug: Data to save: {type(data)}")
            print(f"Debug: Data keys: {list(data.keys()) if isinstance(data, dict) else 'not a dict'}")
            
            json_data = json.dumps(data).encode()
            encrypted_data = self.fernet.encrypt(json_data)
            
            # Store with timestamp
            user_data['data'][data_type] = {
                'timestamp': int(time.time()),
                'encrypted_data': encrypted_data.decode('utf-8')
            }
            
            print(f"Debug: Updated data types: {list(user_data['data'].keys())}")
            
            # Write to temporary file first
            temp_file = user_file + '.tmp'
            with open(temp_file, 'w') as f:
                json.dump(user_data, f, indent=2)
            
            # Replace the original file
            os.replace(temp_file, user_file)
            
            print(f"Debug: Successfully saved {data_type}")
            return True
            
        except Exception as e:
            print(f"{Fore.RED}Error saving {data_type}: {str(e)}")
            import traceback
            print(f"Debug: Full traceback:")
            print(traceback.format_exc())
            if 'temp_file' in locals():
                try:
                    os.remove(temp_file)
                except:
                    pass
            return False

    def load_data(self, data_type: str) -> dict:
        """Load data with proper structure"""
        if not hasattr(self, 'user') or not hasattr(self, 'fernet'):
            print(f"Debug: Missing user or fernet attributes")
            return {}

        try:
            user_file = os.path.join(self.users_dir, f"{self.user}.json")
            print(f"\nDebug: Loading {data_type} from {user_file}")
            
            if not os.path.exists(user_file):
                print(f"Debug: User file does not exist")
                return {}
            
            # Read the data
            with open(user_file, 'r') as f:
                user_data = json.loads(f.read())
            
            print(f"Debug: User data keys: {list(user_data.keys())}")
            
            # Check data structure
            if 'data' not in user_data:
                print(f"Debug: No 'data' key in user_data")
                return {}
            
            print(f"Debug: Available data types: {list(user_data['data'].keys())}")
            
            if data_type not in user_data['data']:
                print(f"Debug: {data_type} not found in data")
                return {}
            
            # Get the encrypted data
            data_entry = user_data['data'][data_type]
            print(f"Debug: Data entry keys: {list(data_entry.keys())}")
            
            encrypted_data = data_entry.get('encrypted_data')
            if not encrypted_data:
                print(f"Debug: No encrypted_data found for {data_type}")
                return {}
            
            # Decrypt and return
            decrypted_data = self.fernet.decrypt(encrypted_data.encode())
            loaded_data = json.loads(decrypted_data)
            
            print(f"Debug: Successfully loaded {data_type} data")
            print(f"Debug: Loaded data type: {type(loaded_data)}")
            print(f"Debug: Loaded data keys: {list(loaded_data.keys()) if isinstance(loaded_data, dict) else 'not a dict'}")
            
            return loaded_data
            
        except Exception as e:
            print(f"{Fore.RED}Error loading {data_type}: {str(e)}")
            import traceback
            print(f"Debug: Full traceback:")
            print(traceback.format_exc())
            return {}

    def get_empty_data(self, data_type: str) -> dict:
        """Return empty data structure based on type"""
        today = date.today().isoformat()
        
        if data_type == "tasks":
            return {
                "date": today,
                "tasks": []
            }
        elif data_type == "notes":
            return {
                "modules": {},
                "last_modified": datetime.now().isoformat(),
                "metadata": {
                    "created_at": datetime.now().isoformat(),
                    "version": "1.0"
                }
            }
        elif data_type == "fastnotes":
            return {
                "notes": [],
                "last_modified": datetime.now().isoformat(),
                "metadata": {
                    "created_at": datetime.now().isoformat(),
                    "version": "1.0"
                }
            }
        return {}

    def save_tasks(self, tasks: list) -> bool:
        """Save tasks with date-specific files"""
        task_data = {
            "date": date.today().isoformat(),
            "tasks": tasks,
            "last_modified": datetime.now().isoformat()
        }
        
        # Save to a date-specific file
        filename = f"{self.user}_tasks_{task_data['date']}.enc"
        file_path = os.path.join(self.data_dir, "users", filename)
        
        encrypted_data = self.fernet.encrypt(json.dumps(task_data).encode())
        with open(file_path, 'wb') as f:
            f.write(encrypted_data)
        return True
    
    def load_tasks(self, target_date: str = None) -> dict:
        """Load tasks and handle completed tasks"""
        if target_date is None:
            target_date = date.today().isoformat()

        # First try to load tasks for the specific date
        filename = f"{self.user}_tasks_{target_date}.enc"
        file_path = os.path.join(self.data_dir, "users", filename)
        
        if os.path.exists(file_path):
            try:
                with open(file_path, 'rb') as f:
                    encrypted_data = f.read()
                decrypted_data = self.fernet.decrypt(encrypted_data)
                task_data = json.loads(decrypted_data)
                
                # Add 'mode' field to existing tasks if missing
                if 'tasks' in task_data:
                    for task in task_data['tasks']:
                        if 'mode' not in task:
                            task['mode'] = 'custom'  # Set default mode for existing tasks
                
                return task_data
            except Exception as e:
                print(f"Error loading tasks: {str(e)}")
                return self.get_empty_data("tasks")
        
        # If loading today's tasks and file doesn't exist
        if target_date == date.today().isoformat():
            # Load yesterday's tasks to check for pending ones
            yesterday = (date.today() - timedelta(days=1)).isoformat()
            yesterday_data = self.load_tasks(yesterday)
            
            if yesterday_data and "tasks" in yesterday_data:
                # Only keep pending tasks from yesterday
                pending_tasks = [
                    task for task in yesterday_data["tasks"]
                    if task.get("status", "").strip() in [f"{Fore.YELLOW}Pending", f"{Fore.YELLOW}⏸ Paused"]
                ]
                
                # Ensure all tasks have the 'mode' field
                for task in pending_tasks:
                    if 'mode' not in task:
                        task['mode'] = 'custom'
                
                return {
                    "date": target_date,
                    "tasks": pending_tasks
                }
        
        return self.get_empty_data("tasks")
    
    def save_notes(self, notes_data: dict) -> bool:
        """Save notes modules"""
        if not isinstance(notes_data, dict):
            notes_data = {"modules": {}}
        
        if "modules" not in notes_data:
            notes_data["modules"] = {}
            
        notes_data["last_modified"] = datetime.now().isoformat()
        return self.save_data(notes_data, "notes")

    def load_notes(self) -> dict:
        """Load notes modules"""
        notes_data = self.load_data("notes")
        if not notes_data or not isinstance(notes_data, dict):
            return self.get_empty_data("notes")
        
        # Ensure required structure exists
        if "modules" not in notes_data:
            notes_data["modules"] = {}
        if "metadata" not in notes_data:
            notes_data["metadata"] = {
                "created_at": datetime.now().isoformat(),
                "version": "1.0"
            }
        
        return notes_data

    def save_fastnotes(self, fastnotes: list) -> bool:
        """Save fast notes"""
        notes_data = {
            "notes": fastnotes if isinstance(fastnotes, list) else [],
            "last_modified": datetime.now().isoformat(),
            "metadata": {
                "created_at": datetime.now().isoformat(),
                "version": "1.0"
            }
        }
        return self.save_data(notes_data, "fastnotes")

    def load_fastnotes(self) -> list:
        """Load fast notes"""
        data = self.load_data("fastnotes")
        if not data or not isinstance(data, dict):
            data = self.get_empty_data("fastnotes")
        return data.get("notes", [])

    def save_report(self, report: list, date_str: str = None) -> bool:
        """Save report to a secure location with encryption"""
        if not self.user or not self.fernet:
            return False

        if date_str is None:
            date_str = datetime.now().strftime('%Y-%m-%d')

        # Create reports directory if it doesn't exist
        reports_dir = os.path.join(self.data_dir, "reports")
        if not os.path.exists(reports_dir):
            os.makedirs(reports_dir)
            # Set proper permissions
            if os.name == 'nt':
                self._set_windows_permissions(reports_dir)
            else:
                os.chmod(reports_dir, 0o700)

        # Prepare report data with metadata
        report_data = {
            "date": date_str,
            "content": report,
            "generated_at": datetime.now().isoformat(),
            "username": self.user
        }

        try:
            # Encrypt and save the report
            json_data = json.dumps(report_data).encode()
            encrypted_data = self.fernet.encrypt(json_data)
            
            filename = f"report_{date_str}_{int(time.time())}.enc"
            file_path = os.path.join(reports_dir, filename)
            
            with open(file_path, 'wb') as f:
                f.write(encrypted_data)
            
            # Set proper permissions for the report file
            if os.name == 'nt':
                self._set_windows_permissions(file_path)
            else:
                os.chmod(file_path, 0o600)
            
            return True
        except Exception as e:
            print(f"Error saving report: {str(e)}")
            return False

    def load_report(self, date_str: str) -> dict:
        """Load and decrypt a report for a specific date"""
        if not self.user or not self.fernet:
            return None

        reports_dir = os.path.join(self.data_dir, "reports")
        if not os.path.exists(reports_dir):
            return None

        try:
            # Find the most recent report file for the given date
            report_files = [f for f in os.listdir(reports_dir) 
                           if f.startswith(f"report_{date_str}_") and f.endswith('.enc')]
            
            if not report_files:
                return None

            # Get the most recent report
            latest_report = sorted(report_files)[-1]
            file_path = os.path.join(reports_dir, latest_report)

            with open(file_path, 'rb') as f:
                encrypted_data = f.read()
            
            decrypted_data = self.fernet.decrypt(encrypted_data)
            return json.loads(decrypted_data)
        except Exception as e:
            print(f"Error loading report: {str(e)}")
            return None

    def _set_windows_permissions(self, path):
        """Set Windows-specific file permissions"""
        try:
            import win32security
            import ntsecuritycon as con
            
            # Get current user's SID
            username = os.getenv('USERNAME')
            domain = os.getenv('USERDOMAIN')
            sid = win32security.LookupAccountName(domain, username)[0]
            
            # Create DACL with full control only for current user
            dacl = win32security.ACL()
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION,
                con.FILE_ALL_ACCESS,
                sid
            )
            
            # Set security on file/directory
            security_desc = win32security.SECURITY_DESCRIPTOR()
            security_desc.SetSecurityDescriptorDacl(1, dacl, 0)
            win32security.SetFileSecurity(
                path,
                win32security.DACL_SECURITY_INFORMATION,
                security_desc
            )
        except Exception as e:
            print(f"Warning: Could not set Windows permissions: {str(e)}")

def display_auth_screen():
    """Display an attractive authentication screen"""
    os.system('cls' if os.name == 'nt' else 'clear')
    print(f"""
{Fore.CYAN}╔═══════════════════════════════════════════╗
║             {Fore.GREEN}TASKMAN LOGIN{Fore.CYAN}             ║
╚═══════════════════════════════════════════╝
""")

def get_credentials(mode="login") -> tuple:
    """Get username and password with improved UI"""
    display_auth_screen()
    
    print(f"{Fore.YELLOW}{'Login' if mode == 'login' else 'Register'} to TASKMAN")
    print(f"{Fore.CYAN}{'=' * 40}")
    
    username = input(f"{Fore.CYAN}Username: {Fore.WHITE}").strip()
    password = getpass.getpass(f"{Fore.CYAN}Password: {Fore.WHITE}")
    
    return username, password

def setup_backend():
    """Initialize backend and handle authentication with improved UI"""
    backend = TaskmanBackend()
    max_attempts = 3
    attempts = 0
    
    while attempts < max_attempts:
        display_auth_screen()
        print(f"{Fore.CYAN}1. {Fore.WHITE}Login")
        print(f"{Fore.CYAN}2. {Fore.WHITE}Register")
        print(f"{Fore.CYAN}3. {Fore.WHITE}Exit")
        print(f"{Fore.CYAN}{'=' * 40}")
        
        choice = input(f"{Fore.GREEN}Choice: {Fore.WHITE}").strip()
        
        if choice == "1":
            username, password = get_credentials("login")
            if backend.authenticate(username, password):
                os.system('cls' if os.name == 'nt' else 'clear')  # Clear screen for animation
                return backend
            attempts += 1
            print(f"{Fore.RED}Invalid credentials! {max_attempts - attempts} attempts remaining.")
            time.sleep(1)
            
        elif choice == "2":
            username, password = get_credentials("register")
            if backend.register_user(username, password):
                print(f"{Fore.GREEN}Registration successful! Please login.")
                time.sleep(1)
            else:
                print(f"{Fore.RED}Username already exists!")
                time.sleep(1)
                
        elif choice == "3":
            exit(0)
            
        else:
            print(f"{Fore.RED}Invalid choice!")
            time.sleep(1)
    
    print(f"{Fore.RED}Too many failed attempts. Please try again later.")
    exit(1)