# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['connect', 'connect.devops_testing', 'connect.devops_testing.bdd']

package_data = \
{'': ['*']}

install_requires = \
['behave', 'connect-openapi-client']

setup_kwargs = {
    'name': 'connect-devops-testing-library',
    'version': '24.0',
    'description': 'Testing framework to ease Connect EaaS Processors development.',
    'long_description': '# Connect DevOps Testing Library\n\nTesting library to ease Connect EaaS Processors development.\n\nDevOps Testing Library has a small request builder to ease the manipulation of the connect requests during testing:\n\n````python\nfrom connect.devops_testing import fixures\nimport os\n\ntemplate  = os.path.dirname(__file__) + \'/request.json\'\n\nrequest = fixures.make_request_builder(template) \\\n    .with_type(\'purchase\') \\\n    .with_asset_product(\'PRD-000-000-000\', \'Product Name\') \\\n    .with_asset_configuration_param(\'SOME_ASSET_CFG__PARAM_ID_A\', \'some_cfg_value_a\') \\\n    .with_asset_param(\'SOME_ASSET_PARAM_ID_001\', \'some_value_001\') \\\n    .with_asset_param(\'SOME_ASSET_PARAM_ID_002\', \'some_value_002\') \\\n    .build()\n````\n\nDevOps Testing Library also has several built-in assert functions that can be easily used to evaluate a connect request \nresponse:\n\n```python\nfrom connect.devops_testing import asserts\n\nasserts.request_status(request, \'approved\')\nasserts.asset_status(request, \'active\')\nasserts.asset_params_value_not_equal(request, \'SOME_ASSET_PARAM_ID_001\', \'some_expected_value\')\n```\n\nUsing these two features you can easily create a small test to check a purchase request of your processor:\n\n```python\nfrom connect.devops_testing import fixures, asserts\nfrom my_ext.extension import MyExtension\nimport os\n\ndef test_should_approve_request(mocked_connect_client, mocked_service_client, logger, eaas_config):\n    template  = os.path.dirname(__file__) + \'/request.json\'\n    \n    # prepare the request.\n    request = fixures.make_request_builder(template) \\\n        .with_type(\'purchase\') \\\n        .with_status(\'pending\') \\\n        .with_asset_param(\'subscription_id\', \'\') \\\n        .build()\n\n    # instantiate and execute the extension for the given request.\n    extension = MyExtension(mocked_connect_client, logger, eaas_config)\n    result = extension.process_asset_adjustment_request(request)\n\n    # evaluate the task result and request.\n    asserts.task_response_status(result, \'success\')\n    asserts.request_status(request, \'approved\')\n    asserts.asset_status(request, \'active\')\n    asserts.asset_params_value(request, \'subscription_id\', \'==\', \'ID:123456789\')\n```\n\nAdditionally, you may want to create real end-to-end test calling Connect and evaluating the processed request, for this\nyou should use the built-in request dispatcher. The dispatcher will take automatically the required credentials from the\nenvironment variables in `CONNECT_API_KEY` and `CONNECT_API_URL`. Alternatively, you can pass explicitly the credentials \nto the `make_request_dispatcher(api_key=XXX, api_url=YYY)` function. Let\'s see example:\n\n```python\nfrom connect.devops_testing import asserts, fixures\nimport os\n\ndef test_should_approve_purchase_request_successfully():\n    template = os.path.dirname(__file__) + \'/request.json\'\n    \n    # prepare the request.\n    request = fixures.make_request_builder(template) \\\n        .with_type(\'purchase\') \\\n        .with_status(\'pending\') \\\n        .with_asset_param(\'subscription_id\', \'\') \\\n        .build()\n\n    # dispatch the request to connect and wait some time so the \n    # processor can process the request.\n    request = fixures.make_request_dispatcher() \\\n        .provision_request(request, 10, 20)\n\n    # evaluate the processed request.\n    asserts.request_status(request, \'approved\')\n    asserts.asset_status(request, \'active\')\n    asserts.asset_params_value(request, \'subscription_id\', \'==\', \'ID:123456789\')\n```\n\nOnce the request is dispatched the Dispatcher will reload the request again every `10` seconds a maximum of `20` \nattempts. If the request has not been processed the asserts may fail. The wait time between request reload can be \nconfigured directly in the `.provision_request(api_key=, api_url=)` method call.\n\nObviously, some Connect processors may take a lot of time to process a request, for those type of processors this kind\nof end-to-end test is not suitable.\n\nFinally, the DevOps Testing Library also allows you to easily use Behave! BDD tool for you test. You just need to set \nthe following code in your `features/environment.py` file\n\n```python\nfrom connect.devops_testing.bdd.fixures import use_connect_request_dispatcher, use_connect_request_builder\n# import the built-in steps for e2e testing. \nfrom connect.devops_testing.bdd import steps\n\ndef before_all(context):\n    # attach the request dispatcher to the behave context if you want do e2e test.\n    use_connect_request_dispatcher(context)\n    # attach the request builder to the behave context.\n    use_connect_request_builder(context)\n```\n\nIt\'s time to define the feature file in `features/purchase.feature`:\n\n```gherkin\nFeature: Purchase a new subscription.\n\n  Scenario: Customer buys a subscription.\n    Given a new valid email address\n    When subscription request is processed\n    Then the subscription id is provided\n```\n\nNow let\'s define the steps in `features/steps/purchase.py` file\n\n```python\nfrom behave import given, then\nfrom connect.devops_testing import asserts\nimport os\n\n\n@given("a new valid email address")\ndef step_impl(context):\n    context.request = context.builder \\\n        .from_file(os.path.dirname(__file__) + \'/request.json\') \\\n        .with_asset_param(\'CUSTOMER_EMAIL_ADDRESS\', \'vincent.vega@gmail.com\') \\\n        .build()\n\n\n@then("the subscription id is provided")\ndef step_impl(context):\n    asserts.request_status(context.request, \'approved\')\n    asserts.asset_status(context.request, \'active\')\n    asserts.asset_params_value_not_equal(context.request, \'CUSTOMER_EMAIL_ADDRESS\', \'\')\n```\n\nThe `@when("subscription request is processed")` is provided by the DevOps Testing Library.\n',
    'author': 'CloudBlue LLC',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cloudblue/connect-devops-testing-library',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
