# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['openapydantic',
 'openapydantic.versions',
 'openapydantic.versions.openapi_302']

package_data = \
{'': ['*']}

install_requires = \
['email-validator>=1.1.3,<2.0.0',
 'jsonpath-ng>=1.5.3,<2.0.0',
 'pydantic>=1.9.0,<2.0.0',
 'pyyaml>=5.3.1',
 'types-PyYAML>=6.0.5,<7.0.0']

setup_kwargs = {
    'name': 'openapydantic',
    'version': '0.2.3',
    'description': 'openapi specification parser based on pydantic',
    'long_description': '# Openapydantic\n\n[openapi](https://github.com/OAI/OpenAPI-Specification) specification validator based on [pydantic](https://pydantic-docs.helpmanual.io/).\n\n## Python version support\n\n3.8+\n\n## Openapi versions support\n\n- ❌ 2.0\n- 🟠 3.0.0\n- 🟠 3.0.1\n- ✅ 3.0.2\n- ❌ 3.0.3\n- ❌ 3.1.0\n\nOpenapi versions are retrocompatibles (except for major version).\n\nSo 3.0.2 specification should be able to handle 3.0.0 and 3.0.1 data.\n\nUnit tests handle this case (3.0.2 object automatically try to load previous version fixtures).\n\n## Installation\n\nDepending on your preference...\n\n```\n    pip install openapydantic\n```\n\n...or...\n\n```\n    poetry add openapydantic\n```\n\n## Basic usage\n\n### Api loader\n\nOpenapydantic provide an openapi specification (a.k.a "swagger file") loader.\n\nThis loader returns a pydantic model so you can work with your specification like a common pydantic python object.\n\nFor each openapi specification version, there\'s a dedicated python class.\n\nThe loader can either automatically determine the class to provide...\n\n```python\nimport asyncio\n\nimport openapydantic\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="openapi-spec.yaml",\n    ),\n)\nprint(api.info)\n# if my openapi version is "3.0.2", \'api\' is an instance of OpenApi302\n# if the version is not implemented, it will crash\n```\n\n... or you can also specify a specific version.\n\nIt may be useful for backward compatibility (for eg: create an OpenApi302 object using data from an 3.0.1 openapi specfication ).\n\n```python\nimport asyncio\n\nimport openapydantic\n\nOpenApiVersion = openapydantic.OpenApiVersion\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="openapi-spec-3-0-1.yaml",\n        version=OpenApiVersion.v3_0_2\n    ),\n)\n# Here ,\'api\' is an OpenApi302 object, event if you send an 3.0.1 spec.\n\nprint(api.openapi)\n>> 3.0.1 # version in the spec file\nprint(api.__version__)\n>> 3.0.2 # openapi version supported for the object class\n```\n\n### Reference interpolation\n\nOpenapydantic will interpolate openapi references.\n\nIf your api looks like this:\n\n```yaml\n# my-api.yaml\nopenapi: 3.0.2\ninfo:\n  version: "1.0.0"\n  title: Example\npaths:\n  /user:\n    get:\n      summary: Get user\n      responses:\n        "200":\n          description: successful operation\n          content:\n            application/json:\n              schema:\n                $ref: "#/components/schemas/User"\ncomponents:\n  schemas:\n    User:\n      type: object\n      properties:\n        id:\n          type: integer\n          format: int64\n        name:\n          type: string\n          example: "John Doe"\n```\n\nOnce loaded, it will be usable like if it was ...\n\n```yaml\nopenapi: 3.0.2\ninfo:\n  version: "1.0.0"\n  title: Example\npaths:\n  /user:\n    get:\n      summary: Get user\n      responses:\n        "200":\n          description: successful operation\n          content:\n            application/json:\n              schema:\n                type: object\n                properties:\n                  id:\n                    type: integer\n                    format: int64\n                  name:\n                    type: string\n                    example: "John Doe"\n\n```\n\nAnd so you will be able to do things like...\n\n```python\nimport asyncio\n\nimport openapydantic\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="my-api.yaml",\n    ),\n)\nprint(api.info)\nprint(\n    api.paths["/user"]\n    .get.responses["200"]\n    .content["application/json"]\n    .schema_.properties["name"]\n    .example\n)\n>> John Doe\n```\n\nAs describe in the openapi specification some attributes are fix (\'paths\', \'content\' etc...) and some can be mapping with a free key.\n\nMapping must be accessed like common dict, either by direct key loading, either using .get(\'*key*\')\n\nNote that file reference (e.g: "#/file.yaml" are currently not supported)\n\nReference that reference themself will not be interpolated so ...\n\n```yaml\n# my-api.yaml\nopenapi: 3.0.2\ninfo:\n  version: "1.0.0"\n  title: Example\npaths:\n  /user:\n    get:\n      summary: Get user\n      responses:\n        "200":\n          description: successful operation\n          content:\n            application/json:\n              schema:\n                $ref: "#/components/schemas/User"\ncomponents:\n  schemas:\n    User:\n      type: object\n      properties:\n        id:\n          type: integer\n          format: int64\n        name:\n          type: string\n          example: "John Doe"\n        brother:\n          $ref: "#/components/schemas/User"\n```\n\n.. will stay the same ...\n\n```python\nimport asyncio\n\nimport openapydantic\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="my-api.yaml",\n    ),\n)\nprint(api.components.schemas["User"].properties["brother"].ref)\n>> \'#/components/schemas/User\'\n```\n\n### Attributes name collision\n\nOpenapi specify some attribute which name are already reserved either by pydantic,either by the python language itself.\n\nTo access these attributes, you must use the Openapydantic specific name\n\n| Attribute name | Openapydantic specific name |\n|----------------|------------------------------|\n| schema         | schema_                      |\n| in             | in_                          |\n| not            | not_                         |\n\ne.g:\n\n```python\nprint(\n    api.paths["/user"]\n    .get.responses["200"]\n    .content["application/json"]\n    .schema_\n)\n```\n\n### Model export\n\nYou can access the original api you provided as a dict using the **raw_api** attribute.\n\n```python\nimport asyncio\n\nimport openapydantic\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="my-api.yaml",\n    ),\n)\n\nprint(api.raw_api)\n>> {\'openapi\': \'3.0.2\', \'info\': {\'version\': \'1.0.0\', \'title\': \'Example\'}, \'paths\': {\'/user\': {\'get\': {\'summary\': \'Get user\', \'responses\': {\'200\': {\'description\': \'successful operation\', \'content\': {\'application/json\': {\'schema\': {\'$ref\': \'#/components/schemas/User\'}}}}}}}}, \'components\': {\'schemas\': {\'User\': {\'type\': \'object\', \'properties\': {\'id\': {\'type\': \'integer\', \'format\': \'int64\'}, \'name\': {\'type\': \'string\', \'example\': \'John Doe\'}}}}}}\n```\n\nYou can export your data as json string or as python dict using specific methods:\n\n```python\nimport asyncio\n\nimport openapydantic\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="my-api.yaml",\n    ),\n)\n\nprint(api.as_clean_json())\n>> {"openapi": "3.0.2", "info": {"title": "Example", "version": "1.0.0"}, "paths": {"/user": {"get": {"summary": "Get user", "responses": {"200": {"description": "successful operation", "content": {"application/json": {"schema": {"type": "object", "properties": {"id": {"type": "integer", "format": "int64"}, "name": {"type": "string", "example": "John Doe"}}}}}}}}}}}\n\n\nprint(api.as_clean_dict())\n> {\'openapi\': <OpenApiVersion.v3_0_2: \'3.0.2\'>, \'info\': {\'title\': \'Example\', \'version\': \'1.0.0\'}, \'paths\': {\'/user\': {\'get\': {\'summary\': \'Get user\', \'responses\': {\'200\': {\'description\': \'successful operation\', \'content\': {\'application/json\': {\'schema\': {\'type\': <JsonType.object_: \'object\'>, \'properties\': {\'id\': {\'type\': <JsonType.integer: \'integer\'>, \'format\': \'int64\'}, \'name\': {\'type\': <JsonType.string: \'string\'>, \'example\': \'John Doe\'}}}}}}}}}}}\n```\n\nNote that these functions are just wrapper to .dict() and .json() pydantic model with specific parameters.\n\nBy default, since the references are interpolated, the **components** root key is exclude.\n\nIf you want to have it in the output, you can set the **exclude_components** parameter to False.\n\n```python\nimport asyncio\n\nimport openapydantic\n\napi = asyncio.run(\n    openapydantic.load_api(\n        file_path="my-api.yaml",\n    ),\n)\n\nprint(\n    api.as_clean_json(\n        exclude_components=False,\n    ),\n)\n\n>> {"components": {"schemas": {"User": {"type": "object", "properties": {"id": {"type": "integer", "format": "int64"}, "name": {"type": "string", "example": "John Doe"}}}}}, "openapi": "3.0.2", "info": {"title": "Example", "version": "1.0.0"}, "paths": {"/user": {"get": {"summary": "Get user", "responses": {"200": {"description": "successful operation", "content": {"application/json": {"schema": {"type": "object", "properties": {"id": {"type": "integer", "format": "int64"}, "name": {"type": "string", "example": "John Doe"}}}}}}}}}}, "raw_api": {"openapi": "3.0.2", "info": {"version": "1.0.0", "title": "Example"}, "paths": {"/user": {"get": {"summary": "Get user", "responses": {"200": {"description": "successful operation", "content": {"application/json": {"schema": {"$ref": "#/components/schemas/User"}}}}}}}}, "components": {"schemas": {"User": {"type": "object", "properties": {"id": {"type": "integer", "format": "int64"}, "name": {"type": "string", "example": "John Doe"}}}}}}}\n\n```\n\nIn the same way,the **raw_api** attribute is exclude by default.\n\nIf you want to have it in the output, you can set the **exclude_raw_api** parameter to False.\n',
    'author': 'Richard Devers',
    'author_email': 'ritchiedev25@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/richarddevers/openapydantic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
