# API automated testing in Python
This is a starting point for automating API's in Python using the pytest and the AIOHTTP or HTTPX libraries on macOS.

## Required dependencies
In the terminal run

1. `/bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/master/install.sh)"`
2. `brew install pyenv`
3. `PYTHON_CONFIGURE_OPTS="--enable-framework" pyenv install 3.10.3 && pyenv global 3.10.3 && echo export PATH="$(pyenv root)/shims:$PATH" >> ~/.bash_profile && . ~/.bash_profile && pip install pipenv`

## Pip Installation
In the terminal

1. Cd into the root of your test project
2. Run [`pip install api-automation-tools`](https://pypi.org/project/api-automation-tools/)

## Local Setup
In the terminal

1. Fork and clone this repository
2. Cd into the root of your test project
3. Run `pip install -e path_to_apiautomationtools`

## Usage
### Helpers
```
import apiautomationtools.helpers.dictionary_helpers as dh
import apiautomationtools.helpers.directory_helpers as dh
import apiautomationtools.helpers.json_helpers as js
```

### Pytest
This class setup creates boilerplate data directories for tests and credentials. The teardown performs 
validations and organizational restructuring on any test generated data.
```
import pytest
from apiautomationtools.pytest import ApiPytestHelper

class SomeBasePytest(ApiPytestHelper):

    @pytest.fixture
    def test_app(self, ...):
        ...
```

### Requests with AIOHTTP or HTTPX
These classes wrap easy to use methods for making async requests. All the standard requests request
arguments are still applicable. As requests are completed, their data is saved in two csv files. One 
csv file contains the actual data sent, while the other contains a scrubbed set of data. The scrubbed 
csv file can be stored and used for reference validation of subsequent runs.
```
from apiautomationtools.client import AsyncRequests, HttpxRequests

aiohttp_requests = AsyncRequests()
httpx_requests = HttpxRequests()

batch = {'method': 'get', 'headers': {...}, 'url': '...', ...any classic requests arguments}
response = aiohttp_requests.request(batch)
response = httpx_requests.request(batch)

or

batch = [{'method': 'get', ...}, ...]
responses = aiohttp_requests.request(batch)
responses = httpx_requests.request(batch)

or 

batch = generate_batch("get", {...}, "https://httpbin.org/get")
response = aiohttp_requests.request(batch)
response = httpx_requests.request(batch)

Note: You can indicate where the batch generator will start looking for path parameters by placing
      a semicolon (;) where the path parameters start e.g. https://httpbin.org/get;/param/value.
```

### Validations
This class performs a difference between scrubbed csv files of the stored and live data generated from 
the responses of the request method. Any mismatches can be raised as errors.
```
from apiautomationtools.validations import Validations

validations = Validations()
mismatches = validations.validate_references()
mismatches = validations.validate_references(actual_refs={...})
mismatches => [{'keys': ..., 'actual_refs': {...}, 'expected_refs': {...}, 'unscrubbed_refs': {...}}, ...]
```

### Examples
An example of a test illustrating single and batch style requests - 
[test_get_example.py](examples/test_get_example.py)

An example of a base class showing setups and teardowns - 
[api_base_pytest_example.py](examples/api_base_pytest_example.py)

An example of the csv report file generated from running the test - 
[get_example.csv](examples/run_info/run_logs/pass/get_example.csv)

An example of the scrubbed csv report file generated from running the test - 
[get_example_scrubbed.csv](examples/run_info/run_logs/pass/get_example_scrubbed.csv)

An example of the log file generated from running the test - 
[get_example.log](examples/run_info/run_logs/pass/get_example.log)

An example of the error csv report file generated from running the test - 
[get_example_errors.csv](examples/run_info/run_errors/get_example_errors.csv)

### Directory structure
This package requires the following base structure for the project.
```
.
├── credentials                         # Optional - credentials
│   └── credentials.json                # Optional - credentials as json
├── tests                               # Required - test files
│   ├── data                            # Optional - test data
│   │   └── data.json                   # Optional - test data as json
│   └── test_file.py                    # Required - pytest test
└── validations                         # Optional - validation data
    └── file.json                       # Optional - validation data as json (The validation file's are scrubbed files 
                                                     autogenerated from test runs. However, the validation files 
                                                     organizational structure much match the structure of the test 
                                                     files in the tests directory.)
```