"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const common_1 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
const build_image_function_1 = require("../../lambdas/build-image-function");
const utils_1 = require("../../utils");
const common_2 = require("../common");
const dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
ENV RUNNER_VERSION=___RUNNER_VERSION___
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
/**
 * Image builder recipe for a Docker container image.
 */
class ContainerRecipe extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_1.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ContainerRecipe', name, {
                platform: props.platform,
                components,
                dockerfileTemplate,
            }),
            parentImage: props.parentImage ?? 'mcr.microsoft.com/windows/servercore:ltsc2019-amd64',
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: props.dockerfileTemplate,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An image builder that uses AWS Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild and Fargate.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'CodeBuild provider', {
 *     label: 'custom-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
class ContainerImageBuilder extends common_1.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_2.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_2.Architecture.X86_64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'image',
        });
        this.parentImage = props?.parentImage;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove all but the latest image',
                    tagStatus: aws_ecr_1.TagStatus.ANY,
                    maxImageCount: 1,
                },
            ],
        });
        // add all basic components
        this.addBaseWindowsComponents();
    }
    addBaseWindowsComponents() {
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.platform == 'Linux') {
            this.prependComponent(linux_components_1.LinuxUbuntuComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else if (this.platform == 'Windows') {
            this.prependComponent(windows_components_1.WindowsComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else {
            throw new Error(`Unknown platform: ${this.platform}`);
        }
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_1.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        const recipe = new ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform,
            components: this.components,
            targetRepository: this.repository,
            dockerfileTemplate: dockerfileTemplate.replace('___RUNNER_VERSION___', this.runnerVersion.version),
            parentImage: this.parentImage,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
        ]);
        const image = this.createImage(infra, dist, log, undefined, recipe.arn);
        this.createPipeline(infra, dist, log, undefined, recipe.arn);
        this.imageCleaner(image, recipe.name);
        this.boundImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        return this.boundImage;
    }
    imageCleaner(image, recipeName) {
        const crHandler = utils_1.singletonLambda(build_image_function_1.BuildImageFunction, this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
}
exports.ContainerImageBuilder = ContainerImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
ContainerImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ContainerImageBuilder", version: "0.8.3" };
//# sourceMappingURL=data:application/json;base64,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