"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const crypto = require("crypto");
const AWS = require("aws-sdk");
const helpers_1 = require("./helpers");
const sf = new AWS.StepFunctions();
// TODO use @octokit/webhooks?
function verifyBody(event, secret) {
    const sig = Buffer.from(event.headers['x-hub-signature-256'] || '', 'utf8');
    if (!event.body) {
        throw new Error('No body');
    }
    let body;
    if (event.isBase64Encoded) {
        body = Buffer.from(event.body, 'base64');
    }
    else {
        body = Buffer.from(event.body || '', 'utf8');
    }
    const hmac = crypto.createHmac('sha256', secret);
    hmac.update(body);
    const expectedSig = Buffer.from(`sha256=${hmac.digest('hex')}`, 'utf8');
    console.log('Calculated signature: ', expectedSig.toString());
    if (sig.length !== expectedSig.length || !crypto.timingSafeEqual(sig, expectedSig)) {
        throw new Error(`Signature mismatch. Expected ${expectedSig.toString()} but got ${sig.toString()}`);
    }
    return body.toString();
}
exports.handler = async function (event) {
    if (!process.env.WEBHOOK_SECRET_ARN || !process.env.STEP_FUNCTION_ARN) {
        throw new Error('Missing environment variables');
    }
    const webhookSecret = (await helpers_1.getSecretJsonValue(process.env.WEBHOOK_SECRET_ARN)).webhookSecret;
    let body;
    try {
        body = verifyBody(event, webhookSecret);
    }
    catch (e) {
        console.error(e);
        return {
            statusCode: 403,
            body: 'Bad signature',
        };
    }
    if (event.headers['content-type'] !== 'application/json') {
        console.error(`This webhook only accepts JSON payloads, got ${event.headers['content-type']}`);
        return {
            statusCode: 400,
            body: 'Expecting JSON payload',
        };
    }
    if (event.headers['x-github-event'] === 'ping') {
        return {
            statusCode: 200,
            body: 'Pong',
        };
    }
    // if (event.headers['x-github-event'] !== 'workflow_job' && event.headers['x-github-event'] !== 'workflow_run') {
    //     console.error(`This webhook only accepts workflow_job and workflow_run, got ${event.headers['x-github-event']}`);
    if (event.headers['x-github-event'] !== 'workflow_job') {
        console.error(`This webhook only accepts workflow_job, got ${event.headers['x-github-event']}`);
        return {
            statusCode: 400,
            body: 'Expecting workflow_job',
        };
    }
    const payload = JSON.parse(body);
    if (payload.action !== 'queued') {
        console.log(`Ignoring action "${payload.action}", expecting "queued"`);
        return {
            statusCode: 200,
            body: 'OK. No runner started.',
        };
    }
    if (!payload.workflow_job.labels.includes('self-hosted')) {
        console.log(`Ignoring labels "${payload.workflow_job.labels}", expecting "self-hosted"`);
        return {
            statusCode: 200,
            body: 'OK. No runner started.',
        };
    }
    // it's easier to deal with maps in step functions
    let labels = {};
    payload.workflow_job.labels.forEach((l) => labels[l] = true);
    // set execution name which is also used as runner name which are limited to 64 characters
    let executionName = `${payload.repository.full_name.replace('/', '-')}-${event.headers['x-github-delivery']}`.slice(0, 64);
    // start execution
    const execution = await sf.startExecution({
        stateMachineArn: process.env.STEP_FUNCTION_ARN,
        input: JSON.stringify({
            owner: payload.repository.owner.login,
            repo: payload.repository.name,
            runId: payload.workflow_job.run_id,
            installationId: payload.installation?.id,
            labels: labels,
        }),
        // name is not random so multiple execution of this webhook won't cause multiple builders to start
        name: executionName,
    }).promise();
    console.log(`Started ${execution.executionArn}`);
    return {
        statusCode: 202,
        body: executionName,
    };
};
//# sourceMappingURL=data:application/json;base64,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