"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const crypto = require("crypto");
const fs = require("fs");
const rest_1 = require("@octokit/rest");
const github_1 = require("./github");
const helpers_1 = require("./helpers");
const nonce = crypto.randomBytes(64).toString('hex');
function getHtml(baseUrl, token, domain) {
    return fs.readFileSync('index.html', 'utf-8')
        .replace(/INSERT_WEBHOOK_URL_HERE/g, process.env.WEBHOOK_URL)
        .replace(/INSERT_BASE_URL_HERE/g, baseUrl)
        .replace(/INSERT_TOKEN_HERE/g, token)
        .replace(/INSERT_SECRET_ARN_HERE/g, process.env.SETUP_SECRET_ARN)
        .replace(/INSERT_DOMAIN_HERE/g, domain)
        .replace(/<script/g, `<script nonce="${nonce}"`)
        .replace(/<style/g, `<style nonce="${nonce}"`);
}
function response(code, body) {
    return {
        statusCode: code,
        headers: {
            'Content-Type': 'text/html',
            'Content-Security-Policy': `default-src 'unsafe-inline' 'nonce-${nonce}'; img-src data:; connect-src 'self'; form-action https:; frame-ancestors 'none'; object-src 'none'; base-uri 'self'`,
        },
        body: body,
    };
}
async function handleRoot(event, setupToken) {
    const setupBaseUrl = `https://${event.requestContext.domainName}`;
    const githubSecrets = await helpers_1.getSecretJsonValue(process.env.GITHUB_SECRET_ARN);
    return response(200, getHtml(setupBaseUrl, setupToken, githubSecrets.domain));
}
function decodeBody(event) {
    let body = event.body;
    if (!body) {
        throw new Error('No body found');
    }
    if (event.isBase64Encoded) {
        body = Buffer.from(body, 'base64').toString('utf-8');
    }
    return JSON.parse(body);
}
async function handleDomain(event) {
    const body = decodeBody(event);
    if (!body.domain) {
        return response(400, 'Invalid domain');
    }
    const githubSecrets = await helpers_1.getSecretJsonValue(process.env.GITHUB_SECRET_ARN);
    githubSecrets.domain = body.domain;
    await helpers_1.updateSecretValue(process.env.GITHUB_SECRET_ARN, JSON.stringify(githubSecrets));
    return response(200, 'Domain set');
}
async function handlePat(event) {
    const body = decodeBody(event);
    if (!body.pat || !body.domain) {
        return response(400, 'Invalid personal access token');
    }
    await helpers_1.updateSecretValue(process.env.GITHUB_SECRET_ARN, JSON.stringify({
        domain: body.domain,
        appId: '',
        personalAuthToken: body.pat,
    }));
    await helpers_1.updateSecretValue(process.env.SETUP_SECRET_ARN, JSON.stringify({ token: '' }));
    return response(200, 'Personal access token set');
}
async function handleNewApp(event) {
    if (!event.queryStringParameters) {
        return response(400, 'Invalid code');
    }
    const code = event.queryStringParameters.code;
    if (!code) {
        return response(400, 'Invalid code');
    }
    const githubSecrets = await helpers_1.getSecretJsonValue(process.env.GITHUB_SECRET_ARN);
    const baseUrl = github_1.baseUrlFromDomain(githubSecrets.domain);
    const newApp = await new rest_1.Octokit({ baseUrl }).rest.apps.createFromManifest({ code });
    await helpers_1.updateSecretValue(process.env.GITHUB_SECRET_ARN, JSON.stringify({
        domain: new URL(newApp.data.html_url).host,
        appId: newApp.data.id,
        personalAuthToken: '',
    }));
    await helpers_1.updateSecretValue(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN, newApp.data.pem);
    await helpers_1.updateSecretValue(process.env.WEBHOOK_SECRET_ARN, JSON.stringify({
        webhookSecret: newApp.data.webhook_secret,
    }));
    await helpers_1.updateSecretValue(process.env.SETUP_SECRET_ARN, JSON.stringify({ token: '' }));
    return response(200, `New app set. <a href="${newApp.data.html_url}/installations/new">Install it</a> for your repositories.`);
}
async function handleExistingApp(event) {
    const body = decodeBody(event);
    if (!body.appid || !body.pk || !body.domain) {
        return response(400, 'Missing fields');
    }
    await helpers_1.updateSecretValue(process.env.GITHUB_SECRET_ARN, JSON.stringify({
        domain: body.domain,
        appId: body.appid,
        personalAuthToken: '',
    }));
    await helpers_1.updateSecretValue(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN, body.pk);
    await helpers_1.updateSecretValue(process.env.SETUP_SECRET_ARN, JSON.stringify({ token: '' }));
    return response(200, 'Existing app set. Don\'t forget to set up the webhook.');
}
exports.handler = async function (event) {
    // confirm required environment variables
    if (!process.env.WEBHOOK_URL) {
        throw new Error('Missing environment variables');
    }
    const setupToken = (await helpers_1.getSecretJsonValue(process.env.SETUP_SECRET_ARN)).token;
    // bail out if setup was already completed
    if (!setupToken) {
        return response(200, 'Setup already complete. Put a new token in the setup secret if you want to redo it.');
    }
    if (!event.queryStringParameters) {
        return response(403, 'Wrong setup token.');
    }
    // safely confirm url token matches our secret
    const urlToken = event.queryStringParameters.token || event.queryStringParameters.state || '';
    if (urlToken.length != setupToken.length || !crypto.timingSafeEqual(Buffer.from(urlToken, 'utf-8'), Buffer.from(setupToken, 'utf-8'))) {
        return response(403, 'Wrong setup token.');
    }
    // handle requests
    try {
        if (event.requestContext.http.path == '/') {
            return await handleRoot(event, setupToken);
        }
        else if (event.requestContext.http.path == '/domain' && event.requestContext.http.method == 'POST') {
            return await handleDomain(event);
        }
        else if (event.requestContext.http.path == '/pat' && event.requestContext.http.method == 'POST') {
            return await handlePat(event);
        }
        else if (event.requestContext.http.path == '/complete-new-app' && event.requestContext.http.method == 'GET') {
            return await handleNewApp(event);
        }
        else if (event.requestContext.http.path == '/app' && event.requestContext.http.method == 'POST') {
            return await handleExistingApp(event);
        }
        else {
            return response(404, 'Not found');
        }
    }
    catch (e) {
        return response(500, `<b>Error:</b> ${e}`);
    }
};
//# sourceMappingURL=data:application/json;base64,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