"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const constructs_1 = require("constructs");
const common_1 = require("./common");
class EcsFargateSpotLaunchTarget {
    /**
     * Called when the Fargate launch type configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                CapacityProviderStrategy: [
                    {
                        CapacityProvider: 'FARGATE_SPOT',
                    },
                ],
            },
        };
    }
}
/**
 * GitHub Actions runner provider using Fargate to execute the actions.
 *
 * Creates a task definition with a single container that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class FargateRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.label = props.label || 'fargate';
        this.vpc = props.vpc || aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'default vpc', { isDefault: true });
        this.securityGroup = props.securityGroup || new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'security group', { vpc: this.vpc });
        this.connections = this.securityGroup.connections;
        this.assignPublicIp = props.assignPublicIp || true;
        this.cluster = props.cluster ? props.cluster : new aws_cdk_lib_1.aws_ecs.Cluster(this, 'cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        this.spot = props.spot ?? false;
        this.task = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'task', {
            cpu: props.cpu || 1024,
            memoryLimitMiB: props.memoryLimitMiB || 2048,
            ephemeralStorageGiB: props.ephemeralStorageGiB || 25,
        });
        this.container = this.task.addContainer('runner', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromAsset(path.join(__dirname, 'docker-images', 'fargate'), {
                buildArgs: {
                    RUNNER_VERSION: props.runnerVersion ? props.runnerVersion.version : common_1.RunnerVersion.latest().version,
                },
            }),
            logging: aws_cdk_lib_1.aws_ecs.AwsLogDriver.awsLogs({
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, 'logs', {
                    retention: props.logRetention || aws_logs_1.RetentionDays.ONE_MONTH,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
                streamPrefix: 'runner',
            }),
        });
        this.grantPrincipal = new aws_cdk_lib_1.aws_iam.UnknownPrincipal({ resource: this.task.taskRole });
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsRunTask(this, 'Fargate Runner', {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
            taskDefinition: this.task,
            cluster: this.cluster,
            launchTarget: this.spot ? new EcsFargateSpotLaunchTarget() : new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsFargateLaunchTarget(),
            assignPublicIp: this.assignPublicIp,
            securityGroups: this.securityGroup ? [this.securityGroup] : undefined,
            containerOverrides: [
                {
                    containerDefinition: this.container,
                    environment: [
                        {
                            name: 'RUNNER_TOKEN',
                            value: parameters.runnerTokenPath,
                        },
                        {
                            name: 'RUNNER_NAME',
                            value: parameters.runnerNamePath,
                        },
                        {
                            name: 'RUNNER_LABEL',
                            value: this.label,
                        },
                        {
                            name: 'GITHUB_DOMAIN',
                            value: parameters.githubDomainPath,
                        },
                        {
                            name: 'OWNER',
                            value: parameters.ownerPath,
                        },
                        {
                            name: 'REPO',
                            value: parameters.repoPath,
                        },
                    ],
                },
            ],
        });
    }
}
exports.FargateRunner = FargateRunner;
_a = JSII_RTTI_SYMBOL_1;
FargateRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.FargateRunner", version: "0.2.0" };
//# sourceMappingURL=data:application/json;base64,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