# -*- coding: utf-8 -*-

"""
Implements the free biclosed monoidal category.
"""

from discopy import messages, monoidal
from discopy.cat import AxiomError


class Ty(monoidal.Ty):
    """
    Objects in a free biclosed monoidal category.
    Generated by the following grammar:

        ty ::= Ty(name) | ty @ ty | ty >> ty | ty << ty

    Examples
    --------
    >>> x, y = Ty('x'), Ty('y')
    >>> print(y << x >> y)
    ((y << x) >> y)
    >>> print((y << x >> y) @ x)
    ((y << x) >> y) @ x
    """
    @staticmethod
    def upgrade(ty):
        if len(ty) == 1 and isinstance(ty[0], (Over, Under)):
            return ty[0]
        return Ty(*ty.objects)

    def __init__(self, *objects, left=None, right=None):
        self.left, self.right = left, right
        super().__init__(*objects)

    def __lshift__(self, other):
        return Over(self, other)

    def __rshift__(self, other):
        return Under(self, other)


class Over(Ty):
    """ Forward slash types. """
    def __init__(self, left, right):
        super().__init__(self, left=left, right=right)

    def __repr__(self):
        return "Over({}, {})".format(repr(self.left), repr(self.right))

    def __str__(self):
        return "({} << {})".format(self.left, self.right)

    def __eq__(self, other):
        if not isinstance(other, Over):
            return False
        return self.left == other.left and self.right == other.right

    def __hash__(self):
        return hash(repr(self))


class Under(Ty):
    """ Backward slash types. """
    def __init__(self, left, right):
        super().__init__(self, left=left, right=right)

    def __repr__(self):
        return "Under({}, {})".format(repr(self.left), repr(self.right))

    def __str__(self):
        return "({} >> {})".format(self.left, self.right)

    def __eq__(self, other):
        if not isinstance(other, Under):
            return False
        return self.left == other.left and self.right == other.right

    def __hash__(self):
        return hash(repr(self))


class Diagram(monoidal.Diagram):
    """ Diagrams in a biclosed monoidal category. """
    @staticmethod
    def upgrade(diagram):
        return Diagram(
            diagram.dom, diagram.cod, diagram.boxes,
            diagram.offsets, diagram.layers)

    @staticmethod
    def id(dom):
        return Id(dom)

    @staticmethod
    def fa(left, right):
        """ Forward application. """
        return FA(left, right)

    @staticmethod
    def ba(left, right):
        """ Backward application. """
        return BA(left, right)


class Id(monoidal.Id, Diagram):
    """ Identity diagram in a biclosed monoidal category. """


class Box(monoidal.Box, Diagram):
    """ Boxes in a biclosed monoidal category. """


class FA(Box):
    """ Forward application. """
    def __init__(self, left, right):
        self.left, self.right = left, right
        if not isinstance(left, Over) or left.right != right:
            raise AxiomError(messages.are_not_adjoints(left, right))
        dom, cod = left @ right, left.left
        super().__init__("FA({}, {})".format(left, right), dom, cod)

    def __repr__(self):
        return "FA({}, {})".format(repr(self.left), repr(self.right))


class BA(Box):
    """ Backward application. """
    def __init__(self, left, right):
        self.left, self.right = left, right
        if not isinstance(right, Under) or right.left != left:
            raise AxiomError(messages.are_not_adjoints(left, right))
        dom, cod = left @ right, right.right
        super().__init__("BA({}, {})".format(left, right), dom, cod)

    def __repr__(self):
        return "BA({}, {})".format(repr(self.left), repr(self.right))


class Functor(monoidal.Functor):
    """
    Functors into biclosed monoidal categories.

    Examples
    --------
    >>> from discopy import rigid
    >>> x, y = Ty('x'), Ty('y')
    >>> F = Functor(
    ...     ob={x: x, y: y}, ar={},
    ...     ob_factory=rigid.Ty,
    ...     ar_factory=rigid.Diagram)
    >>> print(F(y >> x << y))
    y.r @ x @ y.l
    >>> assert F((y >> x) << y) == F(y >> (x << y))
    """
    def __init__(self, ob, ar, ob_factory=Ty, ar_factory=Diagram):
        super().__init__(ob, ar, ob_factory, ar_factory)

    def __call__(self, diagram):
        if isinstance(diagram, Over):
            return self(diagram.left) << self(diagram.right)
        if isinstance(diagram, Under):
            return self(diagram.left) >> self(diagram.right)
        if isinstance(diagram, Ty) and len(diagram) > 1:
            return self.ob_factory.tensor(*[
                self(diagram[i: i + 1]) for i in range(len(diagram))])
        if isinstance(diagram, FA):
            return self.ar_factory.fa(self(diagram.left), self(diagram.right))
        if isinstance(diagram, BA):
            return self.ar_factory.ba(self(diagram.left), self(diagram.right))
        return super().__call__(diagram)
