"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
// App setup
const app = new cdk.App();
const stack = new cdk.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-apigateway-iot with overriden params';
const apiGatewayProps = {
    restApiName: 'RestApi-Regional',
    description: 'Description for the Regional Rest Api',
    endpointConfiguration: { types: [api.EndpointType.REGIONAL] },
    apiKeySourceType: api.ApiKeySourceType.HEADER,
    defaultMethodOptions: {
        authorizationType: api.AuthorizationType.NONE,
    }
};
const policyJSON = {
    Version: "2012-10-17",
    Statement: [
        {
            Action: [
                "iot:UpdateThingShadow"
            ],
            Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
            Effect: "Allow"
        },
        {
            Action: [
                "iot:Publish"
            ],
            Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
            Effect: "Allow"
        }
    ]
};
const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
const iamRoleProps = {
    assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
    path: '/',
    inlinePolicies: { testPolicy: policyDocument }
};
// Create a policy that overrides the default policy that gets created with the construct
const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
// Api gateway setup
const props = {
    iotEndpoint: 'a1234567890123-ats',
    apiGatewayCreateApiKey: true,
    apiGatewayExecutionRole,
    apiGatewayProps
};
// Instantiate construct
new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot', props);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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