# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_google_sso', 'django_google_sso.migrations', 'django_google_sso.tests']

package_data = \
{'': ['*'], 'django_google_sso': ['templates/admin_sso/*']}

install_requires = \
['django>=4.0',
 'google-auth',
 'google-auth-httplib2',
 'google-auth-oauthlib',
 'loguru']

setup_kwargs = {
    'name': 'django-google-sso',
    'version': '2.2.0',
    'description': 'Easily add Google SSO login to Django Admin',
    'long_description': '## Welcome to Django Google SSO\n\n[![PyPI](https://img.shields.io/pypi/v/django-google-sso)](https://pypi.org/project/django-google-sso/)\n[![Build](https://github.com/chrismaille/django-google-sso/workflows/tests/badge.svg)](https://github.com/chrismaille/django-google-sso/actions)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/django-google-sso)](https://www.python.org)\n[![PyPI - Django Version](https://img.shields.io/pypi/djversions/django-google-sso)](https://www.djangoproject.com/)\n[![Conventional Commits](https://img.shields.io/badge/Conventional%20Commits-1.0.0-yellow.svg)](https://conventionalcommits.org)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n\nThis library aims to simplify the process of authenticating users with Google in Django Admin pages,\ninspired by libraries like [django_microsoft_auth](https://github.com/AngellusMortis/django_microsoft_auth)\nand [django-admin-sso](https://github.com/matthiask/django-admin-sso/)\n\n### Why another library?\n\n* This library aims for _simplicity_ and ease of use. [django-allauth](https://github.com/pennersr/django-allauth) is _\n  de facto_ solution for Authentication in Django,\n  but add lots of boilerplate, specially the html templates. **Django-Google-SSO** just add the "Login with Google"\n  button in the default login page.\n* [django-admin-sso](https://github.com/matthiask/django-admin-sso/) is a good solution, but it uses a deprecated\n  google `auth2client` version.\n\n### Install\n\n```shell\n$ pip install django-google-sso\n```\n\n### Versions\n\nFor django 4.x use version `2.x`\nFor django 3.x use version `1.x`\n\n### Configuration\n\n1. Add the following to your `settings.py` `INSTALLED_APPS`:\n\n```python\nINSTALLED_APPS = [\n    # other django apps\n    "django.contrib.messages",  # Need for Auth messages\n    "django_google_sso",  # Add django_google_sso\n]\n```\n\n2. In [Google Console](https://console.cloud.google.com/apis/credentials) at _Api -> Credentials_, retrieve your\n   Project Credentials and add them in your `settings.py`:\n\n```python\nGOOGLE_SSO_CLIENT_ID = "your client id here"\nGOOGLE_SSO_PROJECT_ID = "your project id here"\nGOOGLE_SSO_CLIENT_SECRET = "your client secret here"\n```\n\n3. Add the default site and allowed domains to auto-create users:\n\n```python\nGOOGLE_SSO_ALLOWABLE_DOMAINS = ["example.com"]\n```\n\n4. Add the netloc for the redirect url:\n\n```python\nGOOGLE_SSO_CALLBACK_DOMAIN = "localhost:8000"\n```\n\n5. In `urls.py` please add the **Django-Google-SSO** views:\n\n```python\nfrom django.urls import include, path\n\nurlpatterns = [\n    # other urlpatterns...\n    path(\n        "google_sso/", include("django_google_sso.urls", namespace="django_google_sso")\n    ),\n]\n```\n\n6. In [Google Console](https://console.cloud.google.com/apis/credentials) at _Api -> Credentials -> Oauth2 Client_,\n   please add **Django-Google-SSO** callback url, using this format: `https://your-domain.com/google_sso/callback/`,\n   where `your-domain.com` is the domain you defined in Django Sites Framework. For example, if you change your\n   Site object domain to `localhost:8000`, then your callback must be `http://localhost:8000/google_sso/callback/`. If\n   you don\'t want to use Sites Framework, or need a different domain, you can use the `GOOGLE_SSO_CALLBACK_DOMAIN`\n   setting to define the callback domain. For example `GOOGLE_SSO_CALLBACK_DOMAIN="my-other-domain.com"`.\n\n7. Run migrations:\n\n```shell\n$ python manage.py migrate\n```\n\n### Using Sites Framework\n\nIf you want to use the [Sites Framework](https://docs.djangoproject.com/en/3.2/ref/contrib/sites/), you can define the\nGoogle OAuth\ncallback url using the domain defined in Site model. To do this change your `settings.py`:\n\n```python\nINSTALLED_APPS = [\n    # other django apps\n    "django.contrib.messages",  # Need for Auth messages\n    "django.contrib.sites",  # Add Sites framework\n    "django_google_sso",  # Add django_google_sso\n]\n\nSITE_ID = 1  # Optional, just add if you want to use sites without request.\n\n# Comment or remove GOOGLE_SSO_CALLBACK_DOMAIN\n# GOOGLE_SSO_CALLBACK_DOMAIN = "localhost:8000"\n```\n\nPlease make note if you maintain the `GOOGLE_SSO_CALLBACK_DOMAIN` settings, his value will be used instead Sites\nFramework current domain.\n\n### How Django-Google-SSO works\n\nFirst, the user is redirected to the Django login page. If settings `GOOGLE_SSO_ENABLED` is True, the\n"Login with Google" button will be added to default form.\n\nOn click, **Django-Google-SSO** will add, in current Django Cache system, the `next_path` and Google Flow `state`.\nThis data will expire in 10 minutes. Then user will be redirected to Google login page.\n\nTo generate the callback url, **Django-Google-SSO** will use this format: `{scheme}://{netloc}{uri}` where:\n\n* `scheme` is retrieved first from `HTTP_X_FORWARDED_PROTO` header, then from request.\n* `netloc` is retrieved first from `GOOGLE_SSO_CALLBACK_DOMAIN` settings, then from\n  the registered domain in current Site or from the request, as\n  per [get_current_site](https://docs.djangoproject.com/en/4.1/ref/contrib/sites/#get-current-site-shortcut) rules.\n* ``uri`` is the [reverse](https://docs.djangoproject.com/en/4.1/ref/urlresolvers/#reverse) url from\n  view `django_google_sso:oauth_callback` imported in your `urls.py`\n\nOn callback, **Django-Google-SSO** will check `code` and `state` received. If they are valid,\nGoogle\'s UserInfo will be retrieved. If the user is already registered in Django, the user\nwill be logged in.\n\nOtherwise, the user will be created and logged in, if his email domain,\nmatches one of the `GOOGLE_SSO_ALLOWABLE_DOMAINS`. On creation only, this user can be set the\n`staff` or `superuser` status, if his email are in `GOGGLE_SSO_STAFF_LIST` or\n`GOGGLE_SSO_SUPERUSER_LIST` respectively. Please note if you add an email to one of these lists, the email domain\nmust be added to `GOOGLE_SSO_ALLOWABLE_DOMAINS`\ntoo. This session will expire in 1 hour, or the time defined, in seconds, in `GOOGLE_SSO_SESSION_COOKIE_AGE`.\n\nBrowser will be redirected to `next_path` if operation succeeds, or the `login` page, if operation fails.\n\n### Further customization\n\nPlease add the following variables to your `settings.py`:\n\n```python\nGOOGLE_SSO_ENABLED = True  # default value\nGOOGLE_SSO_SESSION_COOKIE_AGE = 3600  # default value\n\n# Mark as True, to add superuser status to first user\n# created with email domain in `GOOGLE_SSO_ALLOWABLE_DOMAINS`\nGOGGLE_SSO_AUTO_CREATE_FIRST_SUPERUSER = False\n\nGOGGLE_SSO_STAFF_LIST = ["email@example.com"]\nGOGGLE_SSO_SUPERUSER_LIST = ["another-email@example.com"]\nGOOGLE_SSO_TIMEOUT = 10  # Time before timeout Google requests. Default value: 10 seconds\nGOOGLE_SSO_SCOPES = [  # Google default scope\n    "openid",\n    "https://www.googleapis.com/auth/userinfo.email",\n    "https://www.googleapis.com/auth/userinfo.profile",\n]\n```\n\n### Running behind a Reverse Proxy\n\nPlease make sure you\'re passing the correct `X-Forwarded-Proto` header.\n\n### Using the `login_required` decorator\n\nTo use the `login_required` decorator, or his Class Based View equivalent, you can redirect the `accounts/login` route\nto the modified login form page, adding this to your `urls.py`:\n\n````python\nfrom django.conf.urls import url\nfrom django.contrib.auth.views import LoginView\n\nurlpatterns = [\n    url(\n        r"^accounts/login/$",\n        LoginView.as_view(\n            template_name="admin_sso/login.html"  # The modified form with Google button\n        ),\n    ),\n]\n````\n\n### Example App\n\nTo test this library please check the `Example App` provided [here](/example_app).\n\n### Not working?\n\nDon\'t panic. Get a towel and, please, open an [issue](https://github.com/chrismaille/django-google-sso/issues).\n',
    'author': 'Chris Maillefaud',
    'author_email': 'chrismaillefaud@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/chrismaille/django-google-sso',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
