# test_fzf.py

import pytest
from pyselector.menus.fzf import Fzf


@pytest.fixture
def fzf() -> Fzf:
    return Fzf()


def test_check_command(fzf) -> None:
    assert "fzf" in fzf.command


def test_build_command_args(fzf) -> None:
    args = fzf._build_command(
        case_sensitive=True,
        multi_select=True,
        prompt="Testing>",
    )
    assert "+i" in args
    assert "--prompt" in args
    assert "--multi" in args


def test_build_command_kwargs(fzf) -> None:
    args = fzf._build_command(
        case_sensitive=True,
        multi_select=False,
        prompt="Testing>",
        mesg="Testing...",
        height="40%",
        cycle=True,
        preview=True,
    )
    assert "--prompt" in args
    assert "--header" in args
    assert "--cycle" in args
    assert "--no-preview" not in args
    assert "--multi" not in args
    assert "--height" in args


def test_build_command_header(fzf) -> None:
    alt_r = fzf.keybind.add(
        key="alt-r",
        description="Testing add keybind",
        callback=lambda: None,
    )
    args = fzf._build_command(
        case_sensitive=True,
        multi_select=False,
        prompt="Testing>",
    )
    assert "--header" in args
    assert f"--bind={alt_r.bind}:" in args


def test_build_command_multi_select(fzf) -> None:
    args = fzf._build_command(
        case_sensitive=True,
        multi_select=True,
        prompt="Testing>",
        preview=False,
        height="40%",
        invalid_arg=True,
    )
    assert "--no-preview" in args
    assert "--multi" in args
    assert "--height" in args


def test_prompt_single_selection(fzf) -> None:
    item, keycode = fzf.prompt(
        case_sensitive=False,
        multi_select=False,
        items=["A", "B", "C"],
    )
    assert len(item) == 1


def test_prompt_multiple_selection(fzf) -> None:
    items, keycode = fzf.prompt(
        case_sensitive=False,
        multi_select=True,
        items=["A", "B", "C"],
        mesg="> Use <tab> to select all items",
    )
    assert len(items) == 3


def test_prompt_empty_items(fzf) -> None:
    with pytest.raises(SystemExit):
        item, keycode = fzf.prompt(
            case_sensitive=False, multi_select=False, mesg="> Hit <Escape>"
        )
