"use strict";
const sinon = require("sinon");
const consts_1 = require("../lib/cluster-resource-handler/consts");
const fargate_1 = require("../lib/cluster-resource-handler/fargate");
function newRequestMock(props = {}) {
    return {
        RequestType: 'Create',
        ServiceToken: 'ServiceTokenMock',
        LogicalResourceId: 'LogicalResourceIdMock',
        RequestId: 'RequestIdMock',
        ResourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
        ResponseURL: 'ResponseURLMock',
        StackId: 'StackIdMock',
        PhysicalResourceId: 'PhysicalResourceIdMock',
        ResourceProperties: {
            ServiceToken: 'ServiceTokenMock',
            AssumeRoleArn: 'AssumeRoleArn',
            Config: {
                clusterName: 'MockClusterName',
            },
        },
        ...props,
    };
}
function newEksClientMock() {
    return {
        createCluster: sinon.fake.throws('not implemented'),
        deleteCluster: sinon.fake.throws('not implemented'),
        describeCluster: sinon.fake.throws('not implemented'),
        describeUpdate: sinon.fake.throws('not implemented'),
        updateClusterConfig: sinon.fake.throws('not implemented'),
        updateClusterVersion: sinon.fake.throws('not implemented'),
        configureAssumeRole: sinon.fake(),
        createFargateProfile: sinon.fake.returns({
            fargateProfile: {
                fargateProfileName: 'MockProfileName',
                fargateProfileArn: 'MockProfileArn',
            },
        }),
        deleteFargateProfile: sinon.fake(),
        describeFargateProfile: sinon.fake.throws('not implemented'),
    };
}
module.exports = {
    create: {
        async 'calls createFargateProfile with a generated name'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.configureAssumeRole, {
                RoleArn: 'AssumeRoleArn',
                RoleSessionName: 'AWSCDK.EKSCluster.Create.RequestIdMock',
            });
            sinon.assert.calledWithExactly(client.createFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'LogicalResourceIdMock-RequestIdMock',
            });
            test.deepEqual(onEventResponse, {
                PhysicalResourceId: 'MockProfileName',
                Data: { fargateProfileArn: 'MockProfileArn' },
            });
            test.done();
        },
        async 'respects physical name if provided'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                ResourceProperties: {
                    AssumeRoleArn: 'AssumeRoleArnMock',
                    Config: {
                        fargateProfileName: 'MyProfileNameBoom',
                        clusterName: 'MockClusterName',
                    },
                },
            }));
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.createFargateProfile, {
                fargateProfileName: 'MyProfileNameBoom',
                clusterName: 'MockClusterName',
            });
            test.deepEqual(onEventResponse, {
                PhysicalResourceId: 'MockProfileName',
                Data: { fargateProfileArn: 'MockProfileArn' },
            });
            test.done();
        },
        async 'isComplete returns true if fargate profile is ACTIVE'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'ACTIVE',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            const resp = await handler.isComplete();
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(resp.IsComplete, true);
            test.done();
        },
        async 'isComplete returns false as long as fargate profile is CREATING'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'CREATING',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            const resp = await handler.isComplete();
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(resp.IsComplete, false);
            test.done();
        },
        async 'isComplete throws an exception if the status is CREATE_FAILED'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'CREATE_FAILED',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(error.message, 'CREATE_FAILED');
            test.done();
        },
    },
    update: {
        async 'calls createFargateProfile with a new name'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                RequestType: 'Update',
            }));
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.configureAssumeRole, {
                RoleArn: 'AssumeRoleArn',
                RoleSessionName: 'AWSCDK.EKSCluster.Update.RequestIdMock',
            });
            sinon.assert.calledWithExactly(client.createFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'LogicalResourceIdMock-RequestIdMock',
            });
            test.deepEqual(onEventResponse, {
                PhysicalResourceId: 'MockProfileName',
                Data: { fargateProfileArn: 'MockProfileArn' },
            });
            test.done();
        },
    },
    delete: {
        async 'calls deleteFargateProfile'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                RequestType: 'Delete',
            }));
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.configureAssumeRole, {
                RoleArn: 'AssumeRoleArn',
                RoleSessionName: 'AWSCDK.EKSCluster.Delete.RequestIdMock',
            });
            sinon.assert.calledWithExactly(client.deleteFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(onEventResponse, undefined);
            test.done();
        },
        async 'isComplete returns true when describeFargateProfile throws ResourceNotFoundException'(test) {
            // GIVEN
            const client = newEksClientMock();
            const resourceNotFoundError = new Error();
            resourceNotFoundError.code = 'ResourceNotFoundException';
            client.describeFargateProfile = sinon.fake.throws(resourceNotFoundError);
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                RequestType: 'Delete',
            }));
            const resp = await handler.isComplete();
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.deepEqual(resp, {
                IsComplete: true,
            });
            test.done();
        },
        async 'isComplete throws an exception if the status is DELETE_FAILED'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'DELETE_FAILED',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(error.message, 'DELETE_FAILED');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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