"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Service Account
 */
class ServiceAccount extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const { cluster } = props;
        this.serviceAccountName = (_a = props.name) !== null && _a !== void 0 ? _a : this.node.uniqueId.toLowerCase();
        this.serviceAccountNamespace = (_b = props.namespace) !== null && _b !== void 0 ? _b : 'default';
        /* Add conditions to the role to improve security. This prevents other pods in the same namespace to assume the role.
        * See documentation: https://docs.aws.amazon.com/eks/latest/userguide/create-service-account-iam-policy-and-role.html
        */
        const conditions = new core_1.CfnJson(this, 'ConditionJson', {
            value: {
                [`${cluster.clusterOpenIdConnectIssuer}:aud`]: 'sts.amazonaws.com',
                [`${cluster.clusterOpenIdConnectIssuer}:sub`]: `system:serviceaccount:${this.serviceAccountNamespace}:${this.serviceAccountName}`,
            },
        });
        const principal = new aws_iam_1.OpenIdConnectPrincipal(cluster.openIdConnectProvider).withConditions({
            StringEquals: conditions,
        });
        this.role = new aws_iam_1.Role(this, 'Role', { assumedBy: principal });
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.grantPrincipal = this.role.grantPrincipal;
        this.policyFragment = this.role.policyFragment;
        cluster.addResource(`${id}ServiceAccountResource`, {
            apiVersion: 'v1',
            kind: 'ServiceAccount',
            metadata: {
                name: this.serviceAccountName,
                namespace: this.serviceAccountNamespace,
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                },
                annotations: {
                    'eks.amazonaws.com/role-arn': this.role.roleArn,
                },
            },
        });
    }
    addToPolicy(statement) {
        return this.role.addToPolicy(statement);
    }
    addToPrincipalPolicy(statement) {
        return this.role.addToPrincipalPolicy(statement);
    }
}
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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