#!/usr/bin/env python
# coding=utf-8
"""Colormaps for visualization."""

import numpy as np

colormap_names = [
    'Autumn', 'Bone', 'Cold', 'Disparity',
    'Earth', 'Grayscale', 'Hot', 'HSV', 'Inferno', 'Jet',
    'Magma', 'Parula', 'Pastel', 'Plasma', 'Sepia',
    'Temperature', 'Thermal', 'Turbo', 'Viridis'
]


def by_name(colormap_name, return_rgb=True):
    if colormap_name is None:
        return None
    cmn = colormap_name.lower()
    if cmn == 'gray' or cmn == 'grayscale' or cmn == 'grey' or cmn == 'greyscale':
        return colormap_gray
    return globals()['colormap_{:s}_{:s}'.format(
            cmn,
            'rgb' if return_rgb else 'bgr'
        )]


def sample(colormap_name, N, return_rgb=True):
    """Returns N uniformly spread colors from the given colormap (specified by its name).
    For example, to sample 10 uniformly spaced colors from HSV: colors = sample('hsv', 10)
    If N > len(colormap), the colormap entries will be repeated.
    """
    cmap = by_name(colormap_name, return_rgb=return_rgb)
    if N < 2:
        raise ValueError('N must be > 1')

    if N <= len(cmap):
        # Truncate indices to get our intended behavior (e.g. 3 entries yields #0, #127, #255)
        indices = np.floor(np.linspace(0, len(cmap)-1, num=min(len(cmap), N), endpoint=True)).astype(np.int32)
        return [cmap[idx] for idx in indices]
    else:
        # Repeat the colormap as often as needed
        reps = int(N / len(cmap))
        rem = N % len(cmap)
        return cmap*reps + cmap[0:rem]


def make_flow_color_wheel():
    """Returns the standard color wheel (i.e. a 55x3 color map) used for
    optical flow visualization. This is a port of the C++/MATLAB code from
    https://people.csail.mit.edu/celiu/OpticalFlow, thus using similar/same
    names for variables.
    """
    # Color encoding schema
    RY = 15
    YG = 6
    GC = 4
    CB = 11
    BM = 13
    MR = 6
    ncols = RY + YG + GC + CB + BM + MR
    colorwheel = np.zeros([ncols, 3])
    col = 0
    # RY
    colorwheel[0:RY, 0] = 255
    colorwheel[0:RY, 1] = np.floor(255*np.arange(0, RY, 1)/RY)
    col += RY
    # YG
    colorwheel[col:YG+col, 0] = 255 - np.floor(255 * np.arange(0, YG, 1) / YG)
    colorwheel[col:YG+col, 1] = 255
    col += YG
    # GC
    colorwheel[col:GC+col, 1] = 255
    colorwheel[col:GC+col, 2] = np.floor(255 * np.arange(0, GC, 1) / GC)
    col += GC
    # CB
    colorwheel[col:CB+col, 1] = 255 - np.floor(255 * np.arange(0, CB, 1) / CB)
    colorwheel[col:CB+col, 2] = 255
    col += CB
    # BM
    colorwheel[col:BM+col, 2] = 255
    colorwheel[col:BM+col, 0] = np.floor(255 * np.arange(0, BM, 1) / BM)
    col += BM
    # MR
    colorwheel[col:MR+col, 2] = 255 - np.floor(255 * np.arange(0, MR, 1) / MR)
    colorwheel[col:MR+col, 0] = 255
    return colorwheel


colormap_autumn_bgr = [
    (0, 0, 255), (0, 1, 255), (0, 2, 255), (0, 3, 255), (0, 4, 255), (0, 5, 255),
    (0, 6, 255), (0, 7, 255), (0, 8, 255), (0, 9, 255), (0, 10, 255), (0, 11, 255),
    (0, 12, 255), (0, 13, 255), (0, 14, 255), (0, 15, 255), (0, 16, 255), (0, 17, 255),
    (0, 18, 255), (0, 19, 255), (0, 20, 255), (0, 21, 255), (0, 22, 255), (0, 23, 255),
    (0, 24, 255), (0, 25, 255), (0, 26, 255), (0, 27, 255), (0, 28, 255), (0, 29, 255),
    (0, 30, 255), (0, 31, 255), (0, 32, 255), (0, 33, 255), (0, 34, 255), (0, 35, 255),
    (0, 36, 255), (0, 37, 255), (0, 38, 255), (0, 39, 255), (0, 40, 255), (0, 41, 255),
    (0, 42, 255), (0, 43, 255), (0, 44, 255), (0, 45, 255), (0, 46, 255), (0, 47, 255),
    (0, 48, 255), (0, 49, 255), (0, 50, 255), (0, 51, 255), (0, 52, 255), (0, 53, 255),
    (0, 54, 255), (0, 55, 255), (0, 56, 255), (0, 57, 255), (0, 58, 255), (0, 59, 255),
    (0, 60, 255), (0, 61, 255), (0, 62, 255), (0, 63, 255), (0, 64, 255), (0, 65, 255),
    (0, 66, 255), (0, 67, 255), (0, 68, 255), (0, 69, 255), (0, 70, 255), (0, 71, 255),
    (0, 72, 255), (0, 73, 255), (0, 74, 255), (0, 75, 255), (0, 76, 255), (0, 77, 255),
    (0, 78, 255), (0, 79, 255), (0, 80, 255), (0, 81, 255), (0, 82, 255), (0, 83, 255),
    (0, 84, 255), (0, 85, 255), (0, 86, 255), (0, 87, 255), (0, 88, 255), (0, 89, 255),
    (0, 90, 255), (0, 91, 255), (0, 92, 255), (0, 93, 255), (0, 94, 255), (0, 95, 255),
    (0, 96, 255), (0, 97, 255), (0, 98, 255), (0, 99, 255), (0, 100, 255), (0, 101, 255),
    (0, 102, 255), (0, 103, 255), (0, 104, 255), (0, 105, 255), (0, 106, 255), (0, 107, 255),
    (0, 108, 255), (0, 109, 255), (0, 110, 255), (0, 111, 255), (0, 112, 255), (0, 113, 255),
    (0, 114, 255), (0, 115, 255), (0, 116, 255), (0, 117, 255), (0, 118, 255), (0, 119, 255),
    (0, 120, 255), (0, 121, 255), (0, 122, 255), (0, 123, 255), (0, 124, 255), (0, 125, 255),
    (0, 126, 255), (0, 127, 255), (0, 128, 255), (0, 129, 255), (0, 130, 255), (0, 131, 255),
    (0, 132, 255), (0, 133, 255), (0, 134, 255), (0, 135, 255), (0, 136, 255), (0, 137, 255),
    (0, 138, 255), (0, 139, 255), (0, 140, 255), (0, 141, 255), (0, 142, 255), (0, 143, 255),
    (0, 144, 255), (0, 145, 255), (0, 146, 255), (0, 147, 255), (0, 148, 255), (0, 149, 255),
    (0, 150, 255), (0, 151, 255), (0, 152, 255), (0, 153, 255), (0, 154, 255), (0, 155, 255),
    (0, 156, 255), (0, 157, 255), (0, 158, 255), (0, 159, 255), (0, 160, 255), (0, 161, 255),
    (0, 162, 255), (0, 163, 255), (0, 164, 255), (0, 165, 255), (0, 166, 255), (0, 167, 255),
    (0, 168, 255), (0, 169, 255), (0, 170, 255), (0, 171, 255), (0, 172, 255), (0, 173, 255),
    (0, 174, 255), (0, 175, 255), (0, 176, 255), (0, 177, 255), (0, 178, 255), (0, 179, 255),
    (0, 180, 255), (0, 181, 255), (0, 182, 255), (0, 183, 255), (0, 184, 255), (0, 185, 255),
    (0, 186, 255), (0, 187, 255), (0, 188, 255), (0, 189, 255), (0, 190, 255), (0, 191, 255),
    (0, 192, 255), (0, 193, 255), (0, 194, 255), (0, 195, 255), (0, 196, 255), (0, 197, 255),
    (0, 198, 255), (0, 199, 255), (0, 200, 255), (0, 201, 255), (0, 202, 255), (0, 203, 255),
    (0, 204, 255), (0, 205, 255), (0, 206, 255), (0, 207, 255), (0, 208, 255), (0, 209, 255),
    (0, 210, 255), (0, 211, 255), (0, 212, 255), (0, 213, 255), (0, 214, 255), (0, 215, 255),
    (0, 216, 255), (0, 217, 255), (0, 218, 255), (0, 219, 255), (0, 220, 255), (0, 221, 255),
    (0, 222, 255), (0, 223, 255), (0, 224, 255), (0, 225, 255), (0, 226, 255), (0, 227, 255),
    (0, 228, 255), (0, 229, 255), (0, 230, 255), (0, 231, 255), (0, 232, 255), (0, 233, 255),
    (0, 234, 255), (0, 235, 255), (0, 236, 255), (0, 237, 255), (0, 238, 255), (0, 239, 255),
    (0, 240, 255), (0, 241, 255), (0, 242, 255), (0, 243, 255), (0, 244, 255), (0, 245, 255),
    (0, 246, 255), (0, 247, 255), (0, 248, 255), (0, 249, 255), (0, 250, 255), (0, 251, 255),
    (0, 252, 255), (0, 253, 255), (0, 254, 255), (0, 255, 255)]
colormap_autumn_rgb = [(c[2], c[1], c[0]) for c in colormap_autumn_bgr]

colormap_bone_bgr = [
    (0, 0, 0), (1, 1, 1), (2, 2, 2), (4, 3, 3), (5, 4, 4), (6, 4, 4),
    (7, 5, 5), (8, 6, 6), (10, 7, 7), (11, 8, 8), (12, 9, 9), (13, 10, 10),
    (15, 11, 11), (16, 11, 11), (17, 12, 12), (18, 13, 13), (19, 14, 14), (21, 15, 15),
    (22, 16, 16), (23, 17, 17), (24, 17, 17), (25, 18, 18), (27, 19, 19), (28, 20, 20),
    (29, 21, 21), (30, 22, 22), (31, 23, 23), (33, 24, 24), (34, 24, 24), (35, 25, 25),
    (36, 26, 26), (37, 27, 27), (39, 28, 28), (40, 29, 29), (41, 30, 30), (42, 31, 31),
    (44, 32, 32), (45, 32, 32), (46, 33, 33), (47, 34, 34), (48, 35, 35), (50, 36, 36),
    (51, 37, 37), (52, 38, 38), (53, 39, 39), (54, 39, 39), (56, 40, 40), (57, 41, 41),
    (58, 42, 42), (59, 43, 43), (60, 44, 44), (62, 45, 45), (63, 45, 45), (64, 46, 46),
    (65, 47, 47), (66, 48, 48), (68, 49, 49), (69, 50, 50), (70, 51, 51), (71, 52, 52),
    (73, 53, 53), (74, 53, 53), (75, 54, 54), (76, 55, 55), (77, 56, 56), (79, 57, 57),
    (80, 58, 58), (81, 59, 59), (82, 59, 59), (83, 60, 60), (85, 61, 61), (86, 62, 62),
    (87, 63, 63), (88, 64, 64), (89, 65, 65), (91, 66, 66), (92, 66, 66), (93, 67, 67),
    (94, 68, 68), (95, 69, 69), (97, 70, 70), (98, 71, 71), (99, 72, 72), (100, 73, 73),
    (102, 74, 74), (103, 74, 74), (104, 75, 75), (105, 76, 76), (106, 77, 77), (108, 78, 78),
    (109, 79, 79), (110, 80, 80), (111, 81, 81), (112, 81, 81), (114, 82, 82), (115, 83, 83),
    (116, 84, 84), (117, 85, 85), (118, 87, 86), (119, 88, 87), (119, 89, 88), (120, 90, 88),
    (121, 91, 89), (122, 93, 90), (123, 94, 91), (124, 95, 92), (125, 96, 93), (126, 97, 94),
    (126, 99, 94), (127, 100, 95), (128, 101, 96), (129, 102, 97), (130, 103, 98), (131, 105, 99),
    (132, 106, 100), (133, 107, 101), (133, 108, 102), (134, 110, 102), (135, 111, 103), (136, 112, 104),
    (137, 113, 105), (138, 114, 106), (139, 116, 107), (140, 117, 108), (140, 118, 109), (141, 119, 109),
    (142, 120, 110), (143, 122, 111), (144, 123, 112), (145, 124, 113), (146, 125, 114), (147, 126, 115),
    (147, 128, 115), (148, 129, 116), (149, 130, 117), (150, 131, 118), (151, 132, 119), (152, 134, 120),
    (153, 135, 121), (153, 136, 122), (154, 137, 123), (155, 139, 123), (156, 140, 124), (157, 141, 125),
    (158, 142, 126), (159, 143, 127), (160, 145, 128), (161, 146, 129), (161, 147, 130), (162, 148, 130),
    (163, 149, 131), (164, 151, 132), (165, 152, 133), (166, 153, 134), (167, 154, 135), (167, 155, 136),
    (168, 157, 137), (169, 158, 137), (170, 159, 138), (171, 160, 139), (172, 161, 140), (173, 163, 141),
    (174, 164, 142), (174, 165, 143), (175, 166, 144), (176, 168, 144), (177, 169, 145), (178, 170, 146),
    (179, 171, 147), (180, 172, 148), (181, 174, 149), (182, 175, 150), (182, 176, 151), (183, 177, 151),
    (184, 178, 152), (185, 180, 153), (186, 181, 154), (187, 182, 155), (188, 183, 156), (188, 184, 157),
    (189, 186, 157), (190, 187, 158), (191, 188, 159), (192, 189, 160), (193, 190, 161), (194, 192, 162),
    (195, 193, 163), (195, 194, 164), (196, 195, 165), (197, 197, 165), (198, 198, 166), (199, 199, 167),
    (200, 200, 168), (201, 201, 170), (202, 202, 171), (203, 203, 172), (203, 203, 174), (204, 204, 175),
    (205, 205, 177), (206, 206, 178), (207, 207, 179), (208, 208, 181), (209, 209, 182), (210, 210, 183),
    (210, 210, 185), (211, 211, 186), (212, 212, 188), (213, 213, 189), (214, 214, 190), (215, 215, 192),
    (216, 216, 193), (217, 217, 195), (217, 217, 196), (218, 218, 197), (219, 219, 199), (220, 220, 200),
    (221, 221, 201), (222, 222, 203), (223, 223, 204), (224, 224, 205), (224, 224, 207), (225, 225, 208),
    (226, 226, 210), (227, 227, 211), (228, 228, 212), (229, 229, 214), (230, 230, 215), (230, 230, 216),
    (231, 231, 218), (232, 232, 219), (233, 233, 221), (234, 234, 222), (235, 235, 223), (236, 236, 225),
    (237, 237, 226), (238, 238, 227), (238, 238, 229), (239, 239, 230), (240, 240, 232), (241, 241, 233),
    (242, 242, 234), (243, 243, 236), (244, 244, 237), (244, 244, 238), (245, 245, 240), (246, 246, 241),
    (247, 247, 243), (248, 248, 244), (249, 249, 245), (250, 250, 247), (251, 251, 248), (251, 251, 250),
    (252, 252, 251), (253, 253, 252), (254, 254, 254), (255, 255, 255)]
colormap_bone_rgb = [(c[2], c[1], c[0]) for c in colormap_bone_bgr]

colormap_cold_bgr = [
    (0, 0, 0), (3, 0, 0), (6, 0, 0), (9, 0, 0), (12, 0, 0), (15, 0, 0),
    (18, 0, 0), (21, 0, 0), (24, 0, 0), (27, 0, 0), (30, 0, 0), (33, 0, 0),
    (36, 0, 0), (39, 0, 0), (42, 0, 0), (45, 0, 0), (48, 0, 0), (51, 0, 0),
    (54, 0, 0), (57, 0, 0), (60, 0, 0), (63, 0, 0), (66, 0, 0), (69, 0, 0),
    (72, 0, 0), (75, 0, 0), (78, 0, 0), (81, 0, 0), (84, 0, 0), (87, 0, 0),
    (90, 0, 0), (93, 0, 0), (96, 0, 0), (99, 0, 0), (102, 0, 0), (105, 0, 0),
    (108, 0, 0), (111, 0, 0), (114, 0, 0), (117, 0, 0), (120, 0, 0), (123, 0, 0),
    (126, 0, 0), (129, 0, 0), (132, 0, 0), (135, 0, 0), (138, 0, 0), (141, 0, 0),
    (144, 0, 0), (147, 0, 0), (150, 0, 0), (153, 0, 0), (156, 0, 0), (159, 0, 0),
    (162, 0, 0), (165, 0, 0), (168, 0, 0), (171, 0, 0), (174, 0, 0), (177, 0, 0),
    (180, 0, 0), (183, 0, 0), (186, 0, 0), (189, 0, 0), (192, 0, 0), (195, 0, 0),
    (198, 0, 0), (201, 0, 0), (204, 0, 0), (207, 0, 0), (210, 0, 0), (213, 0, 0),
    (216, 0, 0), (219, 0, 0), (222, 0, 0), (225, 0, 0), (228, 0, 0), (231, 0, 0),
    (234, 0, 0), (237, 0, 0), (240, 0, 0), (243, 0, 0), (246, 0, 0), (249, 0, 0),
    (252, 0, 0), (255, 0, 0), (255, 3, 0), (255, 6, 0), (255, 9, 0), (255, 12, 0),
    (255, 15, 0), (255, 18, 0), (255, 21, 0), (255, 24, 0), (255, 27, 0), (255, 30, 0),
    (255, 33, 0), (255, 36, 0), (255, 39, 0), (255, 42, 0), (255, 45, 0), (255, 48, 0),
    (255, 51, 0), (255, 54, 0), (255, 57, 0), (255, 60, 0), (255, 63, 0), (255, 66, 0),
    (255, 69, 0), (255, 72, 0), (255, 75, 0), (255, 78, 0), (255, 81, 0), (255, 84, 0),
    (255, 87, 0), (255, 90, 0), (255, 93, 0), (255, 96, 0), (255, 99, 0), (255, 102, 0),
    (255, 105, 0), (255, 108, 0), (255, 111, 0), (255, 114, 0), (255, 117, 0), (255, 120, 0),
    (255, 123, 0), (255, 126, 0), (255, 129, 0), (255, 132, 0), (255, 135, 0), (255, 138, 0),
    (255, 141, 0), (255, 144, 0), (255, 147, 0), (255, 150, 0), (255, 153, 0), (255, 156, 0),
    (255, 159, 0), (255, 162, 0), (255, 165, 0), (255, 168, 0), (255, 171, 0), (255, 174, 0),
    (255, 177, 0), (255, 180, 0), (255, 183, 0), (255, 186, 0), (255, 189, 0), (255, 192, 0),
    (255, 195, 0), (255, 198, 0), (255, 201, 0), (255, 204, 0), (255, 207, 0), (255, 210, 0),
    (255, 213, 0), (255, 216, 0), (255, 219, 0), (255, 222, 0), (255, 225, 0), (255, 228, 0),
    (255, 231, 0), (255, 234, 0), (255, 237, 0), (255, 240, 0), (255, 243, 0), (255, 246, 0),
    (255, 249, 0), (255, 252, 0), (255, 255, 0), (255, 255, 3), (255, 255, 6), (255, 255, 9),
    (255, 255, 12), (255, 255, 15), (255, 255, 18), (255, 255, 21), (255, 255, 24), (255, 255, 27),
    (255, 255, 30), (255, 255, 33), (255, 255, 36), (255, 255, 39), (255, 255, 42), (255, 255, 45),
    (255, 255, 48), (255, 255, 51), (255, 255, 54), (255, 255, 57), (255, 255, 60), (255, 255, 63),
    (255, 255, 66), (255, 255, 69), (255, 255, 72), (255, 255, 75), (255, 255, 78), (255, 255, 81),
    (255, 255, 84), (255, 255, 87), (255, 255, 90), (255, 255, 93), (255, 255, 96), (255, 255, 99),
    (255, 255, 102), (255, 255, 105), (255, 255, 108), (255, 255, 111), (255, 255, 114), (255, 255, 117),
    (255, 255, 120), (255, 255, 123), (255, 255, 126), (255, 255, 129), (255, 255, 132), (255, 255, 135),
    (255, 255, 138), (255, 255, 141), (255, 255, 144), (255, 255, 147), (255, 255, 150), (255, 255, 153),
    (255, 255, 156), (255, 255, 159), (255, 255, 162), (255, 255, 165), (255, 255, 168), (255, 255, 171),
    (255, 255, 174), (255, 255, 177), (255, 255, 180), (255, 255, 183), (255, 255, 186), (255, 255, 189),
    (255, 255, 192), (255, 255, 195), (255, 255, 198), (255, 255, 201), (255, 255, 204), (255, 255, 207),
    (255, 255, 210), (255, 255, 213), (255, 255, 216), (255, 255, 219), (255, 255, 222), (255, 255, 225),
    (255, 255, 228), (255, 255, 231), (255, 255, 234), (255, 255, 237), (255, 255, 240), (255, 255, 243),
    (255, 255, 246), (255, 255, 249), (255, 255, 252), (255, 255, 255)]
colormap_cold_rgb = [(c[2], c[1], c[0]) for c in colormap_cold_bgr]

colormap_disparity_bgr = [
    (0, 0, 0), (9, 0, 0), (18, 0, 0), (26, 0, 0), (35, 0, 0), (44, 0, 0),
    (53, 0, 0), (61, 0, 0), (70, 0, 0), (79, 0, 0), (88, 0, 0), (96, 0, 0),
    (105, 0, 0), (114, 0, 0), (123, 0, 0), (132, 0, 0), (70, 0, 27), (80, 0, 26),
    (91, 0, 25), (102, 0, 25), (113, 0, 24), (123, 0, 23), (134, 0, 22), (145, 0, 22),
    (155, 0, 21), (166, 0, 20), (177, 0, 20), (188, 0, 19), (198, 0, 18), (209, 0, 17),
    (220, 0, 17), (230, 0, 16), (167, 1, 42), (163, 2, 45), (158, 3, 48), (153, 3, 52),
    (149, 4, 55), (144, 5, 58), (140, 6, 62), (135, 7, 65), (131, 8, 68), (126, 9, 71),
    (122, 10, 75), (117, 11, 78), (113, 12, 81), (108, 13, 84), (104, 14, 88), (99, 15, 91),
    (153, 0, 102), (147, 0, 108), (142, 0, 113), (136, 0, 119), (131, 0, 124), (126, 0, 129),
    (120, 0, 135), (115, 0, 140), (109, 0, 146), (104, 0, 151), (99, 0, 156), (93, 0, 162),
    (88, 0, 167), (82, 0, 173), (77, 0, 178), (72, 0, 183), (83, 12, 159), (80, 11, 166),
    (76, 10, 172), (73, 9, 178), (69, 9, 184), (66, 8, 190), (63, 7, 196), (59, 6, 202),
    (56, 6, 208), (53, 5, 214), (49, 4, 220), (46, 3, 226), (43, 3, 233), (39, 2, 239),
    (36, 1, 245), (33, 1, 251), (79, 7, 224), (88, 7, 224), (98, 7, 224), (107, 7, 224),
    (116, 6, 224), (125, 6, 224), (135, 6, 224), (144, 6, 224), (153, 6, 224), (162, 6, 224),
    (171, 6, 224), (181, 6, 224), (190, 6, 224), (199, 6, 224), (208, 5, 224), (217, 5, 224),
    (171, 1, 254), (174, 3, 252), (177, 5, 250), (180, 8, 247), (183, 10, 245), (187, 12, 243),
    (190, 14, 241), (193, 17, 238), (196, 19, 236), (199, 21, 234), (202, 23, 232), (205, 26, 229),
    (208, 28, 227), (211, 30, 225), (214, 32, 223), (217, 35, 220), (178, 54, 201), (175, 59, 196),
    (171, 63, 192), (168, 68, 187), (165, 73, 182), (162, 78, 177), (158, 82, 173), (155, 87, 168),
    (152, 92, 163), (149, 97, 158), (145, 101, 154), (142, 106, 149), (139, 111, 144), (136, 116, 139),
    (132, 120, 135), (129, 125, 130), (50, 133, 147), (49, 139, 140), (48, 145, 133), (47, 150, 126),
    (46, 156, 118), (45, 161, 111), (44, 167, 104), (43, 172, 97), (43, 178, 90), (42, 183, 82),
    (41, 189, 75), (40, 194, 68), (39, 200, 61), (38, 206, 53), (37, 211, 46), (36, 217, 39),
    (48, 206, 58), (51, 209, 55), (53, 213, 51), (56, 216, 47), (58, 219, 43), (60, 222, 39),
    (63, 225, 36), (65, 228, 32), (67, 232, 28), (70, 235, 24), (72, 238, 20), (75, 241, 17),
    (77, 244, 13), (79, 247, 9), (82, 251, 5), (84, 254, 1), (98, 241, 24), (106, 242, 23),
    (114, 243, 21), (123, 244, 20), (131, 245, 18), (139, 246, 16), (147, 247, 15), (156, 247, 13),
    (164, 248, 12), (172, 249, 10), (181, 250, 8), (189, 251, 7), (197, 252, 5), (206, 253, 4),
    (214, 254, 2), (222, 255, 0), (186, 247, 33), (186, 248, 35), (186, 248, 37), (186, 249, 40),
    (186, 249, 42), (187, 250, 44), (187, 250, 46), (187, 251, 49), (187, 251, 51), (187, 252, 53),
    (187, 252, 56), (187, 253, 58), (187, 253, 60), (187, 254, 62), (187, 254, 65), (187, 255, 67),
    (183, 242, 97), (177, 243, 101), (172, 244, 105), (167, 245, 109), (161, 245, 112), (156, 246, 116),
    (151, 247, 120), (145, 248, 123), (140, 249, 127), (134, 250, 131), (129, 251, 134), (124, 251, 138),
    (118, 252, 142), (113, 253, 146), (108, 254, 149), (102, 255, 153), (155, 241, 164), (146, 242, 169),
    (137, 243, 174), (127, 244, 179), (118, 245, 184), (109, 245, 189), (100, 246, 194), (90, 247, 199),
    (81, 248, 204), (72, 249, 209), (63, 250, 214), (53, 251, 219), (44, 252, 224), (35, 253, 229),
    (26, 254, 234), (16, 255, 239), (95, 253, 216), (101, 253, 216), (107, 254, 217), (113, 254, 218),
    (119, 254, 219), (125, 254, 220), (131, 254, 221), (137, 254, 221), (143, 254, 222), (149, 254, 223),
    (155, 254, 224), (162, 255, 225), (168, 255, 225), (174, 255, 226), (180, 255, 227), (186, 255, 228),
    (123, 255, 255), (132, 255, 255), (141, 255, 255), (150, 255, 255), (159, 255, 255), (167, 255, 255),
    (176, 255, 255), (185, 255, 255), (194, 255, 255), (202, 255, 255), (211, 255, 255), (220, 255, 255),
    (229, 255, 255), (237, 255, 255), (246, 255, 255), (255, 255, 255)]
colormap_disparity_rgb = [(c[2], c[1], c[0]) for c in colormap_disparity_bgr]

colormap_earth_bgr = [
    (0, 0, 0), (1, 2, 0), (2, 3, 0), (2, 5, 0), (3, 6, 0), (4, 8, 0),
    (5, 9, 0), (6, 11, 0), (7, 12, 0), (7, 14, 0), (8, 15, 0), (9, 17, 0),
    (10, 19, 0), (11, 20, 0), (11, 22, 0), (12, 23, 0), (13, 25, 0), (14, 26, 0),
    (15, 28, 0), (14, 29, 2), (14, 29, 4), (14, 30, 6), (13, 30, 9), (13, 31, 11),
    (12, 32, 13), (12, 32, 16), (11, 33, 18), (11, 34, 20), (10, 34, 23), (10, 35, 25),
    (9, 35, 27), (9, 36, 30), (8, 37, 32), (8, 37, 34), (7, 38, 37), (7, 38, 39),
    (7, 39, 41), (7, 40, 42), (9, 42, 41), (10, 44, 40), (12, 46, 39), (13, 48, 39),
    (14, 49, 38), (16, 51, 37), (17, 53, 36), (19, 55, 35), (20, 57, 35), (22, 59, 34),
    (23, 60, 33), (25, 62, 32), (26, 64, 31), (27, 66, 31), (29, 68, 30), (30, 70, 29),
    (32, 71, 28), (33, 73, 28), (32, 74, 31), (32, 74, 33), (31, 75, 35), (31, 76, 37),
    (30, 76, 39), (30, 77, 42), (29, 78, 44), (29, 78, 46), (28, 79, 48), (28, 80, 50),
    (27, 80, 52), (27, 81, 55), (26, 82, 57), (26, 82, 59), (25, 83, 61), (25, 84, 63),
    (24, 85, 66), (24, 85, 67), (25, 87, 67), (27, 88, 67), (28, 90, 67), (29, 91, 67),
    (30, 93, 67), (31, 94, 67), (33, 96, 67), (34, 97, 68), (35, 98, 68), (36, 100, 68),
    (38, 101, 68), (39, 103, 68), (40, 104, 68), (41, 106, 68), (42, 107, 68), (44, 109, 68),
    (45, 110, 68), (46, 112, 68), (48, 113, 68), (50, 114, 69), (52, 114, 70), (54, 115, 71),
    (56, 116, 71), (58, 117, 72), (60, 118, 73), (62, 119, 73), (64, 120, 74), (66, 121, 75),
    (68, 122, 76), (70, 123, 76), (72, 124, 77), (74, 125, 78), (76, 126, 79), (78, 127, 79),
    (80, 128, 80), (82, 129, 81), (83, 129, 83), (83, 130, 85), (83, 130, 87), (83, 131, 90),
    (84, 131, 92), (84, 131, 94), (84, 132, 97), (84, 132, 99), (85, 133, 102), (85, 133, 104),
    (85, 134, 106), (85, 134, 109), (86, 135, 111), (86, 135, 113), (86, 136, 116), (86, 136, 118),
    (87, 137, 120), (87, 137, 123), (88, 137, 125), (90, 138, 126), (92, 138, 128), (94, 139, 129),
    (96, 140, 131), (98, 140, 133), (99, 141, 134), (101, 141, 136), (103, 142, 138), (105, 142, 139),
    (107, 143, 141), (109, 143, 142), (111, 144, 144), (113, 144, 146), (115, 145, 147), (117, 145, 149),
    (119, 146, 150), (120, 146, 152), (122, 147, 153), (125, 148, 153), (127, 150, 152), (129, 151, 152),
    (132, 153, 151), (134, 154, 151), (136, 156, 150), (139, 157, 150), (141, 159, 150), (143, 160, 149),
    (145, 162, 149), (148, 163, 148), (150, 165, 148), (152, 166, 147), (155, 168, 147), (157, 169, 147),
    (159, 171, 146), (161, 172, 146), (164, 174, 145), (165, 175, 145), (165, 177, 145), (166, 178, 145),
    (167, 180, 145), (168, 181, 145), (169, 183, 145), (170, 185, 145), (171, 186, 145), (172, 188, 145),
    (172, 189, 144), (173, 191, 144), (174, 193, 144), (175, 194, 144), (176, 196, 144), (177, 197, 144),
    (178, 199, 144), (179, 200, 144), (179, 202, 144), (179, 203, 145), (179, 204, 147), (179, 205, 148),
    (179, 206, 150), (179, 207, 151), (179, 208, 152), (178, 209, 154), (178, 210, 155), (178, 211, 157),
    (178, 212, 158), (178, 213, 160), (178, 214, 161), (178, 215, 163), (177, 216, 164), (177, 217, 166),
    (177, 218, 167), (177, 219, 169), (177, 220, 170), (176, 220, 173), (176, 221, 175), (175, 221, 178),
    (175, 222, 180), (174, 222, 183), (174, 223, 185), (174, 223, 188), (173, 224, 191), (173, 224, 193),
    (172, 225, 196), (172, 225, 198), (171, 226, 201), (171, 226, 204), (170, 226, 206), (170, 227, 209),
    (169, 227, 211), (169, 228, 214), (168, 228, 217), (169, 229, 219), (170, 229, 221), (172, 229, 223),
    (174, 230, 225), (175, 230, 227), (177, 231, 229), (179, 231, 231), (181, 231, 233), (182, 232, 235),
    (184, 232, 237), (186, 232, 239), (187, 233, 241), (189, 233, 243), (191, 233, 245), (193, 234, 247),
    (194, 234, 249), (196, 234, 251), (198, 235, 253), (200, 235, 255), (203, 236, 255), (206, 237, 255),
    (209, 238, 255), (212, 240, 255), (215, 241, 255), (218, 242, 255), (221, 243, 255), (224, 244, 255),
    (227, 245, 255), (230, 246, 255), (233, 247, 255), (237, 248, 255), (240, 249, 255), (243, 251, 255),
    (246, 252, 255), (249, 253, 255), (252, 254, 255), (255, 255, 255)]
colormap_earth_rgb = [(c[2], c[1], c[0]) for c in colormap_earth_bgr]

colormap_gray = [
    (0, 0, 0), (1, 1, 1), (2, 2, 2), (3, 3, 3), (4, 4, 4), (5, 5, 5),
    (6, 6, 6), (7, 7, 7), (8, 8, 8), (9, 9, 9), (10, 10, 10), (11, 11, 11),
    (12, 12, 12), (13, 13, 13), (14, 14, 14), (15, 15, 15), (16, 16, 16), (17, 17, 17),
    (18, 18, 18), (19, 19, 19), (20, 20, 20), (21, 21, 21), (22, 22, 22), (23, 23, 23),
    (24, 24, 24), (25, 25, 25), (26, 26, 26), (27, 27, 27), (28, 28, 28), (29, 29, 29),
    (30, 30, 30), (31, 31, 31), (32, 32, 32), (33, 33, 33), (34, 34, 34), (35, 35, 35),
    (36, 36, 36), (37, 37, 37), (38, 38, 38), (39, 39, 39), (40, 40, 40), (41, 41, 41),
    (42, 42, 42), (43, 43, 43), (44, 44, 44), (45, 45, 45), (46, 46, 46), (47, 47, 47),
    (48, 48, 48), (49, 49, 49), (50, 50, 50), (51, 51, 51), (52, 52, 52), (53, 53, 53),
    (54, 54, 54), (55, 55, 55), (56, 56, 56), (57, 57, 57), (58, 58, 58), (59, 59, 59),
    (60, 60, 60), (61, 61, 61), (62, 62, 62), (63, 63, 63), (64, 64, 64), (65, 65, 65),
    (66, 66, 66), (67, 67, 67), (68, 68, 68), (69, 69, 69), (70, 70, 70), (71, 71, 71),
    (72, 72, 72), (73, 73, 73), (74, 74, 74), (75, 75, 75), (76, 76, 76), (77, 77, 77),
    (78, 78, 78), (79, 79, 79), (80, 80, 80), (81, 81, 81), (82, 82, 82), (83, 83, 83),
    (84, 84, 84), (85, 85, 85), (86, 86, 86), (87, 87, 87), (88, 88, 88), (89, 89, 89),
    (90, 90, 90), (91, 91, 91), (92, 92, 92), (93, 93, 93), (94, 94, 94), (95, 95, 95),
    (96, 96, 96), (97, 97, 97), (98, 98, 98), (99, 99, 99), (100, 100, 100), (101, 101, 101),
    (102, 102, 102), (103, 103, 103), (104, 104, 104), (105, 105, 105), (106, 106, 106), (107, 107, 107),
    (108, 108, 108), (109, 109, 109), (110, 110, 110), (111, 111, 111), (112, 112, 112), (113, 113, 113),
    (114, 114, 114), (115, 115, 115), (116, 116, 116), (117, 117, 117), (118, 118, 118), (119, 119, 119),
    (120, 120, 120), (121, 121, 121), (122, 122, 122), (123, 123, 123), (124, 124, 124), (125, 125, 125),
    (126, 126, 126), (127, 127, 127), (128, 128, 128), (129, 129, 129), (130, 130, 130), (131, 131, 131),
    (132, 132, 132), (133, 133, 133), (134, 134, 134), (135, 135, 135), (136, 136, 136), (137, 137, 137),
    (138, 138, 138), (139, 139, 139), (140, 140, 140), (141, 141, 141), (142, 142, 142), (143, 143, 143),
    (144, 144, 144), (145, 145, 145), (146, 146, 146), (147, 147, 147), (148, 148, 148), (149, 149, 149),
    (150, 150, 150), (151, 151, 151), (152, 152, 152), (153, 153, 153), (154, 154, 154), (155, 155, 155),
    (156, 156, 156), (157, 157, 157), (158, 158, 158), (159, 159, 159), (160, 160, 160), (161, 161, 161),
    (162, 162, 162), (163, 163, 163), (164, 164, 164), (165, 165, 165), (166, 166, 166), (167, 167, 167),
    (168, 168, 168), (169, 169, 169), (170, 170, 170), (171, 171, 171), (172, 172, 172), (173, 173, 173),
    (174, 174, 174), (175, 175, 175), (176, 176, 176), (177, 177, 177), (178, 178, 178), (179, 179, 179),
    (180, 180, 180), (181, 181, 181), (182, 182, 182), (183, 183, 183), (184, 184, 184), (185, 185, 185),
    (186, 186, 186), (187, 187, 187), (188, 188, 188), (189, 189, 189), (190, 190, 190), (191, 191, 191),
    (192, 192, 192), (193, 193, 193), (194, 194, 194), (195, 195, 195), (196, 196, 196), (197, 197, 197),
    (198, 198, 198), (199, 199, 199), (200, 200, 200), (201, 201, 201), (202, 202, 202), (203, 203, 203),
    (204, 204, 204), (205, 205, 205), (206, 206, 206), (207, 207, 207), (208, 208, 208), (209, 209, 209),
    (210, 210, 210), (211, 211, 211), (212, 212, 212), (213, 213, 213), (214, 214, 214), (215, 215, 215),
    (216, 216, 216), (217, 217, 217), (218, 218, 218), (219, 219, 219), (220, 220, 220), (221, 221, 221),
    (222, 222, 222), (223, 223, 223), (224, 224, 224), (225, 225, 225), (226, 226, 226), (227, 227, 227),
    (228, 228, 228), (229, 229, 229), (230, 230, 230), (231, 231, 231), (232, 232, 232), (233, 233, 233),
    (234, 234, 234), (235, 235, 235), (236, 236, 236), (237, 237, 237), (238, 238, 238), (239, 239, 239),
    (240, 240, 240), (241, 241, 241), (242, 242, 242), (243, 243, 243), (244, 244, 244), (245, 245, 245),
    (246, 246, 246), (247, 247, 247), (248, 248, 248), (249, 249, 249), (250, 250, 250), (251, 251, 251),
    (252, 252, 252), (253, 253, 253), (254, 254, 254), (255, 255, 255)]

colormap_hot_bgr = [
    (0, 0, 0), (0, 0, 3), (0, 0, 5), (0, 0, 8), (0, 0, 11), (0, 0, 13),
    (0, 0, 16), (0, 0, 19), (0, 0, 21), (0, 0, 24), (0, 0, 27), (0, 0, 29),
    (0, 0, 32), (0, 0, 35), (0, 0, 37), (0, 0, 40), (0, 0, 43), (0, 0, 45),
    (0, 0, 48), (0, 0, 51), (0, 0, 53), (0, 0, 56), (0, 0, 59), (0, 0, 61),
    (0, 0, 64), (0, 0, 67), (0, 0, 69), (0, 0, 72), (0, 0, 75), (0, 0, 77),
    (0, 0, 80), (0, 0, 83), (0, 0, 85), (0, 0, 88), (0, 0, 91), (0, 0, 93),
    (0, 0, 96), (0, 0, 99), (0, 0, 101), (0, 0, 104), (0, 0, 107), (0, 0, 109),
    (0, 0, 112), (0, 0, 115), (0, 0, 117), (0, 0, 120), (0, 0, 123), (0, 0, 125),
    (0, 0, 128), (0, 0, 131), (0, 0, 133), (0, 0, 136), (0, 0, 139), (0, 0, 141),
    (0, 0, 144), (0, 0, 147), (0, 0, 149), (0, 0, 152), (0, 0, 155), (0, 0, 157),
    (0, 0, 160), (0, 0, 163), (0, 0, 165), (0, 0, 168), (0, 0, 171), (0, 0, 173),
    (0, 0, 176), (0, 0, 179), (0, 0, 181), (0, 0, 184), (0, 0, 187), (0, 0, 189),
    (0, 0, 192), (0, 0, 195), (0, 0, 197), (0, 0, 200), (0, 0, 203), (0, 0, 205),
    (0, 0, 208), (0, 0, 211), (0, 0, 213), (0, 0, 216), (0, 0, 219), (0, 0, 221),
    (0, 0, 224), (0, 0, 227), (0, 0, 229), (0, 0, 232), (0, 0, 235), (0, 0, 237),
    (0, 0, 240), (0, 0, 243), (0, 0, 245), (0, 0, 248), (0, 0, 251), (0, 0, 253),
    (0, 1, 255), (0, 4, 255), (0, 6, 255), (0, 9, 255), (0, 12, 255), (0, 14, 255),
    (0, 17, 255), (0, 20, 255), (0, 22, 255), (0, 25, 255), (0, 28, 255), (0, 30, 255),
    (0, 33, 255), (0, 36, 255), (0, 38, 255), (0, 41, 255), (0, 44, 255), (0, 46, 255),
    (0, 49, 255), (0, 52, 255), (0, 54, 255), (0, 57, 255), (0, 60, 255), (0, 62, 255),
    (0, 65, 255), (0, 68, 255), (0, 70, 255), (0, 73, 255), (0, 76, 255), (0, 78, 255),
    (0, 81, 255), (0, 84, 255), (0, 86, 255), (0, 89, 255), (0, 92, 255), (0, 94, 255),
    (0, 97, 255), (0, 100, 255), (0, 102, 255), (0, 105, 255), (0, 108, 255), (0, 110, 255),
    (0, 113, 255), (0, 116, 255), (0, 118, 255), (0, 121, 255), (0, 124, 255), (0, 126, 255),
    (0, 129, 255), (0, 132, 255), (0, 134, 255), (0, 137, 255), (0, 140, 255), (0, 142, 255),
    (0, 145, 255), (0, 148, 255), (0, 150, 255), (0, 153, 255), (0, 156, 255), (0, 158, 255),
    (0, 161, 255), (0, 164, 255), (0, 166, 255), (0, 169, 255), (0, 172, 255), (0, 174, 255),
    (0, 177, 255), (0, 180, 255), (0, 182, 255), (0, 185, 255), (0, 188, 255), (0, 190, 255),
    (0, 193, 255), (0, 196, 255), (0, 198, 255), (0, 201, 255), (0, 204, 255), (0, 206, 255),
    (0, 209, 255), (0, 212, 255), (0, 214, 255), (0, 217, 255), (0, 220, 255), (0, 222, 255),
    (0, 225, 255), (0, 228, 255), (0, 230, 255), (0, 233, 255), (0, 236, 255), (0, 238, 255),
    (0, 241, 255), (0, 244, 255), (0, 246, 255), (0, 249, 255), (0, 252, 255), (0, 254, 255),
    (3, 255, 255), (7, 255, 255), (11, 255, 255), (15, 255, 255), (19, 255, 255), (23, 255, 255),
    (27, 255, 255), (31, 255, 255), (35, 255, 255), (39, 255, 255), (43, 255, 255), (47, 255, 255),
    (51, 255, 255), (55, 255, 255), (59, 255, 255), (63, 255, 255), (67, 255, 255), (71, 255, 255),
    (75, 255, 255), (79, 255, 255), (83, 255, 255), (87, 255, 255), (91, 255, 255), (95, 255, 255),
    (99, 255, 255), (103, 255, 255), (107, 255, 255), (111, 255, 255), (115, 255, 255), (119, 255, 255),
    (123, 255, 255), (127, 255, 255), (131, 255, 255), (135, 255, 255), (139, 255, 255), (143, 255, 255),
    (147, 255, 255), (151, 255, 255), (155, 255, 255), (159, 255, 255), (163, 255, 255), (167, 255, 255),
    (171, 255, 255), (175, 255, 255), (179, 255, 255), (183, 255, 255), (187, 255, 255), (191, 255, 255),
    (195, 255, 255), (199, 255, 255), (203, 255, 255), (207, 255, 255), (211, 255, 255), (215, 255, 255),
    (219, 255, 255), (223, 255, 255), (227, 255, 255), (231, 255, 255), (235, 255, 255), (239, 255, 255),
    (243, 255, 255), (247, 255, 255), (251, 255, 255), (255, 255, 255)]
colormap_hot_rgb = [(c[2], c[1], c[0]) for c in colormap_hot_bgr]

colormap_hsv_bgr = [
    (0, 0, 255), (0, 6, 255), (0, 12, 255), (0, 18, 255), (0, 24, 255), (0, 30, 255),
    (0, 36, 255), (0, 42, 255), (0, 48, 255), (0, 54, 255), (0, 60, 255), (0, 66, 255),
    (0, 72, 255), (0, 78, 255), (0, 84, 255), (0, 90, 255), (0, 96, 255), (0, 102, 255),
    (0, 108, 255), (0, 114, 255), (0, 120, 255), (0, 126, 255), (0, 132, 255), (0, 138, 255),
    (0, 144, 255), (0, 150, 255), (0, 156, 255), (0, 162, 255), (0, 168, 255), (0, 174, 255),
    (0, 180, 255), (0, 186, 255), (0, 192, 255), (0, 198, 255), (0, 204, 255), (0, 210, 255),
    (0, 216, 255), (0, 222, 255), (0, 228, 255), (0, 234, 255), (0, 240, 255), (0, 246, 255),
    (0, 252, 255), (0, 255, 252), (0, 255, 246), (0, 255, 240), (0, 255, 234), (0, 255, 228),
    (0, 255, 222), (0, 255, 216), (0, 255, 210), (0, 255, 204), (0, 255, 198), (0, 255, 192),
    (0, 255, 186), (0, 255, 180), (0, 255, 174), (0, 255, 168), (0, 255, 162), (0, 255, 156),
    (0, 255, 150), (0, 255, 144), (0, 255, 138), (0, 255, 132), (0, 255, 126), (0, 255, 120),
    (0, 255, 114), (0, 255, 108), (0, 255, 102), (0, 255, 96), (0, 255, 90), (0, 255, 84),
    (0, 255, 78), (0, 255, 72), (0, 255, 66), (0, 255, 60), (0, 255, 54), (0, 255, 48),
    (0, 255, 42), (0, 255, 36), (0, 255, 30), (0, 255, 24), (0, 255, 18), (0, 255, 12),
    (0, 255, 6), (0, 255, 0), (6, 255, 0), (12, 255, 0), (18, 255, 0), (24, 255, 0),
    (30, 255, 0), (36, 255, 0), (42, 255, 0), (48, 255, 0), (54, 255, 0), (60, 255, 0),
    (66, 255, 0), (72, 255, 0), (78, 255, 0), (84, 255, 0), (90, 255, 0), (96, 255, 0),
    (102, 255, 0), (108, 255, 0), (114, 255, 0), (120, 255, 0), (126, 255, 0), (132, 255, 0),
    (138, 255, 0), (144, 255, 0), (150, 255, 0), (156, 255, 0), (162, 255, 0), (168, 255, 0),
    (174, 255, 0), (180, 255, 0), (186, 255, 0), (192, 255, 0), (198, 255, 0), (204, 255, 0),
    (210, 255, 0), (216, 255, 0), (222, 255, 0), (228, 255, 0), (234, 255, 0), (240, 255, 0),
    (246, 255, 0), (252, 255, 0), (255, 252, 0), (255, 246, 0), (255, 240, 0), (255, 234, 0),
    (255, 228, 0), (255, 222, 0), (255, 216, 0), (255, 210, 0), (255, 204, 0), (255, 198, 0),
    (255, 192, 0), (255, 186, 0), (255, 180, 0), (255, 174, 0), (255, 168, 0), (255, 162, 0),
    (255, 156, 0), (255, 150, 0), (255, 144, 0), (255, 138, 0), (255, 132, 0), (255, 126, 0),
    (255, 120, 0), (255, 114, 0), (255, 108, 0), (255, 102, 0), (255, 96, 0), (255, 90, 0),
    (255, 84, 0), (255, 78, 0), (255, 72, 0), (255, 66, 0), (255, 60, 0), (255, 54, 0),
    (255, 48, 0), (255, 42, 0), (255, 36, 0), (255, 30, 0), (255, 24, 0), (255, 18, 0),
    (255, 12, 0), (255, 6, 0), (255, 0, 0), (255, 0, 6), (255, 0, 12), (255, 0, 18),
    (255, 0, 24), (255, 0, 30), (255, 0, 36), (255, 0, 42), (255, 0, 48), (255, 0, 54),
    (255, 0, 60), (255, 0, 66), (255, 0, 72), (255, 0, 78), (255, 0, 84), (255, 0, 90),
    (255, 0, 96), (255, 0, 102), (255, 0, 108), (255, 0, 114), (255, 0, 120), (255, 0, 126),
    (255, 0, 132), (255, 0, 138), (255, 0, 144), (255, 0, 150), (255, 0, 156), (255, 0, 162),
    (255, 0, 168), (255, 0, 174), (255, 0, 180), (255, 0, 186), (255, 0, 192), (255, 0, 198),
    (255, 0, 204), (255, 0, 210), (255, 0, 216), (255, 0, 222), (255, 0, 228), (255, 0, 234),
    (255, 0, 240), (255, 0, 246), (255, 0, 252), (252, 0, 255), (246, 0, 255), (240, 0, 255),
    (234, 0, 255), (228, 0, 255), (222, 0, 255), (216, 0, 255), (210, 0, 255), (204, 0, 255),
    (198, 0, 255), (192, 0, 255), (186, 0, 255), (180, 0, 255), (174, 0, 255), (168, 0, 255),
    (162, 0, 255), (156, 0, 255), (150, 0, 255), (144, 0, 255), (138, 0, 255), (132, 0, 255),
    (126, 0, 255), (120, 0, 255), (114, 0, 255), (108, 0, 255), (102, 0, 255), (96, 0, 255),
    (90, 0, 255), (84, 0, 255), (78, 0, 255), (72, 0, 255), (66, 0, 255), (60, 0, 255),
    (54, 0, 255), (48, 0, 255), (42, 0, 255), (36, 0, 255), (30, 0, 255), (24, 0, 255),
    (18, 0, 255), (12, 0, 255), (6, 0, 255), (0, 0, 255)]
colormap_hsv_rgb = [(c[2], c[1], c[0]) for c in colormap_hsv_bgr]

colormap_inferno_bgr = [
    (4, 0, 0), (5, 0, 1), (6, 1, 1), (8, 1, 1), (10, 1, 2), (12, 2, 2),
    (14, 2, 2), (16, 2, 3), (18, 3, 4), (20, 3, 4), (23, 4, 5), (25, 4, 6),
    (27, 5, 7), (29, 5, 8), (31, 6, 9), (34, 7, 10), (36, 7, 11), (38, 8, 12),
    (41, 8, 13), (43, 9, 14), (45, 9, 16), (48, 10, 17), (50, 10, 18), (52, 11, 20),
    (55, 11, 21), (57, 11, 22), (60, 12, 24), (62, 12, 25), (65, 12, 27), (67, 12, 28),
    (69, 12, 30), (72, 12, 31), (74, 12, 33), (76, 12, 35), (79, 12, 36), (81, 12, 38),
    (83, 11, 40), (85, 11, 41), (87, 11, 43), (89, 11, 45), (91, 10, 47), (92, 10, 49),
    (94, 10, 50), (95, 10, 52), (97, 9, 54), (98, 9, 56), (99, 9, 57), (100, 9, 59),
    (101, 9, 61), (102, 9, 62), (103, 10, 64), (104, 10, 66), (104, 10, 68), (105, 10, 69),
    (106, 11, 71), (106, 11, 73), (107, 12, 74), (107, 12, 76), (108, 13, 77), (108, 13, 79),
    (108, 14, 81), (109, 14, 82), (109, 15, 84), (109, 15, 85), (110, 16, 87), (110, 16, 89),
    (110, 17, 90), (110, 18, 92), (110, 18, 93), (110, 19, 95), (110, 19, 97), (110, 20, 98),
    (110, 21, 100), (110, 21, 101), (110, 22, 103), (110, 22, 105), (110, 23, 106), (110, 24, 108),
    (110, 24, 109), (110, 25, 111), (110, 25, 113), (110, 26, 114), (110, 26, 116), (110, 27, 117),
    (109, 28, 119), (109, 28, 120), (109, 29, 122), (109, 29, 124), (109, 30, 125), (108, 30, 127),
    (108, 31, 128), (108, 32, 130), (107, 32, 132), (107, 33, 133), (107, 33, 135), (106, 34, 136),
    (106, 34, 138), (105, 35, 140), (105, 35, 141), (105, 36, 143), (104, 37, 144), (104, 37, 146),
    (103, 38, 147), (103, 38, 149), (102, 39, 151), (102, 39, 152), (101, 40, 154), (100, 41, 155),
    (100, 41, 157), (99, 42, 159), (99, 42, 160), (98, 43, 162), (97, 44, 163), (96, 44, 165),
    (96, 45, 166), (95, 46, 168), (94, 46, 169), (94, 47, 171), (93, 48, 173), (92, 48, 174),
    (91, 49, 176), (90, 50, 177), (90, 50, 179), (89, 51, 180), (88, 52, 182), (87, 53, 183),
    (86, 53, 185), (85, 54, 186), (84, 55, 188), (83, 56, 189), (82, 57, 191), (81, 58, 192),
    (80, 58, 193), (79, 59, 195), (78, 60, 196), (77, 61, 198), (76, 62, 199), (75, 63, 200),
    (74, 64, 202), (73, 65, 203), (72, 66, 204), (71, 67, 206), (70, 68, 207), (69, 69, 208),
    (68, 70, 210), (67, 71, 211), (66, 72, 212), (65, 74, 213), (63, 75, 215), (62, 76, 216),
    (61, 77, 217), (60, 78, 218), (59, 80, 219), (58, 81, 221), (56, 82, 222), (55, 83, 223),
    (54, 85, 224), (53, 86, 225), (52, 87, 226), (51, 89, 227), (49, 90, 228), (48, 92, 229),
    (47, 93, 230), (46, 94, 231), (45, 96, 232), (43, 97, 233), (42, 99, 234), (41, 100, 235),
    (40, 102, 235), (38, 103, 236), (37, 105, 237), (36, 106, 238), (35, 108, 239), (33, 110, 239),
    (32, 111, 240), (31, 113, 241), (29, 115, 241), (28, 116, 242), (27, 118, 243), (25, 120, 243),
    (24, 121, 244), (23, 123, 245), (21, 125, 245), (20, 126, 246), (19, 128, 246), (18, 130, 247),
    (16, 132, 247), (15, 133, 248), (14, 135, 248), (12, 137, 248), (11, 139, 249), (10, 140, 249),
    (9, 142, 249), (8, 144, 250), (7, 146, 250), (7, 148, 250), (6, 150, 251), (6, 151, 251),
    (6, 153, 251), (6, 155, 251), (7, 157, 251), (7, 159, 252), (8, 161, 252), (9, 163, 252),
    (10, 165, 252), (12, 166, 252), (13, 168, 252), (15, 170, 252), (17, 172, 252), (18, 174, 252),
    (20, 176, 252), (22, 178, 252), (24, 180, 252), (26, 182, 251), (29, 184, 251), (31, 186, 251),
    (33, 188, 251), (35, 190, 251), (38, 192, 250), (40, 194, 250), (42, 196, 250), (45, 198, 250),
    (47, 199, 249), (50, 201, 249), (53, 203, 249), (55, 205, 248), (58, 207, 248), (61, 209, 247),
    (64, 211, 247), (67, 213, 246), (70, 215, 246), (73, 217, 245), (76, 219, 245), (79, 221, 244),
    (83, 223, 244), (86, 225, 244), (90, 227, 243), (93, 229, 243), (97, 230, 242), (101, 232, 242),
    (105, 234, 242), (109, 236, 241), (113, 237, 241), (117, 239, 241), (121, 241, 241), (125, 242, 242),
    (130, 244, 242), (134, 245, 243), (138, 246, 243), (142, 248, 244), (146, 249, 245), (150, 250, 246),
    (154, 251, 248), (157, 252, 249), (161, 253, 250), (164, 255, 252)]
colormap_inferno_rgb = [(c[2], c[1], c[0]) for c in colormap_inferno_bgr]

colormap_jet_bgr = [
    (128, 0, 0), (132, 0, 0), (136, 0, 0), (140, 0, 0), (144, 0, 0), (147, 0, 0),
    (152, 0, 0), (156, 0, 0), (160, 0, 0), (163, 0, 0), (168, 0, 0), (172, 0, 0),
    (176, 0, 0), (179, 0, 0), (184, 0, 0), (188, 0, 0), (192, 0, 0), (195, 0, 0),
    (200, 0, 0), (204, 0, 0), (208, 0, 0), (211, 0, 0), (216, 0, 0), (220, 0, 0),
    (224, 0, 0), (227, 0, 0), (232, 0, 0), (236, 0, 0), (240, 0, 0), (243, 0, 0),
    (248, 0, 0), (252, 0, 0), (255, 0, 0), (255, 4, 0), (255, 8, 0), (255, 13, 0),
    (255, 16, 0), (255, 20, 0), (255, 24, 0), (255, 29, 0), (255, 33, 0), (255, 36, 0),
    (255, 41, 0), (255, 45, 0), (255, 49, 0), (255, 52, 0), (255, 57, 0), (255, 61, 0),
    (255, 65, 0), (255, 68, 0), (255, 73, 0), (255, 77, 0), (255, 81, 0), (255, 84, 0),
    (255, 89, 0), (255, 93, 0), (255, 97, 0), (255, 100, 0), (255, 105, 0), (255, 109, 0),
    (255, 113, 0), (255, 116, 0), (255, 121, 0), (255, 125, 0), (255, 129, 0), (255, 132, 0),
    (255, 136, 0), (255, 141, 0), (255, 145, 0), (255, 148, 0), (255, 153, 0), (255, 157, 0),
    (255, 161, 0), (255, 164, 0), (255, 168, 0), (255, 173, 0), (255, 177, 0), (255, 180, 0),
    (255, 185, 0), (255, 189, 0), (255, 193, 0), (255, 196, 0), (255, 200, 0), (255, 205, 0),
    (255, 209, 0), (255, 212, 0), (255, 217, 0), (255, 221, 0), (255, 225, 0), (255, 228, 0),
    (255, 232, 0), (255, 237, 0), (255, 241, 0), (255, 244, 0), (255, 249, 0), (255, 253, 0),
    (254, 255, 1), (250, 255, 5), (246, 255, 9), (242, 255, 14), (238, 255, 17), (234, 255, 21),
    (229, 255, 26), (226, 255, 30), (222, 255, 33), (218, 255, 37), (214, 255, 41), (210, 255, 46),
    (206, 255, 49), (202, 255, 53), (197, 255, 58), (194, 255, 62), (190, 255, 66), (186, 255, 69),
    (182, 255, 73), (178, 255, 78), (174, 255, 82), (170, 255, 85), (165, 255, 90), (162, 255, 94),
    (158, 255, 98), (154, 255, 101), (150, 255, 105), (146, 255, 110), (142, 255, 114), (138, 255, 117),
    (133, 255, 122), (130, 255, 126), (126, 255, 130), (122, 255, 133), (118, 255, 137), (114, 255, 141),
    (110, 255, 145), (105, 255, 150), (101, 255, 154), (98, 255, 158), (94, 255, 162), (90, 255, 165),
    (86, 255, 169), (82, 255, 173), (77, 255, 178), (73, 255, 182), (69, 255, 186), (66, 255, 190),
    (62, 255, 194), (58, 255, 197), (54, 255, 201), (50, 255, 205), (46, 255, 209), (41, 255, 214),
    (37, 255, 218), (33, 255, 222), (30, 255, 226), (26, 255, 229), (22, 255, 233), (18, 255, 237),
    (13, 255, 242), (9, 255, 246), (5, 255, 250), (1, 255, 254), (0, 253, 255), (0, 249, 255),
    (0, 245, 255), (0, 241, 255), (0, 237, 255), (0, 232, 255), (0, 228, 255), (0, 225, 255),
    (0, 221, 255), (0, 217, 255), (0, 213, 255), (0, 209, 255), (0, 204, 255), (0, 200, 255),
    (0, 196, 255), (0, 193, 255), (0, 189, 255), (0, 185, 255), (0, 181, 255), (0, 177, 255),
    (0, 173, 255), (0, 168, 255), (0, 164, 255), (0, 161, 255), (0, 157, 255), (0, 153, 255),
    (0, 149, 255), (0, 145, 255), (0, 140, 255), (0, 136, 255), (0, 132, 255), (0, 129, 255),
    (0, 125, 255), (0, 121, 255), (0, 117, 255), (0, 113, 255), (0, 109, 255), (0, 104, 255),
    (0, 100, 255), (0, 97, 255), (0, 93, 255), (0, 89, 255), (0, 85, 255), (0, 81, 255),
    (0, 76, 255), (0, 72, 255), (0, 68, 255), (0, 65, 255), (0, 61, 255), (0, 57, 255),
    (0, 53, 255), (0, 49, 255), (0, 45, 255), (0, 40, 255), (0, 36, 255), (0, 33, 255),
    (0, 29, 255), (0, 25, 255), (0, 21, 255), (0, 17, 255), (0, 12, 255), (0, 8, 255),
    (0, 4, 255), (0, 0, 255), (0, 0, 252), (0, 0, 248), (0, 0, 244), (0, 0, 240),
    (0, 0, 236), (0, 0, 231), (0, 0, 227), (0, 0, 224), (0, 0, 220), (0, 0, 216),
    (0, 0, 212), (0, 0, 208), (0, 0, 203), (0, 0, 199), (0, 0, 195), (0, 0, 192),
    (0, 0, 188), (0, 0, 184), (0, 0, 180), (0, 0, 176), (0, 0, 172), (0, 0, 167),
    (0, 0, 163), (0, 0, 160), (0, 0, 156), (0, 0, 152), (0, 0, 148), (0, 0, 144),
    (0, 0, 139), (0, 0, 135), (0, 0, 132), (0, 0, 128)]
colormap_jet_rgb = [(c[2], c[1], c[0]) for c in colormap_jet_bgr]

colormap_magma_bgr = [
    (4, 0, 0), (5, 0, 1), (6, 1, 1), (8, 1, 1), (9, 1, 2), (11, 2, 2),
    (13, 2, 2), (15, 3, 3), (18, 3, 3), (20, 4, 4), (22, 4, 5), (24, 5, 6),
    (26, 5, 6), (28, 6, 7), (30, 7, 8), (32, 7, 9), (34, 8, 10), (36, 9, 11),
    (38, 9, 12), (41, 10, 13), (43, 11, 14), (45, 11, 16), (47, 12, 17), (49, 13, 18),
    (52, 13, 19), (54, 14, 20), (56, 14, 21), (59, 15, 22), (61, 15, 24), (63, 16, 25),
    (66, 16, 26), (68, 16, 28), (71, 17, 29), (73, 17, 30), (75, 17, 32), (78, 17, 33),
    (80, 17, 34), (83, 18, 36), (85, 18, 37), (88, 18, 39), (90, 17, 41), (92, 17, 42),
    (95, 17, 44), (97, 17, 45), (99, 17, 47), (101, 17, 49), (103, 16, 51), (105, 16, 52),
    (107, 16, 54), (108, 16, 56), (110, 15, 57), (112, 15, 59), (113, 15, 61), (114, 15, 63),
    (116, 15, 64), (117, 15, 66), (118, 15, 68), (119, 16, 69), (120, 16, 71), (120, 16, 73),
    (121, 16, 74), (122, 17, 76), (123, 17, 78), (123, 18, 79), (124, 18, 81), (124, 19, 82),
    (125, 19, 84), (125, 20, 86), (126, 21, 87), (126, 21, 89), (126, 22, 90), (127, 22, 92),
    (127, 23, 93), (127, 24, 95), (128, 24, 96), (128, 25, 98), (128, 26, 100), (128, 26, 101),
    (128, 27, 103), (129, 28, 104), (129, 28, 106), (129, 29, 107), (129, 29, 109), (129, 30, 110),
    (129, 31, 112), (129, 31, 114), (129, 32, 115), (129, 33, 117), (129, 33, 118), (129, 34, 120),
    (130, 34, 121), (130, 35, 123), (130, 35, 124), (130, 36, 126), (130, 37, 128), (129, 37, 129),
    (129, 38, 131), (129, 38, 132), (129, 39, 134), (129, 39, 136), (129, 40, 137), (129, 41, 139),
    (129, 41, 140), (129, 42, 142), (129, 42, 144), (129, 43, 145), (128, 43, 147), (128, 44, 148),
    (128, 44, 150), (128, 45, 152), (128, 45, 153), (127, 46, 155), (127, 46, 156), (127, 47, 158),
    (127, 47, 160), (126, 48, 161), (126, 48, 163), (126, 49, 165), (125, 49, 166), (125, 50, 168),
    (125, 51, 170), (124, 51, 171), (124, 52, 173), (123, 52, 174), (123, 53, 176), (123, 53, 178),
    (122, 54, 179), (122, 54, 181), (121, 55, 183), (121, 55, 184), (120, 56, 186), (120, 57, 188),
    (119, 57, 189), (119, 58, 191), (118, 58, 192), (117, 59, 194), (117, 60, 196), (116, 60, 197),
    (115, 61, 199), (115, 62, 200), (114, 62, 202), (113, 63, 204), (113, 64, 205), (112, 64, 207),
    (111, 65, 208), (111, 66, 210), (110, 67, 211), (109, 68, 213), (108, 69, 214), (108, 69, 216),
    (107, 70, 217), (106, 71, 219), (105, 72, 220), (104, 73, 222), (104, 74, 223), (103, 76, 224),
    (102, 77, 226), (101, 78, 227), (100, 79, 228), (100, 80, 229), (99, 82, 231), (98, 83, 232),
    (98, 84, 233), (97, 86, 234), (96, 87, 235), (96, 88, 236), (95, 90, 237), (94, 91, 238),
    (94, 93, 239), (94, 95, 240), (93, 96, 241), (93, 98, 242), (92, 100, 242), (92, 101, 243),
    (92, 103, 244), (92, 105, 244), (92, 107, 245), (92, 108, 246), (92, 110, 246), (92, 112, 247),
    (92, 114, 247), (92, 116, 248), (92, 118, 248), (93, 120, 249), (93, 121, 249), (93, 123, 249),
    (94, 125, 250), (94, 127, 250), (95, 129, 250), (95, 131, 251), (96, 133, 251), (97, 135, 251),
    (97, 137, 252), (98, 138, 252), (99, 140, 252), (100, 142, 252), (101, 144, 252), (102, 146, 253),
    (103, 148, 253), (104, 150, 253), (105, 152, 253), (106, 154, 253), (107, 155, 253), (108, 157, 254),
    (109, 159, 254), (110, 161, 254), (111, 163, 254), (113, 165, 254), (114, 167, 254), (115, 169, 254),
    (116, 170, 254), (118, 172, 254), (119, 174, 254), (120, 176, 254), (122, 178, 254), (123, 180, 254),
    (124, 182, 254), (126, 183, 254), (127, 185, 254), (129, 187, 254), (130, 189, 254), (132, 191, 254),
    (133, 193, 254), (135, 194, 254), (136, 196, 254), (138, 198, 254), (140, 200, 254), (141, 202, 254),
    (143, 204, 254), (144, 205, 254), (146, 207, 254), (148, 209, 254), (149, 211, 254), (151, 213, 254),
    (153, 215, 254), (154, 216, 254), (156, 218, 253), (158, 220, 253), (160, 222, 253), (161, 224, 253),
    (163, 226, 253), (165, 227, 253), (167, 229, 253), (169, 231, 253), (170, 233, 253), (172, 235, 253),
    (174, 236, 252), (176, 238, 252), (178, 240, 252), (180, 242, 252), (182, 244, 252), (184, 246, 252),
    (185, 247, 252), (187, 249, 252), (189, 251, 252), (191, 253, 252)]
colormap_magma_rgb = [(c[2], c[1], c[0]) for c in colormap_magma_bgr]

colormap_parula_bgr = [
    (168, 38, 62), (172, 39, 62), (175, 40, 63), (178, 41, 63), (180, 42, 64), (183, 43, 64),
    (186, 44, 65), (189, 45, 65), (191, 46, 66), (194, 47, 66), (197, 48, 67), (200, 49, 67),
    (202, 50, 67), (205, 51, 68), (208, 52, 68), (210, 53, 69), (213, 55, 69), (215, 56, 69),
    (217, 57, 70), (220, 58, 70), (222, 59, 70), (224, 61, 70), (225, 62, 71), (227, 63, 71),
    (229, 65, 71), (230, 66, 71), (232, 68, 71), (233, 69, 71), (235, 70, 71), (236, 72, 72),
    (237, 73, 72), (238, 75, 72), (240, 76, 72), (241, 78, 72), (242, 79, 72), (243, 80, 72),
    (244, 82, 72), (245, 83, 72), (246, 84, 72), (247, 86, 71), (247, 87, 71), (248, 89, 71),
    (249, 90, 71), (250, 91, 71), (250, 93, 71), (251, 94, 70), (251, 96, 70), (252, 97, 70),
    (252, 98, 69), (253, 100, 69), (253, 101, 68), (253, 103, 67), (254, 104, 67), (254, 106, 66),
    (254, 107, 65), (254, 109, 64), (255, 110, 63), (255, 112, 62), (255, 113, 60), (255, 115, 59),
    (255, 116, 57), (254, 118, 56), (254, 119, 54), (253, 121, 53), (253, 122, 51), (252, 124, 50),
    (252, 125, 49), (251, 127, 48), (250, 128, 47), (250, 130, 47), (249, 131, 46), (248, 132, 46),
    (248, 134, 46), (247, 135, 46), (246, 136, 45), (245, 138, 45), (244, 139, 45), (243, 140, 45),
    (242, 142, 45), (241, 143, 44), (240, 144, 44), (239, 145, 43), (238, 147, 42), (237, 148, 41),
    (236, 149, 40), (235, 151, 39), (234, 152, 39), (233, 153, 38), (232, 154, 38), (232, 155, 37),
    (231, 156, 37), (230, 158, 36), (229, 159, 36), (229, 160, 35), (228, 161, 35), (228, 162, 34),
    (227, 163, 33), (227, 165, 32), (226, 166, 31), (225, 167, 30), (225, 168, 29), (224, 169, 29),
    (223, 170, 28), (222, 171, 27), (221, 172, 26), (220, 173, 25), (218, 174, 23), (217, 175, 22),
    (216, 176, 20), (214, 177, 18), (213, 178, 16), (212, 179, 14), (210, 179, 11), (209, 180, 8),
    (207, 181, 6), (206, 182, 4), (204, 183, 2), (202, 183, 1), (201, 184, 0), (199, 185, 0),
    (198, 186, 0), (196, 186, 1), (194, 187, 2), (193, 187, 4), (191, 188, 6), (189, 189, 9),
    (188, 189, 13), (186, 190, 16), (184, 190, 20), (182, 191, 23), (181, 192, 26), (179, 192, 29),
    (177, 193, 32), (175, 193, 35), (174, 194, 37), (172, 194, 39), (170, 195, 41), (168, 195, 43),
    (166, 196, 44), (165, 196, 46), (163, 197, 47), (161, 197, 49), (159, 198, 50), (157, 199, 51),
    (155, 199, 53), (153, 200, 54), (150, 200, 56), (148, 201, 57), (146, 201, 59), (144, 202, 61),
    (141, 202, 64), (139, 202, 66), (137, 203, 69), (134, 203, 72), (132, 203, 75), (129, 204, 78),
    (127, 204, 81), (124, 204, 84), (122, 204, 87), (119, 204, 90), (116, 205, 94), (114, 205, 97),
    (111, 205, 100), (108, 205, 103), (105, 205, 107), (102, 205, 110), (100, 205, 114), (97, 204, 118),
    (94, 204, 121), (91, 204, 125), (89, 204, 129), (86, 204, 132), (83, 203, 136), (81, 203, 139),
    (78, 203, 143), (75, 202, 147), (72, 202, 150), (70, 201, 154), (67, 201, 157), (64, 200, 161),
    (62, 200, 164), (59, 199, 167), (57, 199, 171), (55, 198, 174), (53, 198, 178), (51, 197, 181),
    (49, 196, 184), (47, 196, 187), (45, 195, 190), (44, 195, 194), (42, 194, 197), (41, 193, 200),
    (40, 193, 203), (39, 192, 206), (39, 191, 208), (39, 191, 211), (39, 190, 214), (40, 190, 217),
    (40, 189, 219), (41, 188, 222), (42, 188, 225), (43, 188, 227), (45, 187, 230), (46, 187, 232),
    (48, 186, 234), (50, 186, 236), (53, 186, 239), (55, 186, 241), (57, 186, 243), (59, 186, 245),
    (61, 186, 247), (62, 186, 249), (62, 187, 251), (62, 188, 252), (61, 189, 254), (60, 190, 254),
    (59, 192, 254), (58, 193, 254), (57, 194, 254), (56, 196, 254), (55, 197, 254), (53, 199, 254),
    (52, 200, 254), (51, 202, 254), (50, 203, 253), (49, 205, 253), (49, 206, 253), (48, 208, 252),
    (47, 210, 251), (46, 211, 251), (46, 213, 250), (45, 214, 249), (44, 216, 249), (43, 217, 248),
    (42, 219, 247), (42, 221, 247), (41, 222, 246), (40, 224, 246), (40, 225, 245), (39, 227, 245),
    (38, 229, 245), (38, 230, 245), (37, 232, 245), (36, 233, 245), (35, 235, 245), (34, 236, 245),
    (33, 238, 245), (32, 239, 246), (31, 241, 246), (30, 242, 246), (28, 244, 247), (27, 245, 247),
    (26, 247, 248), (24, 248, 248), (22, 249, 249), (21, 251, 249)]
colormap_parula_rgb = [(c[2], c[1], c[0]) for c in colormap_parula_bgr]

colormap_pastel_bgr = [
    (0, 0, 0), (0, 0, 3), (0, 0, 7), (0, 0, 10), (1, 0, 13), (1, 0, 16),
    (1, 0, 20), (1, 0, 23), (1, 0, 26), (1, 0, 30), (1, 0, 33), (1, 0, 36),
    (2, 0, 40), (2, 0, 43), (2, 0, 46), (2, 0, 49), (2, 0, 53), (2, 0, 56),
    (2, 0, 59), (2, 0, 63), (3, 0, 66), (3, 0, 69), (3, 0, 73), (3, 0, 76),
    (3, 0, 79), (3, 0, 82), (3, 0, 86), (3, 0, 89), (4, 0, 92), (4, 0, 96),
    (4, 0, 99), (4, 0, 102), (4, 0, 105), (4, 0, 109), (4, 0, 112), (4, 0, 115),
    (5, 0, 119), (7, 1, 118), (12, 4, 115), (16, 6, 112), (21, 9, 108), (26, 11, 105),
    (30, 14, 102), (35, 16, 98), (39, 19, 95), (44, 21, 92), (49, 24, 89), (53, 26, 85),
    (58, 29, 82), (62, 31, 79), (67, 34, 75), (72, 36, 72), (76, 39, 69), (81, 41, 65),
    (85, 44, 62), (90, 46, 59), (95, 49, 56), (99, 51, 52), (104, 54, 49), (108, 56, 46),
    (113, 59, 42), (118, 61, 39), (122, 64, 36), (127, 66, 32), (131, 69, 29), (136, 71, 26),
    (141, 74, 23), (145, 76, 19), (150, 79, 16), (154, 81, 13), (159, 84, 9), (164, 86, 6),
    (168, 89, 3), (172, 90, 1), (173, 89, 7), (175, 87, 13), (176, 86, 19), (177, 84, 24),
    (178, 83, 30), (180, 81, 36), (181, 80, 42), (182, 78, 48), (183, 77, 54), (185, 75, 60),
    (186, 74, 66), (187, 72, 72), (188, 70, 77), (190, 69, 83), (191, 67, 89), (192, 66, 95),
    (193, 64, 101), (195, 63, 107), (196, 61, 113), (197, 60, 119), (198, 58, 125), (200, 57, 131),
    (201, 55, 136), (202, 54, 142), (203, 52, 148), (205, 50, 154), (206, 49, 160), (207, 47, 166),
    (208, 46, 172), (210, 44, 178), (211, 43, 184), (212, 41, 189), (213, 40, 195), (215, 38, 201),
    (216, 37, 207), (217, 35, 213), (215, 37, 213), (211, 41, 210), (207, 45, 208), (203, 49, 205),
    (199, 52, 202), (196, 56, 200), (192, 60, 197), (188, 64, 195), (184, 67, 192), (180, 71, 189),
    (176, 75, 187), (173, 79, 184), (169, 83, 182), (165, 86, 179), (161, 90, 176), (157, 94, 174),
    (153, 98, 171), (150, 102, 168), (146, 105, 166), (142, 109, 163), (138, 113, 161), (134, 117, 158),
    (130, 120, 155), (127, 124, 153), (123, 128, 150), (119, 132, 148), (115, 136, 145), (111, 139, 142),
    (107, 143, 140), (104, 147, 137), (100, 151, 134), (96, 154, 132), (92, 158, 129), (88, 162, 127),
    (85, 166, 124), (81, 170, 121), (78, 172, 121), (76, 172, 124), (75, 172, 128), (73, 173, 132),
    (72, 173, 135), (70, 173, 139), (69, 173, 143), (67, 173, 147), (66, 173, 150), (64, 173, 154),
    (63, 173, 158), (61, 173, 162), (60, 173, 165), (58, 174, 169), (57, 174, 173), (55, 174, 176),
    (54, 174, 180), (52, 174, 184), (51, 174, 188), (49, 174, 191), (48, 174, 195), (46, 174, 199),
    (45, 174, 202), (43, 174, 206), (42, 175, 210), (40, 175, 214), (39, 175, 217), (37, 175, 221),
    (36, 175, 225), (34, 175, 228), (33, 175, 232), (31, 175, 236), (30, 175, 240), (28, 175, 243),
    (27, 176, 247), (26, 176, 251), (24, 176, 254), (23, 178, 254), (23, 180, 254), (22, 182, 253),
    (21, 184, 252), (21, 186, 252), (20, 189, 251), (19, 191, 250), (19, 193, 250), (18, 195, 249),
    (17, 197, 248), (17, 199, 248), (16, 202, 247), (15, 204, 246), (15, 206, 246), (14, 208, 245),
    (13, 210, 244), (13, 212, 244), (12, 215, 243), (11, 217, 242), (11, 219, 242), (10, 221, 241),
    (10, 223, 240), (9, 225, 240), (8, 228, 239), (8, 230, 238), (7, 232, 238), (6, 234, 237),
    (6, 236, 236), (5, 239, 236), (4, 241, 235), (4, 243, 234), (3, 245, 233), (2, 247, 233),
    (2, 249, 232), (1, 252, 231), (0, 254, 231), (3, 255, 231), (10, 255, 231), (17, 255, 232),
    (24, 255, 233), (31, 255, 233), (38, 255, 234), (45, 255, 235), (52, 255, 235), (59, 255, 236),
    (66, 255, 237), (73, 255, 237), (80, 255, 238), (87, 255, 239), (94, 255, 239), (101, 255, 240),
    (108, 255, 241), (115, 255, 241), (122, 255, 242), (129, 255, 243), (136, 255, 244), (143, 255, 244),
    (150, 255, 245), (157, 255, 246), (164, 255, 246), (171, 255, 247), (178, 255, 248), (185, 255, 248),
    (192, 255, 249), (199, 255, 250), (206, 255, 250), (213, 255, 251), (220, 255, 252), (227, 255, 252),
    (234, 255, 253), (241, 255, 254), (248, 255, 254), (255, 255, 255)]
colormap_pastel_rgb = [(c[2], c[1], c[0]) for c in colormap_pastel_bgr]

colormap_plasma_bgr = [
    (135, 8, 13), (136, 7, 16), (137, 7, 19), (138, 7, 22), (140, 6, 25), (141, 6, 27),
    (142, 6, 29), (143, 6, 32), (144, 6, 34), (145, 6, 36), (145, 5, 38), (146, 5, 40),
    (147, 5, 42), (148, 5, 44), (149, 5, 46), (150, 5, 47), (151, 5, 49), (151, 5, 51),
    (152, 4, 53), (153, 4, 55), (154, 4, 56), (154, 4, 58), (155, 4, 60), (156, 4, 62),
    (156, 4, 63), (157, 4, 65), (158, 3, 67), (158, 3, 68), (159, 3, 70), (159, 3, 72),
    (160, 3, 73), (161, 3, 75), (161, 2, 76), (162, 2, 78), (162, 2, 80), (163, 2, 81),
    (163, 2, 83), (164, 2, 85), (164, 1, 86), (164, 1, 88), (165, 1, 89), (165, 1, 91),
    (166, 1, 92), (166, 1, 94), (166, 1, 96), (167, 0, 97), (167, 0, 99), (167, 0, 100),
    (167, 0, 102), (168, 0, 103), (168, 0, 105), (168, 0, 106), (168, 0, 108), (168, 0, 110),
    (168, 0, 111), (168, 0, 113), (168, 1, 114), (168, 1, 116), (168, 1, 117), (168, 1, 119),
    (168, 1, 120), (168, 2, 122), (168, 2, 123), (168, 3, 125), (168, 3, 126), (168, 4, 128),
    (167, 4, 129), (167, 5, 131), (167, 5, 132), (166, 6, 134), (166, 7, 135), (166, 8, 136),
    (165, 9, 138), (165, 10, 139), (165, 11, 141), (164, 12, 142), (164, 13, 143), (163, 14, 145),
    (163, 15, 146), (162, 16, 148), (161, 17, 149), (161, 19, 150), (160, 20, 152), (159, 21, 153),
    (159, 22, 154), (158, 23, 156), (157, 24, 157), (157, 25, 158), (156, 26, 160), (155, 27, 161),
    (154, 29, 162), (154, 30, 163), (153, 31, 165), (152, 32, 166), (151, 33, 167), (150, 34, 168),
    (149, 35, 170), (148, 36, 171), (148, 38, 172), (147, 39, 173), (146, 40, 174), (145, 41, 176),
    (144, 42, 177), (143, 43, 178), (142, 44, 179), (141, 46, 180), (140, 47, 181), (139, 48, 182),
    (138, 49, 183), (137, 50, 184), (136, 51, 186), (136, 52, 187), (135, 53, 188), (134, 55, 189),
    (133, 56, 190), (132, 57, 191), (131, 58, 192), (130, 59, 193), (129, 60, 194), (128, 61, 195),
    (127, 62, 196), (126, 64, 197), (125, 65, 198), (124, 66, 199), (123, 67, 200), (122, 68, 201),
    (122, 69, 202), (121, 70, 203), (120, 71, 204), (119, 73, 204), (118, 74, 205), (117, 75, 206),
    (116, 76, 207), (115, 77, 208), (114, 78, 209), (113, 79, 210), (113, 81, 211), (112, 82, 212),
    (111, 83, 213), (110, 84, 213), (109, 85, 214), (108, 86, 215), (107, 87, 216), (106, 88, 217),
    (106, 90, 218), (105, 91, 218), (104, 92, 219), (103, 93, 220), (102, 94, 221), (101, 95, 222),
    (100, 97, 222), (99, 98, 223), (99, 99, 224), (98, 100, 225), (97, 101, 226), (96, 102, 226),
    (95, 104, 227), (94, 105, 228), (93, 106, 229), (93, 107, 229), (92, 108, 230), (91, 110, 231),
    (90, 111, 231), (89, 112, 232), (88, 113, 233), (87, 114, 233), (87, 116, 234), (86, 117, 235),
    (85, 118, 235), (84, 119, 236), (83, 121, 237), (82, 122, 237), (81, 123, 238), (81, 124, 239),
    (80, 126, 239), (79, 127, 240), (78, 128, 240), (77, 129, 241), (76, 131, 241), (75, 132, 242),
    (75, 133, 243), (74, 135, 243), (73, 136, 244), (72, 137, 244), (71, 139, 245), (70, 140, 245),
    (69, 141, 246), (68, 143, 246), (68, 144, 247), (67, 145, 247), (66, 147, 247), (65, 148, 248),
    (64, 149, 248), (63, 151, 249), (62, 152, 249), (62, 154, 249), (61, 155, 250), (60, 156, 250),
    (59, 158, 250), (58, 159, 251), (57, 161, 251), (56, 162, 251), (56, 163, 252), (55, 165, 252),
    (54, 166, 252), (53, 168, 252), (52, 169, 252), (51, 171, 253), (51, 172, 253), (50, 174, 253),
    (49, 175, 253), (48, 177, 253), (47, 178, 253), (47, 180, 253), (46, 181, 253), (45, 183, 254),
    (44, 184, 254), (44, 186, 254), (43, 187, 254), (42, 189, 254), (42, 190, 254), (41, 192, 254),
    (41, 194, 253), (40, 195, 253), (39, 197, 253), (39, 198, 253), (39, 200, 253), (38, 202, 253),
    (38, 203, 253), (37, 205, 252), (37, 206, 252), (37, 208, 252), (37, 210, 252), (36, 211, 251),
    (36, 213, 251), (36, 215, 251), (36, 216, 250), (36, 218, 250), (36, 220, 249), (37, 221, 249),
    (37, 223, 248), (37, 225, 248), (37, 226, 247), (37, 228, 247), (38, 230, 246), (38, 232, 246),
    (38, 233, 245), (39, 235, 245), (39, 237, 244), (39, 238, 243), (39, 240, 243), (39, 242, 242),
    (38, 244, 241), (37, 245, 241), (36, 247, 240), (33, 249, 240)]
colormap_plasma_rgb = [(c[2], c[1], c[0]) for c in colormap_plasma_bgr]

colormap_sepia_bgr = [
    (0, 0, 0), (1, 1, 2), (2, 3, 4), (3, 4, 6), (4, 5, 7), (6, 7, 9),
    (7, 8, 11), (8, 9, 13), (9, 11, 14), (10, 12, 16), (11, 13, 17), (12, 14, 18),
    (13, 15, 19), (14, 16, 21), (15, 17, 22), (16, 18, 23), (16, 19, 24), (17, 20, 25),
    (18, 21, 26), (19, 21, 27), (19, 22, 27), (20, 23, 28), (21, 24, 29), (21, 24, 30),
    (22, 25, 31), (23, 26, 32), (23, 27, 33), (24, 27, 34), (24, 28, 35), (25, 29, 36),
    (26, 29, 37), (26, 30, 38), (27, 31, 39), (27, 32, 40), (28, 32, 41), (28, 33, 42),
    (29, 34, 43), (30, 35, 44), (30, 35, 45), (31, 36, 46), (31, 37, 47), (32, 38, 48),
    (33, 39, 49), (33, 39, 50), (34, 40, 51), (34, 41, 52), (35, 42, 53), (36, 42, 55),
    (36, 43, 56), (37, 44, 57), (37, 45, 58), (38, 46, 59), (39, 46, 60), (39, 47, 61),
    (40, 48, 62), (40, 49, 63), (41, 50, 64), (42, 51, 65), (42, 51, 66), (43, 52, 67),
    (43, 53, 68), (44, 54, 69), (45, 55, 70), (45, 55, 71), (46, 56, 72), (46, 57, 73),
    (47, 58, 75), (48, 59, 76), (48, 60, 77), (49, 61, 78), (50, 61, 79), (50, 62, 80),
    (51, 63, 81), (51, 64, 82), (52, 65, 83), (53, 66, 84), (53, 67, 85), (54, 67, 86),
    (54, 68, 87), (55, 69, 89), (56, 70, 90), (56, 71, 91), (57, 72, 92), (58, 73, 93),
    (58, 73, 94), (59, 74, 95), (59, 75, 96), (60, 76, 97), (61, 77, 98), (61, 78, 99),
    (62, 79, 101), (63, 80, 102), (63, 81, 103), (64, 81, 104), (64, 82, 105), (65, 83, 106),
    (66, 84, 107), (66, 85, 108), (67, 86, 109), (68, 87, 111), (68, 88, 112), (69, 89, 113),
    (69, 90, 114), (70, 90, 115), (71, 91, 116), (71, 92, 117), (72, 93, 118), (73, 94, 120),
    (73, 95, 121), (74, 96, 122), (75, 97, 123), (75, 98, 124), (76, 99, 125), (77, 100, 126),
    (77, 101, 127), (78, 102, 128), (78, 103, 129), (79, 104, 130), (80, 105, 131), (80, 106, 132),
    (81, 107, 133), (82, 108, 134), (82, 109, 135), (83, 110, 136), (84, 111, 137), (85, 112, 138),
    (86, 113, 139), (87, 114, 140), (88, 115, 140), (89, 116, 141), (90, 117, 142), (91, 118, 143),
    (92, 119, 143), (93, 120, 144), (94, 122, 145), (95, 123, 146), (96, 124, 147), (97, 125, 147),
    (98, 126, 148), (99, 127, 149), (100, 128, 150), (101, 129, 151), (102, 130, 152), (103, 131, 153),
    (105, 132, 153), (106, 133, 154), (107, 134, 155), (108, 135, 156), (110, 136, 157), (111, 137, 158),
    (112, 138, 159), (113, 140, 160), (115, 141, 160), (116, 142, 161), (117, 143, 162), (118, 144, 163),
    (119, 145, 164), (121, 146, 164), (122, 147, 165), (123, 148, 166), (124, 149, 167), (126, 150, 168),
    (127, 152, 168), (128, 153, 169), (129, 154, 170), (131, 155, 171), (132, 156, 172), (133, 157, 173),
    (135, 158, 174), (136, 159, 175), (137, 160, 176), (139, 161, 177), (140, 162, 178), (141, 163, 179),
    (142, 164, 180), (144, 165, 181), (145, 166, 182), (146, 167, 184), (148, 168, 185), (149, 169, 186),
    (150, 170, 187), (152, 171, 188), (153, 172, 189), (154, 173, 190), (156, 174, 191), (157, 175, 192),
    (158, 176, 193), (160, 177, 194), (161, 179, 195), (162, 180, 196), (164, 181, 197), (165, 182, 198),
    (166, 183, 199), (168, 184, 200), (169, 185, 201), (171, 186, 202), (172, 187, 203), (173, 188, 204),
    (175, 189, 205), (176, 190, 206), (178, 192, 207), (179, 193, 208), (180, 194, 209), (182, 195, 210),
    (183, 196, 211), (184, 197, 212), (186, 198, 213), (187, 199, 213), (189, 200, 214), (190, 202, 215),
    (191, 203, 216), (193, 204, 217), (194, 205, 218), (196, 206, 219), (197, 207, 220), (198, 208, 221),
    (200, 209, 221), (201, 211, 222), (203, 212, 223), (204, 213, 224), (205, 214, 225), (207, 215, 226),
    (208, 216, 227), (210, 217, 228), (211, 219, 228), (212, 220, 229), (214, 221, 230), (215, 222, 231),
    (217, 223, 232), (218, 224, 233), (219, 226, 234), (221, 227, 235), (222, 228, 235), (224, 229, 236),
    (225, 230, 237), (226, 231, 238), (228, 233, 239), (229, 234, 240), (231, 235, 241), (232, 236, 241),
    (233, 237, 242), (235, 238, 243), (236, 240, 244), (238, 241, 245), (239, 242, 246), (241, 243, 247),
    (242, 244, 247), (243, 245, 248), (245, 247, 249), (246, 248, 250), (248, 249, 251), (249, 250, 252),
    (251, 251, 252), (252, 253, 253), (254, 254, 254), (255, 255, 255)]
colormap_sepia_rgb = [(c[2], c[1], c[0]) for c in colormap_sepia_bgr]

colormap_temperature_bgr = [
    (217, 0, 36), (219, 2, 35), (221, 4, 35), (223, 6, 34), (225, 8, 33), (227, 10, 32),
    (229, 11, 32), (231, 13, 31), (233, 15, 30), (235, 17, 29), (237, 19, 29), (239, 21, 28),
    (241, 23, 27), (243, 25, 26), (245, 27, 26), (247, 29, 25), (248, 32, 26), (248, 36, 27),
    (249, 40, 28), (249, 44, 29), (250, 48, 30), (250, 52, 31), (251, 56, 32), (251, 60, 33),
    (252, 64, 34), (252, 68, 35), (253, 72, 37), (253, 75, 38), (254, 79, 39), (254, 83, 40),
    (255, 87, 41), (255, 90, 42), (255, 94, 44), (255, 97, 45), (255, 100, 46), (255, 103, 48),
    (255, 106, 49), (255, 110, 50), (255, 113, 52), (255, 116, 53), (255, 119, 54), (255, 123, 56),
    (255, 126, 57), (255, 129, 58), (255, 132, 60), (255, 135, 61), (255, 138, 63), (255, 141, 65),
    (255, 144, 66), (255, 146, 68), (255, 149, 70), (255, 152, 71), (255, 155, 73), (255, 157, 75),
    (255, 160, 77), (255, 163, 78), (255, 166, 80), (255, 168, 82), (255, 171, 83), (255, 174, 85),
    (255, 176, 87), (255, 179, 89), (255, 181, 91), (255, 183, 93), (255, 186, 95), (255, 188, 97),
    (255, 190, 99), (255, 193, 101), (255, 195, 103), (255, 197, 105), (255, 200, 107), (255, 202, 109),
    (255, 204, 111), (255, 207, 113), (255, 209, 115), (255, 211, 117), (255, 213, 120), (255, 214, 122),
    (255, 216, 124), (255, 218, 127), (255, 219, 129), (255, 221, 132), (255, 222, 134), (255, 224, 136),
    (255, 225, 139), (255, 227, 141), (255, 228, 143), (255, 230, 146), (255, 232, 148), (255, 233, 151),
    (255, 235, 153), (255, 236, 155), (255, 237, 158), (255, 238, 160), (255, 239, 163), (255, 240, 165),
    (255, 241, 167), (255, 242, 170), (255, 242, 172), (255, 243, 174), (255, 244, 177), (255, 245, 179),
    (255, 246, 182), (255, 247, 184), (255, 248, 186), (255, 249, 189), (255, 250, 192), (255, 250, 195),
    (255, 251, 198), (255, 251, 201), (255, 251, 204), (255, 252, 207), (255, 252, 210), (255, 252, 213),
    (255, 253, 216), (255, 253, 219), (255, 254, 222), (255, 254, 225), (255, 254, 228), (255, 255, 232),
    (255, 255, 235), (254, 255, 236), (252, 255, 237), (251, 255, 239), (250, 255, 240), (248, 255, 241),
    (247, 255, 243), (245, 255, 244), (244, 255, 245), (243, 255, 247), (241, 255, 248), (240, 255, 250),
    (239, 255, 251), (237, 255, 252), (236, 255, 254), (235, 255, 255), (232, 254, 255), (228, 253, 255),
    (225, 252, 255), (222, 251, 255), (219, 251, 255), (216, 250, 255), (213, 249, 255), (210, 248, 255),
    (207, 247, 255), (204, 246, 255), (201, 245, 255), (198, 244, 255), (195, 244, 255), (192, 243, 255),
    (189, 242, 255), (186, 240, 255), (184, 238, 255), (182, 236, 255), (179, 234, 255), (177, 233, 255),
    (174, 231, 255), (172, 229, 255), (170, 227, 255), (167, 225, 255), (165, 223, 255), (163, 222, 255),
    (160, 220, 255), (158, 218, 255), (155, 216, 255), (153, 214, 255), (151, 211, 255), (148, 209, 255),
    (146, 206, 255), (143, 203, 255), (141, 200, 255), (139, 197, 255), (136, 195, 255), (134, 192, 255),
    (132, 189, 255), (129, 186, 255), (127, 184, 255), (124, 181, 255), (122, 178, 255), (120, 175, 255),
    (117, 172, 255), (115, 169, 255), (113, 165, 255), (111, 162, 255), (109, 159, 255), (107, 155, 255),
    (105, 152, 255), (103, 148, 255), (101, 145, 255), (99, 141, 255), (97, 138, 255), (95, 134, 255),
    (93, 131, 255), (91, 127, 255), (89, 124, 255), (87, 120, 255), (85, 116, 255), (83, 112, 255),
    (82, 109, 255), (80, 105, 255), (78, 101, 255), (77, 97, 255), (75, 93, 255), (73, 89, 255),
    (71, 85, 255), (70, 81, 255), (68, 77, 255), (66, 73, 255), (65, 69, 255), (63, 65, 255),
    (61, 61, 255), (61, 60, 254), (60, 58, 254), (60, 57, 253), (59, 55, 253), (59, 54, 252),
    (58, 53, 252), (58, 51, 251), (57, 50, 251), (57, 48, 250), (56, 47, 250), (56, 45, 249),
    (55, 44, 249), (55, 42, 248), (54, 41, 248), (54, 40, 247), (53, 38, 245), (53, 37, 243),
    (52, 36, 241), (52, 35, 239), (52, 34, 237), (51, 32, 235), (51, 31, 233), (50, 30, 231),
    (50, 29, 229), (50, 28, 227), (49, 26, 225), (49, 25, 223), (48, 24, 221), (48, 23, 219),
    (48, 22, 217), (47, 20, 213), (46, 19, 210), (45, 17, 207), (44, 16, 203), (43, 14, 200),
    (42, 13, 196), (41, 12, 193), (40, 10, 190), (39, 9, 186), (38, 7, 183), (37, 6, 179),
    (36, 4, 176), (35, 3, 173), (34, 1, 169), (33, 0, 166)]
colormap_temperature_rgb = [(c[2], c[1], c[0]) for c in colormap_temperature_bgr]

colormap_thermal_bgr = [
    (0, 0, 0), (4, 1, 3), (8, 2, 5), (12, 2, 8), (15, 3, 10), (18, 4, 13),
    (21, 5, 15), (23, 6, 17), (25, 7, 18), (27, 7, 20), (29, 8, 21), (32, 9, 23),
    (34, 10, 24), (36, 11, 24), (38, 12, 25), (40, 13, 26), (43, 13, 27), (45, 14, 28),
    (47, 14, 29), (50, 15, 30), (52, 15, 31), (54, 15, 31), (57, 15, 32), (59, 16, 33),
    (62, 16, 34), (64, 16, 35), (66, 16, 36), (69, 16, 37), (71, 17, 38), (74, 17, 39),
    (76, 17, 39), (79, 17, 40), (82, 18, 41), (84, 18, 42), (87, 18, 42), (90, 18, 43),
    (92, 19, 43), (95, 19, 44), (98, 19, 45), (100, 19, 45), (103, 20, 46), (106, 20, 46),
    (109, 20, 47), (111, 20, 47), (114, 20, 48), (117, 20, 49), (119, 20, 50), (122, 20, 51),
    (125, 20, 52), (127, 20, 53), (129, 19, 55), (131, 18, 58), (133, 17, 61), (135, 15, 65),
    (136, 13, 68), (137, 10, 72), (138, 7, 75), (139, 4, 79), (139, 0, 83), (139, 0, 87),
    (138, 0, 91), (138, 0, 95), (137, 0, 99), (136, 0, 102), (135, 0, 105), (134, 0, 109),
    (133, 0, 112), (132, 0, 115), (131, 0, 117), (130, 0, 120), (129, 0, 122), (128, 0, 125),
    (127, 0, 127), (126, 0, 130), (124, 0, 132), (123, 0, 134), (122, 0, 137), (121, 0, 139),
    (120, 0, 141), (119, 0, 143), (118, 0, 145), (116, 0, 147), (115, 0, 149), (114, 0, 151),
    (113, 0, 153), (112, 0, 155), (111, 0, 157), (109, 0, 159), (108, 0, 161), (107, 0, 163),
    (106, 0, 165), (105, 0, 167), (103, 0, 168), (102, 0, 170), (101, 0, 172), (100, 0, 174),
    (99, 0, 176), (97, 0, 178), (96, 0, 179), (95, 0, 181), (94, 0, 183), (93, 0, 185),
    (91, 1, 186), (90, 2, 188), (89, 4, 190), (88, 6, 192), (86, 8, 193), (85, 10, 195),
    (84, 12, 197), (83, 15, 198), (81, 16, 200), (80, 18, 202), (79, 20, 203), (77, 22, 205),
    (76, 23, 206), (75, 25, 208), (73, 26, 210), (72, 28, 211), (71, 29, 213), (69, 31, 214),
    (68, 32, 216), (67, 34, 218), (65, 35, 219), (64, 36, 221), (62, 38, 222), (61, 39, 224),
    (59, 41, 225), (58, 42, 227), (56, 44, 228), (55, 45, 229), (53, 47, 231), (52, 48, 232),
    (50, 50, 234), (48, 52, 235), (46, 53, 236), (45, 55, 237), (43, 57, 239), (41, 58, 240),
    (39, 60, 241), (37, 62, 242), (35, 64, 243), (32, 66, 244), (30, 68, 245), (27, 70, 246),
    (25, 72, 247), (22, 74, 247), (19, 76, 248), (15, 78, 249), (10, 80, 249), (7, 83, 249),
    (3, 86, 250), (0, 88, 250), (0, 91, 250), (0, 93, 249), (0, 96, 249), (0, 99, 249),
    (0, 101, 249), (0, 104, 249), (0, 106, 249), (0, 109, 248), (0, 111, 248), (0, 114, 248),
    (0, 116, 247), (0, 118, 247), (0, 121, 246), (0, 123, 246), (0, 125, 245), (0, 128, 245),
    (0, 130, 244), (0, 132, 244), (0, 134, 243), (0, 137, 243), (0, 139, 242), (0, 141, 241),
    (0, 143, 241), (0, 145, 240), (0, 148, 239), (0, 150, 239), (0, 152, 238), (0, 154, 237),
    (0, 156, 236), (0, 158, 236), (0, 160, 235), (0, 162, 234), (0, 164, 233), (0, 166, 233),
    (0, 168, 232), (0, 170, 231), (0, 172, 230), (4, 174, 229), (11, 176, 228), (18, 178, 228),
    (23, 179, 227), (27, 181, 226), (31, 183, 225), (35, 185, 224), (39, 187, 223), (43, 188, 223),
    (47, 190, 222), (50, 192, 221), (54, 194, 220), (57, 195, 219), (61, 197, 218), (64, 199, 218),
    (68, 200, 217), (71, 202, 216), (75, 204, 215), (79, 205, 215), (83, 207, 214), (86, 208, 214),
    (90, 210, 213), (94, 211, 213), (98, 213, 212), (101, 214, 212), (105, 216, 212), (109, 217, 212),
    (113, 218, 212), (117, 220, 212), (121, 221, 212), (125, 222, 212), (128, 223, 212), (132, 224, 213),
    (136, 226, 213), (140, 227, 214), (144, 228, 214), (147, 229, 215), (151, 230, 216), (155, 231, 217),
    (159, 232, 218), (162, 233, 219), (166, 234, 220), (170, 235, 221), (173, 236, 222), (177, 237, 224),
    (181, 238, 225), (184, 238, 226), (188, 239, 228), (191, 240, 229), (195, 241, 230), (199, 242, 231),
    (202, 243, 233), (206, 244, 234), (209, 245, 236), (213, 245, 237), (216, 246, 238), (220, 247, 240),
    (223, 248, 241), (227, 249, 243), (230, 249, 244), (234, 250, 246), (237, 251, 247), (241, 252, 249),
    (244, 253, 250), (248, 253, 252), (251, 254, 253), (255, 255, 255)]
colormap_thermal_rgb = [(c[2], c[1], c[0]) for c in colormap_thermal_bgr]


# Google's rainbow "Turbo" colormap, see
# https://ai.googleblog.com/2019/08/turbo-improved-rainbow-colormap-for.html
# Adapted from https://gist.github.com/mikhailov-work/ee72ba4191942acecc03fe6da94fc73f
colormap_turbo_bgr = [
    (59, 18, 48), (66, 21, 49), (74, 24, 50), (81, 27, 52), (88, 30, 53), (95, 33, 54),
    (101, 35, 55), (108, 38, 56), (114, 41, 57), (121, 44, 58), (127, 47, 59), (133, 50, 60),
    (139, 53, 60), (145, 55, 61), (150, 58, 62), (156, 61, 63), (161, 64, 64), (166, 67, 64),
    (171, 69, 65), (176, 72, 65), (181, 75, 66), (186, 78, 67), (190, 80, 67), (194, 83, 67),
    (199, 86, 68), (203, 88, 68), (206, 91, 69), (210, 94, 69), (214, 96, 69), (217, 99, 69),
    (221, 102, 70), (224, 104, 70), (227, 107, 70), (230, 109, 70), (232, 112, 70), (235, 115, 70),
    (237, 117, 70), (240, 120, 70), (242, 122, 70), (244, 125, 70), (246, 127, 70), (248, 130, 70),
    (249, 132, 69), (251, 135, 69), (252, 137, 69), (253, 140, 68), (253, 142, 67), (254, 145, 66),
    (254, 147, 65), (254, 150, 64), (254, 152, 63), (254, 155, 62), (253, 157, 60), (252, 160, 59),
    (252, 162, 57), (251, 165, 56), (249, 168, 54), (248, 170, 52), (246, 172, 51), (245, 175, 49),
    (243, 177, 47), (241, 180, 45), (239, 182, 43), (237, 185, 42), (235, 187, 40), (233, 189, 38),
    (230, 192, 37), (228, 194, 35), (225, 196, 33), (223, 198, 32), (220, 201, 30), (218, 203, 29),
    (215, 205, 28), (212, 207, 27), (210, 209, 26), (207, 211, 25), (204, 213, 24), (202, 215, 24),
    (199, 217, 23), (196, 218, 23), (194, 220, 23), (191, 222, 23), (189, 224, 24), (186, 225, 24),
    (184, 227, 25), (182, 228, 26), (180, 229, 27), (177, 231, 29), (175, 232, 30), (172, 233, 32),
    (169, 235, 34), (166, 236, 36), (163, 237, 39), (160, 238, 41), (157, 239, 44), (154, 240, 47),
    (151, 241, 50), (148, 243, 53), (145, 244, 56), (141, 244, 59), (138, 245, 63), (135, 246, 66),
    (131, 247, 70), (128, 248, 74), (124, 249, 77), (121, 249, 81), (118, 250, 85), (114, 251, 89),
    (111, 251, 93), (108, 252, 97), (104, 252, 101), (101, 253, 105), (98, 253, 109), (95, 253, 113),
    (92, 254, 116), (89, 254, 120), (86, 254, 124), (83, 254, 128), (80, 254, 132), (77, 254, 135),
    (75, 254, 139), (72, 254, 142), (70, 254, 146), (68, 254, 149), (66, 254, 152), (64, 253, 155),
    (62, 253, 158), (61, 252, 161), (59, 252, 164), (58, 251, 166), (57, 251, 169), (55, 250, 172),
    (55, 249, 174), (54, 248, 177), (53, 248, 179), (53, 247, 182), (52, 245, 185), (52, 244, 187),
    (52, 243, 190), (51, 242, 192), (51, 241, 195), (51, 239, 197), (51, 238, 200), (51, 237, 202),
    (52, 235, 205), (52, 234, 207), (52, 232, 209), (53, 231, 212), (53, 229, 214), (53, 227, 216),
    (54, 226, 218), (54, 224, 221), (54, 222, 223), (55, 220, 225), (55, 218, 227), (56, 216, 229),
    (56, 215, 231), (56, 213, 232), (57, 211, 234), (57, 209, 236), (57, 207, 237), (57, 205, 239),
    (58, 203, 240), (58, 200, 242), (58, 198, 243), (58, 196, 244), (58, 194, 246), (57, 192, 247),
    (57, 190, 248), (57, 188, 249), (56, 186, 249), (55, 183, 250), (55, 181, 251), (54, 179, 251),
    (53, 176, 252), (52, 174, 252), (51, 171, 253), (50, 169, 253), (49, 166, 253), (48, 163, 253),
    (47, 161, 254), (46, 158, 254), (45, 155, 254), (44, 152, 254), (43, 149, 253), (41, 146, 253),
    (40, 143, 253), (39, 140, 253), (38, 137, 252), (36, 134, 252), (35, 131, 251), (34, 128, 251),
    (32, 125, 250), (31, 122, 250), (30, 119, 249), (28, 116, 248), (27, 113, 247), (26, 110, 247),
    (24, 107, 246), (23, 104, 245), (22, 101, 244), (21, 99, 243), (20, 96, 242), (19, 93, 241),
    (17, 90, 239), (16, 88, 238), (15, 85, 237), (14, 82, 236), (13, 80, 234), (13, 77, 233),
    (12, 75, 232), (11, 73, 230), (10, 70, 229), (10, 68, 227), (9, 66, 226), (8, 64, 224),
    (8, 62, 222), (7, 60, 221), (7, 58, 219), (6, 56, 217), (6, 54, 215), (5, 52, 214),
    (5, 50, 212), (5, 48, 210), (4, 47, 208), (4, 45, 206), (3, 43, 203), (3, 41, 201),
    (3, 40, 199), (2, 38, 197), (2, 36, 195), (2, 35, 192), (2, 33, 190), (1, 31, 187),
    (1, 30, 185), (1, 28, 182), (1, 27, 180), (1, 25, 177), (1, 24, 174), (1, 22, 172),
    (1, 21, 169), (1, 20, 166), (1, 18, 163), (1, 17, 160), (1, 16, 157), (1, 14, 154),
    (1, 13, 151), (1, 12, 148), (1, 11, 145), (1, 10, 142), (1, 9, 139), (1, 8, 135),
    (1, 7, 132), (2, 6, 129), (2, 5, 125), (2, 4, 122)
]
colormap_turbo_rgb = [(c[2], c[1], c[0]) for c in colormap_turbo_bgr]


colormap_viridis_bgr = [
    (84, 1, 68), (86, 2, 68), (87, 4, 69), (89, 5, 69), (90, 7, 70), (92, 8, 70),
    (93, 10, 70), (94, 11, 70), (96, 13, 71), (97, 14, 71), (99, 16, 71), (100, 17, 71),
    (101, 19, 71), (103, 20, 72), (104, 22, 72), (105, 23, 72), (106, 24, 72), (108, 26, 72),
    (109, 27, 72), (110, 28, 72), (111, 29, 72), (112, 31, 72), (113, 32, 72), (115, 33, 72),
    (116, 35, 72), (117, 36, 72), (118, 37, 72), (119, 38, 72), (120, 40, 72), (121, 41, 72),
    (122, 42, 71), (122, 44, 71), (123, 45, 71), (124, 46, 71), (125, 47, 71), (126, 48, 70),
    (126, 50, 70), (127, 51, 70), (128, 52, 70), (129, 53, 69), (129, 55, 69), (130, 56, 69),
    (131, 57, 68), (131, 58, 68), (132, 59, 68), (132, 61, 67), (133, 62, 67), (133, 63, 66),
    (134, 64, 66), (134, 65, 66), (135, 66, 65), (135, 68, 65), (136, 69, 64), (136, 70, 64),
    (136, 71, 63), (137, 72, 63), (137, 73, 62), (137, 74, 62), (138, 76, 62), (138, 77, 61),
    (138, 78, 61), (138, 79, 60), (139, 80, 60), (139, 81, 59), (139, 82, 59), (139, 83, 58),
    (140, 84, 58), (140, 85, 57), (140, 86, 57), (140, 88, 56), (140, 89, 56), (140, 90, 55),
    (141, 91, 55), (141, 92, 54), (141, 93, 54), (141, 94, 53), (141, 95, 53), (141, 96, 52),
    (141, 97, 52), (141, 98, 51), (141, 99, 51), (142, 100, 50), (142, 101, 50), (142, 102, 49),
    (142, 103, 49), (142, 104, 49), (142, 105, 48), (142, 106, 48), (142, 107, 47), (142, 108, 47),
    (142, 109, 46), (142, 110, 46), (142, 111, 46), (142, 112, 45), (142, 113, 45), (142, 113, 44),
    (142, 114, 44), (142, 115, 44), (142, 116, 43), (142, 117, 43), (142, 118, 42), (142, 119, 42),
    (142, 120, 42), (142, 121, 41), (142, 122, 41), (142, 123, 41), (142, 124, 40), (142, 125, 40),
    (142, 126, 39), (142, 127, 39), (142, 128, 39), (142, 129, 38), (142, 130, 38), (142, 130, 38),
    (142, 131, 37), (142, 132, 37), (142, 133, 37), (142, 134, 36), (142, 135, 36), (142, 136, 35),
    (142, 137, 35), (141, 138, 35), (141, 139, 34), (141, 140, 34), (141, 141, 34), (141, 142, 33),
    (141, 143, 33), (141, 144, 33), (140, 145, 33), (140, 146, 32), (140, 146, 32), (140, 147, 32),
    (140, 148, 31), (139, 149, 31), (139, 150, 31), (139, 151, 31), (139, 152, 31), (138, 153, 31),
    (138, 154, 31), (138, 155, 30), (137, 156, 30), (137, 157, 30), (137, 158, 31), (136, 159, 31),
    (136, 160, 31), (136, 161, 31), (135, 161, 31), (135, 162, 31), (134, 163, 32), (134, 164, 32),
    (133, 165, 33), (133, 166, 33), (133, 167, 34), (132, 168, 34), (131, 169, 35), (131, 170, 36),
    (130, 171, 37), (130, 172, 37), (129, 173, 38), (129, 173, 39), (128, 174, 40), (127, 175, 41),
    (127, 176, 42), (126, 177, 44), (125, 178, 45), (124, 179, 46), (124, 180, 47), (123, 181, 49),
    (122, 182, 50), (121, 182, 52), (121, 183, 53), (120, 184, 55), (119, 185, 56), (118, 186, 58),
    (117, 187, 59), (116, 188, 61), (115, 188, 63), (114, 189, 64), (113, 190, 66), (112, 191, 68),
    (111, 192, 70), (110, 193, 72), (109, 193, 74), (108, 194, 76), (107, 195, 78), (106, 196, 80),
    (105, 197, 82), (104, 197, 84), (103, 198, 86), (101, 199, 88), (100, 200, 90), (99, 200, 92),
    (98, 201, 94), (96, 202, 96), (95, 203, 99), (94, 203, 101), (92, 204, 103), (91, 205, 105),
    (90, 205, 108), (88, 206, 110), (87, 207, 112), (86, 208, 115), (84, 208, 117), (83, 209, 119),
    (81, 209, 122), (80, 210, 124), (78, 211, 127), (77, 211, 129), (75, 212, 132), (73, 213, 134),
    (72, 213, 137), (70, 214, 139), (69, 214, 142), (67, 215, 144), (65, 215, 147), (64, 216, 149),
    (62, 216, 152), (60, 217, 155), (59, 217, 157), (57, 218, 160), (55, 218, 162), (54, 219, 165),
    (52, 219, 168), (50, 220, 170), (48, 220, 173), (47, 221, 176), (45, 221, 178), (43, 222, 181),
    (41, 222, 184), (40, 222, 186), (38, 223, 189), (37, 223, 192), (35, 223, 194), (33, 224, 197),
    (32, 224, 200), (31, 225, 202), (29, 225, 205), (28, 225, 208), (27, 226, 210), (26, 226, 213),
    (25, 226, 216), (25, 227, 218), (24, 227, 221), (24, 227, 223), (24, 228, 226), (25, 228, 229),
    (25, 228, 231), (26, 229, 234), (27, 229, 236), (28, 229, 239), (29, 229, 241), (30, 230, 244),
    (32, 230, 246), (33, 230, 248), (35, 231, 251), (37, 231, 253)]
colormap_viridis_rgb = [(c[2], c[1], c[0]) for c in colormap_viridis_bgr]
