"""TBApi v3 Parser"""

import os
import sys
import json
import datetime
import requests
from SQLiteHelper import SQLiteHelper as sq # SQLite3 Wrapper for easier reading and programming
from .exceptions import *
from .models import *


# Main Parser Class:  All requests routed through this class's methods.
class TBAParser:
    """TBA Parser Class: Routes API requests and handles caching. Requires v3 API Key from TBA.

    :param str api_key: API Key used for connection to The Blue Alliance API Servers.  Get an API Key from your `TBA Account Settings <tba_account_settings_>`_.
    :param bool cache: *(opt)* Enabling of the In-Built Caching System. *(default: True)*
    :param bool local: *(opt)* Store cache Database in the local directory rather than globally in the python directory. *(default: False)*
    :param bool force_cache: *(opt)* Force the :class:`Parser` to pull data for the In-Built Cache before attempting TBA Connection. *(default: False)*
    :param bool log_cache: *(opt)* Log Caching Information to the Terminal while attempting to interact with the In-Built Cache. *(default: False)*
    :param int cache_multiplier: *(opt)* Multiplier for die-times of cache data. Allows data to be used by the standard caching methods after their standard accuracy period. *(default: 1)*
    
    All Parser Class Methods have the following optional arguments:

    :param bool force_new: *(opt)* Force a new request from The Blue Alliance servers, ignoring the cache response or 304 Response Headers. *(default: False)*
    :param bool force_cache: *(opt)* Force a call to the cache, regardless of global cache settings. *(default: False)*
    :param bool log_cache: *(opt)* Log Caching Information to the Terminal while attempting to interact with the In-Built Cache. *(default: False)*"""

    # Initiate Information needed for connection to TBA v3 (api_key)
    def __init__(self, api_key, *, cache=True, local=False, force_cache=False, log_cache=False, cache_multiplier=1):
        self.api_key = api_key # TBA API v3 API key
        self.cache = cache
        self.force_cache = force_cache
        self.log_cache = log_cache
        self.cache_multiplier = cache_multiplier
        self.base_url = 'http://www.thebluealliance.com/api/v3'

        if self.cache:
            if local is False:
                self.storage_path = os.path.dirname(sys.executable) + '/.tbapi'

                if not os.path.exists(self.storage_path):
                    try:
                        os.makedirs(self.storage_path)
                    except:
                        self.storage_path = os.getcwd()

                self.cache_db = sq.Connect(self.storage_path + '/cache')
            else:
                self.cache_db = sq.Connect('tbapi-cache')


            if not self.cache_db.table('cache_data').tableExists():
                cache_preset = CacheTable()
                self.cache_db.table('cache_data').init(cache_preset)

    # Modify the response when outputted in the terminal or when converted to a string.
    def __repr__(self):
        """Modified Response Return"""
        return '<tbapi.Parser: \'{}\'>'.format(self.api_key)

    # Method to pull JSON array from TBA v3 API.  Includes Caching and Offline Support.
    def pull_response_json(self, path, *, force_new=False, force_cache=False, log_cache=False):
        """Pull the JSON response to a given path from the TBA API servers or the response cache.
        
        :param string path: The URL to pull data from, without the base string 'http://www.thebluealliance.com/api/v3'."""
        
        if not path.startswith('/'):
            path = '/' + path

        current_time = datetime.datetime.now() # Get the Current Time at the start of method execution

        if (self.cache or force_cache) and force_new == False: # If caching is enabled, and a new response is not forced, attempt pulling from cache

            cache_reply = self.cache_db.table('cache_data').select('RESPONSE', 'REQUESTED', 'LAST_MODIFIED', 'MAX_AGE').where('REQUEST').equals(path).execute() # Query Cache Database for last stored response

            # Default Values
            cache_response = None
            cache_requested = None
            cache_last_modified = None
            cache_last_modified_str = ''
            cache_max_age = ''

            # Pull Data from Database:  Queried Variable is primary key, so only one row is possible.
            for entry in cache_reply:
                cache_response = json.loads(entry[0]) # Convert stored response back to Python Dictionaries and Lists
                cache_requested = datetime.datetime.strptime(entry[1].strip(), '%a, %d %b %Y %H:%M:%S') # Convert Strings to Datetime objects
                cache_last_modified = datetime.datetime.strptime(entry[2].strip(), '%a, %d %b %Y %H:%M:%S %Z') # Convert Strings to Datetime objects
                cache_last_modified_str = entry[2]
                cache_max_age = entry[3]

            if not cache_response is None: # If Database Response NOT blank:  AKA, if database value for given path not null
                request = (self.base_url + path) # Full Request URL generation based on base_url as set in __init__
                header = {'X-TBA-Auth-Key': self.api_key, 'If-Modified-Since': cache_last_modified_str} # Form headers, but substitute in last_modified string from database to see if data has changed since last database pull.

                if force_cache or self.force_cache:
                    if self.log_cache or log_cache:
                        print('Forcing Read from Cache...')

                    if cache_response == []:
                        raise EmptyError

                    return cache_response

                try:
                    response = requests.get(request, headers=header)
                except:
                    cache_expire_offset = float(cache_max_age) * float(self.cache_multiplier)
                    cache_expire = cache_requested + datetime.timedelta(seconds=int(cache_expire_offset))

                    if current_time <= cache_expire:
                        if self.log_cache or log_cache:
                            print('Offline: Reading from Cache...')

                        if cache_response == []:
                            raise EmptyError

                        return cache_response
                    else:
                        raise OfflineError

                if response.status_code == 304:
                    if self.log_cache or log_cache:
                        print('304 NOT MODIFIED: Reading from Cache...')

                    if cache_response == []:
                        raise EmptyError

                    return cache_response
                else:
                    if self.log_cache or log_cache:
                        print('Data Modified, Ignoring Cache...')

                    try:
                        json_array = response.json()
                    except:
                        raise ParseError

                    response_string = json.dumps(json_array)
                    response_headers = response.headers
                    response_last_modified = response_headers['Last-Modified']
                    response_cache_control = response_headers['Cache-Control']
                    current_time_str = current_time.strftime('%a, %d %b %Y %H:%M:%S')

                    response_max_age = ''
                    cache_control_array = response_cache_control.split(',')

                    for cache_control_item in cache_control_array:
                        cache_control_item = cache_control_item.strip()
                        if cache_control_item.startswith('max-age='):
                            response_max_age = cache_control_item.strip()[8:]

                    self.cache_db.table('cache_data').update('RESPONSE', 'REQUESTED', 'LAST_MODIFIED', 'MAX_AGE').insert(response_string, current_time_str, response_last_modified, response_max_age).where('REQUEST').equals(path).execute()

                    if json_array == []:
                        raise EmptyError

                    return json_array

        request = (self.base_url + path)
        header = {'X-TBA-Auth-Key': self.api_key}

        try:
            response = requests.get(request, headers=header)

            if self.log_cache or log_cache:
                print('No Cache Available, Pulling New Data...')
        except:
            raise OfflineError

        try:
            json_array = response.json()
        except:
            raise ParseError

        if self.cache:
            response_string = json.dumps(json_array)
            response_headers = response.headers
            last_modified = response_headers['Last-Modified']
            cache_control = response_headers['Cache-Control']
            current_time_str = current_time.strftime('%a, %d %b %Y %H:%M:%S')

            response_max_age = ''
            cache_control_array = cache_control.split(',')

            for cache_control_item in cache_control_array:
                cache_control_item = cache_control_item.strip()
                if cache_control_item.startswith('max-age='):
                    response_max_age = cache_control_item.strip()[8:]

            self.cache_db.table('cache_data').insert(path, response_string, current_time_str, last_modified, response_max_age).into('REQUEST', 'RESPONSE', 'REQUESTED', 'LAST_MODIFIED', 'MAX_AGE')

        if json.dumps(json_array) == '[]':
            raise EmptyError

        return json_array


    ### CALL METHODS
    # Get TBA API Status
    def get_status(self, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`Status`: the Status of The Blue Alliance's API and its Apps."""
        return API_Status(self.pull_response_json('/status', force_new=False, force_cache=False, log_cache=False), self, {'force_new': force_new, 'force_cache': force_cache, 'log_cache': log_cache})

    # Get a List of FRC Teams
    def get_team_list(self, *, page=None, year=None, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Team` objects.
        
        :param int page: *(opt)* The page of teams to fetch.  Argument must be named explicitly when passing.  If not included, method will pull all available pages.
        :param int year: *(opt)* The year to fetch competing teams for. Argument must be named explicitly when passing.  If not included, method will pull all teams that have ever competed and are recorded in the TBA databases."""
        if not page is None:
            return self.__get_team_list_page(page, year=year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        else:
            team_list = DataList()

            for prospective_page in range(0,100):
                try:
                    partial_list = self.__get_team_list_page(prospective_page, year=year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
                except EmptyError:
                    break

                team_list += partial_list
                team_list.raw += partial_list.raw

        return team_list

    # HELPER: get single page of team data
    def __get_team_list_page(self, page, *, year=None, force_new=False, force_cache=False, log_cache=False):
        """HELPER METHOD: Get a single page of teams."""
        return_array = self.pull_response_json('/teams/{}{}'.format('{}/'.format(year) if not year is None else '', str(page)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)

        page_list = DataList(objects=[], raw=return_array)

        for item in return_array:
            page_list.append(Team(item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}))

        return page_list

    # Get a List of FRC Team Keys
    def get_team_key_list(self, *, page=None, year=None, force_new=False, force_cache=False, log_cache=False):
        """Get a list of team keys.

        :param int page: *(opt)* The page of teams to fetch.  Argument must be named explicitly when passing.  If not included, method will pull all available pages.
        :param int year: *(opt)* The year to fetch competing teams for. Argument must be named explicitly when passing.  If not included, method will pull all teams that have ever competed and are recorded in the TBA databases."""
        if not page is None:
            return self.__get_team_list_page(page, year=year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        else:
            key_list = []

            for prospective_page in range(0,100):
                try:
                    partial_list = self.__get_team_key_list_page(prospective_page, year=year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
                except EmptyError:
                    break

                key_list += partial_list

        return key_list
    
    # HELPER: get single page of team keys
    def __get_team_key_list_page(self, page, *, year=None, force_new=False, force_cache=False, log_cache=False):
        """HELPER METHOD: Get a single page of team keys."""
        return self.pull_response_json('/teams/{}{}/keys'.format('{}/'.format(year) if not year is None else '', str(page)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)

    # HELPER: convert team_number or team_key input to uniform team_key
    def __get_team_key(self, team_identifier):
        """Convert team_number or team_key to uniform team_key for request path generation."""
        if str(team_identifier).startswith('frc'):
            return team_identifier
        else:
            try:
                int(team_identifier)
            except ValueError:
                raise KeyInputError

            return 'frc' + str(team_identifier)

    # Get information about a single FRC Team by team number or team key
    def get_team(self, team_identifier, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`Team`: Get a single team.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`."""
        return Team(self.pull_response_json('/team/{}'.format(self.__get_team_key(team_identifier)), force_new=force_new, force_cache=force_cache, log_cache=log_cache), self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache})

    # Get a list containing the years in which a given team has participated
    def get_team_years_participated(self, team_identifier, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`list`: Get a list containing the years in which a given team has participated.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`."""
        return self.pull_response_json('/team/{}/years_participated'.format(self.__get_team_key(team_identifier)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
    
    # Get a list of Districts that the given team has competed in
    def get_team_districts(self, team_identifier, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`District` objects in which the given team has competed in.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`."""
        district_list = self.pull_response_json('/team/{}/districts'.format(self.__get_team_key(team_identifier)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[District(district_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for district_item in district_list], raw=district_list)

    # Get a list of Robots built and operated by a given team
    def get_team_robots(self, team_identifier, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Team_Robot` objects built and operated by a given team.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`."""
        robot_list = self.pull_response_json('/team/{}/robots'.format(self.__get_team_key(team_identifier)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Team_Robot(robot_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for robot_item in robot_list], raw=robot_list)

    # Get a list of Social Media Presences operated by a given team.
    def get_team_social_media(self, team_identifier, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Team_Media` objects (Social Media Prescenses) operated by a given team.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
            
        Alias: *get_team_social*"""
        social_list = self.pull_response_json('/team/{}/social_media'.format(self.__get_team_key(team_identifier)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Team_Media(social_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for social_item in social_list], raw=social_list)

    # ALIAS: get_team_social_media
    def get_team_social(self, team_identifier, *, force_new=False, force_cache=False, log_cache=False):
        return self.get_team_social_media(team_identifier, force_new=force_new, force_cache=force_cache, log_cache=log_cache)

    # Get a list of events attended by a given team
    def get_team_events(self, team_identifier, year=None, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Event` objects attended by a given team.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg int year: *(opt)* The Year to pull events for.  If not included, method will pull all events in which a team has competed in."""
        if not year:
            event_list = self.pull_response_json('/team/{}/events'.format(self.__get_team_key(team_identifier)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
            return DataList(objects=[Event(event_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for event_item in event_list], raw=event_list)
        else:
            event_list = self.pull_response_json('/team/{}/events/{}'.format(self.__get_team_key(team_identifier), str(year)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
            return DataList(objects=[Event(event_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for event_item in event_list], raw=event_list)

    # Get a list of Matches played by a given Team at a given Event
    def get_team_event_matches(self, team_identifier, event_key, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Event_Match` objects played by a given Team at a given Event.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg str event_key: The :attr:`event_key` of the desired Event, following the format `YYYY[EVENT_CODE]`, where `YYYY` is the the year and `EVENT_CODE` is the `event code <https://docs.google.com/spreadsheet/ccc?key=0ApRO2Yzh2z01dExFZEdieV9WdTJsZ25HSWI3VUxsWGc>`_ of the event."""
        match_list = self.pull_response_json('/team/{}/event/{}/matches'.format(self.__get_team_key(team_identifier), event_key), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Event_Match(match_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for match_item in match_list], raw=match_list)

    # Get a list of Awards presented to a given Team at a given Event
    def get_team_event_awards(self, team_identifier, event_key, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Award` objects presented to a given Team at a given Event.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg str event_key: The :attr:`event_key` of the desired Event, following the format `YYYY[EVENT_CODE]`, where `YYYY` is the the year and `EVENT_CODE` is the `event code <https://docs.google.com/spreadsheet/ccc?key=0ApRO2Yzh2z01dExFZEdieV9WdTJsZ25HSWI3VUxsWGc>`_ of the event."""
        award_list = self.pull_response_json('/team/{}/event/{}/awards'.format(self.__get_team_key(team_identifier), event_key), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Award(award_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for award_item in award_list], raw=award_list)

    # Get information about the status of a given team at a given event
    def get_team_event_status(self, team_identifier, event_key, *, force_new=False, force_cache=False, log_cache=False):
        """Get information about the status of a given team at a given event.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg str event_key: The :attr:`event_key` of the desired Event, following the format `YYYY[EVENT_CODE]`, where `YYYY` is the the year and `EVENT_CODE` is the `event code <https://docs.google.com/spreadsheet/ccc?key=0ApRO2Yzh2z01dExFZEdieV9WdTJsZ25HSWI3VUxsWGc>`_ of the event."""
        return self.pull_response_json('/team/{}/event/{}/status'.format(self.__get_team_key(team_identifier), event_key), force_new=force_new, force_cache=force_cache, log_cache=log_cache)

    # Get a list of Awards presented to a given Team throughout their History
    def get_team_awards(self, team_identifier, year=None, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Award` objects presented to a given Team throughout their History.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg int year: *(opt)* The Year to pull awards for.  If not included, method will pull all awards won by the team."""
        award_list = self.pull_response_json('/team/{}/awards{}'.format(self.__get_team_key(team_identifier), '/{}'.format(year) if year else ''), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Award(award_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for award_item in award_list], raw=award_list)

    # Get a list of Matches played by a given Team throughout their History
    def get_team_matches(self, team_identifier, year=None, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Event_Match` objects played by a given Team throughout their History.
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg int year: *(opt)* The Year to pull matches for.  If not included, method will pull all matches played by the team."""
        if year:
            return self.__get_team_year_matches(team_identifier, year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        else:
            try:
                rookie_year = self.get_team(team_identifier, force_new=force_new, force_cache=force_cache, log_cache=log_cache).rookie_year
            except:
                raise Exception('YearParseError:  Could not find team\'s rookie year.')

            try:
                current_year = self.get_status(force_new=force_new, force_cache=force_cache, log_cache=log_cache).current_season
            except:
                raise Exception('YearParseError:  Could not find current FRC Season.')

            final_data = DataList()

            for test_year in range(rookie_year, current_year + 1):
                try:
                    partial_data = self.__get_team_year_matches(team_identifier, test_year)

                    final_data += partial_data
                    final_data.raw += partial_data.raw
                except EmptyError:
                    pass

            return final_data

    # HELPER: Get a single page of Team Matches from a given year
    def __get_team_year_matches(self, team_identifier, year, *, force_new=False, force_cache=False, log_cache=False):
        """HELPER: Get a single page of Team Matches from a given year."""
        match_list = self.pull_response_json('/team/{}/matches/{}'.format(self.__get_team_key(team_identifier), str(year)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Event_Match(match_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for match_item in match_list], raw=match_list)

    # Get a list of Media provided by a given Team throughout their History
    def get_team_media(self, team_identifier, year=None, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: a list of :class:`Team_Media` objects provided by a given Team throughout their History
        
        :arg int/str team_identifier: Either a team's :attr:`team_key` or their :attr:`team_number`.
        :arg int year: *(opt)* The Year to pull media for.  If not included, method will pull all media provided by the team."""
        if year:
            return self.__get_team_year_media(team_identifier, year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        else:
            try:
                rookie_year = self.get_team(team_identifier, force_new=force_new, force_cache=force_cache, log_cache=log_cache).rookie_year
            except:
                raise Exception('YearParseError:  Could not find team\'s rookie year.')

            try:
                current_year = self.get_status(force_new=force_new, force_cache=force_cache, log_cache=log_cache).current_season
            except:
                raise Exception('YearParseError:  Could not find current FRC Season.')

            final_data = DataList()

            for test_year in range(rookie_year, current_year + 1):
                try:
                    partial_data = self.__get_team_year_media(team_identifier, test_year)

                    final_data += partial_data
                    final_data.raw += partial_data.raw
                except EmptyError:
                    pass

            return final_data

    # HELPER: Get a single page of Team Media from a given year
    def __get_team_year_media(self, team_identifier, year, *, force_new=False, force_cache=False, log_cache=False):
        """HELPER: Get a single page of Team Media from a given year."""
        media_list = self.pull_response_json('/team/{}/media/{}'.format(self.__get_team_key(team_identifier), str(year)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Team_Media(media_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for media_item in media_list], raw=media_list)

    # Get a list of Events that took place in a given year or throughout FIRST history
    def get_event_list(self, year=None, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`DataList`: Get a list of :class:`Event` objects.
        
        :arg int year: The year to grab events for."""
        if year:
            return self.__get_event_list_year(year, force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        else:
            try:
                current_year = int(self.get_status(force_new=force_new, force_cache=force_cache, log_cache=log_cache).current_season)
            except:
                raise Exception('YearParseError:  Could not find current FRC Season.')

            final_data = DataList([], [])

            for test_year in range(1992, current_year + 1):
                try:
                    partial_data = self.__get_event_list_year(test_year)

                    final_data += partial_data
                    final_data.raw += partial_data.raw
                except EmptyError:
                    pass

            return final_data

    
    # HELPER: Get a single page of Events from a given year
    def __get_event_list_year(self, year, *, force_new=False, force_cache=False, log_cache=False):
        """HELPER: Get a single page of Events from a given year."""
        event_list = self.pull_response_json('/events/{}'.format(str(year)), force_new=force_new, force_cache=force_cache, log_cache=log_cache)
        return DataList(objects=[Event(event_item, self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache}) for event_item in event_list], raw=event_list)

    # Get a single event, by event key.
    def get_event(self, event_key, *, force_new=False, force_cache=False, log_cache=False):
        """:class:`Event`: Get a single event by it's event_key.
        
        :arg str event_key: The :attr:`event_key` of the desired Event, following the format `YYYY[EVENT_CODE]`, where `YYYY` is the the year and `EVENT_CODE` is the `event code <https://docs.google.com/spreadsheet/ccc?key=0ApRO2Yzh2z01dExFZEdieV9WdTJsZ25HSWI3VUxsWGc>`_ of the event."""
        return Event(self.pull_response_json('/event/{}'.format(event_key), force_new=force_new, force_cache=force_cache, log_cache=log_cache), self, {'force_new':force_new, 'force_cache':force_cache, 'log_cache':log_cache})
