"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.backends = new Array();
        this.listeners = new Array();
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
function renderHealthCheck(hc, pm) {
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
    return healthCheck;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: {
                    accessLog: {
                        file: {
                            path: '/dev/stdout',
                        },
                    },
                },
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read it's properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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