"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mesh = exports.MeshFilterType = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
const virtual_service_1 = require("./virtual-service");
/**
 * A utility enum defined for the egressFilter type property, the default of DROP_ALL,
 * allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 */
var MeshFilterType;
(function (MeshFilterType) {
    /**
     * Allows all outbound traffic
     */
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    /**
     * Allows traffic only to other resources inside the mesh, or API calls to amazon resources
     */
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * Adds a VirtualRouter to the Mesh with the given id and props
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualService with the given id
     */
    addVirtualService(id, props = {}) {
        return new virtual_service_1.VirtualService(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualNode to the Mesh
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * Define a new AppMesh mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 */
class Mesh extends MeshBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing mesh by arn
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing mesh by name
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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