# Copyright (C) 2021-2022 Greenbone Networks GmbH
#
# SPDX-License-Identifier: AGPL-3.0-or-later
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

from datetime import datetime
from unittest import TestCase

from notus.scanner.models.advisory import Advisory, Severity
from notus.scanner.models.packages.rpm import RPMPackage
from notus.scanner.models.vulnerability import PackageVulnerability


class PackageVulnerabilityTestCase(TestCase):
    def test_constructor(self):
        host_ip = "123.456.789.10"
        host_name = "foo"

        severity_date = datetime(year=2021, month=1, day=2, hour=11, minute=11)
        creation_date = datetime(year=2021, month=3, day=21, hour=10, minute=0)
        last_modification = datetime.now()
        severity = Severity(
            origin="foo",
            date=severity_date,
        )
        advisory = Advisory(
            oid="1.2.3.4.5",
            title="Foo Bar",
            creation_date=creation_date,
            last_modification=last_modification,
            advisory_id="1234",
            advisory_xref="http://foo/1234",
            severity=severity,
        )

        package = RPMPackage.from_full_name(
            "foo-bar-1.2.3-4.x86_64",
        )

        fixed_package = RPMPackage.from_full_name(
            "foo-bar-1.2.4-4.x86_64",
        )

        package_vulnerability = PackageVulnerability(
            host_ip=host_ip,
            host_name=host_name,
            advisory=advisory,
            package=package,
            fixed_package=fixed_package,
        )

        self.assertEqual(package_vulnerability.host_ip, host_ip)
        self.assertEqual(package_vulnerability.host_name, host_name)

        advisory = package_vulnerability.advisory
        self.assertEqual(advisory.oid, "1.2.3.4.5")

        package, fixed_package = next(
            iter(package_vulnerability.packages.items())
        )
        self.assertEqual(package.name, "foo-bar")
        self.assertEqual(package.version, "1.2.3")  # pylint: disable=no-member
        self.assertEqual(package.release, "4")  # pylint: disable=no-member
        self.assertEqual(package.full_name, "foo-bar-1.2.3-4.x86_64")

        self.assertEqual(fixed_package.name, "foo-bar")
        self.assertEqual(
            fixed_package.version, "1.2.4"  # pylint: disable=no-member
        )
        self.assertEqual(
            fixed_package.release, "4"  # pylint: disable=no-member
        )
        self.assertEqual(fixed_package.full_name, "foo-bar-1.2.4-4.x86_64")
