"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InterfaceVpcEndpoint = exports.InterfaceVpcEndpointAwsService = exports.InterfaceVpcEndpointService = exports.GatewayVpcEndpoint = exports.GatewayVpcEndpointAwsService = exports.VpcEndpointType = exports.VpcEndpoint = void 0;
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
/**
 *
 */
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add.
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    VpcEndpointType["INTERFACE"] = "Interface";
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    /**
     *
     */
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
/**
 *
 */
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
/**
 *
 */
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 *
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     *
     */
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    /**
     *
     */
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    /**
     *
     */
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        const region = core_1.Lazy.stringValue({
            produce: (context) => core_1.Stack.of(context.scope).region,
        });
        this.name = `${prefix || 'com.amazonaws'}.${region}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
/**
 *
 */
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
/**
 *
 */
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
/**
 *
 */
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
/**
 *
 */
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
/**
 *
 */
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
/**
 *
 */
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
/**
 *
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
/**
 *
 */
InterfaceVpcEndpointAwsService.KINESIS_FIREHOSE = new InterfaceVpcEndpointAwsService('kinesis-firehose');
/**
 *
 */
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
/**
 *
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
/**
 *
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
/**
 *
 */
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
/**
 *
 */
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
/**
 *
 */
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
/**
 *
 */
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
/**
 *
 */
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
/**
 *
 */
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
/**
 *
 */
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
/**
 *
 */
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
/**
 *
 */
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
/**
 *
 */
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 *
 */
InterfaceVpcEndpointAwsService.STEP_FUNCTIONS = new InterfaceVpcEndpointAwsService('states');
/**
 * A interface VPC endpoint.
 *
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        // Determine which subnets to place the endpoint in
        const subnetIds = this.endpointSubnets(props);
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Determine which subnets to place the endpoint in. This is in its own function
     * because there's a lot of code.
     */
    endpointSubnets(props) {
        var _a;
        const lookupSupportedAzs = (_a = props.lookupSupportedAzs) !== null && _a !== void 0 ? _a : false;
        const subnetSelection = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        // If we don't have an account/region, we will not be able to do filtering on AZs since
        // they will be undefined
        const agnosticAcct = core_1.Token.isUnresolved(this.stack.account);
        const agnosticRegion = core_1.Token.isUnresolved(this.stack.region);
        // Some service names, such as AWS service name references, use Tokens to automatically
        // fill in the region
        // If it is an InterfaceVpcEndpointAwsService, then the reference will be resolvable since
        // only references the region
        const isAwsService = core_1.Token.isUnresolved(props.service.name) && props.service instanceof InterfaceVpcEndpointAwsService;
        // Determine what name we pass to the context provider, either the verbatim name
        // or a resolved version if it is an AWS service reference
        let lookupServiceName = props.service.name;
        if (isAwsService && !agnosticRegion) {
            lookupServiceName = core_1.Stack.of(this).resolve(props.service.name);
        }
        else {
            // It's an agnostic service and we don't know how to resolve it.
            // This is ok if the stack is region agnostic and we're not looking up
            // AZs
            lookupServiceName = props.service.name;
        }
        // Check if lookup is impossible and throw an appropriate error
        // Context provider cannot make an AWS call without an account/region
        if ((agnosticAcct || agnosticRegion) && lookupSupportedAzs) {
            throw new Error('Cannot look up VPC endpoint availability zones if account/region are not specified');
        }
        // Context provider doesn't know the name of the service if there is a Token
        // in the name
        const agnosticService = core_1.Token.isUnresolved(lookupServiceName);
        if (agnosticService && lookupSupportedAzs) {
            throw new Error(`Cannot lookup AZs for a service name with a Token: ${props.service.name}`);
        }
        // Here we do the actual lookup for AZs, if told to do so
        let subnets;
        if (lookupSupportedAzs) {
            const availableAZs = this.availableAvailabilityZones(lookupServiceName);
            subnets = subnetSelection.subnets.filter(s => availableAZs.includes(s.availabilityZone));
        }
        else {
            subnets = subnetSelection.subnets;
        }
        const subnetIds = subnets.map(s => s.subnetId);
        return subnetIds;
    }
    availableAvailabilityZones(serviceName) {
        // Here we check what AZs the endpoint service is available in
        // If for whatever reason we can't retrieve the AZs, and no context is set,
        // we will fall back to all AZs
        const availableAZs = core_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
            dummyValue: this.stack.availabilityZones,
            props: { serviceName },
        }).value;
        if (!Array.isArray(availableAZs)) {
            throw new Error(`Discovered AZs for endpoint service ${serviceName} must be an array`);
        }
        return availableAZs;
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBjLWVuZHBvaW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidnBjLWVuZHBvaW50LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUN4QywyREFBMkQ7QUFDM0Qsd0NBQThGO0FBRTlGLCtDQUEwRDtBQUMxRCxtREFBaUQ7QUFDakQsaUNBQThCO0FBQzlCLGlDQUE4QjtBQUM5QixxREFBaUU7QUFDakUsaUNBQW1EOzs7O0FBY25ELE1BQXNCLFdBQVksU0FBUSxlQUFROzs7Ozs7Ozs7SUFjekMsV0FBVyxDQUFDLFNBQThCO1FBQy9DLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0NBQW9DLENBQUMsQ0FBQztTQUN2RDtRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3hCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDaEQ7UUFFRCxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMvQyxDQUFDO0NBQ0Y7QUF6QkQsa0NBeUJDOzs7O0FBV0QsSUFBWSxlQWlCWDtBQWpCRCxXQUFZLGVBQWU7SUFRekIsMENBQXVCLENBQUE7SUFRdkIsc0NBQW1CLENBQUE7QUFDckIsQ0FBQyxFQWpCVyxlQUFlLEdBQWYsdUJBQWUsS0FBZix1QkFBZSxRQWlCMUI7Ozs7QUFlRCxNQUFhLDRCQUE0Qjs7OztJQVN2QyxZQUFZLElBQVksRUFBRSxNQUFlO1FBQ3ZDLElBQUksQ0FBQyxJQUFJLEdBQUcsR0FBRyxNQUFNLElBQUksZUFBZSxJQUFJLFVBQUcsQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFLENBQUM7SUFDbkUsQ0FBQzs7QUFYSCxvRUFZQzs7OztBQVh3QixxQ0FBUSxHQUFHLElBQUksNEJBQTRCLENBQUMsVUFBVSxDQUFDLENBQUM7Ozs7QUFDeEQsK0JBQUUsR0FBRyxJQUFJLDRCQUE0QixDQUFDLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUF5RHJFLE1BQWEsa0JBQW1CLFNBQVEsV0FBVzs7OztJQStCakQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4QjtRQUN0RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sT0FBTyxHQUFjLEtBQUssQ0FBQyxPQUFPO1lBQ3RDLENBQUMsQ0FBQyxjQUFPLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUNyRSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsY0FBYyxFQUFFLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLEVBQUUsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzVGLE1BQU0sYUFBYSxHQUFHLHVCQUFnQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWhELElBQUksYUFBYSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RUFBeUUsQ0FBQyxDQUFDO1NBQzVGO1FBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSw4QkFBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsY0FBYyxFQUFFLFdBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ3JFLGFBQWE7WUFDYixXQUFXLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJO1lBQy9CLGVBQWUsRUFBRSxlQUFlLENBQUMsT0FBTztZQUN4QyxLQUFLLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxLQUFLO1NBQ3ZCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxhQUFhLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUNsQyxJQUFJLENBQUMsNEJBQTRCLEdBQUcsUUFBUSxDQUFDLHFCQUFxQixDQUFDO1FBQ25FLElBQUksQ0FBQyxxQkFBcUIsR0FBRyxRQUFRLENBQUMsY0FBYyxDQUFDO1FBQ3JELElBQUksQ0FBQyw4QkFBOEIsR0FBRyxRQUFRLENBQUMsdUJBQXVCLENBQUM7SUFDekUsQ0FBQzs7OztJQXJETSxNQUFNLENBQUMsd0JBQXdCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsb0JBQTRCO1FBQy9GLE1BQU0sTUFBTyxTQUFRLFdBQVc7WUFBaEM7O2dCQUNTLGtCQUFhLEdBQUcsb0JBQW9CLENBQUM7WUFDOUMsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQztDQWdERjtBQXhERCxnREF3REM7Ozs7QUF5QkQsTUFBYSwyQkFBMkI7Ozs7SUFpQnRDLFlBQVksSUFBWSxFQUFFLElBQWE7Ozs7UUFGdkIsc0JBQWlCLEdBQWEsS0FBSyxDQUFDO1FBR2xELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2pCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxJQUFJLEdBQUcsQ0FBQztJQUMxQixDQUFDO0NBQ0Y7QUFyQkQsa0VBcUJDOzs7O0FBS0QsTUFBYSw4QkFBOEI7Ozs7SUE2RHpDLFlBQVksSUFBWSxFQUFFLE1BQWUsRUFBRSxJQUFhOzs7O1FBRnhDLHNCQUFpQixHQUFhLElBQUksQ0FBQztRQUdqRCxNQUFNLE1BQU0sR0FBRyxXQUFJLENBQUMsV0FBVyxDQUFDO1lBQzlCLE9BQU8sRUFBRSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsWUFBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTTtTQUNyRCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxHQUFHLEdBQUcsTUFBTSxJQUFJLGVBQWUsSUFBSSxNQUFNLElBQUksSUFBSSxFQUFFLENBQUM7UUFDN0QsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLElBQUksR0FBRyxDQUFDO0lBQzFCLENBQUM7O0FBbkVILHdFQW9FQzs7OztBQW5Fd0IsaURBQWtCLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxVQUFVLEVBQUUsZUFBZSxDQUFDLENBQUM7Ozs7QUFDckYsNkNBQWMsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGdCQUFnQixDQUFDLENBQUM7Ozs7QUFDdEUseUNBQVUsR0FBRyxJQUFJLDhCQUE4QixDQUFDLFlBQVksQ0FBQyxDQUFDOzs7O0FBQzlELHdDQUFTLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxXQUFXLENBQUMsQ0FBQzs7OztBQUM1RCw2Q0FBYyxHQUFHLElBQUksOEJBQThCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQzs7OztBQUN0RSx5Q0FBVSxHQUFHLElBQUksOEJBQThCLENBQUMsWUFBWSxDQUFDLENBQUM7Ozs7QUFDOUQsOENBQWUsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGlCQUFpQixDQUFDLENBQUM7Ozs7QUFDeEUsMkNBQVksR0FBRyxJQUFJLDhCQUE4QixDQUFDLGNBQWMsQ0FBQyxDQUFDOzs7O0FBQ2xFLHFDQUFNLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxRQUFRLENBQUMsQ0FBQzs7OztBQUN0RCxrQ0FBRyxHQUFHLElBQUksOEJBQThCLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7QUFDaEQsMkNBQVksR0FBRyxJQUFJLDhCQUE4QixDQUFDLGFBQWEsQ0FBQyxDQUFDOzs7O0FBQ2pFLGtDQUFHLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxTQUFTLENBQUMsQ0FBQzs7OztBQUNwRCx5Q0FBVSxHQUFHLElBQUksOEJBQThCLENBQUMsU0FBUyxDQUFDLENBQUM7Ozs7QUFDM0Qsa0NBQUcsR0FBRyxJQUFJLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxDQUFDOzs7O0FBQ2hELHdDQUFTLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxXQUFXLENBQUMsQ0FBQzs7OztBQUM1RCw0Q0FBYSxHQUFHLElBQUksOEJBQThCLENBQUMsZUFBZSxDQUFDLENBQUM7Ozs7QUFDcEUsaURBQWtCLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDOzs7O0FBQzdFLHNEQUF1QixHQUFHLElBQUksOEJBQThCLENBQUMsd0JBQXdCLENBQUMsQ0FBQzs7OztBQUN2Rix3REFBeUIsR0FBRyxJQUFJLDhCQUE4QixDQUFDLDJCQUEyQixDQUFDLENBQUM7Ozs7QUFDNUYscURBQXNCLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDOzs7O0FBQ3BGLGdEQUFpQixHQUFHLElBQUksOEJBQThCLENBQUMsUUFBUSxDQUFDLENBQUM7Ozs7QUFDakUseUNBQVUsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGFBQWEsQ0FBQyxDQUFDOzs7O0FBQy9ELDZDQUFjLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDOzs7O0FBQ3RFLGtEQUFtQixHQUFHLElBQUksOEJBQThCLENBQUMscUJBQXFCLENBQUMsQ0FBQzs7OztBQUNoRiw4Q0FBZSxHQUFHLElBQUksOEJBQThCLENBQUMsaUJBQWlCLENBQUMsQ0FBQzs7OztBQUN4RSwrQ0FBZ0IsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGtCQUFrQixDQUFDLENBQUM7Ozs7QUFDMUUsa0NBQUcsR0FBRyxJQUFJLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxDQUFDOzs7O0FBQ2hELDhDQUFlLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxNQUFNLENBQUMsQ0FBQzs7OztBQUM3RCx5Q0FBVSxHQUFHLElBQUksOEJBQThCLENBQUMsWUFBWSxDQUFDLENBQUM7Ozs7QUFDOUQsNENBQWEsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGVBQWUsQ0FBQyxDQUFDOzs7O0FBQ3BFLGdEQUFpQixHQUFHLElBQUksOEJBQThCLENBQUMsbUJBQW1CLENBQUMsQ0FBQzs7OztBQUM1RSxxREFBc0IsR0FBRyxJQUFJLDhCQUE4QixDQUFDLHdCQUF3QixDQUFDLENBQUM7Ozs7QUFDdEYsOENBQWUsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGdCQUFnQixDQUFDLENBQUM7Ozs7QUFDdkUsOENBQWUsR0FBRyxJQUFJLDhCQUE4QixDQUFDLGdCQUFnQixDQUFDLENBQUM7Ozs7QUFDdkUsa0NBQUcsR0FBRyxJQUFJLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxDQUFDOzs7O0FBQ2hELGtDQUFHLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7OztBQUNoRCxrQ0FBRyxHQUFHLElBQUksOEJBQThCLENBQUMsS0FBSyxDQUFDLENBQUM7Ozs7QUFDaEQsMkNBQVksR0FBRyxJQUFJLDhCQUE4QixDQUFDLGFBQWEsQ0FBQyxDQUFDOzs7O0FBQ2pFLGtDQUFHLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7OztBQUNoRCx1Q0FBUSxHQUFHLElBQUksOEJBQThCLENBQUMsaUJBQWlCLENBQUMsQ0FBQzs7OztBQUNqRSw4Q0FBZSxHQUFHLElBQUksOEJBQThCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQzs7OztBQUN2RSwwQ0FBVyxHQUFHLElBQUksOEJBQThCLENBQUMsYUFBYSxDQUFDLENBQUM7Ozs7QUFDaEUsK0NBQWdCLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDOzs7O0FBQzFFLDZDQUFjLEdBQUcsSUFBSSw4QkFBOEIsQ0FBQyxRQUFRLENBQUMsQ0FBQzs7Ozs7O0FBb0d2RixNQUFhLG9CQUFxQixTQUFRLFdBQVc7Ozs7SUE0RG5ELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBZ0M7O1FBQ3hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLElBQUksOEJBQWEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO2dCQUN2RixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7YUFDZixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxlQUFlLEdBQUcsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztRQUN6RCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUkseUJBQVcsQ0FBQztZQUNqQyxXQUFXLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztZQUN6QyxjQUFjO1NBQ2YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMsSUFBSSxLQUFLLEtBQUssRUFBRTtZQUN4QixJQUFJLENBQUMsV0FBVyxDQUFDLG9CQUFvQixDQUFDLFdBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO1NBQzFFO1FBRUQsbURBQW1EO1FBQ25ELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFOUMsTUFBTSxRQUFRLEdBQUcsSUFBSSw4QkFBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsaUJBQWlCLGNBQUUsS0FBSyxDQUFDLGlCQUFpQixtQ0FBSSxLQUFLLENBQUMsT0FBTyxDQUFDLGlCQUFpQixtQ0FBSSxJQUFJO1lBQ3JGLGNBQWMsRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztZQUNyRSxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztZQUM1RCxXQUFXLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJO1lBQy9CLGVBQWUsRUFBRSxlQUFlLENBQUMsU0FBUztZQUMxQyxTQUFTO1lBQ1QsS0FBSyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsS0FBSztTQUN2QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFDbEMsSUFBSSxDQUFDLDRCQUE0QixHQUFHLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztRQUNuRSxJQUFJLENBQUMscUJBQXFCLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQztRQUNyRCxJQUFJLENBQUMsOEJBQThCLEdBQUcsUUFBUSxDQUFDLHVCQUF1QixDQUFDO0lBQ3pFLENBQUM7Ozs7SUExRk0sTUFBTSxDQUFDLGtDQUFrQyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFDO1FBQ2xILElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxDQUFDLENBQUM7U0FDcEY7UUFFRCxNQUFNLGNBQWMsR0FBRyxLQUFLLENBQUMsZUFBZTtZQUMxQyxDQUFDLENBQUMsQ0FBQyw4QkFBYSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1lBQ3BGLENBQUMsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBRXpCLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixrQkFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7Z0JBQ3BDLGdCQUFXLEdBQUcsSUFBSSx5QkFBVyxDQUFDO29CQUM1QyxXQUFXLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO29CQUNqQyxjQUFjO2lCQUNmLENBQUMsQ0FBQztZQUNMLENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUEwRUQ7OztPQUdHO0lBQ0ssZUFBZSxDQUFDLEtBQWdDOztRQUN0RCxNQUFNLGtCQUFrQixTQUFHLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksS0FBSyxDQUFDO1FBQzdELE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRXRGLHVGQUF1RjtRQUN2Rix5QkFBeUI7UUFDekIsTUFBTSxZQUFZLEdBQUcsWUFBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzVELE1BQU0sY0FBYyxHQUFHLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUU3RCx1RkFBdUY7UUFDdkYscUJBQXFCO1FBQ3JCLDBGQUEwRjtRQUMxRiw2QkFBNkI7UUFDN0IsTUFBTSxZQUFZLEdBQUcsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssQ0FBQyxPQUFPLFlBQVksOEJBQThCLENBQUM7UUFFdkgsZ0ZBQWdGO1FBQ2hGLDBEQUEwRDtRQUMxRCxJQUFJLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQzNDLElBQUksWUFBWSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ25DLGlCQUFpQixHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDaEU7YUFBTTtZQUNMLGdFQUFnRTtZQUNoRSxzRUFBc0U7WUFDdEUsTUFBTTtZQUNOLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1NBQ3hDO1FBRUQsK0RBQStEO1FBQy9ELHFFQUFxRTtRQUNyRSxJQUFJLENBQUMsWUFBWSxJQUFJLGNBQWMsQ0FBQyxJQUFJLGtCQUFrQixFQUFFO1lBQzFELE1BQU0sSUFBSSxLQUFLLENBQUMsb0ZBQW9GLENBQUMsQ0FBQztTQUN2RztRQUNELDRFQUE0RTtRQUM1RSxjQUFjO1FBQ2QsTUFBTSxlQUFlLEdBQUcsWUFBSyxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzlELElBQUksZUFBZSxJQUFJLGtCQUFrQixFQUFFO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUM3RjtRQUVELHlEQUF5RDtRQUN6RCxJQUFJLE9BQU8sQ0FBQztRQUNaLElBQUksa0JBQWtCLEVBQUU7WUFDdEIsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLDBCQUEwQixDQUFDLGlCQUFpQixDQUFDLENBQUM7WUFDeEUsT0FBTyxHQUFHLGVBQWUsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1NBQzFGO2FBQU07WUFDTCxPQUFPLEdBQUcsZUFBZSxDQUFDLE9BQU8sQ0FBQztTQUNuQztRQUNELE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDL0MsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVPLDBCQUEwQixDQUFDLFdBQW1CO1FBQ3BELDhEQUE4RDtRQUM5RCwyRUFBMkU7UUFDM0UsK0JBQStCO1FBQy9CLE1BQU0sWUFBWSxHQUFHLHNCQUFlLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRTtZQUNsRCxRQUFRLEVBQUUsUUFBUSxDQUFDLGVBQWUsQ0FBQywyQ0FBMkM7WUFDOUUsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCO1lBQ3hDLEtBQUssRUFBRSxFQUFFLFdBQVcsRUFBRTtTQUN2QixDQUFDLENBQUMsS0FBSyxDQUFDO1FBQ1QsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsV0FBVyxtQkFBbUIsQ0FBQyxDQUFDO1NBQ3hGO1FBQ0QsT0FBTyxZQUFZLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBcktELG9EQXFLQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGN4c2NoZW1hIGZyb20gJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSc7XG5pbXBvcnQgeyBBd3MsIENvbnRleHRQcm92aWRlciwgSVJlc291cmNlLCBMYXp5LCBSZXNvdXJjZSwgU3RhY2ssIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENvbm5lY3Rpb25zLCBJQ29ubmVjdGFibGUgfSBmcm9tICcuL2Nvbm5lY3Rpb25zJztcbmltcG9ydCB7IENmblZQQ0VuZHBvaW50IH0gZnJvbSAnLi9lYzIuZ2VuZXJhdGVkJztcbmltcG9ydCB7IFBlZXIgfSBmcm9tICcuL3BlZXInO1xuaW1wb3J0IHsgUG9ydCB9IGZyb20gJy4vcG9ydCc7XG5pbXBvcnQgeyBJU2VjdXJpdHlHcm91cCwgU2VjdXJpdHlHcm91cCB9IGZyb20gJy4vc2VjdXJpdHktZ3JvdXAnO1xuaW1wb3J0IHsgYWxsUm91dGVUYWJsZUlkcywgZmxhdHRlbiB9IGZyb20gJy4vdXRpbCc7XG5pbXBvcnQgeyBJU3VibmV0LCBJVnBjLCBTdWJuZXRTZWxlY3Rpb24gfSBmcm9tICcuL3ZwYyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElWcGNFbmRwb2ludCBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwY0VuZHBvaW50SWQ6IHN0cmluZztcbn1cblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFZwY0VuZHBvaW50IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVnBjRW5kcG9pbnQge1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdnBjRW5kcG9pbnRJZDogc3RyaW5nO1xuXG4gIHByb3RlY3RlZCBwb2xpY3lEb2N1bWVudD86IGlhbS5Qb2xpY3lEb2N1bWVudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkVG9Qb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KSB7XG4gICAgaWYgKCFzdGF0ZW1lbnQuaGFzUHJpbmNpcGFsKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1N0YXRlbWVudCBtdXN0IGhhdmUgYSBgUHJpbmNpcGFsYC4nKTtcbiAgICB9XG5cbiAgICBpZiAoIXRoaXMucG9saWN5RG9jdW1lbnQpIHtcbiAgICAgIHRoaXMucG9saWN5RG9jdW1lbnQgPSBuZXcgaWFtLlBvbGljeURvY3VtZW50KCk7XG4gICAgfVxuXG4gICAgdGhpcy5wb2xpY3lEb2N1bWVudC5hZGRTdGF0ZW1lbnRzKHN0YXRlbWVudCk7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJR2F0ZXdheVZwY0VuZHBvaW50IGV4dGVuZHMgSVZwY0VuZHBvaW50IHtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBWcGNFbmRwb2ludFR5cGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIElOVEVSRkFDRSA9ICdJbnRlcmZhY2UnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBHQVRFV0FZID0gJ0dhdGV3YXknXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJR2F0ZXdheVZwY0VuZHBvaW50U2VydmljZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZSBpbXBsZW1lbnRzIElHYXRld2F5VnBjRW5kcG9pbnRTZXJ2aWNlIHtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBEWU5BTU9EQiA9IG5ldyBHYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdkeW5hbW9kYicpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFMzID0gbmV3IEdhdGV3YXlWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ3MzJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IobmFtZTogc3RyaW5nLCBwcmVmaXg/OiBzdHJpbmcpIHtcbiAgICB0aGlzLm5hbWUgPSBgJHtwcmVmaXggfHwgJ2NvbS5hbWF6b25hd3MnfS4ke0F3cy5SRUdJT059LiR7bmFtZX1gO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBHYXRld2F5VnBjRW5kcG9pbnRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzZXJ2aWNlOiBJR2F0ZXdheVZwY0VuZHBvaW50U2VydmljZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3VibmV0cz86IFN1Ym5ldFNlbGVjdGlvbltdXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBHYXRld2F5VnBjRW5kcG9pbnRQcm9wcyBleHRlbmRzIEdhdGV3YXlWcGNFbmRwb2ludE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM6IElWcGNcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgR2F0ZXdheVZwY0VuZHBvaW50IGV4dGVuZHMgVnBjRW5kcG9pbnQgaW1wbGVtZW50cyBJR2F0ZXdheVZwY0VuZHBvaW50IHtcblxuICBwdWJsaWMgc3RhdGljIGZyb21HYXRld2F5VnBjRW5kcG9pbnRJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBnYXRld2F5VnBjRW5kcG9pbnRJZDogc3RyaW5nKTogSUdhdGV3YXlWcGNFbmRwb2ludCB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgVnBjRW5kcG9pbnQge1xuICAgICAgcHVibGljIHZwY0VuZHBvaW50SWQgPSBnYXRld2F5VnBjRW5kcG9pbnRJZDtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB2cGNFbmRwb2ludElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHZwY0VuZHBvaW50Q3JlYXRpb25UaW1lc3RhbXA6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB2cGNFbmRwb2ludE5ldHdvcmtJbnRlcmZhY2VJZHM6IHN0cmluZ1tdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHZwY0VuZHBvaW50RG5zRW50cmllczogc3RyaW5nW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEdhdGV3YXlWcGNFbmRwb2ludFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHN1Ym5ldHM6IElTdWJuZXRbXSA9IHByb3BzLnN1Ym5ldHNcbiAgICAgID8gZmxhdHRlbihwcm9wcy5zdWJuZXRzLm1hcChzID0+IHByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKHMpLnN1Ym5ldHMpKVxuICAgICAgOiBbLi4ucHJvcHMudnBjLnByaXZhdGVTdWJuZXRzLCAuLi5wcm9wcy52cGMucHVibGljU3VibmV0cywgLi4ucHJvcHMudnBjLmlzb2xhdGVkU3VibmV0c107XG4gICAgY29uc3Qgcm91dGVUYWJsZUlkcyA9IGFsbFJvdXRlVGFibGVJZHMoc3VibmV0cyk7XG5cbiAgICBpZiAocm91dGVUYWJsZUlkcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuXFwndCBhZGQgYSBnYXRld2F5IGVuZHBvaW50IHRvIFZQQzsgcm91dGUgdGFibGUgSURzIGFyZSBub3QgYXZhaWxhYmxlJyk7XG4gICAgfVxuXG4gICAgY29uc3QgZW5kcG9pbnQgPSBuZXcgQ2ZuVlBDRW5kcG9pbnQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgcG9saWN5RG9jdW1lbnQ6IExhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnBvbGljeURvY3VtZW50IH0pLFxuICAgICAgcm91dGVUYWJsZUlkcyxcbiAgICAgIHNlcnZpY2VOYW1lOiBwcm9wcy5zZXJ2aWNlLm5hbWUsXG4gICAgICB2cGNFbmRwb2ludFR5cGU6IFZwY0VuZHBvaW50VHlwZS5HQVRFV0FZLFxuICAgICAgdnBjSWQ6IHByb3BzLnZwYy52cGNJZCxcbiAgICB9KTtcblxuICAgIHRoaXMudnBjRW5kcG9pbnRJZCA9IGVuZHBvaW50LnJlZjtcbiAgICB0aGlzLnZwY0VuZHBvaW50Q3JlYXRpb25UaW1lc3RhbXAgPSBlbmRwb2ludC5hdHRyQ3JlYXRpb25UaW1lc3RhbXA7XG4gICAgdGhpcy52cGNFbmRwb2ludERuc0VudHJpZXMgPSBlbmRwb2ludC5hdHRyRG5zRW50cmllcztcbiAgICB0aGlzLnZwY0VuZHBvaW50TmV0d29ya0ludGVyZmFjZUlkcyA9IGVuZHBvaW50LmF0dHJOZXR3b3JrSW50ZXJmYWNlSWRzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJSW50ZXJmYWNlVnBjRW5kcG9pbnRTZXJ2aWNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByaXZhdGVEbnNEZWZhdWx0PzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBJbnRlcmZhY2VWcGNFbmRwb2ludFNlcnZpY2UgaW1wbGVtZW50cyBJSW50ZXJmYWNlVnBjRW5kcG9pbnRTZXJ2aWNlIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHBvcnQ6IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcHJpdmF0ZURuc0RlZmF1bHQ/OiBib29sZWFuID0gZmFsc2U7XG5cbiAgY29uc3RydWN0b3IobmFtZTogc3RyaW5nLCBwb3J0PzogbnVtYmVyKSB7XG4gICAgdGhpcy5uYW1lID0gbmFtZTtcbiAgICB0aGlzLnBvcnQgPSBwb3J0IHx8IDQ0MztcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSBpbXBsZW1lbnRzIElJbnRlcmZhY2VWcGNFbmRwb2ludFNlcnZpY2Uge1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNBR0VNQUtFUl9OT1RFQk9PSyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ25vdGVib29rJywgJ2F3cy5zYWdlbWFrZXInKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDTE9VREZPUk1BVElPTiA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2Nsb3VkZm9ybWF0aW9uJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ0xPVURUUkFJTCA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2Nsb3VkdHJhaWwnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDT0RFQlVJTEQgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdjb2RlYnVpbGQnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDT0RFQlVJTERfRklQUyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2NvZGVidWlsZC1maXBzJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ09ERUNPTU1JVCA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2NvZGVjb21taXQnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDT0RFQ09NTUlUX0ZJUFMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdjb2RlY29tbWl0LWZpcHMnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDT0RFUElQRUxJTkUgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdjb2RlcGlwZWxpbmUnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDT05GSUcgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdjb25maWcnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFQzIgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdlYzInKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFQzJfTUVTU0FHRVMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdlYzJtZXNzYWdlcycpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVDUiA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2Vjci5hcGknKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFQ1JfRE9DS0VSID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnZWNyLmRrcicpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVDUyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2VjcycpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVDU19BR0VOVCA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2Vjcy1hZ2VudCcpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVDU19URUxFTUVUUlkgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdlY3MtdGVsZW1ldHJ5Jyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgRUxBU1RJQ19GSUxFU1lTVEVNID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnZWxhc3RpY2ZpbGVzeXN0ZW0nKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFTEFTVElDX0ZJTEVTWVNURU1fRklQUyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2VsYXN0aWNmaWxlc3lzdGVtLWZpcHMnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFTEFTVElDX0lORkVSRU5DRV9SVU5USU1FID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnZWxhc3RpYy1pbmZlcmVuY2UucnVudGltZScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVMQVNUSUNfTE9BRF9CQUxBTkNJTkcgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdlbGFzdGljbG9hZGJhbGFuY2luZycpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENMT1VEV0FUQ0hfRVZFTlRTID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnZXZlbnRzJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQVBJR0FURVdBWSA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2V4ZWN1dGUtYXBpJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ09ERUNPTU1JVF9HSVQgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdnaXQtY29kZWNvbW1pdCcpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENPREVDT01NSVRfR0lUX0ZJUFMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdnaXQtY29kZWNvbW1pdC1maXBzJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgS0lORVNJU19TVFJFQU1TID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgna2luZXNpcy1zdHJlYW1zJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgS0lORVNJU19GSVJFSE9TRSA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ2tpbmVzaXMtZmlyZWhvc2UnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBLTVMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdrbXMnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDTE9VRFdBVENIX0xPR1MgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdsb2dzJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ0xPVURXQVRDSCA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ21vbml0b3JpbmcnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTQUdFTUFLRVJfQVBJID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnc2FnZW1ha2VyLmFwaScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNBR0VNQUtFUl9SVU5USU1FID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnc2FnZW1ha2VyLnJ1bnRpbWUnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTQUdFTUFLRVJfUlVOVElNRV9GSVBTID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnc2FnZW1ha2VyLnJ1bnRpbWUtZmlwcycpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNFQ1JFVFNfTUFOQUdFUiA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ3NlY3JldHNtYW5hZ2VyJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU0VSVklDRV9DQVRBTE9HID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnc2VydmljZWNhdGFsb2cnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTTlMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdzbnMnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTUVMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdzcXMnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTU00gPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdzc20nKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTU01fTUVTU0FHRVMgPSBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnRBd3NTZXJ2aWNlKCdzc21tZXNzYWdlcycpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNUUyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ3N0cycpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFRSQU5TRkVSID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgndHJhbnNmZXIuc2VydmVyJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU1RPUkFHRV9HQVRFV0FZID0gbmV3IEludGVyZmFjZVZwY0VuZHBvaW50QXdzU2VydmljZSgnc3RvcmFnZWdhdGV3YXknKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSRUtPR05JVElPTiA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ3Jla29nbml0aW9uJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUkVLT0dOSVRJT05fRklQUyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ3Jla29nbml0aW9uLWZpcHMnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTVEVQX0ZVTkNUSU9OUyA9IG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UoJ3N0YXRlcycpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBwcml2YXRlRG5zRGVmYXVsdD86IGJvb2xlYW4gPSB0cnVlO1xuXG4gIGNvbnN0cnVjdG9yKG5hbWU6IHN0cmluZywgcHJlZml4Pzogc3RyaW5nLCBwb3J0PzogbnVtYmVyKSB7XG4gICAgY29uc3QgcmVnaW9uID0gTGF6eS5zdHJpbmdWYWx1ZSh7XG4gICAgICBwcm9kdWNlOiAoY29udGV4dCkgPT4gU3RhY2sub2YoY29udGV4dC5zY29wZSkucmVnaW9uLFxuICAgIH0pO1xuICAgIHRoaXMubmFtZSA9IGAke3ByZWZpeCB8fCAnY29tLmFtYXpvbmF3cyd9LiR7cmVnaW9ufS4ke25hbWV9YDtcbiAgICB0aGlzLnBvcnQgPSBwb3J0IHx8IDQ0MztcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSW50ZXJmYWNlVnBjRW5kcG9pbnRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlcnZpY2U6IElJbnRlcmZhY2VWcGNFbmRwb2ludFNlcnZpY2U7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJpdmF0ZURuc0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdWJuZXRzPzogU3VibmV0U2VsZWN0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogSVNlY3VyaXR5R3JvdXBbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG9wZW4/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvb2t1cFN1cHBvcnRlZEF6cz86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJbnRlcmZhY2VWcGNFbmRwb2ludFByb3BzIGV4dGVuZHMgSW50ZXJmYWNlVnBjRW5kcG9pbnRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM6IElWcGNcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJSW50ZXJmYWNlVnBjRW5kcG9pbnQgZXh0ZW5kcyBJVnBjRW5kcG9pbnQsIElDb25uZWN0YWJsZSB7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgSW50ZXJmYWNlVnBjRW5kcG9pbnQgZXh0ZW5kcyBWcGNFbmRwb2ludCBpbXBsZW1lbnRzIElJbnRlcmZhY2VWcGNFbmRwb2ludCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tSW50ZXJmYWNlVnBjRW5kcG9pbnRBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBJbnRlcmZhY2VWcGNFbmRwb2ludEF0dHJpYnV0ZXMpOiBJSW50ZXJmYWNlVnBjRW5kcG9pbnQge1xuICAgIGlmICghYXR0cnMuc2VjdXJpdHlHcm91cHMgJiYgIWF0dHJzLnNlY3VyaXR5R3JvdXBJZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdFaXRoZXIgYHNlY3VyaXR5R3JvdXBzYCBvciBgc2VjdXJpdHlHcm91cElkYCBtdXN0IGJlIHNwZWNpZmllZC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBzZWN1cml0eUdyb3VwcyA9IGF0dHJzLnNlY3VyaXR5R3JvdXBJZFxuICAgICAgPyBbU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKHNjb3BlLCAnU2VjdXJpdHlHcm91cCcsIGF0dHJzLnNlY3VyaXR5R3JvdXBJZCldXG4gICAgICA6IGF0dHJzLnNlY3VyaXR5R3JvdXBzO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJSW50ZXJmYWNlVnBjRW5kcG9pbnQge1xuICAgICAgcHVibGljIHJlYWRvbmx5IHZwY0VuZHBvaW50SWQgPSBhdHRycy52cGNFbmRwb2ludElkO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zID0gbmV3IENvbm5lY3Rpb25zKHtcbiAgICAgICAgZGVmYXVsdFBvcnQ6IFBvcnQudGNwKGF0dHJzLnBvcnQpLFxuICAgICAgICBzZWN1cml0eUdyb3VwcyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdnBjRW5kcG9pbnRJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHZwY0VuZHBvaW50Q3JlYXRpb25UaW1lc3RhbXA6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHZwY0VuZHBvaW50RG5zRW50cmllczogc3RyaW5nW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB2cGNFbmRwb2ludE5ldHdvcmtJbnRlcmZhY2VJZHM6IHN0cmluZ1tdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBzZWN1cml0eUdyb3VwSWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBDb25uZWN0aW9ucztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSW50ZXJmYWNlVnBjRW5kcG9pbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBzZWN1cml0eUdyb3VwcyA9IHByb3BzLnNlY3VyaXR5R3JvdXBzIHx8IFtuZXcgU2VjdXJpdHlHcm91cCh0aGlzLCAnU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgIH0pXTtcblxuICAgIHRoaXMuc2VjdXJpdHlHcm91cElkID0gc2VjdXJpdHlHcm91cHNbMF0uc2VjdXJpdHlHcm91cElkO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgQ29ubmVjdGlvbnMoe1xuICAgICAgZGVmYXVsdFBvcnQ6IFBvcnQudGNwKHByb3BzLnNlcnZpY2UucG9ydCksXG4gICAgICBzZWN1cml0eUdyb3VwcyxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy5vcGVuICE9PSBmYWxzZSkge1xuICAgICAgdGhpcy5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbShQZWVyLmlwdjQocHJvcHMudnBjLnZwY0NpZHJCbG9jaykpO1xuICAgIH1cblxuICAgIC8vIERldGVybWluZSB3aGljaCBzdWJuZXRzIHRvIHBsYWNlIHRoZSBlbmRwb2ludCBpblxuICAgIGNvbnN0IHN1Ym5ldElkcyA9IHRoaXMuZW5kcG9pbnRTdWJuZXRzKHByb3BzKTtcblxuICAgIGNvbnN0IGVuZHBvaW50ID0gbmV3IENmblZQQ0VuZHBvaW50KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHByaXZhdGVEbnNFbmFibGVkOiBwcm9wcy5wcml2YXRlRG5zRW5hYmxlZCA/PyBwcm9wcy5zZXJ2aWNlLnByaXZhdGVEbnNEZWZhdWx0ID8/IHRydWUsXG4gICAgICBwb2xpY3lEb2N1bWVudDogTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucG9saWN5RG9jdW1lbnQgfSksXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiBzZWN1cml0eUdyb3Vwcy5tYXAocyA9PiBzLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICBzZXJ2aWNlTmFtZTogcHJvcHMuc2VydmljZS5uYW1lLFxuICAgICAgdnBjRW5kcG9pbnRUeXBlOiBWcGNFbmRwb2ludFR5cGUuSU5URVJGQUNFLFxuICAgICAgc3VibmV0SWRzLFxuICAgICAgdnBjSWQ6IHByb3BzLnZwYy52cGNJZCxcbiAgICB9KTtcblxuICAgIHRoaXMudnBjRW5kcG9pbnRJZCA9IGVuZHBvaW50LnJlZjtcbiAgICB0aGlzLnZwY0VuZHBvaW50Q3JlYXRpb25UaW1lc3RhbXAgPSBlbmRwb2ludC5hdHRyQ3JlYXRpb25UaW1lc3RhbXA7XG4gICAgdGhpcy52cGNFbmRwb2ludERuc0VudHJpZXMgPSBlbmRwb2ludC5hdHRyRG5zRW50cmllcztcbiAgICB0aGlzLnZwY0VuZHBvaW50TmV0d29ya0ludGVyZmFjZUlkcyA9IGVuZHBvaW50LmF0dHJOZXR3b3JrSW50ZXJmYWNlSWRzO1xuICB9XG5cbiAgLyoqXG4gICAqIERldGVybWluZSB3aGljaCBzdWJuZXRzIHRvIHBsYWNlIHRoZSBlbmRwb2ludCBpbi4gVGhpcyBpcyBpbiBpdHMgb3duIGZ1bmN0aW9uXG4gICAqIGJlY2F1c2UgdGhlcmUncyBhIGxvdCBvZiBjb2RlLlxuICAgKi9cbiAgcHJpdmF0ZSBlbmRwb2ludFN1Ym5ldHMocHJvcHM6IEludGVyZmFjZVZwY0VuZHBvaW50UHJvcHMpIHtcbiAgICBjb25zdCBsb29rdXBTdXBwb3J0ZWRBenMgPSBwcm9wcy5sb29rdXBTdXBwb3J0ZWRBenMgPz8gZmFsc2U7XG4gICAgY29uc3Qgc3VibmV0U2VsZWN0aW9uID0gcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMoeyAuLi5wcm9wcy5zdWJuZXRzLCBvbmVQZXJBejogdHJ1ZSB9KTtcblxuICAgIC8vIElmIHdlIGRvbid0IGhhdmUgYW4gYWNjb3VudC9yZWdpb24sIHdlIHdpbGwgbm90IGJlIGFibGUgdG8gZG8gZmlsdGVyaW5nIG9uIEFacyBzaW5jZVxuICAgIC8vIHRoZXkgd2lsbCBiZSB1bmRlZmluZWRcbiAgICBjb25zdCBhZ25vc3RpY0FjY3QgPSBUb2tlbi5pc1VucmVzb2x2ZWQodGhpcy5zdGFjay5hY2NvdW50KTtcbiAgICBjb25zdCBhZ25vc3RpY1JlZ2lvbiA9IFRva2VuLmlzVW5yZXNvbHZlZCh0aGlzLnN0YWNrLnJlZ2lvbik7XG5cbiAgICAvLyBTb21lIHNlcnZpY2UgbmFtZXMsIHN1Y2ggYXMgQVdTIHNlcnZpY2UgbmFtZSByZWZlcmVuY2VzLCB1c2UgVG9rZW5zIHRvIGF1dG9tYXRpY2FsbHlcbiAgICAvLyBmaWxsIGluIHRoZSByZWdpb25cbiAgICAvLyBJZiBpdCBpcyBhbiBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2UsIHRoZW4gdGhlIHJlZmVyZW5jZSB3aWxsIGJlIHJlc29sdmFibGUgc2luY2VcbiAgICAvLyBvbmx5IHJlZmVyZW5jZXMgdGhlIHJlZ2lvblxuICAgIGNvbnN0IGlzQXdzU2VydmljZSA9IFRva2VuLmlzVW5yZXNvbHZlZChwcm9wcy5zZXJ2aWNlLm5hbWUpICYmIHByb3BzLnNlcnZpY2UgaW5zdGFuY2VvZiBJbnRlcmZhY2VWcGNFbmRwb2ludEF3c1NlcnZpY2U7XG5cbiAgICAvLyBEZXRlcm1pbmUgd2hhdCBuYW1lIHdlIHBhc3MgdG8gdGhlIGNvbnRleHQgcHJvdmlkZXIsIGVpdGhlciB0aGUgdmVyYmF0aW0gbmFtZVxuICAgIC8vIG9yIGEgcmVzb2x2ZWQgdmVyc2lvbiBpZiBpdCBpcyBhbiBBV1Mgc2VydmljZSByZWZlcmVuY2VcbiAgICBsZXQgbG9va3VwU2VydmljZU5hbWUgPSBwcm9wcy5zZXJ2aWNlLm5hbWU7XG4gICAgaWYgKGlzQXdzU2VydmljZSAmJiAhYWdub3N0aWNSZWdpb24pIHtcbiAgICAgIGxvb2t1cFNlcnZpY2VOYW1lID0gU3RhY2sub2YodGhpcykucmVzb2x2ZShwcm9wcy5zZXJ2aWNlLm5hbWUpO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBJdCdzIGFuIGFnbm9zdGljIHNlcnZpY2UgYW5kIHdlIGRvbid0IGtub3cgaG93IHRvIHJlc29sdmUgaXQuXG4gICAgICAvLyBUaGlzIGlzIG9rIGlmIHRoZSBzdGFjayBpcyByZWdpb24gYWdub3N0aWMgYW5kIHdlJ3JlIG5vdCBsb29raW5nIHVwXG4gICAgICAvLyBBWnNcbiAgICAgIGxvb2t1cFNlcnZpY2VOYW1lID0gcHJvcHMuc2VydmljZS5uYW1lO1xuICAgIH1cblxuICAgIC8vIENoZWNrIGlmIGxvb2t1cCBpcyBpbXBvc3NpYmxlIGFuZCB0aHJvdyBhbiBhcHByb3ByaWF0ZSBlcnJvclxuICAgIC8vIENvbnRleHQgcHJvdmlkZXIgY2Fubm90IG1ha2UgYW4gQVdTIGNhbGwgd2l0aG91dCBhbiBhY2NvdW50L3JlZ2lvblxuICAgIGlmICgoYWdub3N0aWNBY2N0IHx8IGFnbm9zdGljUmVnaW9uKSAmJiBsb29rdXBTdXBwb3J0ZWRBenMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGxvb2sgdXAgVlBDIGVuZHBvaW50IGF2YWlsYWJpbGl0eSB6b25lcyBpZiBhY2NvdW50L3JlZ2lvbiBhcmUgbm90IHNwZWNpZmllZCcpO1xuICAgIH1cbiAgICAvLyBDb250ZXh0IHByb3ZpZGVyIGRvZXNuJ3Qga25vdyB0aGUgbmFtZSBvZiB0aGUgc2VydmljZSBpZiB0aGVyZSBpcyBhIFRva2VuXG4gICAgLy8gaW4gdGhlIG5hbWVcbiAgICBjb25zdCBhZ25vc3RpY1NlcnZpY2UgPSBUb2tlbi5pc1VucmVzb2x2ZWQobG9va3VwU2VydmljZU5hbWUpO1xuICAgIGlmIChhZ25vc3RpY1NlcnZpY2UgJiYgbG9va3VwU3VwcG9ydGVkQXpzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBsb29rdXAgQVpzIGZvciBhIHNlcnZpY2UgbmFtZSB3aXRoIGEgVG9rZW46ICR7cHJvcHMuc2VydmljZS5uYW1lfWApO1xuICAgIH1cblxuICAgIC8vIEhlcmUgd2UgZG8gdGhlIGFjdHVhbCBsb29rdXAgZm9yIEFacywgaWYgdG9sZCB0byBkbyBzb1xuICAgIGxldCBzdWJuZXRzO1xuICAgIGlmIChsb29rdXBTdXBwb3J0ZWRBenMpIHtcbiAgICAgIGNvbnN0IGF2YWlsYWJsZUFacyA9IHRoaXMuYXZhaWxhYmxlQXZhaWxhYmlsaXR5Wm9uZXMobG9va3VwU2VydmljZU5hbWUpO1xuICAgICAgc3VibmV0cyA9IHN1Ym5ldFNlbGVjdGlvbi5zdWJuZXRzLmZpbHRlcihzID0+IGF2YWlsYWJsZUFacy5pbmNsdWRlcyhzLmF2YWlsYWJpbGl0eVpvbmUpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgc3VibmV0cyA9IHN1Ym5ldFNlbGVjdGlvbi5zdWJuZXRzO1xuICAgIH1cbiAgICBjb25zdCBzdWJuZXRJZHMgPSBzdWJuZXRzLm1hcChzID0+IHMuc3VibmV0SWQpO1xuICAgIHJldHVybiBzdWJuZXRJZHM7XG4gIH1cblxuICBwcml2YXRlIGF2YWlsYWJsZUF2YWlsYWJpbGl0eVpvbmVzKHNlcnZpY2VOYW1lOiBzdHJpbmcpOiBzdHJpbmdbXSB7XG4gICAgLy8gSGVyZSB3ZSBjaGVjayB3aGF0IEFacyB0aGUgZW5kcG9pbnQgc2VydmljZSBpcyBhdmFpbGFibGUgaW5cbiAgICAvLyBJZiBmb3Igd2hhdGV2ZXIgcmVhc29uIHdlIGNhbid0IHJldHJpZXZlIHRoZSBBWnMsIGFuZCBubyBjb250ZXh0IGlzIHNldCxcbiAgICAvLyB3ZSB3aWxsIGZhbGwgYmFjayB0byBhbGwgQVpzXG4gICAgY29uc3QgYXZhaWxhYmxlQVpzID0gQ29udGV4dFByb3ZpZGVyLmdldFZhbHVlKHRoaXMsIHtcbiAgICAgIHByb3ZpZGVyOiBjeHNjaGVtYS5Db250ZXh0UHJvdmlkZXIuRU5EUE9JTlRfU0VSVklDRV9BVkFJTEFCSUxJVFlfWk9ORV9QUk9WSURFUixcbiAgICAgIGR1bW15VmFsdWU6IHRoaXMuc3RhY2suYXZhaWxhYmlsaXR5Wm9uZXMsXG4gICAgICBwcm9wczogeyBzZXJ2aWNlTmFtZSB9LFxuICAgIH0pLnZhbHVlO1xuICAgIGlmICghQXJyYXkuaXNBcnJheShhdmFpbGFibGVBWnMpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYERpc2NvdmVyZWQgQVpzIGZvciBlbmRwb2ludCBzZXJ2aWNlICR7c2VydmljZU5hbWV9IG11c3QgYmUgYW4gYXJyYXlgKTtcbiAgICB9XG4gICAgcmV0dXJuIGF2YWlsYWJsZUFacztcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJbnRlcmZhY2VWcGNFbmRwb2ludEF0dHJpYnV0ZXMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGNFbmRwb2ludElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogSVNlY3VyaXR5R3JvdXBbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xufVxuIl19