"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration.
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements.
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config.
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets.
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object.
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object.
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * @internal
     */
    _attach(attachedResource, attachOptions) {
        var _a, _b, _c;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata('AWS::CloudFormation::Init', bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_a = attachOptions.configSets) !== null && _a !== void 0 ? _a : ['default']).join(',');
        const printLog = (_b = attachOptions.printLog) !== null && _b !== void 0 ? _b : true;
        if ((_c = attachOptions.embedFingerprint) !== null && _c !== void 0 ? _c : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
/**
 * A collection of configuration elements.
 */
class InitConfig {
    /**
     *
     */
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not.
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config.
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _a;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_a = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _a !== void 0 ? _a : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _a, _b;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_a = target[key]) !== null && _a !== void 0 ? _a : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_b = target[key]) !== null && _b !== void 0 ? _b : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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