"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RouteSpec = exports.GrpcRetryEvent = exports.TcpRetryEvent = exports.HttpRetryEvent = exports.HttpRouteProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const http_route_path_match_1 = require("./http-route-path-match");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * Supported :scheme options for HTTP2
 */
var HttpRouteProtocol;
(function (HttpRouteProtocol) {
    /**
     * Match HTTP requests
     */
    HttpRouteProtocol["HTTP"] = "http";
    /**
     * Match HTTPS requests
     */
    HttpRouteProtocol["HTTPS"] = "https";
})(HttpRouteProtocol = exports.HttpRouteProtocol || (exports.HttpRouteProtocol = {}));
/**
 * HTTP events on which to retry.
 */
var HttpRetryEvent;
(function (HttpRetryEvent) {
    /**
     * HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
     */
    HttpRetryEvent["SERVER_ERROR"] = "server-error";
    /**
     * HTTP status codes 502, 503, and 504
     */
    HttpRetryEvent["GATEWAY_ERROR"] = "gateway-error";
    /**
     * HTTP status code 409
     */
    HttpRetryEvent["CLIENT_ERROR"] = "client-error";
    /**
     * Retry on refused stream
     */
    HttpRetryEvent["STREAM_ERROR"] = "stream-error";
})(HttpRetryEvent = exports.HttpRetryEvent || (exports.HttpRetryEvent = {}));
/**
 * TCP events on which you may retry
 */
var TcpRetryEvent;
(function (TcpRetryEvent) {
    /**
     * A connection error
     */
    TcpRetryEvent["CONNECTION_ERROR"] = "connection-error";
})(TcpRetryEvent = exports.TcpRetryEvent || (exports.TcpRetryEvent = {}));
/**
 * gRPC events
 */
var GrpcRetryEvent;
(function (GrpcRetryEvent) {
    /**
     * Request was cancelled
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["CANCELLED"] = "cancelled";
    /**
     * The deadline was exceeded
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["DEADLINE_EXCEEDED"] = "deadline-exceeded";
    /**
     * Internal error
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["INTERNAL_ERROR"] = "internal";
    /**
     * A resource was exhausted
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["RESOURCE_EXHAUSTED"] = "resource-exhausted";
    /**
     * The service is unavailable
     *
     * @see https://grpc.github.io/grpc/core/md_doc_statuscodes.html
     */
    GrpcRetryEvent["UNAVAILABLE"] = "unavailable";
})(GrpcRetryEvent = exports.GrpcRetryEvent || (exports.GrpcRetryEvent = {}));
/**
 * Used to generate specs with different protocols for a RouteSpec
 */
class RouteSpec {
    /**
     * Creates an HTTP Based RouteSpec
     */
    static http(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.http);
            }
            throw error;
        }
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * Creates an HTTP2 Based RouteSpec
     *
     */
    static http2(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.http2);
            }
            throw error;
        }
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * Creates a TCP Based RouteSpec
     */
    static tcp(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_TcpRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.tcp);
            }
            throw error;
        }
        return new TcpRouteSpec(options);
    }
    /**
     * Creates a GRPC Based RouteSpec
     */
    static grpc(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_GrpcRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grpc);
            }
            throw error;
        }
        return new GrpcRouteSpec(options);
    }
}
exports.RouteSpec = RouteSpec;
_a = JSII_RTTI_SYMBOL_1;
RouteSpec[_a] = { fqn: "@aws-cdk/aws-appmesh.RouteSpec", version: "1.156.1" };
class HttpRouteSpec extends RouteSpec {
    constructor(props, protocol) {
        var _b, _c;
        super();
        this.protocol = protocol;
        this.match = props.match;
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
        this.priority = props.priority;
        if (props.retryPolicy) {
            const httpRetryEvents = (_b = props.retryPolicy.httpRetryEvents) !== null && _b !== void 0 ? _b : [];
            const tcpRetryEvents = (_c = props.retryPolicy.tcpRetryEvents) !== null && _c !== void 0 ? _c : [];
            if (httpRetryEvents.length + tcpRetryEvents.length === 0) {
                throw new Error('You must specify one value for at least one of `httpRetryEvents` or `tcpRetryEvents`');
            }
            this.retryPolicy = {
                ...props.retryPolicy,
                httpRetryEvents: httpRetryEvents.length > 0 ? httpRetryEvents : undefined,
                tcpRetryEvents: tcpRetryEvents.length > 0 ? tcpRetryEvents : undefined,
            };
        }
    }
    bind(scope) {
        var _b, _c, _d, _e, _f, _g;
        const pathMatchConfig = ((_c = (_b = this.match) === null || _b === void 0 ? void 0 : _b.path) !== null && _c !== void 0 ? _c : http_route_path_match_1.HttpRoutePathMatch.startsWith('/')).bind(scope);
        // Set prefix path match to '/' if none of path matches are defined.
        const headers = (_d = this.match) === null || _d === void 0 ? void 0 : _d.headers;
        const queryParameters = (_e = this.match) === null || _e === void 0 ? void 0 : _e.queryParameters;
        utils_1.validateHttpMatchArrayLength(headers, queryParameters);
        const httpConfig = {
            action: {
                weightedTargets: renderWeightedTargets(this.weightedTargets),
            },
            match: {
                prefix: pathMatchConfig.prefixPathMatch,
                path: pathMatchConfig.wholePathMatch,
                headers: headers === null || headers === void 0 ? void 0 : headers.map(header => header.bind(scope).headerMatch),
                method: (_f = this.match) === null || _f === void 0 ? void 0 : _f.method,
                scheme: (_g = this.match) === null || _g === void 0 ? void 0 : _g.protocol,
                queryParameters: queryParameters === null || queryParameters === void 0 ? void 0 : queryParameters.map(queryParameter => queryParameter.bind(scope).queryParameterMatch),
            },
            timeout: renderTimeout(this.timeout),
            retryPolicy: this.retryPolicy ? renderHttpRetryPolicy(this.retryPolicy) : undefined,
        };
        return {
            priority: this.priority,
            httpRouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2RouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class TcpRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
        this.priority = props.priority;
    }
    bind(_scope) {
        return {
            priority: this.priority,
            tcpRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
class GrpcRouteSpec extends RouteSpec {
    constructor(props) {
        var _b, _c, _d;
        super();
        this.weightedTargets = props.weightedTargets;
        this.match = props.match;
        this.timeout = props.timeout;
        this.priority = props.priority;
        if (props.retryPolicy) {
            const grpcRetryEvents = (_b = props.retryPolicy.grpcRetryEvents) !== null && _b !== void 0 ? _b : [];
            const httpRetryEvents = (_c = props.retryPolicy.httpRetryEvents) !== null && _c !== void 0 ? _c : [];
            const tcpRetryEvents = (_d = props.retryPolicy.tcpRetryEvents) !== null && _d !== void 0 ? _d : [];
            if (grpcRetryEvents.length + httpRetryEvents.length + tcpRetryEvents.length === 0) {
                throw new Error('You must specify one value for at least one of `grpcRetryEvents`, `httpRetryEvents` or `tcpRetryEvents`');
            }
            this.retryPolicy = {
                ...props.retryPolicy,
                grpcRetryEvents: grpcRetryEvents.length > 0 ? grpcRetryEvents : undefined,
                httpRetryEvents: httpRetryEvents.length > 0 ? httpRetryEvents : undefined,
                tcpRetryEvents: tcpRetryEvents.length > 0 ? tcpRetryEvents : undefined,
            };
        }
    }
    bind(scope) {
        const serviceName = this.match.serviceName;
        const methodName = this.match.methodName;
        const metadata = this.match.metadata;
        utils_1.validateGrpcRouteMatch(this.match);
        utils_1.validateGrpcMatchArrayLength(metadata);
        if (methodName && !serviceName) {
            throw new Error('If you specify a method name, you must also specify a service name');
        }
        return {
            priority: this.priority,
            grpcRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                match: {
                    serviceName: serviceName,
                    methodName: methodName,
                    metadata: metadata === null || metadata === void 0 ? void 0 : metadata.map(singleMetadata => singleMetadata.bind(scope).headerMatch),
                },
                timeout: renderTimeout(this.timeout),
                retryPolicy: this.retryPolicy ? renderGrpcRetryPolicy(this.retryPolicy) : undefined,
            },
        };
    }
}
/**
 * Utility method to add weighted route targets to an existing route
 */
function renderWeightedTargets(weightedTargets) {
    const renderedTargets = [];
    for (const t of weightedTargets) {
        renderedTargets.push({
            virtualNode: t.virtualNode.virtualNodeName,
            weight: t.weight || 1,
        });
    }
    return renderedTargets;
}
/**
 * Utility method to construct a route timeout object
 */
function renderTimeout(timeout) {
    return timeout
        ? {
            idle: (timeout === null || timeout === void 0 ? void 0 : timeout.idle) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.idle.toMilliseconds(),
                }
                : undefined,
            perRequest: (timeout === null || timeout === void 0 ? void 0 : timeout.perRequest) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.perRequest.toMilliseconds(),
                }
                : undefined,
        }
        : undefined;
}
function renderHttpRetryPolicy(retryPolicy) {
    return {
        maxRetries: retryPolicy.retryAttempts,
        perRetryTimeout: {
            unit: 'ms',
            value: retryPolicy.retryTimeout.toMilliseconds(),
        },
        httpRetryEvents: retryPolicy.httpRetryEvents,
        tcpRetryEvents: retryPolicy.tcpRetryEvents,
    };
}
function renderGrpcRetryPolicy(retryPolicy) {
    return {
        ...renderHttpRetryPolicy(retryPolicy),
        grpcRetryEvents: retryPolicy.grpcRetryEvents,
    };
}
//# sourceMappingURL=data:application/json;base64,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