#!/usr/bin/env python3
from __future__ import annotations

import glob
import argparse

from praatio import textgrid as tg
from praatio.data_classes.textgrid import Textgrid

from dynamicfluency.repetitions import make_repetitions_tier, make_freqdist_tier


def parse_arguments() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Processes alligned pos_tags .TextGrid files generated by pos_tagging.py to a .TextGrid with reptition information in it."
    )
    parser.add_argument(
        "-d",
        "--directory",
        nargs="?",
        default="output",
        help="The directory the pos_tags .TextGrid is expected in, and the output is saved to",
    )
    parser.add_argument(
        "-m",
        "--max_read",
        nargs="?",
        default=300,
        type=int,
        help="The maximum amount of words the detector reads back to check for repetitions",
    )
    parser.add_argument(
        "-i",
        "--to_ignore",
        nargs="?",
        help="The words to ignore and not assign any value, seperated by commas.",
    )

    args: argparse.Namespace = parser.parse_args()
    args.to_ignore.split(",")
    return args


def main():
    args: argparse.Namespace = parse_arguments()

    tagged_files = glob.glob(f"./{args.directory}/*.pos_tags.TextGrid")
    for file in tagged_files:
        tagged_grid = tg.openTextgrid(file, includeEmptyIntervals=True)

        repetition_tier = make_repetitions_tier(
            pos_tier=tagged_grid.tierDict["POStags"],
            max_cache=args.max_read,
            to_ignore=args.to_ignore,
        )
        freqdist_tier = make_freqdist_tier(
            pos_tier=tagged_grid.tierDict["POStags"], to_ignore=args.to_ignore
        )

        repetition_grid = Textgrid()
        repetition_grid.addTier(repetition_tier)
        repetition_grid.addTier(freqdist_tier)

        name = file.replace(".pos_tags.TextGrid", ".repetitions.TextGrid")
        repetition_grid.save(name, format="long_textgrid", includeBlankSpaces=True)


if __name__ == "__main__":
    main()
