#!/usr/bin/env python3
"""
Automated test suite for test-v350 MCP server
Generated by KEN-MCP

INSTRUCTIONS FOR CLAUDE:
1. Run this test file immediately: python test.py
2. If import errors occur, run diagnostics first: python diagnose.py
3. Fix any failing tests in server.py
4. Add domain-specific test cases based on the requirements
5. Ensure all tests pass before considering the MCP complete

TROUBLESHOOTING:
- Import errors? Run: python diagnose.py
- Tests fail? Check if server.py has print() statements (use logger instead)
- Still issues? The diagnose.py script will identify common problems

Original Requirements:
Create a simple greeting MCP with a single say_hello tool
"""

import asyncio
import json
import sys
from pathlib import Path

# Add the parent directory to the path so we can import the server
sys.path.insert(0, str(Path(__file__).parent))

# Import the generated server
try:
    from server import mcp
except ImportError as e:
    # DO NOT use print() in MCP servers - it breaks the protocol
    # Log to stderr instead
    import sys
    sys.stderr.write(f"❌ Failed to import server: {e}\n")
    sys.stderr.write("Make sure server.py exists and has no syntax errors\n")
    sys.exit(1)

# Test utilities
class MockContext:
    """Mock context for testing MCP tools"""
    def __init__(self):
        self.logs = []
        self.progress = []
    
    async def info(self, msg):
        self.logs.append(msg)
    
    async def report_progress(self, current, total, msg):
        self.progress.append((current, total, msg))
    
    async def read_resource(self, uri):
        return {"uri": uri, "content": "mock resource content"}


# Test functions
async def test_server_initialization():
    """Test that the MCP server can be initialized"""
    print("Testing server initialization...")
    try:
        assert mcp.name == "test-v350"
        assert hasattr(mcp, 'run')
        print("  ✅ Server initialization test passed")
        return True
    except Exception as e:
        print(f"  ❌ Server initialization failed: {e}")
        return False


async def test_fetch_data():
    """Test fetch_data: Read operation for data.

Domain: general
Related operations: filtering

TODO: Claude, implement thi..."""
    print(f"\nTesting fetch_data...")
    
    # Get the tool from the MCP server
    try:
        tool = await mcp.get_tool("fetch_data")
        if not tool:
            print(f"  ❌ Tool fetch_data not found in MCP server")
            return False
        tool_func = tool.fn
    except Exception as e:
        print(f"  ❌ Could not access fetch_data: {e}")
        return False
    
    ctx = MockContext()
    passed = 0
    failed = 0
    
    # Test 1: Valid inputs
    try:
        result = await tool_func(
            ctx=ctx,
        data_id="test_id_123"
        )
        # Check result structure
        assert isinstance(result, dict), "Result should be a dictionary"
        assert any(key in result for key in ["success", "status", "data", "result"]), \
            "Result should contain success, status, data, or result key"
        print("  ✅ Valid input test passed")
        passed += 1
    except Exception as e:
        print(f"  ❌ Valid input test failed: {e}")
        failed += 1

    
    # Test 2: Missing required parameters
    try:
        # Call without required parameter: data_id
        result = await tool_func(ctx=ctx)
        print(f"  ❌ Should have failed with missing required parameter")
        failed += 1
    except TypeError as e:
        if "data_id" in str(e):
            print(f"  ✅ Missing parameter validation passed")
            passed += 1
        else:
            print(f"  ❌ Wrong error for missing parameter: {e}")
            failed += 1
    except Exception as e:
        print(f"  ❌ Unexpected error for missing parameter: {e}")
        failed += 1

    
    # Test 3: Edge cases
    # TODO: Claude, add more specific edge case tests based on the requirements:
    # - Test with empty strings for string parameters
    # - Test with None for optional parameters  
    # - Test with boundary values for numeric parameters
    # - Test with special characters if applicable
    # - Test error conditions specific to this tool's purpose
    
    print(f"  📊 fetch_data tests: {passed} passed, {failed} failed")
    return failed == 0


async def test_process_data():
    """Test process_data: Process operation for data.

Domain: general
Related operations: filtering

TODO: Claude, implement..."""
    print(f"\nTesting process_data...")
    
    # Get the tool from the MCP server
    try:
        tool = await mcp.get_tool("process_data")
        if not tool:
            print(f"  ❌ Tool process_data not found in MCP server")
            return False
        tool_func = tool.fn
    except Exception as e:
        print(f"  ❌ Could not access process_data: {e}")
        return False
    
    ctx = MockContext()
    passed = 0
    failed = 0
    
    # Test 1: Valid inputs
    try:
        result = await tool_func(
            ctx=ctx,
        input_data={"test": "data"},
        options="test_value"
        )
        # Check result structure
        assert isinstance(result, dict), "Result should be a dictionary"
        assert any(key in result for key in ["success", "status", "data", "result"]), \
            "Result should contain success, status, data, or result key"
        print("  ✅ Valid input test passed")
        passed += 1
    except Exception as e:
        print(f"  ❌ Valid input test failed: {e}")
        failed += 1

    
    # Test 2: Missing required parameters
    try:
        # Call without required parameter: input_data
        result = await tool_func(ctx=ctx)
        print(f"  ❌ Should have failed with missing required parameter")
        failed += 1
    except TypeError as e:
        if "input_data" in str(e):
            print(f"  ✅ Missing parameter validation passed")
            passed += 1
        else:
            print(f"  ❌ Wrong error for missing parameter: {e}")
            failed += 1
    except Exception as e:
        print(f"  ❌ Unexpected error for missing parameter: {e}")
        failed += 1

    
    # Test 3: Edge cases
    # TODO: Claude, add more specific edge case tests based on the requirements:
    # - Test with empty strings for string parameters
    # - Test with None for optional parameters  
    # - Test with boundary values for numeric parameters
    # - Test with special characters if applicable
    # - Test error conditions specific to this tool's purpose
    
    print(f"  📊 process_data tests: {passed} passed, {failed} failed")
    return failed == 0


async def test_store_result():
    """Test store_result: Write operation for result.

Domain: general
Related operations: filtering

TODO: Claude, implement..."""
    print(f"\nTesting store_result...")
    
    # Get the tool from the MCP server
    try:
        tool = await mcp.get_tool("store_result")
        if not tool:
            print(f"  ❌ Tool store_result not found in MCP server")
            return False
        tool_func = tool.fn
    except Exception as e:
        print(f"  ❌ Could not access store_result: {e}")
        return False
    
    ctx = MockContext()
    passed = 0
    failed = 0
    
    # Test 1: Valid inputs
    try:
        result = await tool_func(
            ctx=ctx,
        data={"test": "data"},
        destination={"test": "data"}
        )
        # Check result structure
        assert isinstance(result, dict), "Result should be a dictionary"
        assert any(key in result for key in ["success", "status", "data", "result"]), \
            "Result should contain success, status, data, or result key"
        print("  ✅ Valid input test passed")
        passed += 1
    except Exception as e:
        print(f"  ❌ Valid input test failed: {e}")
        failed += 1

    
    # Test 2: Missing required parameters
    try:
        # Call without required parameter: data
        result = await tool_func(ctx=ctx)
        print(f"  ❌ Should have failed with missing required parameter")
        failed += 1
    except TypeError as e:
        if "data" in str(e):
            print(f"  ✅ Missing parameter validation passed")
            passed += 1
        else:
            print(f"  ❌ Wrong error for missing parameter: {e}")
            failed += 1
    except Exception as e:
        print(f"  ❌ Unexpected error for missing parameter: {e}")
        failed += 1

    
    # Test 3: Edge cases
    # TODO: Claude, add more specific edge case tests based on the requirements:
    # - Test with empty strings for string parameters
    # - Test with None for optional parameters  
    # - Test with boundary values for numeric parameters
    # - Test with special characters if applicable
    # - Test error conditions specific to this tool's purpose
    
    print(f"  📊 store_result tests: {passed} passed, {failed} failed")
    return failed == 0


async def test_resources():
    """Test that MCP resources are accessible"""
    print(f"\nTesting resources...")
    
    try:
        # Get available resources
        resources = await mcp.get_resources()
        assert len(resources) > 0, "No resources defined"
        print(f"  ✅ Found {len(resources)} resources")
        
        # Test each resource
        for uri, resource in resources.items():
            print(f"  Testing resource: {uri}")
            try:
                result = await resource.fn()
                print(f"    ✅ Resource {uri} returned data")
            except Exception as e:
                print(f"    ❌ Resource {uri} failed: {e}")
        
        return True
    except Exception as e:
        print(f"  ❌ Resource test failed: {e}")
        return False


async def test_prompts():
    """Test that MCP prompts are defined"""
    print(f"\nTesting prompts...")
    
    try:
        # Get available prompts
        prompts = await mcp.get_prompts()
        assert len(prompts) > 0, "No prompts defined"
        print(f"  ✅ Found {len(prompts)} prompts")
        
        # Test each prompt
        for name, prompt in prompts.items():
            print(f"  Testing prompt: {name}")
            try:
                # Prompts typically return strings, not async
                result = prompt.fn()
                print(f"    ✅ Prompt {name} returned template")
            except Exception as e:
                print(f"    ❌ Prompt {name} failed: {e}")
        
        return True
    except Exception as e:
        print(f"  ❌ Prompt test failed: {e}")
        return False


async def run_all_tests():
    """Run all test cases"""
    print("=" * 50)
    print(f"🧪 Running MCP Server Tests for test-v350")
    print("=" * 50)
    
    # List all tests to run
    tests = [
        ("Server Initialization", test_server_initialization),
        ("fetch_data", test_fetch_data),
        ("process_data", test_process_data),
        ("store_result", test_store_result),
        ("Resources", test_resources),
        ("Prompts", test_prompts),
    ]
    
    total_passed = 0
    total_failed = 0
    
    for test_name, test_func in tests:
        try:
            result = await test_func()
            if result:
                total_passed += 1
            else:
                total_failed += 1
        except Exception as e:
            print(f"\n❌ {test_name} crashed: {e}")
            total_failed += 1
    
    # Summary
    print("\n" + "=" * 50)
    print(f"📊 Test Summary: {total_passed}/{len(tests)} passed")
    print("=" * 50)
    
    if total_failed > 0:
        print(f"\n⚠️  {total_failed} test(s) failed!")
        print("\nNext steps:")
        print("1. Check the error messages above")
        print("2. Fix the implementation in server.py")
        print("3. Run the tests again: python test.py")
        print("4. All tests must pass before the MCP is ready")
        return 1
    else:
        print("\n✅ All tests passed! The MCP server is ready to use.")
        print("\nYou can now:")
        print("1. Add it to Claude Desktop (see help.md)")
        print("2. Add more specific test cases based on your use case")
        print("3. Test with real data")
        return 0

if __name__ == "__main__":
    exit_code = asyncio.run(run_all_tests())
    sys.exit(exit_code)
