#!/usr/bin/env python3
"""
Automated test suite for audit-test MCP server
Generated by KEN-MCP

INSTRUCTIONS FOR CLAUDE:
1. Run this test file immediately: python test.py
2. If import errors occur, run diagnostics first: python diagnose.py
3. Fix any failing tests in server.py
4. Add domain-specific test cases based on the requirements
5. Ensure all tests pass before considering the MCP complete

TROUBLESHOOTING:
- Import errors? Run: python diagnose.py
- Tests fail? Check if server.py has print() statements (use logger instead)
- Still issues? The diagnose.py script will identify common problems

Original Requirements:
Create a simple audit test MCP to verify all documentation and commands are correct
"""

import asyncio
import json
import sys
from pathlib import Path

# Add the parent directory to the path so we can import the server
sys.path.insert(0, str(Path(__file__).parent))

# Import the generated server
try:
    from server import mcp
except ImportError as e:
    # DO NOT use print() in MCP servers - it breaks the protocol
    # Log to stderr instead
    import sys
    sys.stderr.write(f"❌ Failed to import server: {e}\n")
    sys.stderr.write("Make sure server.py exists and has no syntax errors\n")
    sys.exit(1)

# Test utilities
class MockContext:
    """Mock context for testing MCP tools"""
    def __init__(self):
        self.logs = []
        self.progress = []
    
    async def info(self, msg):
        self.logs.append(msg)
    
    async def report_progress(self, current, total, msg):
        self.progress.append((current, total, msg))
    
    async def read_resource(self, uri):
        return {"uri": uri, "content": "mock resource content"}


# Test functions
async def test_server_initialization():
    """Test that the MCP server can be initialized"""
    print("Testing server initialization...")
    try:
        assert mcp.name == "audit-test"
        assert hasattr(mcp, 'run')
        print("  ✅ Server initialization test passed")
        return True
    except Exception as e:
        print(f"  ❌ Server initialization failed: {e}")
        return False


async def test_read_file():
    """Test read_file: Read operation for file.

Domain: file_system
Related operations: validating

TODO: Claude, implemen..."""
    print(f"\nTesting read_file...")
    
    # Get the tool from the MCP server
    try:
        tool = await mcp.get_tool("read_file")
        if not tool:
            print(f"  ❌ Tool read_file not found in MCP server")
            return False
        tool_func = tool.fn
    except Exception as e:
        print(f"  ❌ Could not access read_file: {e}")
        return False
    
    ctx = MockContext()
    passed = 0
    failed = 0
    
    # Test 1: Valid inputs
    try:
        result = await tool_func(
            ctx=ctx,
        file_id="/tmp/test_file.txt"
        )
        # Check result structure
        assert isinstance(result, dict), "Result should be a dictionary"
        assert any(key in result for key in ["success", "status", "data", "result"]), \
            "Result should contain success, status, data, or result key"
        print("  ✅ Valid input test passed")
        passed += 1
    except Exception as e:
        print(f"  ❌ Valid input test failed: {e}")
        failed += 1

    
    # Test 2: Missing required parameters
    try:
        # Call without required parameter: file_id
        result = await tool_func(ctx=ctx)
        print(f"  ❌ Should have failed with missing required parameter")
        failed += 1
    except TypeError as e:
        if "file_id" in str(e):
            print(f"  ✅ Missing parameter validation passed")
            passed += 1
        else:
            print(f"  ❌ Wrong error for missing parameter: {e}")
            failed += 1
    except Exception as e:
        print(f"  ❌ Unexpected error for missing parameter: {e}")
        failed += 1

    
    # Test 3: Edge cases
    # TODO: Claude, add more specific edge case tests based on the requirements:
    # - Test with empty strings for string parameters
    # - Test with None for optional parameters  
    # - Test with boundary values for numeric parameters
    # - Test with special characters if applicable
    # - Test error conditions specific to this tool's purpose
    
    print(f"  📊 read_file tests: {passed} passed, {failed} failed")
    return failed == 0


async def test_write_file():
    """Test write_file: Write operation for file.

Domain: file_system
Related operations: validating

TODO: Claude, impleme..."""
    print(f"\nTesting write_file...")
    
    # Get the tool from the MCP server
    try:
        tool = await mcp.get_tool("write_file")
        if not tool:
            print(f"  ❌ Tool write_file not found in MCP server")
            return False
        tool_func = tool.fn
    except Exception as e:
        print(f"  ❌ Could not access write_file: {e}")
        return False
    
    ctx = MockContext()
    passed = 0
    failed = 0
    
    # Test 1: Valid inputs
    try:
        result = await tool_func(
            ctx=ctx,
        path="/tmp/test_file.txt",
        content={"test": "data"}
        )
        # Check result structure
        assert isinstance(result, dict), "Result should be a dictionary"
        assert any(key in result for key in ["success", "status", "data", "result"]), \
            "Result should contain success, status, data, or result key"
        print("  ✅ Valid input test passed")
        passed += 1
    except Exception as e:
        print(f"  ❌ Valid input test failed: {e}")
        failed += 1

    
    # Test 2: Missing required parameters
    try:
        # Call without required parameter: path
        result = await tool_func(ctx=ctx)
        print(f"  ❌ Should have failed with missing required parameter")
        failed += 1
    except TypeError as e:
        if "path" in str(e):
            print(f"  ✅ Missing parameter validation passed")
            passed += 1
        else:
            print(f"  ❌ Wrong error for missing parameter: {e}")
            failed += 1
    except Exception as e:
        print(f"  ❌ Unexpected error for missing parameter: {e}")
        failed += 1

    
    # Test 3: Edge cases
    # TODO: Claude, add more specific edge case tests based on the requirements:
    # - Test with empty strings for string parameters
    # - Test with None for optional parameters  
    # - Test with boundary values for numeric parameters
    # - Test with special characters if applicable
    # - Test error conditions specific to this tool's purpose
    
    print(f"  📊 write_file tests: {passed} passed, {failed} failed")
    return failed == 0


async def test_list_directory():
    """Test list_directory: List operation for directory.

Domain: file_system
Related operations: validating

TODO: Claude, imp..."""
    print(f"\nTesting list_directory...")
    
    # Get the tool from the MCP server
    try:
        tool = await mcp.get_tool("list_directory")
        if not tool:
            print(f"  ❌ Tool list_directory not found in MCP server")
            return False
        tool_func = tool.fn
    except Exception as e:
        print(f"  ❌ Could not access list_directory: {e}")
        return False
    
    ctx = MockContext()
    passed = 0
    failed = 0
    
    # Test 1: Valid inputs
    try:
        result = await tool_func(
            ctx=ctx,
        limit=42,
        offset=42,
        filter="test_value"
        )
        # Check result structure
        assert isinstance(result, dict), "Result should be a dictionary"
        assert any(key in result for key in ["success", "status", "data", "result"]), \
            "Result should contain success, status, data, or result key"
        print("  ✅ Valid input test passed")
        passed += 1
    except Exception as e:
        print(f"  ❌ Valid input test failed: {e}")
        failed += 1

    
    # Test 3: Edge cases
    # TODO: Claude, add more specific edge case tests based on the requirements:
    # - Test with empty strings for string parameters
    # - Test with None for optional parameters  
    # - Test with boundary values for numeric parameters
    # - Test with special characters if applicable
    # - Test error conditions specific to this tool's purpose
    
    print(f"  📊 list_directory tests: {passed} passed, {failed} failed")
    return failed == 0


async def test_resources():
    """Test that MCP resources are accessible"""
    print(f"\nTesting resources...")
    
    try:
        # Get available resources
        resources = await mcp.get_resources()
        assert len(resources) > 0, "No resources defined"
        print(f"  ✅ Found {len(resources)} resources")
        
        # Test each resource
        for uri, resource in resources.items():
            print(f"  Testing resource: {uri}")
            try:
                result = await resource.fn()
                print(f"    ✅ Resource {uri} returned data")
            except Exception as e:
                print(f"    ❌ Resource {uri} failed: {e}")
        
        return True
    except Exception as e:
        print(f"  ❌ Resource test failed: {e}")
        return False


async def test_prompts():
    """Test that MCP prompts are defined"""
    print(f"\nTesting prompts...")
    
    try:
        # Get available prompts
        prompts = await mcp.get_prompts()
        assert len(prompts) > 0, "No prompts defined"
        print(f"  ✅ Found {len(prompts)} prompts")
        
        # Test each prompt
        for name, prompt in prompts.items():
            print(f"  Testing prompt: {name}")
            try:
                # Prompts typically return strings, not async
                result = prompt.fn()
                print(f"    ✅ Prompt {name} returned template")
            except Exception as e:
                print(f"    ❌ Prompt {name} failed: {e}")
        
        return True
    except Exception as e:
        print(f"  ❌ Prompt test failed: {e}")
        return False


async def run_all_tests():
    """Run all test cases"""
    print("=" * 50)
    print(f"🧪 Running MCP Server Tests for audit-test")
    print("=" * 50)
    
    # List all tests to run
    tests = [
        ("Server Initialization", test_server_initialization),
        ("read_file", test_read_file),
        ("write_file", test_write_file),
        ("list_directory", test_list_directory),
        ("Resources", test_resources),
        ("Prompts", test_prompts),
    ]
    
    total_passed = 0
    total_failed = 0
    
    for test_name, test_func in tests:
        try:
            result = await test_func()
            if result:
                total_passed += 1
            else:
                total_failed += 1
        except Exception as e:
            print(f"\n❌ {test_name} crashed: {e}")
            total_failed += 1
    
    # Summary
    print("\n" + "=" * 50)
    print(f"📊 Test Summary: {total_passed}/{len(tests)} passed")
    print("=" * 50)
    
    if total_failed > 0:
        print(f"\n⚠️  {total_failed} test(s) failed!")
        print("\nNext steps:")
        print("1. Check the error messages above")
        print("2. Fix the implementation in server.py")
        print("3. Run the tests again: python test.py")
        print("4. All tests must pass before the MCP is ready")
        return 1
    else:
        print("\n✅ All tests passed! The MCP server is ready to use.")
        print("\nYou can now:")
        print("1. Add it to Claude Desktop (see help.md)")
        print("2. Add more specific test cases based on your use case")
        print("3. Test with real data")
        return 0

if __name__ == "__main__":
    exit_code = asyncio.run(run_all_tests())
    sys.exit(exit_code)
