"""
Iteralign-centrifuge.
Design spec.
    1. Grab fastq files from a location.
    2. Classify reads that uniquely align to a reference genome.
    3. Generate a minimap2 index from the previously found reference genomes.
    4. Align the files ONCE to a reference.
    5. Calculate cumulative coverage information.
    6. Write a list of genomes that are covered at a particular threshold.
    7. Rinse and repeat
"""
import logging
import logging.handlers
import os
import sys
import time
from argparse import Namespace


import toml
from watchdog.observers.polling import PollingObserver as Observer

from ru.utils import print_args, send_message, Severity, get_device
from ru.run_until_utils import FastQMonitor
from ru.ru_gen import _cli as BASE
from ru.ru_gen import run as dnrun
from minknow_api.acquisition_pb2 import MinknowStatus

DEFAULT_SERVER_HOST = "127.0.0.1"
DEFAULT_LOG_FORMAT = "%(asctime)s - %(name)-20s - %(message)s"
LOG_LEVELS = ("debug", "info", "warning", "error", "critical")
DEFAULT_COVERAGE_DEPTH = 30
# DEFAULT_PERCENTAGE_COVERED = 0.99
DEFAULT_CORES = 2

DEFAULT_THRESHOLD = 2000
DEFAULT_PREFIX = "iteraligncentrifuge"
# DEFAULT_REJECT = "reject.tsv"
DEFAULT_CREPORT = "centrifuge_report.tsv"
# DEFAULT_READS = "out.tsv"
# DEFAULT_MINDEX = "mindex.mmi"
DEFAULT_GENOME = "genome.fna.gz"
# DEFAULT_TIDFILE = "taxids.toml"
# DEFAULT_COVERAGE_FILE = "coverage.tsv"
DEFAULT_SEQUENCE_LENGTH = 100000

_help = "ReadFish and Run Until, using centrifuge"
_cli = BASE + (
    (
        # In use by event_handler - passed as args.path
        # ToDo: Consider forcing from API.
        "--watch",
        dict(
            metavar="FOLDER",
            help="Top Level Folder containing fastq reads.",
            default=None,
        ),
    ),
    # (
    # Not used ToDo: Delete
    #    "--percent",
    #    dict(
    #        metavar="PERCENT",
    #        help="Default percent of target covered at given depth (default {})".format(
    #            DEFAULT_PERCENTAGE_COVERED
    #        ),
    #        default=DEFAULT_PERCENTAGE_COVERED,
    #        type=float,
    #    ),
    # ),
    (
        # in use by fastqhandler.
        "--depth",
        dict(
            metavar="DEPTH",
            help="Desired coverage depth (default {})".format(DEFAULT_COVERAGE_DEPTH),
            default=DEFAULT_COVERAGE_DEPTH,
            type=int,
        ),
    ),
    (
        # ToDo: Rename to centrifuge workers.
        # in use by fastqhandler.
        "--threads",
        dict(
            metavar="THREADS",
            help="Set the number of default threads to use for threaded tasks (default {})".format(
                DEFAULT_CORES
            ),
            default=DEFAULT_CORES,
            type=int,
        ),
    ),
    (
        # In use by fastqhandler and centrifug class. Absolutely required.
        "--cindex",
        dict(
            required=True,
            action="store",
            help="The centrifuge index required for read alignment. Must only have the prefix of the index names before the full stop and number of each index",
        ),
    ),
    # (
    #    # Not in use anywhere. ToDo: Remove.
    #    "--readfile",
    #    dict(
    #        action="store",
    #        help="The name of the file that will contain the metrics for the reads analysed",
    #        default=DEFAULT_READS,
    #    ),
    # ),
    (
        # In use by centrifuge class
        # ToDo: See https://github.com/LooseLab/Thomas_project/blob/master/ftp_df_assembler.py
        "--csummary",
        dict(
            required=True,
            action="store",
            help="The file path to the custom centrifuge summary file with the ftp path for the reference genomes",
        ),
    ),
    (
        # In use by centrifuge class
        "--gfasta",
        dict(
            action="store",
            help="The file name for the genome references downloaded after centrifuge identification",
            default=DEFAULT_GENOME,
        ),
    ),
    (
        # In use by centrifuge class
        "--creport",
        dict(
            action="store",
            help="The file name of the centrifuge report file 'centrifuge_report.tsv' made each iteration. Will be created in this scripts source directory",
            default=DEFAULT_CREPORT,
        ),
    ),
    (
        #  In use in iteralign centrifuge
        "--keepfiles",
        dict(
            action="store_true",
            help="Set argument to retain the files generated by the centrifuge section of the script",
        ),
    ),
    # (
    # not used. ToDo: Delete
    #    "--reject",
    #    dict(
    #        action="store",
    #        help="The file name containing reads not utilised within the 'call' file and the reason why.",
    #        default=DEFAULT_REJECT,
    #    ),
    # ),
    # (
    # not used. ToDo: Delete
    #    "--mindex",
    #    dict(
    #        action="store",
    #        help="The file name where the minimap index will be stored",
    #        default=DEFAULT_MINDEX,
    #    ),
    # ),
    # (
    # not used. ToDo: Delete
    #    "--tidfile",
    #    dict(
    #        action="store",
    #        help="The file name where the taxIDs found with centrifuge are stored with the iteration they were found",
    #        default=DEFAULT_TIDFILE,
    #    ),
    # ),
    (
        # In use.
        "--prefix",
        dict(
            action="store",
            help="The file path and prefix for all files generated by this script will generate",
            default=DEFAULT_PREFIX,
        ),
    ),
    (
        # ToDo: There is an urgent risk of variable name collision to resolve.
        "--path",
        dict(
            required=True,
            action="store",
            help="The directory path where all files generated by this script will be stored",
        ),
    ),
    (
        # In use by centrifuge
        "--threshold",
        dict(
            action="store",
            help="The threshold number of reads that are classified before a reference genome is added to the mmi",
            default=DEFAULT_THRESHOLD,
            type=int,
        ),
    ),
    (
        # in use by centrifuge
        "--plasmids",
        dict(
            action="store",
            help="The path and file name of the plasmid file that will be used by the script",
        ),
    ),
    (
        # in use by centrifuge ToDo: Rename
        "--references",
        dict(
            action="store",
            help="At least 1 taxID must be provided to be downloaded for mmi generation.",
            nargs="+",
            type=int,
        ),
    ),
    (
        # In use by centrifuge.
        "--seqlength",
        dict(
            action="store",
            help="The threshold length of an assembly to be incorporated into the reference file",
            default=DEFAULT_SEQUENCE_LENGTH,
            type=int,
        ),
    ),
    # (
    # Not used. ToDo: Delete
    #    "--coveragefile",
    #    dict(
    #        action="store",
    #        help="the file suffix containing 3 tab separated columns: iteration, referenceid, coverage (percent)",
    #        default=DEFAULT_COVERAGE_FILE,
    #    ),
    # ),
)


def main():
    sys.exit("This entry point is deprecated, please use 'readfish centrifuge' instead")


def run(parser, args):
    args_copy = Namespace(
        **vars(args)
    )  ## Make a copy of args to pass on to the read until client later.

    args.tomlfile = args.toml
    args.toml = toml.load(args.toml)

    # TODO: Move logging config to separate configuration file
    # set up logging to file
    logging.basicConfig(
        level=logging.DEBUG,
        format="%(levelname)s::%(asctime)s::%(name)s::%(message)s",
        filename=args.log_file,
        filemode="w",
    )

    # define a Handler that writes INFO messages or higher to the sys.stderr
    console = logging.StreamHandler()
    console.setLevel(logging.INFO)

    # set a format which is simpler for console use
    formatter = logging.Formatter("%(name)-15s: %(levelname)-8s %(message)s")
    console.setFormatter(formatter)

    # add the handler to the root logger
    logging.getLogger("").addHandler(console)

    # Start by logging sys.argv and the parameters used
    logger = logging.getLogger("Manager")
    logger.info(" ".join(sys.argv))
    print_args(args, logger=logger)

    logger.info("Initialising ReadFish centrifuge.")

    logger.info("Setting up FastQ monitoring.")

    #### Check if a run is active - if not, wait.

    args.simulation = True
    connection = None

    # set default message severity level.
    severity = 2

    if args.watch is None:
        args.simulation = False
        logger.info("Creating rpc connection for device {}.".format(args.device))
        try:
            connection = get_device(args.device).connect()
            # connection, messageport = get_rpc_connection(args.device)
        except ValueError as e:
            print(e)
            sys.exit(1)

        # send_message_port("Iteralign Connected to MinKNOW", args.host, messageport)
        send_message(
            connection, "ReadFish Centrifuge Connected to MinKNOW.", Severity.WARN
        )

        logger.info("Loaded RPC")
        while (
            connection.acquisition.current_status().status != MinknowStatus.PROCESSING
        ):
            time.sleep(1)
        #### Check if we know where data is being written to , if not... wait
        args.watch = (
            connection.acquisition.get_acquisition_info().config_summary.reads_directory
        )

    else:
        messageport = ""

    ### Here we configure the code to run either iteralign or itercent. If centrifuge is False it will run iteralign.
    event_handler = FastQMonitor(
        args, connection, centrifuge=True, mapper=True, rununtil=False
    )
    # This block handles the fastq
    observer = Observer()
    observer.schedule(event_handler, path=args.watch, recursive=True)
    observer.daemon = True

    try:

        observer.start()
        logger.info("FastQ Monitoring Running.")

        if not args.simulation:
            dnrun(parser, args_copy)
        else:
            while 1:
                time.sleep(1)

    except KeyboardInterrupt:

        logger.info("Exiting - Will take a few seconds to clean up!")

        observer.stop()
        observer.join()

        if args.keepfiles:
            logging.info(
                "The 'keepfiles' argument was set, files generated by classifier have been retained"
            )
        else:
            if os.path.isdir(args.path):
                for path, dirs, files in os.walk(args.path):
                    for f in files:
                        if f.startswith(args.prefix):
                            os.unlink(f)
                            logging.info("file removed: {}".format(f))

            if os.path.isdir("./"):
                for path, dirs, files in os.walk("./"):
                    for f in files:
                        if f.endswith(args.creport):
                            os.unlink(f)
                            logging.info("file removed: {}".format(f))

        logging.info("All files generated by classifier have been removed.")

        os._exit(0)


if __name__ == "__main__":
    main()
