"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeFunction = exports.InvocationType = exports.LogType = exports.AwsApiCall = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const assertions_1 = require("./assertions");
const common_1 = require("./common");
const providers_1 = require("./providers");
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
class AwsApiCall extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.flattenResponse = 'false';
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AwsApiCallProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsApiCall);
            }
            throw error;
        }
        this.provider = new providers_1.AssertionsProvider(this, 'SdkProvider');
        this.provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.name = `${props.service}${props.api}`;
        this.sdkCallResource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                parameters: this.provider.encode(props.parameters),
                flattenResponse: aws_cdk_lib_1.Lazy.string({ produce: () => this.flattenResponse }),
                salt: Date.now().toString(),
            },
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${this.name}`.substring(0, 60),
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.sdkCallResource.node.addDependency(this.provider);
    }
    getAtt(attributeName) {
        this.flattenResponse = 'true';
        return this.sdkCallResource.getAtt(`apiCallResponse.${attributeName}`);
    }
    getAttString(attributeName) {
        this.flattenResponse = 'true';
        return this.sdkCallResource.getAttString(`apiCallResponse.${attributeName}`);
    }
    expect(expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.expect);
            }
            throw error;
        }
        new assertions_1.EqualsAssertion(this, `AssertEquals${this.name}`, {
            expected,
            actual: common_1.ActualResult.fromCustomResource(this.sdkCallResource, 'apiCallResponse'),
        });
    }
    assertAtPath(path, expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.assertAtPath);
            }
            throw error;
        }
        new assertions_1.EqualsAssertion(this, `AssertEquals${this.name}`, {
            expected,
            actual: common_1.ActualResult.fromAwsApiCall(this, path),
        });
    }
}
exports.AwsApiCall = AwsApiCall;
_a = JSII_RTTI_SYMBOL_1;
AwsApiCall[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AwsApiCall", version: "2.43.0-alpha.0" };
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
var LogType;
(function (LogType) {
    /**
     * The log messages are not returned in the response
     */
    LogType["NONE"] = "None";
    /**
     * The log messages are returned in the response
     */
    LogType["TAIL"] = "Tail";
})(LogType = exports.LogType || (exports.LogType = {}));
/**
 * The type of invocation. Default is REQUEST_RESPONE
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONE"] = "RequestResponse";
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
/**
 * An AWS Lambda Invoke function API call.
 * Use this istead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope, id, props) {
        super(scope, id, {
            api: 'invoke',
            service: 'Lambda',
            parameters: {
                FunctionName: props.functionName,
                InvocationType: props.invocationType,
                LogType: props.logType,
                Payload: props.payload,
            },
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_LambdaInvokeFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvokeFunction);
            }
            throw error;
        }
        const stack = aws_cdk_lib_1.Stack.of(this);
        // need to give the assertion lambda permission to invoke
        new aws_cdk_lib_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: props.functionName,
                Principal: this.provider.handlerRoleArn,
            },
        });
        // the api call is 'invoke', but the permission is 'invokeFunction'
        // so need to handle it specially
        this.provider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                service: 'lambda',
                resource: 'function',
                arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                resourceName: props.functionName,
            })]);
    }
}
exports.LambdaInvokeFunction = LambdaInvokeFunction;
_b = JSII_RTTI_SYMBOL_1;
LambdaInvokeFunction[_b] = { fqn: "@aws-cdk/integ-tests-alpha.LambdaInvokeFunction", version: "2.43.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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