var _a;
import { UIElement, UIElementView } from "../ui/ui_element";
import { Menu } from "../menus/menu";
import { Signal } from "../../core/signaling";
import { Align, SizingMode } from "../../core/enums";
import { position, classes, extents, undisplayed } from "../../core/dom";
import { logger } from "../../core/logging";
import { isNumber, isArray } from "../../core/util/types";
import { color2css } from "../../core/util/color";
import { assign } from "../../core/util/object";
import { parse_css_font_size } from "../../core/util/text";
import { build_views } from "../../core/build_views";
import { SizingPolicy } from "../../core/layout";
import { CanvasLayer } from "../../core/util/canvas";
export class LayoutDOMView extends UIElementView {
    constructor() {
        super(...arguments);
        this._offset_parent = null;
        this._viewport = {};
        this.mouseenter = new Signal(this, "mouseenter");
        this.mouseleave = new Signal(this, "mouseleave");
    }
    get is_layout_root() {
        return this.is_root || !(this.parent instanceof LayoutDOMView);
    }
    get base_font_size() {
        const font_size = getComputedStyle(this.el).fontSize;
        const result = parse_css_font_size(font_size);
        if (result != null) {
            const { value, unit } = result;
            if (unit == "px")
                return value;
        }
        return null;
    }
    initialize() {
        super.initialize();
        this.el.style.position = this.is_layout_root ? "relative" : "absolute";
        this._child_views = new Map();
    }
    async lazy_initialize() {
        await super.lazy_initialize();
        await this.build_child_views();
    }
    remove() {
        for (const child_view of this.child_views)
            child_view.remove();
        this._child_views.clear();
        super.remove();
    }
    connect_signals() {
        super.connect_signals();
        this.el.addEventListener("mouseenter", (event) => {
            this.mouseenter.emit(event);
        });
        this.el.addEventListener("mouseleave", (event) => {
            this.mouseleave.emit(event);
        });
        this.el.addEventListener("contextmenu", (event) => {
            if (this.model.context_menu != null) {
                console.log("context menu");
                event.preventDefault();
            }
        });
        if (this.is_layout_root) {
            this._on_resize = () => this.resize_layout();
            window.addEventListener("resize", this._on_resize);
            this._parent_observer = setInterval(() => {
                const offset_parent = this.el.offsetParent;
                if (this._offset_parent != offset_parent) {
                    this._offset_parent = offset_parent;
                    if (offset_parent != null) {
                        this.compute_viewport();
                        this.invalidate_layout();
                    }
                }
            }, 250);
        }
        const p = this.model.properties;
        this.on_change([
            p.width, p.height,
            p.min_width, p.min_height,
            p.max_width, p.max_height,
            p.margin,
            p.width_policy, p.height_policy, p.sizing_mode,
            p.aspect_ratio,
            p.visible,
        ], () => this.invalidate_layout());
        this.on_change([
            p.background,
            p.css_classes,
            p.style,
            p.stylesheets,
        ], () => this.invalidate_render());
    }
    disconnect_signals() {
        if (this._parent_observer != null)
            clearTimeout(this._parent_observer);
        if (this._on_resize != null)
            window.removeEventListener("resize", this._on_resize);
        super.disconnect_signals();
    }
    css_classes() {
        return [...super.css_classes(), ...this.model.css_classes];
    }
    styles() {
        return [...super.styles(), ...this.model.stylesheets];
    }
    *children() {
        yield* super.children();
        yield* this.child_views;
    }
    get child_views() {
        return this.child_models.map((child) => this._child_views.get(child));
    }
    async build_child_views() {
        await build_views(this._child_views, this.child_models, { parent: this });
    }
    render() {
        this.empty();
        assign(this.el.style, this.model.style);
        const { background } = this.model;
        this.el.style.backgroundColor = background != null ? color2css(background) : "";
        classes(this.el).clear().add(...this.css_classes());
        for (const child_view of this.child_views) {
            this.shadow_el.appendChild(child_view.el);
            child_view.render();
        }
    }
    update_layout() {
        for (const child_view of this.child_views)
            child_view.update_layout();
        this._update_layout();
    }
    update_position() {
        this.el.style.display = this.model.visible ? "block" : "none";
        const margin = this.is_layout_root ? this.layout.sizing.margin : undefined;
        position(this.el, this.layout.bbox, margin);
        for (const child_view of this.child_views)
            child_view.update_position();
    }
    after_layout() {
        for (const child_view of this.child_views)
            child_view.after_layout();
        this._has_finished = true;
    }
    compute_viewport() {
        this._viewport = this._viewport_size();
    }
    renderTo(element) {
        element.appendChild(this.el);
        this._offset_parent = this.el.offsetParent;
        this.compute_viewport();
        this.build();
        this.notify_finished();
    }
    build() {
        if (!this.is_layout_root)
            throw new Error(`${this.toString()} is not a root layout`);
        this.render();
        this.update_layout();
        this.compute_layout();
        return this;
    }
    async rebuild() {
        await this.build_child_views();
        this.invalidate_render();
    }
    compute_layout() {
        const start = Date.now();
        this.layout.compute(this._viewport);
        this.update_position();
        this.after_layout();
        logger.debug(`layout computed in ${Date.now() - start} ms`);
    }
    resize_layout() {
        this.root.compute_viewport();
        this.root.compute_layout();
    }
    invalidate_layout() {
        this.root.update_layout();
        this.root.compute_layout();
    }
    invalidate_render() {
        this.render();
        this.invalidate_layout();
    }
    has_finished() {
        if (!super.has_finished())
            return false;
        for (const child_view of this.child_views) {
            if (!child_view.has_finished())
                return false;
        }
        return true;
    }
    _width_policy() {
        return this.model.width != null ? "fixed" : "fit";
    }
    _height_policy() {
        return this.model.height != null ? "fixed" : "fit";
    }
    box_sizing() {
        let { width_policy, height_policy, aspect_ratio } = this.model;
        if (width_policy == "auto")
            width_policy = this._width_policy();
        if (height_policy == "auto")
            height_policy = this._height_policy();
        const { sizing_mode } = this.model;
        if (sizing_mode != null) {
            if (sizing_mode == "fixed")
                width_policy = height_policy = "fixed";
            else if (sizing_mode == "stretch_both")
                width_policy = height_policy = "max";
            else if (sizing_mode == "stretch_width")
                width_policy = "max";
            else if (sizing_mode == "stretch_height")
                height_policy = "max";
            else {
                if (aspect_ratio == null)
                    aspect_ratio = "auto";
                switch (sizing_mode) {
                    case "scale_width":
                        width_policy = "max";
                        height_policy = "min";
                        break;
                    case "scale_height":
                        width_policy = "min";
                        height_policy = "max";
                        break;
                    case "scale_both":
                        width_policy = "max";
                        height_policy = "max";
                        break;
                }
            }
        }
        const sizing = { width_policy, height_policy };
        const { min_width, min_height } = this.model;
        if (min_width != null)
            sizing.min_width = min_width;
        if (min_height != null)
            sizing.min_height = min_height;
        const { width, height } = this.model;
        if (width != null)
            sizing.width = width;
        if (height != null)
            sizing.height = height;
        const { max_width, max_height } = this.model;
        if (max_width != null)
            sizing.max_width = max_width;
        if (max_height != null)
            sizing.max_height = max_height;
        if (aspect_ratio == "auto" && width != null && height != null)
            sizing.aspect = width / height;
        else if (isNumber(aspect_ratio))
            sizing.aspect = aspect_ratio;
        const { margin } = this.model;
        if (margin != null) {
            if (isNumber(margin))
                sizing.margin = { top: margin, right: margin, bottom: margin, left: margin };
            else if (margin.length == 2) {
                const [vertical, horizontal] = margin;
                sizing.margin = { top: vertical, right: horizontal, bottom: vertical, left: horizontal };
            }
            else {
                const [top, right, bottom, left] = margin;
                sizing.margin = { top, right, bottom, left };
            }
        }
        sizing.visible = this.model.visible;
        const { align } = this.model;
        if (isArray(align))
            [sizing.halign, sizing.valign] = align;
        else
            sizing.halign = sizing.valign = align;
        return sizing;
    }
    _viewport_size() {
        return undisplayed(this.el, () => {
            let measuring = this.el;
            while (measuring = measuring.parentElement) {
                // .bk-root element doesn't bring any value
                if (measuring.classList.contains("bk-root"))
                    continue;
                // we reached <body> element, so use viewport size
                if (measuring == document.body) {
                    const { margin: { left, right, top, bottom } } = extents(document.body);
                    const width = Math.ceil(document.documentElement.clientWidth - left - right);
                    const height = Math.ceil(document.documentElement.clientHeight - top - bottom);
                    return { width, height };
                }
                // stop on first element with sensible dimensions
                const { padding: { left, right, top, bottom } } = extents(measuring);
                const { width, height } = measuring.getBoundingClientRect();
                const inner_width = Math.ceil(width - left - right);
                const inner_height = Math.ceil(height - top - bottom);
                if (inner_width > 0 || inner_height > 0)
                    return {
                        width: inner_width > 0 ? inner_width : undefined,
                        height: inner_height > 0 ? inner_height : undefined,
                    };
            }
            // this element is detached from DOM
            return {};
        });
    }
    export(type, hidpi = true) {
        const output_backend = type == "png" ? "canvas" : "svg";
        const composite = new CanvasLayer(output_backend, hidpi);
        const { width, height } = this.layout.bbox;
        composite.resize(width, height);
        for (const view of this.child_views) {
            const region = view.export(type, hidpi);
            const { x, y } = view.layout.bbox;
            composite.ctx.drawImage(region.canvas, x, y);
        }
        return composite;
    }
    serializable_state() {
        return {
            ...super.serializable_state(),
            bbox: this.layout.bbox.box,
            children: this.child_views.map((child) => child.serializable_state()),
        };
    }
}
LayoutDOMView.__name__ = "LayoutDOMView";
export class LayoutDOM extends UIElement {
    constructor(attrs) {
        super(attrs);
    }
}
_a = LayoutDOM;
LayoutDOM.__name__ = "LayoutDOM";
(() => {
    _a.define((types) => {
        const { Boolean, Number, String, Auto, Color, Array, Tuple, Dict, Or, Null, Nullable, Ref } = types;
        const Number2 = Tuple(Number, Number);
        const Number4 = Tuple(Number, Number, Number, Number);
        return {
            width: [Nullable(Number), null],
            height: [Nullable(Number), null],
            min_width: [Nullable(Number), null],
            min_height: [Nullable(Number), null],
            max_width: [Nullable(Number), null],
            max_height: [Nullable(Number), null],
            margin: [Nullable(Or(Number, Number2, Number4)), [0, 0, 0, 0]],
            width_policy: [Or(SizingPolicy, Auto), "auto"],
            height_policy: [Or(SizingPolicy, Auto), "auto"],
            aspect_ratio: [Or(Number, Auto, Null), null],
            sizing_mode: [Nullable(SizingMode), null],
            disabled: [Boolean, false],
            align: [Or(Align, Tuple(Align, Align)), "start"],
            background: [Nullable(Color), null],
            css_classes: [Array(String), []],
            style: [Dict(String), {}],
            stylesheets: [Array(String), []],
            context_menu: [Nullable(Ref(Menu)), null],
        };
    });
})();
//# sourceMappingURL=layout_dom.js.map