# Copyright 2021 Cognite AS


from typing import Literal

import pandas as pd

from scipy.signal import butter, sosfilt

from indsl import versioning
from indsl.type_check import check_types

from . import butterworth_  # noqa


# noinspection SpellCheckingInspection
@check_types
@versioning.register(
    version="2.0",
    changelog="Unused or irrelevant lines of code removed and filter output parameter "
    "removed from function signature and set to `sos`.",
)
def butterworth(
    data: pd.Series,
    N: int = 50,
    Wn: float = 0.1,
    btype: Literal["lowpass", "highpass"] = "lowpass",
):
    """Butterworth

    This is signal processing filter designed to have a frequency response as flat as possible in the passband and
    roll-offs towards zero in the stopband. In other words, this filter is designed not to modify much the signal at the
    in the passband and attenuate as much as possible the signal at the stopband. At the moment onlylow and high pass
    filtering is supported.

    Args:
        data: Time series.
        N: Order.
            Defaults to 50.
        Wn: Critical frequency.
            Number between 0 and 1, with 1 representing one-half of the sampling rate (Nyquist frequency).
            Defaults to 0.1.
        btype: Filter type.
            The options are: "lowpass" and "highpass"
            Defaults to "lowpass".

    Returns:
        pandas.Series: Filtered signal.
    """
    data = data.dropna()
    filter_output = butter(N=N, Wn=Wn, output="sos", btype=btype)
    # Apply second order segments
    filtered = sosfilt(filter_output, data, axis=0)

    return pd.Series(filtered, index=data.index)
