"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubOIDCStack = exports.TokenActions = exports.ProviderUrl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const utils_1 = require("../common/utils");
const stack_1 = require("../stack");
var ProviderUrl;
(function (ProviderUrl) {
    ProviderUrl["GITHUB"] = "https://token.actions.githubusercontent.com";
})(ProviderUrl = exports.ProviderUrl || (exports.ProviderUrl = {}));
var TokenActions;
(function (TokenActions) {
    TokenActions[TokenActions["ALL"] = 0] = "ALL";
    TokenActions[TokenActions["ALL_BRANCH"] = 1] = "ALL_BRANCH";
    TokenActions[TokenActions["ALL_TAGS"] = 2] = "ALL_TAGS";
    TokenActions[TokenActions["CUSTOM"] = 3] = "CUSTOM";
})(TokenActions = exports.TokenActions || (exports.TokenActions = {}));
class GithubOIDCStack extends stack_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.githubUser = props.githubUser;
        this.githubRepository = props.githubRepository;
        this.tokenAction = props.tokenAction;
        let token = this.createTokenAction(props.tokenAction, props.githubUser, props.githubRepository, props.tokenActionCustom);
        this.oidcRole = this.createOidcRole(ProviderUrl.GITHUB, token);
        this.cdkBootstrapRole = this.createCdkBootstrapRole();
        this.cdkDeployRoleManagedPolicies = props.cdkDeployRoleManagedPolicies;
        this.cdkDeployRolePolicyStatements = props.cdkDeployRolePolicyStatements;
        this.cdkDeployRole = this.createCdkDeployRole(this.cdkDeployRoleManagedPolicies, this.cdkDeployRolePolicyStatements);
    }
    createTokenAction(tokenAction, githubUser, githubRepository, tokenActionCustom) {
        if (tokenAction === TokenActions.ALL) {
            return `repo:${githubUser}/*`;
        }
        if (tokenAction === TokenActions.ALL_BRANCH) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/heads/*`;
        }
        if (tokenAction === TokenActions.ALL_TAGS) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/tags/*`;
        }
        if (tokenAction === TokenActions.CUSTOM) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/${tokenActionCustom}`;
        }
        throw new Error('tokenAction not found');
    }
    createCdkDeployRole(managed_policies, policy_statements) {
        let basePolicy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'iam:PassRole',
                        'iam:CreateRole',
                        'iam:DeleteRole',
                        'iam:UpdateRole',
                        'iam:DetachRolePolicy',
                        'iam:AttachRolePolicy',
                        'iam:DeleteRolePolicy',
                        'iam:PutRolePolicy',
                        'ssm:GetParameters',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-deploy-role', {
            roleName: 'cdk-oidc-deploy-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKDeployBasePolicy: basePolicy,
            },
        });
        if (policy_statements) {
            for (let index = 0; index < policy_statements.length; index++) {
                role.addToPolicy(policy_statements[index]);
            }
        }
        if (managed_policies) {
            for (let index = 0; index < managed_policies.length; index++) {
                role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName(managed_policies[index]));
            }
        }
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-deploy-role-output', {
            value: role.roleArn,
            description: 'Role for cdk deploy role arn',
            exportName: 'cdk-oidc-deploy-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createCdkBootstrapRole() {
        let policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                        'iam:*Role*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:cloudformation:${this.region}:${this.account}:stack/CDKToolkit/*`],
                    actions: [
                        'cloudformation:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        's3:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ecr:${this.region}:${this.account}:repository/cdk-*`],
                    actions: [
                        'ecr:SetRepositoryPolicy',
                        'ecr:GetLifecyclePolicy',
                        'ecr:PutImageScanningConfiguration',
                        'ecr:DescribeRepositories',
                        'ecr:CreateRepository',
                        'ecr:DeleteRepository',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ssm:${this.region}:${this.account}:parameter/cdk-bootstrap/*`],
                    actions: [
                        'ssm:GetParameter*',
                        'ssm:PutParameter*',
                        'ssm:DeleteParameter*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-bootstrap-role', {
            roleName: 'cdk-oidc-bootstrap-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKBootstrapPolicy: policy,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-bootstrap-role-output', {
            value: role.roleArn,
            description: 'Role for cdk bootstrap role arn',
            exportName: 'cdk-oidc-bootstrap-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createOidcRole(providerUrl, token) {
        const oidcProvider = new iam.OpenIdConnectProvider(this, 'OIDCProvider', {
            url: providerUrl,
            clientIds: ['sts.amazonaws.com'],
            thumbprints: ['6938fd4d98bab03faadb97b34396831e3780aea1'],
        });
        let role = new iam.Role(this, 'oidc-role', {
            roleName: 'github-oidc-workflow-role',
            assumedBy: new iam.WebIdentityPrincipal(oidcProvider.openIdConnectProviderArn, {
                StringLike: {
                    'token.actions.githubusercontent.com:sub': token,
                },
            }),
        });
        role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:PassRole',
                'ssm:GetParameter*',
                'cloudformation:*',
                's3:*',
                'ecr:*',
            ],
            effect: iam.Effect.ALLOW,
        }));
        role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
            actions: ['sts:AssumeRole'],
            effect: iam.Effect.ALLOW,
        }));
        new aws_cdk_lib_1.CfnOutput(this, 'github-oidc-workflow-role-output', {
            value: role.roleArn,
            description: 'Role for OIDC Github Workflow role arn',
            exportName: 'github-oidc-workflow-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
}
exports.GithubOIDCStack = GithubOIDCStack;
_a = JSII_RTTI_SYMBOL_1;
GithubOIDCStack[_a] = { fqn: "neulabs-cdk-constructs.oidc.GithubOIDCStack", version: "0.1.14" };
//# sourceMappingURL=data:application/json;base64,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