"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GatewayRouteSpec = exports.GatewayRouteHostnameMatch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const http_route_path_match_1 = require("./http-route-path-match");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * Used to generate host name matching methods.
 *
 * @stability stable
 */
class GatewayRouteHostnameMatch {
    /**
     * The value of the host name must match the specified value exactly.
     *
     * @param name The exact host name to match on.
     * @stability stable
     */
    static exactly(name) {
        return new GatewayRouteHostnameMatchImpl({ exact: name });
    }
    /**
     * The value of the host name with the given name must end with the specified characters.
     *
     * @param suffix The specified ending characters of the host name to match on.
     * @stability stable
     */
    static endsWith(suffix) {
        return new GatewayRouteHostnameMatchImpl({ suffix });
    }
}
exports.GatewayRouteHostnameMatch = GatewayRouteHostnameMatch;
_a = JSII_RTTI_SYMBOL_1;
GatewayRouteHostnameMatch[_a] = { fqn: "@aws-cdk/aws-appmesh.GatewayRouteHostnameMatch", version: "1.123.0" };
class GatewayRouteHostnameMatchImpl extends GatewayRouteHostnameMatch {
    constructor(matchProperty) {
        super();
        this.matchProperty = matchProperty;
    }
    bind(_scope) {
        return {
            hostnameMatch: this.matchProperty,
        };
    }
}
/**
 * Used to generate specs with different protocols for a GatewayRoute.
 *
 * @stability stable
 */
class GatewayRouteSpec {
    /**
     * Creates an HTTP Based GatewayRoute.
     *
     * @param options - no http gateway route.
     * @stability stable
     */
    static http(options) {
        return new HttpGatewayRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * Creates an HTTP2 Based GatewayRoute.
     *
     * @param options - no http2 gateway route.
     * @stability stable
     */
    static http2(options) {
        return new HttpGatewayRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * Creates an gRPC Based GatewayRoute.
     *
     * @param options - no grpc gateway route.
     * @stability stable
     */
    static grpc(options) {
        return new GrpcGatewayRouteSpec(options);
    }
}
exports.GatewayRouteSpec = GatewayRouteSpec;
_b = JSII_RTTI_SYMBOL_1;
GatewayRouteSpec[_b] = { fqn: "@aws-cdk/aws-appmesh.GatewayRouteSpec", version: "1.123.0" };
class HttpGatewayRouteSpec extends GatewayRouteSpec {
    constructor(options, protocol) {
        super();
        this.routeTarget = options.routeTarget;
        this.routeType = protocol;
        this.match = options.match;
    }
    bind(scope) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        const pathMatchConfig = ((_d = (_c = this.match) === null || _c === void 0 ? void 0 : _c.path) !== null && _d !== void 0 ? _d : http_route_path_match_1.HttpGatewayRoutePathMatch.startsWith('/')).bind(scope);
        const rewriteRequestHostname = (_e = this.match) === null || _e === void 0 ? void 0 : _e.rewriteRequestHostname;
        const prefixPathRewrite = pathMatchConfig.prefixPathRewrite;
        const wholePathRewrite = pathMatchConfig.wholePathRewrite;
        const httpConfig = {
            match: {
                prefix: pathMatchConfig.prefixPathMatch,
                path: pathMatchConfig.wholePathMatch,
                hostname: (_g = (_f = this.match) === null || _f === void 0 ? void 0 : _f.hostname) === null || _g === void 0 ? void 0 : _g.bind(scope).hostnameMatch,
                method: (_h = this.match) === null || _h === void 0 ? void 0 : _h.method,
                headers: (_k = (_j = this.match) === null || _j === void 0 ? void 0 : _j.headers) === null || _k === void 0 ? void 0 : _k.map(header => header.bind(scope).headerMatch),
                queryParameters: (_m = (_l = this.match) === null || _l === void 0 ? void 0 : _l.queryParameters) === null || _m === void 0 ? void 0 : _m.map(queryParameter => queryParameter.bind(scope).queryParameterMatch),
            },
            action: {
                target: {
                    virtualService: {
                        virtualServiceName: this.routeTarget.virtualServiceName,
                    },
                },
                rewrite: rewriteRequestHostname !== undefined || prefixPathRewrite || wholePathRewrite
                    ? {
                        hostname: rewriteRequestHostname === undefined
                            ? undefined
                            : {
                                defaultTargetHostname: rewriteRequestHostname ? 'ENABLED' : 'DISABLED',
                            },
                        prefix: prefixPathRewrite,
                        path: wholePathRewrite,
                    }
                    : undefined,
            },
        };
        return {
            httpSpecConfig: this.routeType === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2SpecConfig: this.routeType === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class GrpcGatewayRouteSpec extends GatewayRouteSpec {
    constructor(options) {
        super();
        this.match = options.match;
        this.routeTarget = options.routeTarget;
    }
    bind(scope) {
        var _c;
        const metadataMatch = this.match.metadata;
        utils_1.validateGrpcGatewayRouteMatch(this.match);
        utils_1.validateGrpcMatchArrayLength(metadataMatch);
        return {
            grpcSpecConfig: {
                match: {
                    serviceName: this.match.serviceName,
                    hostname: (_c = this.match.hostname) === null || _c === void 0 ? void 0 : _c.bind(scope).hostnameMatch,
                    metadata: metadataMatch === null || metadataMatch === void 0 ? void 0 : metadataMatch.map(metadata => metadata.bind(scope).headerMatch),
                },
                action: {
                    target: {
                        virtualService: {
                            virtualServiceName: this.routeTarget.virtualServiceName,
                        },
                    },
                    rewrite: this.match.rewriteRequestHostname === undefined
                        ? undefined
                        : {
                            hostname: {
                                defaultTargetHostname: this.match.rewriteRequestHostname ? 'ENABLED' : 'DISABLED',
                            },
                        },
                },
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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