# vanilla-roll
[![Build][build-shiled]][build-url]
[![Version][version-shield]][version-url]
[![Downloads][download-shield]][download-url]
[![Contributors][contributors-shield]][contributors-url]
[![Issues][issues-shield]][issues-url]
[![Codecov][codecov-shield]][codecov-url]
[![Apache License 2.0 License][license-shield]][license-url]

vanilla-roll is volume renderer using array-api as backend.

## Why vanilla-roll ?
[VTK](https://vtk.org/) is one of the most excellent volume renderers in the world.
It has good performance and many features.
But I think that the installation process is not easy.
So vanilla-roll is motivated to solve this problem.

## Features
- [ ] IO
  - [x] MRA
  - [x] NIFTI
  - [x] DICOM
  - [ ] NRRD
- [ ] Rendering Algorithm
  - [x] Sampling
  - [x] Shear-Warp
  - [ ] Raycast
- [ ] Rendering Mode
  - [x] MIP
  - [x] MinP
  - [x] Average
  - [ ] VolumeRendering
      - [x] Ambient
      - [ ] Shading
- [ ] Backend
  - [x] numpy
  - [x] pytorch
  - [x] cupy
  - [ ] jax
  - [ ] numbda


## Installation
```bash
$ pip install vanilla-roll
```
vanilla-roll supports following extras

* torch
* dicom
* mha
* nifti


## Example
Code

```python
import urllib.request
from pathlib import Path
from tempfile import TemporaryDirectory

import numpy as np
import skimage.io

import vanilla_roll as vr

# from A high-resolution 7-Tesla fMRI dataset from complex natural stimulation with an audio movie
# https://www.openfmri.org/dataset/ds000113/
MRA_FILE_URL = "https://s3.amazonaws.com/openneuro/ds000113/ds000113_unrevisioned/uncompressed/sub003/angio/angio001.nii.gz"  # noqa: E501


def fetch_mra_volume() -> vr.volume.Volume:
    with TemporaryDirectory() as tmpdir:
        mra_file = Path(tmpdir) / "mra.nii.gz"
        urllib.request.urlretrieve(MRA_FILE_URL, mra_file)
        return vr.io.read_nifti(mra_file)


def save_result(ret: vr.rendering.types.RenderingResult, path: str):
    img_array = vr.rendering.convert_image_to_array(ret.image)
    skimage.io.imsave(path, np.from_dlpack(img_array))  # type: ignore


def main():
    volume = fetch_mra_volume()
    ret = vr.render(volume, mode=vr.rendering.mode.MIP())
    save_result(ret, f"result.png")


if __name__ == "__main__":
    main()
```

Output

![output](https://raw.githubusercontent.com/ar90n/vanilla-roll/assets/images/simple.png)

If you need more exmplaes, please check the [examples](https://github.com/ar90n/vanilla-roll/tree/main/examples).

## For development
### Install Poery plugins
```bash
$ poetry self add 'poethepoet[poetry_plugin]'
```

### Install all extra packages
```bash
$ poetry poe install-all-extras
```

### Run tests
```bash
$ poetry poe test
```

### Run linter and formatter
```bash
$ poetry poe check
```

## See Also

## License
[Apache-2.0](https://github.com/ar90n/vanilla-roll/blob/main/LICENSE)

[download-shield]: https://img.shields.io/pypi/dm/vanilla-roll?style=flat
[download-url]: https://pypi.org/project/vanilla-roll/
[version-shield]: https://img.shields.io/pypi/v/vanilla-roll?style=flat
[version-url]: https://pypi.org/project/vanilla-roll/
[build-shiled]: https://img.shields.io/github/workflow/status/ar90n/vanilla-roll/CI%20testing/main
[build-url]: https://github.com/ar90n/vanilla-roll/actions/workflows/ci-testing.yml
[contributors-shield]: https://img.shields.io/github/contributors/ar90n/vanilla-roll.svg?style=flat
[contributors-url]: https://github.com/ar90n/vanilla-roll/graphs/contributors
[issues-shield]: https://img.shields.io/github/issues/ar90n/vanilla-roll.svg?style=flat
[issues-url]: https://github.com/ar90n/vanilla-roll/issues
[license-shield]: https://img.shields.io/github/license/ar90n/vanilla-roll.svg?style=flat
[license-url]: https://github.com/ar90n/vanilla-roll/blob/main/LICENSE
[codecov-shield]: https://codecov.io/gh/ar90n/vanilla-roll/branch/main/graph/badge.svg?token=8GKU96ODLY
[codecov-url]: https://codecov.io/gh/ar90n/vanilla-roll
