import sqlite3
import sys
from pathlib import Path
from tempfile import NamedTemporaryFile

sys.path.insert(0, str(Path(__file__).resolve().parents[1] / "src"))

from limen.core import Atom, Constant, FormulaNode, KnowledgeBase, Predicate, WeightedFormula, Operator
from limen.storage import load_knowledge_base, save_knowledge_base


def _build_sample_kb() -> KnowledgeBase:
    kb = KnowledgeBase()
    pred = Predicate(name="failedLogin", arity=1, description="failed logins")
    alert = Predicate(name="raiseAlert", arity=1)
    user = Constant("alice")

    kb.add_predicate(pred)
    kb.add_predicate(alert)
    kb.add_constant(user)

    atom_failed = Atom(predicate=pred, arguments=(user,))
    atom_alert = Atom(predicate=alert, arguments=(user,))

    rule = FormulaNode(operator=Operator.IMPLIES, children=(
        FormulaNode.atom_node(atom_failed),
        FormulaNode.atom_node(atom_alert),
    ))
    kb.add_formula(WeightedFormula(formula=rule, weight=1.5, name="alert_rule"))
    return kb


def test_save_and_load_roundtrip(tmp_path):
    kb = _build_sample_kb()
    db_file = tmp_path / "kb.sqlite"
    save_knowledge_base(kb, db_file)

    # sanity check: file exists and basic tables present
    with sqlite3.connect(db_file) as conn:
        rows = conn.execute("SELECT COUNT(*) FROM predicates").fetchone()
        assert rows[0] == 2

    restored = load_knowledge_base(db_file)
    assert set(restored.predicates.keys()) == {"failedLogin", "raiseAlert"}
    assert set(restored.constants.keys()) == {"alice"}
    assert len(restored.formulas) == 1
    wf = restored.formulas[0]
    assert wf.name == "alert_rule"
    assert abs(wf.weight - 1.5) < 1e-9
