import sys
from pathlib import Path

import pytest

sys.path.insert(0, str(Path(__file__).resolve().parents[1] / "src"))

from limen import (
    Atom,
    Constant,
    GroundingPlan,
    KnowledgeBase,
    Predicate,
    assignment_from_observations,
    auto_ground,
    generate_ground_atoms,
)


def _build_kb():
    kb = KnowledgeBase()
    user = Constant("u1")
    host = Constant("h1")
    kb.add_constant(user)
    kb.add_constant(host)
    kb.add_predicate(Predicate("fails", 1))
    kb.add_predicate(Predicate("connects", 2))
    return kb


def test_generate_ground_atoms_uses_all_domains():
    kb = _build_kb()
    atoms = generate_ground_atoms(kb, "fails")
    assert len(atoms) == 2  # two constants, arity 1
    assert all(isinstance(atom, Atom) for atom in atoms)


def test_auto_ground_with_explicit_plan():
    kb = _build_kb()
    plan = GroundingPlan(predicate="connects", domains=[["u1"], ["h1"]])
    atoms = auto_ground(kb, [plan])
    assert len(atoms) == 1
    assert atoms[0].predicate.name == "connects"


def test_assignment_from_observations_sets_values():
    kb = _build_kb()
    observations = {
        "fails": [{"arguments": ["u1"], "value": 0.8}],
        "connects": [{"arguments": ["u1", "h1"], "value": 0.3}],
    }
    assignment = assignment_from_observations(kb, observations)
    fails_pred = kb.get_predicate("fails")
    connects_pred = kb.get_predicate("connects")
    assert assignment.get(Atom(fails_pred, (kb.get_constant("u1"),))) == pytest.approx(0.8)
    assert assignment.get(Atom(connects_pred, (kb.get_constant("u1"), kb.get_constant("h1")))) == pytest.approx(0.3)

