import sys
from pathlib import Path

import pytest

sys.path.insert(0, str(Path(__file__).resolve().parents[1] / "src"))

from limen.config import (
    available_truth_function_types,
    load_kb_from_config,
    register_truth_function_factory,
    validate_config,
)
from limen.core import Atom, KnowledgeBase
from limen.truth_functions import ConstantTruthFunction
from limen.semantics import evaluate_formula


def test_load_kb_from_config_constant_truths():
    config = {
        "predicates": [
            {"name": "failed", "arity": 1},
            {"name": "flag", "arity": 1},
        ],
        "constants": ["u"],
        "formulas": [
            {
                "name": "rule",
                "weight": 1.0,
                "formula": {
                    "operator": "implies",
                    "children": [
                        {
                            "operator": "atom",
                            "predicate": "failed",
                            "arguments": ["u"],
                        },
                        {
                            "operator": "atom",
                            "predicate": "flag",
                            "arguments": ["u"],
                        },
                    ],
                },
            }
        ],
        "truth_functions": [
            {"predicate": "failed", "type": "constant", "value": 0.8},
            {"predicate": "flag", "type": "constant", "value": 0.2},
        ],
    }

    kb = load_kb_from_config(config)
    assert isinstance(kb, KnowledgeBase)
    assignment = kb.build_assignment_from_truth_functions()
    value = evaluate_formula(kb.formulas[0].formula, assignment)
    assert value == pytest.approx(0.4, abs=1e-9)


def test_load_kb_from_config_detects_duplicate_predicates():
    config = {
        "predicates": [
            {"name": "failed", "arity": 1},
            {"name": "failed", "arity": 1},
        ],
        "constants": [],
        "formulas": [
            {
                "name": "pref_rule",
                "weight": 1.0,
                "formula": {
                    "operator": "atom",
                    "predicate": "prefers",
                    "arguments": ["alice", "policyA"],
                },
            }
        ],
        "truth_functions": [],
    }
    with pytest.raises(ValueError, match="Duplicate predicate"):
        load_kb_from_config(config)


def test_validate_config_catches_unknown_truth_function_predicate():
    config = {
        "predicates": [{"name": "p", "arity": 0}],
        "constants": [],
        "formulas": [],
        "truth_functions": [{"predicate": "unknown", "type": "constant", "value": 0.3}],
    }
    issues = validate_config(config)
    assert any("unknown" in issue.message for issue in issues)


def test_register_truth_function_factory_supports_custom_type():
    tf_name = "test_constant"

    def factory(spec):
        return ConstantTruthFunction(spec["value"])

    register_truth_function_factory(tf_name, factory)

    config = {
        "predicates": [{"name": "flag", "arity": 1}],
        "constants": ["u"],
        "formulas": [
            {
                "name": "identity",
                "weight": 1.0,
                "formula": {
                    "operator": "atom",
                    "predicate": "flag",
                    "arguments": ["u"],
                },
            }
        ],
        "truth_functions": [{"predicate": "flag", "type": tf_name, "value": 0.55}],
    }

    kb = load_kb_from_config(config)
    assignment = kb.build_assignment_from_truth_functions()
    predicate = kb.get_predicate("flag")
    atom = Atom(predicate=predicate, arguments=(kb.get_constant("u"),))
    assert assignment.get(atom) == pytest.approx(0.55, abs=1e-9)


def test_table_truth_function_via_config():
    config = {
        "predicates": [{"name": "prefers", "arity": 2}],
        "constants": ["alice", "policyA"],
        "formulas": [
            {
                "name": "pref_rule",
                "weight": 1.0,
                "formula": {
                    "operator": "atom",
                    "predicate": "prefers",
                    "arguments": ["alice", "policyA"],
                },
            }
        ],
        "truth_functions": [
            {
                "predicate": "prefers",
                "type": "table",
                "default": 0.1,
                "values": [
                    {"arguments": ["alice", "policyA"], "value": 0.9},
                ],
            }
        ],
    }
    kb = load_kb_from_config(config)
    predicate = kb.get_predicate("prefers")
    atom = Atom(predicate, (kb.get_constant("alice"), kb.get_constant("policyA")))
    assignment = kb.build_assignment_from_truth_functions()
    assert assignment.get(atom) == pytest.approx(0.9, abs=1e-9)
    assert "table" in available_truth_function_types()
