"""Translate natural-language prompts into structured LIMEN-AI queries."""

from __future__ import annotations

import json
from dataclasses import dataclass
from typing import Dict, List, Tuple

from ..core import Atom
from .llm_client import LLMClient
from .prompts import build_query_prompt
from .schema import SchemaRegistry


@dataclass
class StructuredQuery:
    predicate: str
    args: Tuple[str, ...]


class QueryTranslator:
    def __init__(self, registry: SchemaRegistry, llm_client: LLMClient):
        self.registry = registry
        self.llm = llm_client

    def translate(self, prompt: str) -> List[StructuredQuery]:
        completion = self.llm.complete(build_query_prompt(prompt, self.registry))
        payload = self._safe_json(completion)
        queries: List[StructuredQuery] = []
        for entry in payload.get("queries", []):
            predicate = entry.get("predicate")
            args = entry.get("args", [])
            ok, _ = self.registry.validate_fact(predicate, args)
            if ok:
                queries.append(StructuredQuery(predicate, tuple(args)))
        return queries

    def _safe_json(self, completion: str) -> Dict:
        completion = completion.strip()
        if not completion:
            return {"queries": []}
        try:
            return json.loads(completion)
        except json.JSONDecodeError:
            return {"queries": []}

