"""Prompt templates for LIMEN-AI + LLM orchestration."""

from __future__ import annotations

from textwrap import dedent

from .schema import SchemaRegistry


def build_extraction_prompt(chunk: str, registry: SchemaRegistry) -> str:
    """Prompt instructing the LLM to extract structured facts."""

    instruction = registry.as_instruction()
    prompt = f"""
You are an information extraction assistant. Analyse the following text chunk and emit
JSON with the format:
[
  {{
    "predicate": "predicate_name",
    "args": ["arg1", "arg2", ...],
    "confidence": 0.0-1.0,
    "provenance": "optional source snippet"
  }}
]
Only use predicates from the schema below, and make sure the number of arguments matches the definition.
If no facts are found, return an empty JSON list [].

{instruction}

Text chunk:
\"\"\"
{chunk}
\"\"\"

JSON:
"""
    return dedent(prompt).strip()


def build_query_prompt(question: str, registry: SchemaRegistry) -> str:
    """Prompt that translates a user question into predicate calls."""

    instruction = registry.as_instruction()
    prompt = f"""
Translate the user's question into a list of predicate calls from the schema below.
Respond in JSON with the format:
{{
  "queries": [
    {{"predicate": "name", "args": ["arg1", "arg2", ...]}}
  ]
}}
If the question cannot be mapped, respond with {{"queries": []}}.

{instruction}

Question: \"{question}\"
JSON:
"""
    return dedent(prompt).strip()


def build_response_prompt(structured_answer: str, user_question: str) -> str:
    """Prompt instructing the LLM to turn structured data into natural language."""

    prompt = f"""
You are an analyst assistant. Convert the following structured LIMEN-AI answer into
a short, user-friendly explanation. Ensure you mention the key predicates involved and
highlight any provenance.

Question: {user_question}

Structured answer:
{structured_answer}

Response:
"""
    return dedent(prompt).strip()

