"""Lightweight wrappers for invoking large language models."""

from __future__ import annotations

from dataclasses import dataclass
from typing import Callable, Optional


CompletionFn = Callable[[str], str]


@dataclass
class LLMClient:
    """Minimal client that delegates completions to a callable."""

    completion_fn: CompletionFn

    def complete(self, prompt: str) -> str:
        return self.completion_fn(prompt)


class MockLLMClient(LLMClient):
    """Deterministic mock useful for tests and notebooks."""

    def __init__(self, responses: Optional[dict[str, str]] = None):
        self.responses = responses or {}
        super().__init__(self._default_completion)

    def _default_completion(self, prompt: str) -> str:
        for key, value in self.responses.items():
            if key in prompt:
                return value
        # fallback: echo empty JSON structure to avoid crashes
        return self.responses.get("__default__", "[]")

