"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlShortener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const shortener_function_1 = require("./shortener-function");
/**
 * URL shortener
 */
class UrlShortener extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        // Table to save a counter
        const table = new dynamodb.Table(this, 'Table', {
            partitionKey: {
                name: 'key',
                type: dynamodb.AttributeType.STRING,
            },
        });
        // Bucket to save redirects
        const bucket = new s3.Bucket(this, 'Bucket', {
            lifecycleRules: [{
                    expiration: (_b = props.expiration) !== null && _b !== void 0 ? _b : aws_cdk_lib_1.Duration.days(365),
                }],
            websiteIndexDocument: 'index.html',
        });
        // CloudFront distribution
        const domainName = props.recordName ? `${props.recordName}.${props.hostedZone.zoneName}` : props.hostedZone.zoneName;
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(bucket),
            },
            certificate,
            domainNames: [domainName],
        });
        // Route53 records
        new route53.ARecord(this, 'ARecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        // Lambda function to increment counter and write redirect in bucket
        const handler = new shortener_function_1.ShortenerFunction(this, 'handler', {
            logRetention: logs.RetentionDays.ONE_MONTH,
            environment: {
                DOMAIN_NAME: domainName,
                BUCKET_NAME: bucket.bucketName,
                TABLE_NAME: table.tableName,
            },
        });
        if (props.corsAllowOrigins) {
            handler.addEnvironment('CORS_ALLOW_ORIGINS', props.corsAllowOrigins.join(' '));
        }
        bucket.grantPut(handler);
        bucket.grantPutAcl(handler);
        table.grant(handler, 'dynamodb:UpdateItem');
        // API
        this.api = new apigateway.RestApi(this, `UrlShortener${props.hostedZone.zoneName}`, {
            endpointTypes: props.apiGatewayEndpoint ? [apigateway.EndpointType.PRIVATE] : undefined,
            policy: props.apiGatewayEndpoint
                ? new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['execute-api:Invoke'],
                            principals: [new iam.AnyPrincipal()],
                            resources: [aws_cdk_lib_1.Fn.join('', ['execute-api:/', '*'])],
                            conditions: {
                                StringEquals: { 'aws:SourceVpce': props.apiGatewayEndpoint.vpcEndpointId },
                            },
                        }),
                    ],
                })
                : undefined,
            defaultCorsPreflightOptions: props.corsAllowOrigins
                ? { allowOrigins: props.corsAllowOrigins }
                : undefined,
        });
        this.api.root.addMethod('ANY', new apigateway.LambdaIntegration(handler), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.api.root
            .addResource('{proxy+}')
            .addMethod('ANY', new apigateway.LambdaIntegration(handler), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.apiEndpoint = this.api.url;
    }
}
exports.UrlShortener = UrlShortener;
_a = JSII_RTTI_SYMBOL_1;
UrlShortener[_a] = { fqn: "cloudstructs.UrlShortener", version: "0.4.26" };
//# sourceMappingURL=data:application/json;base64,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