"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const codePipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_DEFAULT_REGION,
            account: process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'my-codepipeline-status-stack', { env });
        const pipeline = this.createCodePipeline(stack);
        const codePipelineStatus = new index_1.CodePipelineStatus(stack, 'CodePipelineStatus', {
            pipelineArn: pipeline.pipelineArn,
            gitHubTokenFromSecretsManager: {
                secretsManagerArn: 'arn:aws:secretsmanager:ap-northeast-1:482631629698:secret:codepipeline/lambda/github-token-YWWmII',
                secretKey: 'codepipeline/lambda/github-token',
            },
            notification: {
                stageName: 'production',
                slackWebHookUrl: 'slack url webhook',
            },
        });
        new cdk.CfnOutput(stack, 'BadgeUrl', {
            value: codePipelineStatus.badgeUrl,
        });
        new cdk.CfnOutput(stack, 'CodePipelineLink', {
            value: codePipelineStatus.codePipelineLink,
        });
        app.synth();
        this.stack = [stack];
    }
    createCodePipeline(stack) {
        const pipeline = new codePipeline.Pipeline(stack, 'TestPipeline', {
            pipelineName: 'integTestCodePipeline',
            crossAccountKeys: false,
        });
        const sourceOutput = new codePipeline.Artifact();
        pipeline.addStage({
            stageName: 'Source',
            actions: [
                new codepipeline_actions.CodeStarConnectionsSourceAction({
                    actionName: 'GitHub_Source',
                    owner: 'kimisme9386',
                    repo: 'cdk-codepipeline-status',
                    output: sourceOutput,
                    connectionArn: `arn:aws:codestar-connections:ap-northeast-1:${cdk.Aws.ACCOUNT_ID}:connection/XXxxxxxxxxxx`,
                    variablesNamespace: 'GitHubSourceVariables',
                    branch: 'dev',
                    codeBuildCloneOutput: true,
                }),
            ],
        });
        const project = this.createCodeBuildProjectWithinCodePipeline(stack, 'buildspec.yml');
        const afterBuildArtifact = new codePipeline.Artifact();
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new codepipeline_actions.CodeBuildAction({
                    actionName: 'AWS_CodeBuild',
                    input: sourceOutput,
                    project: project,
                    type: codepipeline_actions.CodeBuildActionType.BUILD,
                    outputs: [afterBuildArtifact],
                }),
            ],
        });
        return pipeline;
    }
    createCodeBuildProjectWithinCodePipeline(stack, buildSpecPath) {
        const project = new codebuild.PipelineProject(stack, 'CodeBuildWithinCodePipeline', {
            buildSpec: codebuild.BuildSpec.fromSourceFilename(buildSpecPath),
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                computeType: codebuild.ComputeType.SMALL,
                privileged: true,
            },
            cache: codebuild.Cache.local(codebuild.LocalCacheMode.DOCKER_LAYER),
        });
        return project;
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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