#!/usr/bin/env python
"""test io"""
import os
import json
import pytest
import dandelion as ddl
import pandas as pd


@pytest.mark.usefixtures("create_testfolder", "airr_10x")
def test_write_airr(create_testfolder, airr_10x):
    """test_write_airr"""
    out_file = str(create_testfolder) + "/test_airr_rearrangements.tsv"
    airr_10x.to_csv(out_file, sep="\t", index=False)
    assert len(list(create_testfolder.iterdir())) == 1


@pytest.mark.usefixtures("create_testfolder")
def test_loaddata(create_testfolder):
    """test_loaddata"""
    file1 = str(create_testfolder) + "/test_airr_rearrangements.tsv"
    file2 = str(create_testfolder) + "/test_airr_rearrangements2.tsv"
    dat = ddl.utl.load_data(file1)
    assert isinstance(dat, pd.DataFrame)
    with pytest.raises(FileNotFoundError):
        dat2 = ddl.utl.load_data(file2)
    with pytest.raises(FileNotFoundError):
        dat2 = ddl.utl.load_data("something.tsv")
    dat2 = pd.read_csv(file1, sep="\t")
    dat2.drop("sequence_id", inplace=True, axis=1)
    with pytest.raises(KeyError):
        dat2 = ddl.utl.load_data(dat2)


@pytest.mark.usefixtures("create_testfolder", "airr_reannotated")
def test_write_annotated(create_testfolder, airr_reannotated):
    """test_write_annotated"""
    out_file = str(create_testfolder) + "/test_airr_reannotated.tsv"
    airr_reannotated.to_csv(out_file, sep="\t", index=False)
    assert not airr_reannotated.np1_length.empty
    assert not airr_reannotated.np2_length.empty
    assert not airr_reannotated.junction_length.empty


@pytest.mark.usefixtures("create_testfolder")
def test_readwrite_h5(create_testfolder):
    """test_readwrite_h5"""
    out_file1 = str(create_testfolder) + "/test_airr_reannotated.tsv"
    out_file2 = str(create_testfolder) + "/test_airr_reannotated.h5"
    vdj = ddl.Dandelion(out_file1)
    assert not vdj.data.np1_length.empty
    assert not vdj.data.np2_length.empty
    assert not vdj.data.junction_length.empty
    vdj.write_h5(out_file2)
    vdj2 = ddl.read_h5(out_file2)
    assert not vdj2.data.np1_length.empty
    assert not vdj2.data.np2_length.empty
    assert not vdj2.data.junction_length.empty
    vdj.write_h5(out_file2, complib="blosc:lz4")
    vdj2 = ddl.read_h5(out_file2)
    assert not vdj2.data.np1_length.empty
    assert not vdj2.data.np2_length.empty
    assert not vdj2.data.junction_length.empty
    vdj.write_h5(out_file2, compression="blosc:lz4")
    vdj2 = ddl.read_h5(out_file2)
    assert not vdj2.data.np1_length.empty
    assert not vdj2.data.np2_length.empty
    assert not vdj2.data.junction_length.empty
    with pytest.raises(ValueError):
        vdj.write_h5(out_file2, complib="blosc:lz4", compression="blosc:lz4")


@pytest.mark.usefixtures("create_testfolder")
def test_readwrite_pkl(create_testfolder):
    """test_readwrite_pkl"""
    out_file1 = str(create_testfolder) + "/test_airr_reannotated.tsv"
    out_file2 = str(create_testfolder) + "/test_airr_reannotated.pkl"
    out_file3 = str(create_testfolder) + "/test_airr_reannotated.pkl.gz"
    out_file4 = str(create_testfolder) + "/test_airr_reannotated.pkl.pbz2"
    vdj = ddl.Dandelion(out_file1)
    assert not vdj.data.np1_length.empty
    assert not vdj.data.np2_length.empty
    assert not vdj.data.junction_length.empty
    vdj.write_pkl(out_file2)
    vdj3 = ddl.read_pkl(out_file2)
    assert not vdj3.data.np1_length.empty
    assert not vdj3.data.np2_length.empty
    assert not vdj3.data.junction_length.empty
    vdj.write_pkl(out_file3)
    vdj4 = ddl.read_pkl(out_file3)
    assert not vdj4.data.np1_length.empty
    assert not vdj4.data.np2_length.empty
    assert not vdj4.data.junction_length.empty
    vdj.write_pkl(out_file4)
    vdj5 = ddl.read_pkl(out_file4)
    assert not vdj5.data.np1_length.empty
    assert not vdj5.data.np2_length.empty
    assert not vdj5.data.junction_length.empty


@pytest.mark.usefixtures("create_testfolder")
def test_readwrite10xairr(create_testfolder):
    """test_readwrite10xairr"""
    airr_file = str(create_testfolder) + "/test_airr_rearrangements.tsv"
    airr_file2 = str(create_testfolder) + "/test_airr_rearrangements2.tsv"
    vdj = ddl.read_10x_airr(airr_file)
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 32
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 27
    vdj.write_airr(airr_file2)
    vdj2 = ddl.read_10x_airr(airr_file2)
    assert vdj2.data.shape[0] == 9
    assert vdj2.data.shape[1] == 32
    assert vdj2.metadata.shape[0] == 5
    assert vdj2.metadata.shape[1] == 27
    os.remove(airr_file2)


@pytest.mark.usefixtures("create_testfolder", "json_10x_cr6")
def test_read10xvdj_json(create_testfolder, json_10x_cr6):
    """test_read10xvdj_json"""
    json_file = str(create_testfolder) + "/test_all_contig_annotations.json"
    with open(json_file, "w") as outfile:
        json.dump(json_10x_cr6, outfile)
    vdj = ddl.read_10x_vdj(json_file)
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 49
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    os.remove(json_file)


@pytest.mark.usefixtures(
    "create_testfolder", "json_10x_cr6", "annotation_10x_cr6", "fasta_10x_cr6"
)
def test_read10xvdj_cr6(
    create_testfolder, json_10x_cr6, annotation_10x_cr6, fasta_10x_cr6
):
    """test_read10xvdj_cr6"""
    fasta_file = str(create_testfolder) + "/test_filtered_contig.fasta"
    json_file = str(create_testfolder) + "/test_all_contig_annotations.json"
    annot_file = (
        str(create_testfolder) + "/test_filtered_contig_annotations.csv"
    )
    annotation_10x_cr6.to_csv(annot_file, index=False)
    vdj = ddl.read_10x_vdj(annot_file)
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 31
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    with open(json_file, "w") as outfile:
        json.dump(json_10x_cr6, outfile)
    vdj = ddl.read_10x_vdj(annot_file)
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 49
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    assert not vdj.data.sequence.empty
    os.remove(json_file)
    fh = open(fasta_file, "w")
    fh.close()
    out = ""
    for line in fasta_10x_cr6:
        out = ">" + line + "\n" + fasta_10x_cr6[line] + "\n"
        ddl.utl.Write_output(out, fasta_file)
    vdj = ddl.read_10x_vdj(annot_file)
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 32
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    assert not vdj.data.sequence.empty
    os.remove(fasta_file)


@pytest.mark.usefixtures("create_testfolder", "annotation_10x", "fasta_10x")
def test_read10xvdj(create_testfolder, annotation_10x, fasta_10x):
    """test_read10xvdj"""
    fasta_file = str(create_testfolder) + "/test_filtered_contig.fasta"
    annot_file = (
        str(create_testfolder) + "/test_filtered_contig_annotations.csv"
    )
    annotation_10x.to_csv(annot_file, index=False)
    vdj = ddl.read_10x_vdj(annot_file)
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 18
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 29
    fh = open(fasta_file, "w")
    fh.close()
    out = ""
    for line in fasta_10x:
        out = ">" + line + "\n" + fasta_10x[line] + "\n"
        ddl.utl.Write_output(out, fasta_file)
    vdj = ddl.read_10x_vdj(annot_file)
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 19
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 29
    assert not vdj.data.sequence.empty
    os.remove(fasta_file)


@pytest.mark.usefixtures(
    "create_testfolder", "json_10x_cr6", "annotation_10x_cr6", "fasta_10x_cr6"
)
def test_read10xvdj_cr6_folder(
    create_testfolder, json_10x_cr6, annotation_10x_cr6, fasta_10x_cr6
):
    """test_read10xvdj_cr6_folder"""
    fasta_file = str(create_testfolder) + "/test_filtered_contig.fasta"
    json_file = str(create_testfolder) + "/test_all_contig_annotations.json"
    annot_file = (
        str(create_testfolder) + "/test_filtered_contig_annotations.csv"
    )
    annotation_10x_cr6.to_csv(annot_file, index=False)
    vdj = ddl.read_10x_vdj(str(create_testfolder))
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 31
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    with open(json_file, "w") as outfile:
        json.dump(json_10x_cr6, outfile)
    vdj = ddl.read_10x_vdj(str(create_testfolder), filename_prefix="test_all")
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 49
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    assert not vdj.data.sequence.empty
    os.remove(json_file)
    fh = open(fasta_file, "w")
    fh.close()
    out = ""
    for line in fasta_10x_cr6:
        out = ">" + line + "\n" + fasta_10x_cr6[line] + "\n"
        ddl.utl.Write_output(out, fasta_file)
    vdj = ddl.read_10x_vdj(str(create_testfolder))
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 32
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    assert not vdj.data.sequence.empty
    os.remove(fasta_file)


@pytest.mark.usefixtures("create_testfolder", "annotation_10x", "fasta_10x")
def test_read10xvdj_folder(create_testfolder, annotation_10x, fasta_10x):
    """test_read10xvdj_folder"""
    fasta_file = str(create_testfolder) + "/test_filtered_contig.fasta"
    annot_file = (
        str(create_testfolder) + "/test_filtered_contig_annotations.csv"
    )
    annotation_10x.to_csv(annot_file, index=False)
    vdj = ddl.read_10x_vdj(str(create_testfolder))
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 18
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 29
    fh = open(fasta_file, "w")
    fh.close()
    out = ""
    for line in fasta_10x:
        out = ">" + line + "\n" + fasta_10x[line] + "\n"
        ddl.utl.Write_output(out, fasta_file)
    vdj = ddl.read_10x_vdj(str(create_testfolder))
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 19
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 29
    assert not vdj.data.sequence.empty
    os.remove(fasta_file)


@pytest.mark.usefixtures("create_testfolder", "annotation_10x", "fasta_10x")
def test_to_scirpy(create_testfolder, annotation_10x, fasta_10x):
    """test_to_scirpy"""
    fasta_file = str(create_testfolder) + "/test_filtered_contig.fasta"
    annot_file = (
        str(create_testfolder) + "/test_filtered_contig_annotations.csv"
    )
    annotation_10x.to_csv(annot_file, index=False)
    vdj = ddl.read_10x_vdj(str(create_testfolder))
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 18
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 29
    adata = ddl.to_scirpy(vdj)
    assert adata.obs.shape[0] == 5
    assert adata.obs.shape[1] == 43
    fh = open(fasta_file, "w")
    fh.close()
    out = ""
    for line in fasta_10x:
        out = ">" + line + "\n" + fasta_10x[line] + "\n"
        ddl.utl.Write_output(out, fasta_file)
    vdj = ddl.read_10x_vdj(str(create_testfolder))
    assert vdj.data.shape[0] == 9
    assert vdj.data.shape[1] == 19
    assert vdj.metadata.shape[0] == 5
    assert vdj.metadata.shape[1] == 29
    assert not vdj.data.sequence.empty
    adata = ddl.to_scirpy(vdj)
    assert adata.obs.shape[0] == 5
    assert adata.obs.shape[1] == 43
    os.remove(fasta_file)
    vdjx = ddl.from_scirpy(adata)
    assert vdjx.data.shape[0] == 9


@pytest.mark.usefixtures(
    "create_testfolder", "annotation_10x_cr6", "json_10x_cr6"
)
def test_tofro_scirpy_cr6(create_testfolder, annotation_10x_cr6, json_10x_cr6):
    """test_tofro_scirpy_cr6"""
    json_file = str(create_testfolder) + "/test_all_contig_annotations.json"
    annot_file = str(create_testfolder) + "/test_all_contig_annotations.csv"
    annotation_10x_cr6.to_csv(annot_file, index=False)
    with open(json_file, "w") as outfile:
        json.dump(json_10x_cr6, outfile)
    vdj = ddl.read_10x_vdj(str(create_testfolder), filename_prefix="test_all")
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 31
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    adata = ddl.to_scirpy(vdj)
    assert adata.obs.shape[0] == 10
    assert adata.obs.shape[1] == 43
    vdjx = ddl.from_scirpy(adata)
    assert vdjx.data.shape[0] == 26


@pytest.mark.usefixtures("create_testfolder", "annotation_10x", "json_10x_cr6")
def test_tofro_scirpy_cr6_transfer(
    create_testfolder, annotation_10x_cr6, json_10x_cr6
):
    """test_tofro_scirpy_cr6_transfer"""
    json_file = str(create_testfolder) + "/test_all_contig_annotations.json"
    annot_file = str(create_testfolder) + "/test_all_contig_annotations.csv"
    annotation_10x_cr6.to_csv(annot_file, index=False)
    with open(json_file, "w") as outfile:
        json.dump(json_10x_cr6, outfile)
    vdj = ddl.read_10x_vdj(str(create_testfolder), filename_prefix="test_all")
    assert vdj.data.shape[0] == 26
    assert vdj.data.shape[1] == 31
    assert vdj.metadata.shape[0] == 10
    assert vdj.metadata.shape[1] == 31
    adata = ddl.to_scirpy(vdj, transfer=True)
    assert adata.obs.shape[0] == 10
    assert adata.obs.shape[1] == 74
    vdjx = ddl.from_scirpy(adata)
    assert vdjx.data.shape[0] == 26
