"""fixtures"""
import pandas as pd
import pytest
import scanpy as sc
import scipy.sparse


@pytest.fixture(scope="module")
def create_testfolder(tmp_path_factory):
    """Create test folder."""
    fn = tmp_path_factory.mktemp("testfolder")
    return fn


@pytest.fixture
def database_paths():
    """Database paths for tests."""
    db = {
        "igblast_db": "container/database/igblast/",
        "germline": "container/database/germlines/imgt/human/vdj/",
        "blastdb": "container/database/blast/human/",
        "blastdb_fasta": "container/database/blast/human/human_BCR_C.fasta",
    }
    return db


@pytest.fixture
def processed_files():
    """Database paths for tests."""
    fl = {
        "filtered": "filtered_contig_dandelion.tsv",
        "all": "all_contig_dandelion.tsv",
        "germline-dmask": "filtered_contig_dandelion_germline_dmask.tsv",
    }
    return fl


@pytest.fixture
def processed_files_tr():
    """Database paths for tests."""
    fl = {
        "filtered": "filtered_contig_dandelion.tsv",
        "all": "all_contig_dandelion.tsv",
    }
    return fl


@pytest.fixture
def dummy_adata():
    """Dummy anndata for tests."""
    barcodes = [
        "AAACCTGTCATATCGG-1",
        "AAACCTGTCCGTTGTC-1",
        "AAACCTGTCGAGAACG-1",
        "AAACCTGTCTTGAGAC-1",
        "AAACGGGAGCGACGTA-1",
    ]
    obs = pd.DataFrame(index=barcodes)
    obs["sample_id"] = "sample_test"
    obs["group1"] = "test"
    obs["group1"] = obs["group1"].astype("category")
    obs["group2"] = ["a", "a", "a", "b", "b"]
    obs["group2"] = obs["group2"].astype("category")
    obs["group3"] = ["a", "a", "b", "b", "c"]
    obs["group3"] = obs["group3"].astype("category")

    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format="csr"), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep="X", n_neighbors=3)

    return adata


@pytest.fixture
def dummy_adata2():
    """Dummy anndata for tests."""
    barcodes = [
        "AAACCTGTCATATCGG-1",
        "AAACCTGTCCGTTGTC-1",
        "AAACCTGTCCGTTGTC-2",
        "AAACCTGTCCGTTGTC-3",
        "AAACCTGTCCGTTGTC-4",
        "AAACCTGTCGAGAACG-1",
        "AAACCTGTCTTGAGAC-1",
        "AAACGGGAGCGACGTA-1",
    ]
    obs = pd.DataFrame(index=barcodes)
    obs["sample_id"] = "sample_test"
    obs["group1"] = "test"
    obs["group1"] = obs["group1"].astype("category")
    obs["group2"] = ["a", "a", "a", "a", "a", "a", "b", "b"]
    obs["group2"] = obs["group2"].astype("category")
    obs["group3"] = ["a", "a", "b", "b", "b", "b", "b", "c"]
    obs["group3"] = obs["group3"].astype("category")

    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format="csr"), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep="X", n_neighbors=3)

    return adata


@pytest.fixture
def dummy_adata_cr6():
    """Dummy anndata for tests."""
    barcodes = [
        "AAACCTGAGACTGTAA-1",
        "AAACCTGAGGAATTAC-1",
        "AAACCTGAGTCAATAG-1",
        "AAACCTGCATTTCACT-1",
        "AAACCTGGTAACGTTC-1",
        "AAACCTGTCCAAGTAC-1",
        "AAACGGGCACAACGCC-1",
        "AAAGATGAGAGATGAG-1",
        "AAAGATGAGGGATGGG-1",
        "AAAGATGCAACACCTA-1",
    ]
    obs = pd.DataFrame(index=barcodes)
    obs["sample_id"] = "sample_test"
    obs["group1"] = "test"
    obs["group1"] = obs["group1"].astype("category")
    obs["group2"] = ["a", "a", "a", "b", "b", "c", "c", "d", "b", "a"]
    obs["group2"] = obs["group2"].astype("category")
    obs["group3"] = ["a", "a", "b", "b", "c", "c", "a", "b", "c", "a"]
    obs["group3"] = obs["group3"].astype("category")

    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format="csr"), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep="X", n_neighbors=3)

    return adata


@pytest.fixture
def dummy_adata_tr():
    """Dummy anndata for tests."""
    barcodes = [
        "AAGACCTCACTACAGT-1",
        "AACTGGTTCTTTAGTC-1",
        "AAAGATGCACCCTATC-1",
    ]
    obs = pd.DataFrame(index=barcodes)
    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format="csr"), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep="X", n_neighbors=3)

    return adata


@pytest.fixture
def fasta_10x():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        "AAACCTGTCATATCGG-1_contig_1": "TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGTCCGTTGTC-1_contig_1": "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGTCCGTTGTC-1_contig_2": "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGTCGAGAACG-1_contig_1": "ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
        "AAACCTGTCGAGAACG-1_contig_2": "GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGTCTTGAGAC-1_contig_1": "AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGTCTTGAGAC-1_contig_2": "GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACGGGAGCGACGTA-1_contig_1": "GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACGGGAGCGACGTA-1_contig_2": "TCTGGCACCAGGGGTCCCTTCCAATATCAGCACCATGGCCTGGACTCCTCTCTTTCTGTTCCTCCTCACTTGCTGCCCAGGGTCCAATTCCCAGGCTGTGGTGACTCAGGAGCCCTCACTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACCAGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACAAGCAACAAACACTCCTGGACACCTGCCCGGTTCTCAGGCTCCCTCCTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
    }
    return seq


@pytest.fixture
def annotation_10x():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame(
        [
            [
                "AAACCTGTCATATCGG-1",
                "True",
                "AAACCTGTCATATCGG-1_contig_1",
                "True",
                556,
                "IGK",
                "IGKV1-8",
                "None",
                "IGKJ4",
                "IGKC",
                "True",
                "True",
                "CQQYDELPVTF",
                "TGTCAACAATATGACGAACTTCCCGTCACTTTC",
                9139,
                68,
                "clonotype9",
                "clonotype9_consensus_1",
            ],
            [
                "AAACCTGTCCGTTGTC-1",
                "True",
                "AAACCTGTCCGTTGTC-1_contig_1",
                "True",
                551,
                "IGK",
                "IGKV1-8",
                "None",
                "IGKJ1",
                "IGKC",
                "True",
                "True",
                "CQQYYSYPRTF",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                5679,
                43,
                "clonotype10",
                "clonotype10_consensus_1",
            ],
            [
                "AAACCTGTCCGTTGTC-1",
                "True",
                "AAACCTGTCCGTTGTC-1_contig_2",
                "True",
                565,
                "IGH",
                "IGHV1-69D",
                "IGHD3-22",
                "IGHJ3",
                "IGHM",
                "True",
                "True",
                "CATTYYYDSSGYYQNDAFDIW",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                4161,
                51,
                "clonotype10",
                "clonotype10_consensus_2",
            ],
            [
                "AAACCTGTCGAGAACG-1",
                "True",
                "AAACCTGTCGAGAACG-1_contig_1",
                "True",
                642,
                "IGL",
                "IGLV5-45",
                "None",
                "IGLJ3",
                "IGLC3",
                "True",
                "True",
                "CMIWHSSAWVV",
                "TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC",
                13160,
                90,
                "clonotype11",
                "clonotype11_consensus_1",
            ],
            [
                "AAACCTGTCGAGAACG-1",
                "True",
                "AAACCTGTCGAGAACG-1_contig_2",
                "True",
                550,
                "IGH",
                "IGHV1-2",
                "None",
                "IGHJ3",
                "IGHM",
                "True",
                "True",
                "CAREIEGDGVFEIW",
                "TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG",
                5080,
                47,
                "clonotype11",
                "clonotype11_consensus_2",
            ],
            [
                "AAACCTGTCTTGAGAC-1",
                "True",
                "AAACCTGTCTTGAGAC-1_contig_1",
                "True",
                551,
                "IGK",
                "IGKV1D-8",
                "None",
                "IGKJ2",
                "IGKC",
                "True",
                "True",
                "CQQYYSFPYTF",
                "TGTCAACAGTATTATAGTTTCCCGTACACTTTT",
                2813,
                22,
                "clonotype12",
                "clonotype12_consensus_1",
            ],
            [
                "AAACCTGTCTTGAGAC-1",
                "True",
                "AAACCTGTCTTGAGAC-1_contig_2",
                "True",
                557,
                "IGH",
                "IGHV5-51",
                "None",
                "IGHJ3",
                "IGHM",
                "True",
                "True",
                "CARHIRGNRFGNDAFDIW",
                "TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG",
                8292,
                80,
                "clonotype12",
                "clonotype12_consensus_2",
            ],
            [
                "AAACGGGAGCGACGTA-1",
                "True",
                "AAACGGGAGCGACGTA-1_contig_1",
                "True",
                534,
                "IGH",
                "IGHV4-59",
                "None",
                "IGHJ3",
                "IGHM",
                "True",
                "True",
                "CARVGYRAAAGTDAFDIW",
                "TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG",
                1235,
                18,
                "clonotype13",
                "clonotype13_consensus_1",
            ],
            [
                "AAACGGGAGCGACGTA-1",
                "True",
                "AAACGGGAGCGACGTA-1_contig_2",
                "True",
                631,
                "IGL",
                "IGLV7-46",
                "None",
                "IGLJ3",
                "IGLC2",
                "True",
                "False",
                "None",
                "None",
                739,
                8,
                "clonotype13",
                "None",
            ],
        ],
        columns=[
            "barcode",
            "is_cell",
            "contig_id",
            "high_confidence",
            "length",
            "chain",
            "v_gene",
            "d_gene",
            "j_gene",
            "c_gene",
            "full_length",
            "productive",
            "cdr3",
            "cdr3_nt",
            "reads",
            "umis",
            "raw_clonotype_id",
            "raw_consensus_id",
        ],
    )

    return annot


@pytest.fixture
def annotation_10x_cr6():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame(
        [
            [
                "AAACCTGAGACTGTAA-1",
                "TRUE",
                "AAACCTGAGACTGTAA-1_contig_1",
                "TRUE",
                639,
                "IGL",
                "IGLV3-1",
                "",
                "IGLJ2",
                "IGLC2",
                "TRUE",
                "TRUE",
                "SYELTQPPSVSVSPGQTASITC",
                "TCCTATGAGCTGACTCAGCCACCCTCAGTGTCCGTGTCCCCAGGACAGACAGCCAGCATCACCTGC",
                "SGDKLGDKYAS",
                "TCTGGAGATAAATTGGGGGATAAATATGCTTCG",
                "WYQQKPGQSPVLVIY",
                "TGGTATCAGCAGAAGCCAGGCCAGTCCCCTGTGCTGGTCATCTAT",
                "QDSKRPS",
                "CAAGATAGCAAGCGGCCCTCA",
                "GIPERFSGSNSGNTATLTISGTQAMDEGDYY",
                "GGGATCCCTGAGCGATTCTCTGGCTCCAACTCTGGGAACACAGCCACTCTGACCATCAGCGGGACCCAGGCTATGGATGAGGGTGACTATTAC",
                "CQAWDSSTVVF",
                "TGTCAGGCGTGGGACAGCAGCACTGTGGTATTC",
                "GGGTKLTVL",
                "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                464,
                16,
                "clonotype1691",
                "clonotype1691_consensus_2",
                1,
            ],
            [
                "AAACCTGAGACTGTAA-1",
                "TRUE",
                "AAACCTGAGACTGTAA-1_contig_2",
                "TRUE",
                550,
                "IGH",
                "IGHV4-39",
                "IGHD3-22",
                "IGHJ4",
                "IGHM",
                "TRUE",
                "TRUE",
                "QLQLQESGPGLVKSSETLSLTCTVS",
                "CAGCTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGTCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCT",
                "GGSISSSSY",
                "GGTGGCTCCATCAGCAGTAGTAGTTAC",
                "YWGWIRQPPGKGLEWIGNI",
                "TACTGGGGCTGGATCCGCCAGCCCCCAGGGAAGGGGCTGGAGTGGATTGGGAATATC",
                "YYSGS",
                "TATTATAGTGGGAGC",
                "TYYNPSLKSRVTISVDTSKNQFSLKVNSVTAADTAVYY",
                "ACCTACTACAACCCGTCCCTCAAGAGTCGAGTCACCATATCCGTAGACACGTCCAAGAACCAATTCTCCCTGAAGGTGAACTCTGTGACCGCCGCAGACACGGCTGTGTATTAC",
                "CAVQYYYDSSGYRRTEVPFDYW",
                "TGTGCGGTCCAGTATTACTATGATAGTAGTGGTTACCGGAGAACTGAGGTGCCCTTTGACTACTGG",
                "GQGTLVTVSS",
                "GGCCAGGGAACGCTGGTCACCGTCTCCTCAG",
                488,
                30,
                "clonotype1691",
                "clonotype1691_consensus_1",
                1,
            ],
            [
                "AAACCTGAGACTGTAA-1",
                "TRUE",
                "AAACCTGAGACTGTAA-1_contig_3",
                "TRUE",
                386,
                "None",
                "",
                "",
                "IGKJ1",
                "IGKC",
                "FALSE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                120,
                3,
                "",
                "",
                "",
            ],
            [
                "AAACCTGAGGAATTAC-1",
                "TRUE",
                "AAACCTGAGGAATTAC-1_contig_1",
                "TRUE",
                566,
                "IGH",
                "IGHV3-64D",
                "",
                "IGHJ4",
                "IGHM",
                "TRUE",
                "TRUE",
                "EVQLVESGGGLVQPGGSLRLSCSAS",
                "GAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCT",
                "GFTFSSY",
                "GGATTCACCTTCAGTAGCTAT",
                "AMHWVRQAPGKGLEYVSTI",
                "GCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAACTATT",
                "SSNGGI",
                "AGTAGTAATGGGGGTATC",
                "TDYADSVKGRFTIFRDNSKNTLYLQMSSLRAEDTAVYY",
                "ACAGACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTTCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTATATTAC",
                "CVKDREYGEVDYW",
                "TGTGTGAAAGATCGAGAATACGGGGAGGTTGACTACTGG",
                "GQGTLVTVSS",
                "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                1101,
                63,
                "clonotype686",
                "clonotype686_consensus_1",
                1,
            ],
            [
                "AAACCTGAGGAATTAC-1",
                "TRUE",
                "AAACCTGAGGAATTAC-1_contig_2",
                "TRUE",
                550,
                "IGK",
                "IGKV1D-39",
                "",
                "IGKJ1",
                "IGKC",
                "TRUE",
                "TRUE",
                "DIQMTQSPSSLSASVGDRVTITC",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGC",
                "RASQSISNYLN",
                "CGGGCAAGTCAGAGCATTAGCAACTATTTAAAT",
                "WYQQKPGKAPKLLIY",
                "TGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "AASSLQS",
                "GCTGCGTCCAGTTTGCAAAGT",
                "GVPSRFSGSISGTDFTLTISSLQPEDFATYY",
                "GGGGTCCCATCGAGGTTCAGTGGCAGTATATCTGGGACAGATTTCACTCTCACCATCAGCAGTCTGCAACCTGAAGATTTTGCAACTTACTAC",
                "CQQSYSTPRTF",
                "TGTCAACAGAGTTACAGTACCCCTCGAACGTTC",
                "GQGTKVEIK",
                "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                2418,
                105,
                "clonotype686",
                "clonotype686_consensus_2",
                1,
            ],
            [
                "AAACCTGAGGAATTAC-1",
                "TRUE",
                "AAACCTGAGGAATTAC-1_contig_3",
                "TRUE",
                371,
                "None",
                "",
                "",
                "IGHJ6",
                "IGHM",
                "FALSE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                463,
                11,
                "",
                "",
                "",
            ],
            [
                "AAACCTGAGTCAATAG-1",
                "TRUE",
                "AAACCTGAGTCAATAG-1_contig_1",
                "TRUE",
                647,
                "IGL",
                "IGLV1-47",
                "",
                "IGLJ2",
                "IGLC2",
                "TRUE",
                "TRUE",
                "QSVLTQPPSASGTPGQRVTISC",
                "CAGTCTGTGCTGACTCAGCCACCCTCAGCGTCTGGGACCCCCGGGCAGAGGGTCACCATCTCTTGT",
                "SGSSSNIGSNYVY",
                "TCTGGAAGCAGCTCCAACATCGGAAGTAATTATGTATAC",
                "WYQQLPGTAPKLLIY",
                "TGGTACCAGCAGCTCCCAGGAACGGCCCCCAAACTCCTCATCTAT",
                "RNNQRPS",
                "AGGAATAATCAGCGGCCCTCA",
                "GVPDRFSGSKSGTSASLAISGLRSEDEADYY",
                "GGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCAGTGGGCTCCGGTCCGAGGATGAGGCTGATTATTAC",
                "CAAWDDSLSGVLF",
                "TGTGCAGCATGGGATGACAGCCTGAGTGGTGTGCTATTC",
                "GGGTKLTVL",
                "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                1877,
                118,
                "clonotype1317",
                "clonotype1317_consensus_2",
                1,
            ],
            [
                "AAACCTGAGTCAATAG-1",
                "TRUE",
                "AAACCTGAGTCAATAG-1_contig_2",
                "TRUE",
                695,
                "IGH",
                "IGHV3-33",
                "IGHD3-22",
                "IGHJ4",
                "IGHG1",
                "TRUE",
                "TRUE",
                "QVQLVESGGGVVQPGRSLRLSCAAS",
                "CAGGTGCAGCTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCT",
                "GFTFSSY",
                "GGATTCACCTTCAGTAGCTAT",
                "AMHWVRQAPGKGLEWVAVI",
                "GCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATA",
                "SYDGSH",
                "TCGTATGATGGAAGTCAT",
                "KYYADSVKGRFTISRDNSKNTLYMQMNSLRAEDTAVYF",
                "AAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATATGCAGATGAACAGCCTGAGAGCTGAGGACACGGCTGTGTATTTC",
                "CATDRGTGYCDSSGYYPDYW",
                "TGTGCGACAGATCGAGGGACTGGTTACTGTGATAGTAGTGGTTATTACCCTGACTACTGG",
                "GQGTLVTVSS",
                "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                375,
                23,
                "clonotype1317",
                "clonotype1317_consensus_1",
                1,
            ],
            [
                "AAACCTGAGTCAATAG-1",
                "TRUE",
                "AAACCTGAGTCAATAG-1_contig_3",
                "TRUE",
                618,
                "IGH",
                "IGHV1-2",
                "",
                "IGHJ6",
                "IGHG1",
                "TRUE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "CLVKDYFPEPVTVSWNSGALTS",
                "TGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGC",
                "",
                "",
                211,
                20,
                "",
                "",
                "",
            ],
            [
                "AAACCTGCATTTCACT-1",
                "TRUE",
                "AAACCTGCATTTCACT-1_contig_1",
                "TRUE",
                565,
                "IGK",
                "IGKV3-20",
                "",
                "IGKJ1",
                "IGKC",
                "TRUE",
                "TRUE",
                "EIVLTQSPGTVSLSPGDRASLYC",
                "GAAATTGTGTTGACGCAGTCTCCAGGCACTGTGTCTTTGTCTCCAGGGGATAGAGCCTCCCTCTACTGC",
                "RATESVDVSNLA",
                "AGGGCCACTGAGAGTGTTGACGTCAGCAACTTAGCC",
                "WYQLKVGQAPRLLIY",
                "TGGTACCAGCTGAAAGTTGGCCAGGCTCCCAGGCTCCTCATCTAT",
                "GASKRAT",
                "GGTGCATCCAAGAGGGCCACT",
                "GVPDRFAGSGSGTEFTLTISRLEPEDFAVYY",
                "GGGGTCCCAGACAGGTTCGCTGGCAGTGGGTCTGGGACAGAATTCACTCTCACCATCAGCAGACTGGAGCCTGAAGACTTTGCAGTTTATTAC",
                "CQHYGRSPWTF",
                "TGTCAACACTATGGGCGCTCACCCTGGACGTTC",
                "GQGTRVEIK",
                "GGCCAAGGGACCAGGGTGGAAATCAAAC",
                1306,
                41,
                "clonotype1692",
                "clonotype1692_consensus_2",
                1,
            ],
            [
                "AAACCTGCATTTCACT-1",
                "TRUE",
                "AAACCTGCATTTCACT-1_contig_2",
                "TRUE",
                641,
                "IGL",
                "IGLV1-40",
                "",
                "IGLJ3",
                "IGLC2",
                "TRUE",
                "TRUE",
                "QSVLTQPPSVSGAPGQRVTISC",
                "CAGTCTGTGCTGACGCAGCCGCCCTCAGTGTCTGGGGCCCCAGGGCAGAGGGTCACCATCTCCTGC",
                "TGSSSNIGAGYDVH",
                "ACTGGGAGCAGCTCCAACATCGGGGCAGGTTATGATGTACAC",
                "WYQQLPGTAPKLLIY",
                "TGGTACCAGCAGCTTCCAGGAACAGCCCCCAAACTCCTCATCTAT",
                "GNSNRPS",
                "GGTAACAGCAATCGGCCCTCA",
                "GVPDRFSGSKSGTSASLAITGLQAEDEADYY",
                "GGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCACTGGGCTCCAGGCTGAGGATGAGGCTGATTATTAC",
                "CQSYDSTVF",
                "TGCCAGTCCTATGACAGCACCGTGTTC",
                "GGGTKLTVL",
                "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                190,
                11,
                "clonotype1692",
                "clonotype1692_consensus_3",
                1,
            ],
            [
                "AAACCTGCATTTCACT-1",
                "TRUE",
                "AAACCTGCATTTCACT-1_contig_3",
                "TRUE",
                576,
                "IGH",
                "IGHV1-3",
                "IGHD2-2",
                "IGHJ4",
                "IGHM",
                "TRUE",
                "TRUE",
                "QVQLVQSGAEVKKPGASVKVSCKAS",
                "CAGGTCCAGCTTGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTTTCCTGCAAGGCTTCT",
                "GYTFTSY",
                "GGATACACCTTCACTAGCTAT",
                "AMHWVRQAPGQRLEWMGWI",
                "GCTATGCATTGGGTGCGCCAGGCCCCCGGACAAAGGCTTGAGTGGATGGGATGGATC",
                "NAGNGN",
                "AACGCTGGCAATGGTAAC",
                "TKYSQKFQGRVTITRDTSASTAYMELSSLRSEDTAVYY",
                "ACAAAATATTCACAGAAGTTCCAGGGCAGAGTCACCATTACCAGGGACACATCCGCGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAAGACACGGCTGTGTATTAC",
                "CARDYRSSKRYQPHIVVVPAAMGSGALGYW",
                "TGTGCGAGAGATTATCGCTCTTCCAAGCGTTATCAGCCGCATATTGTAGTAGTACCAGCTGCTATGGGTAGCGGGGCTCTTGGGTACTGG",
                "GQGTLVTVSS",
                "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                363,
                17,
                "clonotype1692",
                "clonotype1692_consensus_1",
                1,
            ],
            [
                "AAACCTGGTAACGTTC-1",
                "TRUE",
                "AAACCTGGTAACGTTC-1_contig_1",
                "TRUE",
                574,
                "IGK",
                "IGKV3-15",
                "",
                "IGKJ3",
                "IGKC",
                "TRUE",
                "TRUE",
                "EIVMTQSPATLSVSPGERATLSC",
                "GAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGC",
                "RASQSVSSNLA",
                "AGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCC",
                "WYQQKPGQAPRLLIY",
                "TGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTAT",
                "GASTRAT",
                "GGTGCATCCACCAGGGCCACT",
                "GIPARFSGSGSGTEFTLTISSLQSEDFAVYY",
                "GGCATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTAC",
                "CQQYNNWPQFTF",
                "TGTCAGCAGTATAATAACTGGCCTCAATTCACTTTC",
                "GPGTKVDIK",
                "GGCCCTGGGACCAAAGTGGATATCAAAC",
                640,
                27,
                "clonotype1016",
                "clonotype1016_consensus_2",
                1,
            ],
            [
                "AAACCTGGTAACGTTC-1",
                "TRUE",
                "AAACCTGGTAACGTTC-1_contig_2",
                "TRUE",
                554,
                "IGH",
                "IGHV5-51",
                "",
                "IGHJ5",
                "IGHM",
                "TRUE",
                "TRUE",
                "EVQLVQSGAEVKKPGESLKISCKGS",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCT",
                "GYSFTSY",
                "GGATACAGCTTTACCAGCTAC",
                "WIGWVRQMPGKGLEWMGII",
                "TGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATC",
                "YPGDSD",
                "TATCCTGGTGACTCTGAT",
                "TRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYY",
                "ACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTAC",
                "CARRQWGSSWVGWFDPW",
                "TGTGCGAGACGCCAGTGGGGCAGCAGCTGGGTCGGGTGGTTCGACCCCTGG",
                "GQGTLVTVSS",
                "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                536,
                30,
                "clonotype1016",
                "clonotype1016_consensus_1",
                1,
            ],
            [
                "AAACCTGTCCAAGTAC-1",
                "TRUE",
                "AAACCTGTCCAAGTAC-1_contig_1",
                "TRUE",
                646,
                "IGL",
                "IGLV2-14",
                "",
                "IGLJ2",
                "IGLC2",
                "TRUE",
                "TRUE",
                "QSALTQPASVSGSPGQSITIPC",
                "CAGTCTGCCCTGACTCAGCCTGCCTCCGTGTCGGGGTCTCCGGGACAGTCGATCACCATTCCCTGC",
                "TGTSRDIGYYNYVS",
                "ACTGGAACCAGCAGGGACATTGGTTATTACAATTATGTCTCC",
                "WYQHHPGKAPQLIIY",
                "TGGTACCAACACCACCCCGGCAAAGCCCCCCAACTCATTATTTAT",
                "EVSNRPS",
                "GAGGTCAGTAATCGGCCCTCA",
                "GVSNRFSGSKSGNTASLTISGLQAEDEAHYY",
                "GGGGTTTCTAATCGCTTCTCTGGCTCCAAGTCTGGCAACACGGCCTCCCTGACCATCTCTGGGCTCCAGGCTGAGGACGAGGCTCATTATTAC",
                "CNSYTGSRTRYVLF",
                "TGCAACTCATATACAGGCAGCAGAACCCGATATGTGTTGTTC",
                "GGGTKLTVL",
                "GGCGGAGGGACCAAACTGACCGTCCTAG",
                4247,
                293,
                "clonotype1",
                "clonotype1_consensus_2",
                1,
            ],
            [
                "AAACCTGTCCAAGTAC-1",
                "TRUE",
                "AAACCTGTCCAAGTAC-1_contig_2",
                "TRUE",
                682,
                "IGH",
                "IGHV3-33",
                "",
                "IGHJ6",
                "IGHA1",
                "TRUE",
                "TRUE",
                "QVQLVESGGGVVQPGRSLRLSCAAS",
                "CAGGTGCAACTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCT",
                "GFTFSSH",
                "GGATTCACCTTCAGTAGCCAT",
                "AMHWVRQAPGKGLEWVAVI",
                "GCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATA",
                "SYDGSN",
                "TCATATGATGGAAGTAAT",
                "EYYADSVKGRFTISRDNSKDTLYLQVDSLRPEDTAVYY",
                "GAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGGACACCCTGTATCTGCAAGTGGACAGCCTGAGACCTGAGGACACGGCTGTGTATTAT",
                "CARDRIDSSSWPYYYFYYGMDVW",
                "TGTGCGAGAGATCGAATTGATAGCAGCAGCTGGCCATATTACTACTTCTACTACGGAATGGACGTCTGG",
                "GQGTTLTVSS",
                "GGCCAAGGGACCACGCTCACCGTCTCCTCAG",
                282,
                57,
                "clonotype1",
                "clonotype1_consensus_1",
                1,
            ],
            [
                "AAACCTGTCCAAGTAC-1",
                "TRUE",
                "AAACCTGTCCAAGTAC-1_contig_3",
                "TRUE",
                403,
                "IGH",
                "IGHV1-3",
                "",
                "IGHJ5",
                "IGHG1",
                "FALSE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "CARDVGIPGDTVWFGESIPNRRNWFDPW",
                "TGTGCGAGAGATGTGGGGATACCCGGGGACACCGTATGGTTCGGGGAGTCCATACCTAATCGCCGTAACTGGTTCGACCCCTGG",
                "",
                "",
                92,
                3,
                "",
                "",
                "",
            ],
            [
                "AAACGGGCACAACGCC-1",
                "TRUE",
                "AAACGGGCACAACGCC-1_contig_1",
                "TRUE",
                565,
                "IGK",
                "IGKV3-15",
                "",
                "IGKJ1",
                "IGKC",
                "TRUE",
                "TRUE",
                "EIVMTQSPATLSVSPGERATLSC",
                "GAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGC",
                "RASQSVSSNLA",
                "AGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCC",
                "WYQQKPGQAPRLLIY",
                "TGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTAT",
                "GASTRAT",
                "GGTGCATCCACCAGGGCCACT",
                "GIPARFSGSGSGTEFTLTISSLQSEDFAVYY",
                "GGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTAC",
                "CQQYNNWPPWTF",
                "TGTCAGCAGTATAATAACTGGCCTCCGTGGACGTTC",
                "GQGTKVEIK",
                "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                1619,
                66,
                "clonotype610",
                "clonotype610_consensus_2",
                1,
            ],
            [
                "AAACGGGCACAACGCC-1",
                "TRUE",
                "AAACGGGCACAACGCC-1_contig_2",
                "TRUE",
                557,
                "IGH",
                "IGHV3-64D",
                "",
                "IGHJ4",
                "IGHM",
                "TRUE",
                "TRUE",
                "EVQLVESGGGLVQPGGSLRLSCSAS",
                "GAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCT",
                "GFTFSSY",
                "GGATTCACCTTCAGTAGCTAT",
                "AMHWVRQAPGKGLEYVSAI",
                "GCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAGCTATT",
                "SSNGGS",
                "AGTAGTAATGGGGGTAGC",
                "TYYADSVKGRFTISRDNSKNTLYLQMSSLRAEDTAVYY",
                "ACATACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTGTATTAC",
                "CVKDIVATPNW",
                "TGTGTGAAAGATATAGTGGCTACGCCCAACTGG",
                "GQGTLVTVSS",
                "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                500,
                34,
                "clonotype610",
                "clonotype610_consensus_1",
                1,
            ],
            [
                "AAAGATGAGAGATGAG-1",
                "TRUE",
                "AAAGATGAGAGATGAG-1_contig_1",
                "TRUE",
                555,
                "IGK",
                "IGKV1-27",
                "",
                "IGKJ1",
                "IGKC",
                "TRUE",
                "TRUE",
                "DIQMTQSPSSLSASVGDRVTITC",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGC",
                "RASQGISNYLA",
                "CGGGCGAGTCAGGGCATTAGCAATTATTTAGCC",
                "WYQQKPGKVPKLLIY",
                "TGGTATCAGCAGAAACCAGGGAAAGTTCCTAAGCTCCTGATCTAT",
                "AASTLQS",
                "GCTGCATCCACTTTGCAATCA",
                "GVPSRFSGSGSGTDFTLTISSLQPEDVATYY",
                "GGGGTCCCATCTCGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACTTATTAC",
                "CQKYNSAPWTF",
                "TGTCAAAAGTATAACAGTGCGCCGTGGACGTTC",
                "GQGTKVEIK",
                "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                1913,
                81,
                "clonotype372",
                "clonotype372_consensus_1",
                1,
            ],
            [
                "AAAGATGAGAGATGAG-1",
                "TRUE",
                "AAAGATGAGAGATGAG-1_contig_2",
                "TRUE",
                504,
                "None",
                "",
                "",
                "IGHJ4",
                "IGHG3",
                "FALSE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                293,
                13,
                "",
                "",
                "",
            ],
            [
                "AAAGATGAGGGATGGG-1",
                "TRUE",
                "AAAGATGAGGGATGGG-1_contig_1",
                "TRUE",
                562,
                "IGK",
                "IGKV3-15",
                "",
                "IGKJ1",
                "IGKC",
                "TRUE",
                "TRUE",
                "EIVMTQSPATLSVSPGERATLSC",
                "GAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGC",
                "RASQSVSSNLA",
                "AGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCC",
                "WYQQKPRQAPRLLIY",
                "TGGTACCAGCAGAAACCTCGACAGGCTCCCAGGCTCCTCATCTAT",
                "GASTRAT",
                "GGTGCATCCACCAGGGCCACT",
                "GIPARFSGSGSGTEFTLTISSLQSEDFAVYY",
                "GGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTAC",
                "CQQYNNWPRTF",
                "TGTCAGCAGTATAATAACTGGCCTAGGACGTTC",
                "GQGTKVEIK",
                "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                2955,
                124,
                "clonotype127",
                "clonotype127_consensus_1",
                1,
            ],
            [
                "AAAGATGAGGGATGGG-1",
                "TRUE",
                "AAAGATGAGGGATGGG-1_contig_2",
                "TRUE",
                300,
                "IGH",
                "IGHV3-23",
                "",
                "IGHJ4",
                "IGHG1",
                "FALSE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "CAKDFANTYHYDSSGLGGPFDYW",
                "TGTGCGAAAGATTTTGCGAATACCTACCACTATGATAGTAGTGGCCTAGGTGGTCCTTTTGACTACTGG",
                "",
                "",
                18,
                8,
                "",
                "",
                "",
            ],
            [
                "AAAGATGCAACACCTA-1",
                "TRUE",
                "AAAGATGCAACACCTA-1_contig_1",
                "TRUE",
                553,
                "IGH",
                "IGHV3-48",
                "",
                "IGHJ2",
                "IGHM",
                "TRUE",
                "TRUE",
                "EVQLVESGGGLVQPGGSLRLSCAAS",
                "GAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTACAGCCTGGAGGGTCCCTGAGACTCTCCTGTGCAGCCTCT",
                "GFTFSSY",
                "GGATTCACCTTCAGTAGTTAT",
                "EMNWVRQAPGKGLEWVSYI",
                "GAAATGAACTGGGTCCGCCAGGCTCCAGGGAAGGGGCTGGAGTGGGTTTCATACATT",
                "SSSGST",
                "AGTAGTAGTGGTAGTACC",
                "IYYADSVKGRFTISRDNAKNSLYLQMNSLRAEDTAVYY",
                "ATATACTACGCAGACTCTGTGAAGGGCCGATTCACCATCTCCAGAGACAACGCCAAGAACTCACTGTATCTGCAAATGAACAGCCTGAGAGCCGAGGACACGGCTGTTTATTAC",
                "CARPMKHPLW",
                "TGTGCGAGACCAATGAAACATCCACTCTGG",
                "GRGTLVTVSS",
                "GGCCGTGGCACCCTGGTCACTGTCTCCTCAG",
                659,
                34,
                "clonotype599",
                "clonotype599_consensus_1",
                1,
            ],
            [
                "AAAGATGCAACACCTA-1",
                "TRUE",
                "AAAGATGCAACACCTA-1_contig_2",
                "TRUE",
                662,
                "IGL",
                "IGLV6-57",
                "",
                "IGLJ3",
                "IGLC3",
                "TRUE",
                "TRUE",
                "NFMLTQPHSVSESPGKTVTISC",
                "AATTTTATGCTGACTCAGCCCCACTCTGTGTCGGAGTCTCCGGGGAAGACGGTAACCATCTCCTGC",
                "TGSSGSIASNYVQ",
                "ACCGGCAGCAGTGGCAGCATTGCCAGCAACTATGTGCAG",
                "WYQQRPGSAPTTVIY",
                "TGGTACCAGCAGCGCCCGGGCAGTGCCCCCACCACTGTGATCTAT",
                "EDNQRPS",
                "GAGGATAACCAAAGACCCTCT",
                "GVPDRFSGSIDSSSNSASLTISGLKTEDEADYY",
                "GGGGTCCCTGATCGGTTCTCTGGCTCCATCGACAGCTCCTCCAACTCTGCCTCCCTCACCATCTCTGGACTGAAGACTGAGGACGAGGCTGACTACTAC",
                "CQSYDSSNHWVF",
                "TGTCAGTCTTATGATAGCAGCAATCATTGGGTGTTC",
                "GGGTKLTVL",
                "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                657,
                33,
                "clonotype599",
                "clonotype599_consensus_2",
                1,
            ],
            [
                "AAAGATGCAACACCTA-1",
                "TRUE",
                "AAAGATGCAACACCTA-1_contig_3",
                "TRUE",
                498,
                "IGH",
                "IGHV1-69D",
                "",
                "IGHJ3",
                "IGHM",
                "TRUE",
                "FALSE",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "CASRLFIR",
                "TGTGCGAGTAGACTGTTTATAAGG",
                "",
                "",
                100,
                4,
                "",
                "",
                "",
            ],
        ],
        columns=[
            "barcode",
            "is_cell",
            "contig_id",
            "high_confidence",
            "length",
            "chain",
            "v_gene",
            "d_gene",
            "j_gene",
            "c_gene",
            "full_length",
            "productive",
            "fwr1",
            "fwr1_nt",
            "cdr1",
            "cdr1_nt",
            "fwr2",
            "fwr2_nt",
            "cdr2",
            "cdr2_nt",
            "fwr3",
            "fwr3_nt",
            "cdr3",
            "cdr3_nt",
            "fwr4",
            "fwr4_nt",
            "reads",
            "umis",
            "raw_clonotype_id",
            "raw_consensus_id",
            "exact_subclonotype_id",
        ],
    )

    return annot


@pytest.fixture
def fasta_10x_cr6():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        "AAACCTGAGACTGTAA-1_contig_1": "AAGAACCTGCCCAGCCTGGGCCTCAGGAAGCAGCATCGGAGGTGCCTCAGCCATGGCATGGATCCCTCTCTTCCTCGGCGTCCTTGCTTACTGCACAGGATCCGTGGCCTCCTATGAGCTGACTCAGCCACCCTCAGTGTCCGTGTCCCCAGGACAGACAGCCAGCATCACCTGCTCTGGAGATAAATTGGGGGATAAATATGCTTCGTGGTATCAGCAGAAGCCAGGCCAGTCCCCTGTGCTGGTCATCTATCAAGATAGCAAGCGGCCCTCAGGGATCCCTGAGCGATTCTCTGGCTCCAACTCTGGGAACACAGCCACTCTGACCATCAGCGGGACCCAGGCTATGGATGAGGGTGACTATTACTGTCAGGCGTGGGACAGCAGCACTGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
        "AAACCTGAGACTGTAA-1_contig_2": "ATGCTTTCTGAGAGTCATGGATCTCATGTGCAAGAAAATGAAGCACCTGTGGTTCTTCCTCCTGCTGGTGGCGGCTCCCAGATGGGTCCTGTCCCAGCTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGTCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGCAGTAGTAGTTACTACTGGGGCTGGATCCGCCAGCCCCCAGGGAAGGGGCTGGAGTGGATTGGGAATATCTATTATAGTGGGAGCACCTACTACAACCCGTCCCTCAAGAGTCGAGTCACCATATCCGTAGACACGTCCAAGAACCAATTCTCCCTGAAGGTGAACTCTGTGACCGCCGCAGACACGGCTGTGTATTACTGTGCGGTCCAGTATTACTATGATAGTAGTGGTTACCGGAGAACTGAGGTGCCCTTTGACTACTGGGGCCAGGGAACGCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGAGACTGTAA-1_contig_3": "CTCAGTAAGAGCTGAGCTCTTCCTGTGCTGTGAAAACAGACAAACCAACCAAGTCTCACCTAAGAAAATAAACACATGCTATGTCAGAGAAGCCTCAGGGTTTCCACACCTGCTCGAAAAGGGAGTTGAGCTTCAGCAGCTGACCCAGGACTCTGTTCCCCTTTGGTGAGAAGGGTTTTTGTTCAGCAAGACAATGGAGAGCTCTCACTGTGGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGAGGAATTAC-1_contig_1": "GGGAGCTCTGGGAGAGGAGCCCCAGGCCCGGGATTCCCAGGTGTTTCCATTCAGTGATCAGCACTGAAGACAGAAGACTCATCATGGAGTTCTGGCTGAGCTGGGTTCTCCTTGTTGCCATTTTAAAAGATGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCTGGATTCACCTTCAGTAGCTATGCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAACTATTAGTAGTAATGGGGGTATCACAGACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTTCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTATATTACTGTGTGAAAGATCGAGAATACGGGGAGGTTGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGAGGAATTAC-1_contig_2": "GGAGTCAGTCTCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTACTCTGGCTCCGAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCGGGCAAGTCAGAGCATTAGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCGTCCAGTTTGCAAAGTGGGGTCCCATCGAGGTTCAGTGGCAGTATATCTGGGACAGATTTCACTCTCACCATCAGCAGTCTGCAACCTGAAGATTTTGCAACTTACTACTGTCAACAGAGTTACAGTACCCCTCGAACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGAGGAATTAC-1_contig_3": "GGGACTCCTGGGGGTGACTTAGCAGCCGTCTGCTTGCAGTTGGACTTCCCAGGCCGACAGTGGTCTGGCTTCTGAGGGGTCAGGCCAGAATGTGGGGTACGTGGGAGGCCAGCAGAGGGTTCCATGAGAAGGGCAGGACAGGGCCACGGACAGTCAGCTTCCATGTGACGCCCGGAGACAGAAGGTCTCTGGGTGGCTGGGTTTTTGTGGGGTGAGGATGGACATTCTGCCATTGTGATTACTACTACTACTACGGTATGGACGTCTGGGGCCAAGGGACCACGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGAGTCAATAG-1_contig_1": "TCAGCTTCAGCTGTGGTAGAGAAGACAGGATTCAGGACAATCTCCAGCATGGCCGGCTTCCCTCTCCTCCTCACCCTCCTCACTCACTGTGCAGGGTCCTGGGCCCAGTCTGTGCTGACTCAGCCACCCTCAGCGTCTGGGACCCCCGGGCAGAGGGTCACCATCTCTTGTTCTGGAAGCAGCTCCAACATCGGAAGTAATTATGTATACTGGTACCAGCAGCTCCCAGGAACGGCCCCCAAACTCCTCATCTATAGGAATAATCAGCGGCCCTCAGGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCAGTGGGCTCCGGTCCGAGGATGAGGCTGATTATTACTGTGCAGCATGGGATGACAGCCTGAGTGGTGTGCTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
        "AAACCTGAGTCAATAG-1_contig_2": "GAGCTCTGGGAGAGGAGCCCAGCACTGGAAGTCGCCGGTGTTTCCATTCGGTGATCAGCACTGAACACAGAGGACTCACCATGGAGTTTGGGCTGAGCTGGGTTTTCCTCGTCGCTCTTTTAAGAGGTGTCCAGTGTCAGGTGCAGCTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCTGGATTCACCTTCAGTAGCTATGCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATATCGTATGATGGAAGTCATAAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATATGCAGATGAACAGCCTGAGAGCTGAGGACACGGCTGTGTATTTCTGTGCGACAGATCGAGGGACTGGTTACTGTGATAGTAGTGGTTATTACCCTGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCCTCCAAGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
        "AAACCTGAGTCAATAG-1_contig_3": "GAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAACAGTGGTGGCACAAACTCTGCACAGAAGTTTCAGGGCTGGGTCACCATGACCAGGGACACGTCCATCAGCACATCGGGAGGGACTACGGTGACAATCTTACTACTACTACGGTATGGACGTCTGGGGCCAAGGGACCACGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCCTCCAAGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
        "AAACCTGCATTTCACT-1_contig_1": "AGAGCTCTGGAGAAGAGCTGCTCAGTTAGGACCCAGAGGGAACCATGGAAACCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACCGGAGAAATTGTGTTGACGCAGTCTCCAGGCACTGTGTCTTTGTCTCCAGGGGATAGAGCCTCCCTCTACTGCAGGGCCACTGAGAGTGTTGACGTCAGCAACTTAGCCTGGTACCAGCTGAAAGTTGGCCAGGCTCCCAGGCTCCTCATCTATGGTGCATCCAAGAGGGCCACTGGGGTCCCAGACAGGTTCGCTGGCAGTGGGTCTGGGACAGAATTCACTCTCACCATCAGCAGACTGGAGCCTGAAGACTTTGCAGTTTATTACTGTCAACACTATGGGCGCTCACCCTGGACGTTCGGCCAAGGGACCAGGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGCATTTCACT-1_contig_2": "GCTCTGCTTCAGCTGTGGGCACAAGAGGCAGCACTCAGGACAATCTCCAGCATGGCCTGGTCTCCTCTCCTCCTCACTCTCCTCGCTCACTGCACAGGGTCCTGGGCCCAGTCTGTGCTGACGCAGCCGCCCTCAGTGTCTGGGGCCCCAGGGCAGAGGGTCACCATCTCCTGCACTGGGAGCAGCTCCAACATCGGGGCAGGTTATGATGTACACTGGTACCAGCAGCTTCCAGGAACAGCCCCCAAACTCCTCATCTATGGTAACAGCAATCGGCCCTCAGGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCACTGGGCTCCAGGCTGAGGATGAGGCTGATTATTACTGCCAGTCCTATGACAGCACCGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
        "AAACCTGCATTTCACT-1_contig_3": "CCACATCCCTCCTCAGAAGCCCCCAGAGCACAACGCCTCACCATGGACTGGACCTGGAGGATCCTCTTTTTGGTGGCAGCAGCCACAGGTGCCCACTCCCAGGTCCAGCTTGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTTTCCTGCAAGGCTTCTGGATACACCTTCACTAGCTATGCTATGCATTGGGTGCGCCAGGCCCCCGGACAAAGGCTTGAGTGGATGGGATGGATCAACGCTGGCAATGGTAACACAAAATATTCACAGAAGTTCCAGGGCAGAGTCACCATTACCAGGGACACATCCGCGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAAGACACGGCTGTGTATTACTGTGCGAGAGATTATCGCTCTTCCAAGCGTTATCAGCCGCATATTGTAGTAGTACCAGCTGCTATGGGTAGCGGGGCTCTTGGGTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGGTAACGTTC-1_contig_1": "GCCTGGGTCAGAGCTCTGGGGAGGAACTGCTCAGTTAGGACCCAGACGGAACCATGGAAGCCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACTGGAGAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGCAGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCCTGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTATGGTGCATCCACCAGGGCCACTGGCATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTACTGTCAGCAGTATAATAACTGGCCTCAATTCACTTTCGGCCCTGGGACCAAAGTGGATATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACCTGGTAACGTTC-1_contig_2": "GGAGTCTCCCTCACTGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACGCCAGTGGGGCAGCAGCTGGGTCGGGTGGTTCGACCCCTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAACCTGTCCAAGTAC-1_contig_1": "GCTGGGGTCTCAGGAGGCAGCGCTCTCAGGACATCTCCACCATGGCCTGGGCTCTGCTGCTCCTCACCCTCCTCACTCAGGGCACAGGGTCCGGGGCCCAGTCTGCCCTGACTCAGCCTGCCTCCGTGTCGGGGTCTCCGGGACAGTCGATCACCATTCCCTGCACTGGAACCAGCAGGGACATTGGTTATTACAATTATGTCTCCTGGTACCAACACCACCCCGGCAAAGCCCCCCAACTCATTATTTATGAGGTCAGTAATCGGCCCTCAGGGGTTTCTAATCGCTTCTCTGGCTCCAAGTCTGGCAACACGGCCTCCCTGACCATCTCTGGGCTCCAGGCTGAGGACGAGGCTCATTATTACTGCAACTCATATACAGGCAGCAGAACCCGATATGTGTTGTTCGGCGGAGGGACCAAACTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
        "AAACCTGTCCAAGTAC-1_contig_2": "GAGCTCTGGGAGAGGAGCCCAGCACTGGAAGTCGCCGGTGTTTCCATTCGGTGATCAGCACTGAACACAGAGGACTCACCATGGAGTTTGGGCTGAGCTGGGTTTTCCTCGTTGCTCTTTTAAGAGGTGTCCAGTGTCAGGTGCAACTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCTGGATTCACCTTCAGTAGCCATGCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATATCATATGATGGAAGTAATGAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGGACACCCTGTATCTGCAAGTGGACAGCCTGAGACCTGAGGACACGGCTGTGTATTATTGTGCGAGAGATCGAATTGATAGCAGCAGCTGGCCATATTACTACTTCTACTACGGAATGGACGTCTGGGGCCAAGGGACCACGCTCACCGTCTCCTCAGCATCCCCGACCAGCCCCAAGGTCTTCCCGCTGAGCCTCTGCAGCACCCAGCCAGATGGGAACGTGGTCATCGCCTGCCTGGTCCAGGGCTTCTTCCCCCAGGAGCCACTCAGTGTGACCTGGAGCGAAAGCGGACAGGGCGTGACCGCCAGAAACTTCCC",
        "AAACCTGTCCAAGTAC-1_contig_3": "TTCACAGAAGTTCCAGGGCAGAGTCACCATTACCAGGGACACATCCGCGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAAGACACGGCTGTGTATTACTGTGCGAGAGATGTGGGGATACCCGGGGACACCGTATGGTTCGGGGAGTCCATACCTAATCGCCGTAACTGGTTCGACCCCTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCCTCCAAGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
        "AAACGGGCACAACGCC-1_contig_1": "AGAGCTCTGGGGAGGAACTGCTCAGTTAGGACCCAGACGGAACCATGGAAGCCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACTGGAGAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGCAGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCCTGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTATGGTGCATCCACCAGGGCCACTGGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTACTGTCAGCAGTATAATAACTGGCCTCCGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAACGGGCACAACGCC-1_contig_2": "AGCTCTGGGAGAGGAGCCCCAGGCCCGGGATTCCCAGGTGTTTCCATTCAGTGATCAGCACTGAAGACAGAAGACTCATCATGGAGTTCTGGCTGAGCTGGGTTCTCCTTGTTGCCATTTTAAAAGATGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCTGGATTCACCTTCAGTAGCTATGCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAGCTATTAGTAGTAATGGGGGTAGCACATACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTGTATTACTGTGTGAAAGATATAGTGGCTACGCCCAACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAAGATGAGAGATGAG-1_contig_1": "GGGGAGAAGTCTCTCTCAGTCAGGACACAGCATGGACATGAGGGTCCCTGCTCAGCTCCTGGGACTCCTGCTGCTCTGGCTCCCAGATACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCGGGCGAGTCAGGGCATTAGCAATTATTTAGCCTGGTATCAGCAGAAACCAGGGAAAGTTCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAATCAGGGGTCCCATCTCGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACTTATTACTGTCAAAAGTATAACAGTGCGCCGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAAGATGAGAGATGAG-1_contig_2": "CAGGCCCGTCTGGGCAGGTCAGCCTCAATGGGGACGGTGCTCGGGTCCTGCCTGGCTCCTCCTTGGGGTAGAGGGGACCTATCGCCTGGCCCTGGACCCCCTCAGGCCGCCCCTATTCTTGAAGAAACGCAGGACGGTGGAGGGTGCCCTGGAGGGGAGACGTCTGCTGCCCTGGGTGTCAGGTGCGGCCTCAGATCCTAAACGTGTCCTCCCTGCTCTCTCTGCTTCCGCTGACTTGGGACGGGGTTTCTGGCCTGGCGAGTGTCACTGTGTGCCCTGCCGGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCTTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCGCCCTGCTCCAGGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
        "AAAGATGAGGGATGGG-1_contig_1": "AGAGCTCTGGGGAGGAACTGCTCAGTTAGGACCCAGACGGAACCATGGAAGCCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACTGGAGAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGCAGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCCTGGTACCAGCAGAAACCTCGACAGGCTCCCAGGCTCCTCATCTATGGTGCATCCACCAGGGCCACTGGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTACTGTCAGCAGTATAATAACTGGCCTAGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
        "AAAGATGAGGGATGGG-1_contig_2": "AAGGGGCTGGAGTGGGTCTCAGGTATTAGTGGTAGTGGTGGTGGCACATACTACGCAGACTCCGTGAAGGGCCGGTTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATCTGCAAATGACCAGCCTGGGAGCCGACGACACGGCCGTATATTACTGTGCGAAAGATTTTGCGAATACCTACCACTATGATAGTAGTGGCCTAGGTGGTCCTTTTGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCC",
        "AAAGATGCAACACCTA-1_contig_1": "AGCTCTCAGAGAGGTGCCTTAGCCCTGGATTCCAAGGCATTTCCACTTGGTGATCAGCACTGAACACAGAGGACTCACCATGGAGTTGGGGCTGTGCTGGGTTTTCCTTGTTGCTATTTTAGAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTACAGCCTGGAGGGTCCCTGAGACTCTCCTGTGCAGCCTCTGGATTCACCTTCAGTAGTTATGAAATGAACTGGGTCCGCCAGGCTCCAGGGAAGGGGCTGGAGTGGGTTTCATACATTAGTAGTAGTGGTAGTACCATATACTACGCAGACTCTGTGAAGGGCCGATTCACCATCTCCAGAGACAACGCCAAGAACTCACTGTATCTGCAAATGAACAGCCTGAGAGCCGAGGACACGGCTGTTTATTACTGTGCGAGACCAATGAAACATCCACTCTGGGGCCGTGGCACCCTGGTCACTGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
        "AAAGATGCAACACCTA-1_contig_2": "AGGGCTGGTGGGATCAGTCCTGGTGGTAGCTCAGGAAGCAGAGCCTGGAGCATCTCCACTATGGCCTGGGCTCCACTACTTCTCACCCTCCTCGCTCACTGCACAGGTTCTTGGGCCAATTTTATGCTGACTCAGCCCCACTCTGTGTCGGAGTCTCCGGGGAAGACGGTAACCATCTCCTGCACCGGCAGCAGTGGCAGCATTGCCAGCAACTATGTGCAGTGGTACCAGCAGCGCCCGGGCAGTGCCCCCACCACTGTGATCTATGAGGATAACCAAAGACCCTCTGGGGTCCCTGATCGGTTCTCTGGCTCCATCGACAGCTCCTCCAACTCTGCCTCCCTCACCATCTCTGGACTGAAGACTGAGGACGAGGCTGACTACTACTGTCAGTCTTATGATAGCAGCAATCATTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
        "AAAGATGCAACACCTA-1_contig_3": "CCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGAGTAGACTGTTTATAAGGGGTCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
    }
    return seq


@pytest.fixture
def airr_10x():
    """Standard cellranger airr file to test the preprocessing."""
    airr = pd.DataFrame(
        [
            [
                "AAACCTGTCATATCGG-1",
                "",
                "AAACCTGTCATATCGG-1_contig_1",
                "TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "MRVPAQLLGLLLLWLSGARCDIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELPVTFGGGTNVEMRRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
                "T",
                "F",
                "IGKV1-8",
                "38S314M204S",
                "",
                "",
                "IGKJ4",
                "383S37M136S",
                "IGKC",
                "420S136M",
                "TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "ATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCTCACTTTCGGCGGAGGGACCAAGGTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "TGTCAACAATATGACGAACTTCCCGTCACTTTC",
                "CQQYDELPVTF",
                33,
                11,
                39,
                352,
                "",
                "",
                384,
                420,
                421,
                556,
                9139,
                68,
                "T",
            ],
            [
                "AAACCTGTCCGTTGTC-1",
                "",
                "AAACCTGTCCGTTGTC-1_contig_1",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "MRVPAQLLGLLLLWLPGARCAIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
                "T",
                "F",
                "IGKV1-8",
                "33S345M173S",
                "",
                "",
                "IGKJ1",
                "377S38M136S",
                "IGKC",
                "415S136M",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "AGGCTGGACACACTTCATGCAGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                33,
                11,
                34,
                378,
                "",
                "",
                378,
                415,
                416,
                551,
                5679,
                43,
                "T",
            ],
            [
                "AAACCTGTCCGTTGTC-1",
                "",
                "AAACCTGTCCGTTGTC-1_contig_2",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "MDWTWRFLFVVAAATGVQSQVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV",
                "T",
                "F",
                "IGHV1-69D",
                "58S353M154S",
                "IGHD3-22",
                "411S31M123S",
                "IGHJ3",
                "444S50M71S",
                "IGHM",
                "494S71M",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGAGAGAGTATTACTATGATAGTAGTGGTTATTACTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                63,
                21,
                59,
                411,
                412,
                442,
                445,
                494,
                495,
                565,
                4161,
                51,
                "T",
            ],
            [
                "AAACCTGTCGAGAACG-1",
                "",
                "AAACCTGTCGAGAACG-1_contig_1",
                "ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "MAWTPLLLLFLSHCTGSLSQAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSAWVVGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
                "T",
                "F",
                "IGLV5-45",
                "28S369M245S",
                "",
                "",
                "IGLJ3",
                "393S38M211S",
                "IGLC3",
                "431S211M",
                "ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "ACTGCGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTGCGCAGTGGCATCAATGTTGGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTCTTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC",
                "CMIWHSSAWVV",
                33,
                11,
                29,
                397,
                "",
                "",
                394,
                431,
                432,
                642,
                13160,
                90,
                "T",
            ],
            [
                "AAACCTGTCGAGAACG-1",
                "",
                "AAACCTGTCGAGAACG-1_contig_2",
                "GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "MDWTWRILFLVAAATGAHSQVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAREIEGDGVFEIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV",
                "T",
                "F",
                "IGHV1-2",
                "64S353M133S",
                "",
                "",
                "IGHJ3",
                "429S50M71S",
                "IGHM",
                "479S71M",
                "GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GAGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAACAGTGGTGGCACAAACTATGCACAGAAGTTTCAGGGCTGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG",
                "CAREIEGDGVFEIW",
                42,
                14,
                65,
                417,
                "",
                "",
                430,
                479,
                480,
                550,
                5080,
                47,
                "T",
            ],
            [
                "AAACCTGTCTTGAGAC-1",
                "",
                "AAACCTGTCTTGAGAC-1_contig_1",
                "AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "MDMRVPAQLLGLLLLWLPGARCVIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFPYTFGQGTKLEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
                "T",
                "F",
                "IGKV1D-8",
                "27S353M171S",
                "",
                "",
                "IGKJ2",
                "376S39M136S",
                "IGKC",
                "415S136M",
                "AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "GGGACACCTGGGGACACTGAGCTGGTGCTGAGTTACTGAGATGAGCCAGCTCTGCAGCTGTGCCCAGTCAGCCCCATCCCCTGCTCATTTGCATGTTCCCAGAGCACAACCTCCTGCACTGAAGCCTTATTAATAGGCTGGCCACACTTCATGCAGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCTCCTGTGCAGTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "TGTCAACAGTATTATAGTTTCCCGTACACTTTT",
                "CQQYYSFPYTF",
                33,
                11,
                28,
                380,
                "",
                "",
                377,
                415,
                416,
                551,
                2813,
                22,
                "T",
            ],
            [
                "AAACCTGTCTTGAGAC-1",
                "",
                "AAACCTGTCTTGAGAC-1_contig_2",
                "GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "MGSTAILALLLAVLQGVCAEVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCARHIRGNRFGNDAFDIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV",
                "T",
                "F",
                "IGHV5-51",
                "59S353M145S",
                "",
                "",
                "IGHJ3",
                "436S50M71S",
                "IGHM",
                "486S71M",
                "GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "TGAGTCTCCCTCACTGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG",
                "CARHIRGNRFGNDAFDIW",
                54,
                18,
                60,
                412,
                "",
                "",
                437,
                486,
                487,
                557,
                8292,
                80,
                "T",
            ],
            [
                "AAACGGGAGCGACGTA-1",
                "",
                "AAACGGGAGCGACGTA-1_contig_1",
                "GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "MKHLWFFLLLVAAPRWVLSQVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCARVGYRAAAGTDAFDIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV",
                "T",
                "F",
                "IGHV4-59",
                "39S350M145S",
                "",
                "",
                "IGHJ3",
                "413S50M71S",
                "IGHM",
                "463S71M",
                "GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "ATGAAACATCTGTGGTTCTTCCTTCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCCCAGGGAAGGGACTGGAGTGGATTGGGTATATCTATTACAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATATCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCTGCGGACACGGCCGTGTATTACTGTGCGAGAGATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG",
                "CARVGYRAAAGTDAFDIW",
                54,
                18,
                40,
                389,
                "",
                "",
                414,
                463,
                464,
                534,
                1235,
                18,
                "T",
            ],
            [
                "AAACGGGAGCGACGTA-1",
                "",
                "AAACGGGAGCGACGTA-1_contig_2",
                "AGCTGTGGGCTCAGAAGCAGAGTTCTGGGGTGTCTCCACCATGGCCTGGACCCCTCTCTGGCTCACTCTCCTCACTCTTTGCATAGGTTCTGTGGTTTCTTCTGAGCTGACTCAGGACCCTGCTGTGTCTGTGGCCTTGGGACAGACAGTCAGGATCACATGCCAAGGAGACAGCCTCAGAAGCTATTATGCAAGCTGGTACCAGCAGAAGCCAGGACAGGCCCCTGTACTTGTCATCTATGGTAAAAACAACCGGCCCTCAGGGATCCCAGACCGATTCTCTGGCTCCAGCTCAGGAAACACAGCTTCCTTGACCATCACTGGGGCTCAGGCGGAAGATGAGGCTGACTATTACTGTAACTCCCGGGACAGCAGTGGTAACCATGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "MAWTPLWLTLLTLCIGSVVSSELTQDPAVSVALGQTVRITCQGDSLRSYYASWYQQKPGQAPVLVIYGKNNRPSGIPDRFSGSSSGNTASLTITGAQAEDEADYYCNSRDSSGNHVVFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
                "T",
                "F",
                "IGLV3-19",
                "40S337M256S",
                "",
                "",
                "IGLJ2",
                "384S38M211S",
                "IGLC2",
                "422S211M",
                "AGCTGTGGGCTCAGAAGCAGAGTTCTGGGGTGTCTCCACCATGGCCTGGACCCCTCTCTGGCTCACTCTCCTCACTCTTTGCATAGGTTCTGTGGTTTCTTCTGAGCTGACTCAGGACCCTGCTGTGTCTGTGGCCTTGGGACAGACAGTCAGGATCACATGCCAAGGAGACAGCCTCAGAAGCTATTATGCAAGCTGGTACCAGCAGAAGCCAGGACAGGCCCCTGTACTTGTCATCTATGGTAAAAACAACCGGCCCTCAGGGATCCCAGACCGATTCTCTGGCTCCAGCTCAGGAAACACAGCTTCCTTGACCATCACTGGGGCTCAGGCGGAAGATGAGGCTGACTATTACTGTAACTCCCGGGACAGCAGTGGTAACCATGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "AGCTGTGGGCTCAGAAGCAGAGTTCTGGGGTGTCTCCACCATGGCCTGGACCCCTCTCTGGCTCACTCTCCTCACTCTTTGCATAGGTTCTGTGGTTTCTTCTGAGCTGACTCAGGACCCTGCTGTGTCTGTGGCCTTGGGACAGACAGTCAGGATCACATGCCAAGGAGACAGCCTCAGAAGCTATTATGCAAGCTGGTACCAGCAGAAGCCAGGACAGGCCCCTGTACTTGTCATCTATGGTAAAAACAACCGGCCCTCAGGGATCCCAGACCGATTCTCTGGCTCCAGCTCAGGAAACACAGCTTCCTTGACCATCACTGGGGCTCAGGCGGAAGATGAGGCTGACTATTACTGTAACTCCCGGGACAGCAGTGTGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "TGTAACTCCCGGGACAGCAGTGGTAACCATGTGGTATTC",
                "CNSRDSSGNHVVF",
                39,
                13,
                41,
                377,
                "",
                "",
                385,
                422,
                423,
                633,
                1344,
                14,
                "T",
            ],
        ],
        columns=[
            "cell_id",
            "clone_id",
            "sequence_id",
            "sequence",
            "sequence_aa",
            "productive",
            "rev_comp",
            "v_call",
            "v_cigar",
            "d_call",
            "d_cigar",
            "j_call",
            "j_cigar",
            "c_call",
            "c_cigar",
            "sequence_alignment",
            "germline_alignment",
            "junction",
            "junction_aa",
            "junction_length",
            "junction_aa_length",
            "v_sequence_start",
            "v_sequence_end",
            "d_sequence_start",
            "d_sequence_end",
            "j_sequence_start",
            "j_sequence_end",
            "c_sequence_start",
            "c_sequence_end",
            "consensus_count",
            "duplicate_count",
            "is_cell",
        ],
    )
    return airr


@pytest.fixture
def airr_reannotated():
    """airr reannotated"""
    annotated = pd.DataFrame(
        [
            [
                "AAACCTGTCATATCGG-1_contig_1",
                "TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-33*01,IGKV1D-33*01",
                "",
                "IGKJ4*01",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATT..................AACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCA.....................TTGAATTTAGAAATAGGGGTCCCA...TCAAGATTCAGTGGAAGAGGG......TCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGA",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCAGGCGAGTCAGGACATT..................AGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTACGATGCA.....................TCCAATTTGGAAACAGGGGTCCCA...TCAAGGTTCAGTGGAAGTGGA......TCTGGGACAGATTTTACTTTCACCATCAGCAGCCTGCAGCCTGAAGATATTGCAACATATTACTGTCAACAGTATGATAATCTCCCNNTCACTTTCGGCGGAGGGACCAAGGTGGA",
                "TGTCAACAATATGACGAACTTCCCGTCACTTTC",
                "CQQYDELPVTF",
                "98S44=1X31=2X2=1X7=1X3=1X3=1X37=1X19=2X5=1X4=1X15=1X11=1X2=1X10=2X4=1X1=1X29=1X10=1X11=1X5=2X1=1X2=1X2=",
                "",
                "384S2N22=1X5=",
                "F",
                "T",
                "IGK",
                33,
                2,
                "",
                99,
                382,
                1,
                332,
                "",
                "",
                "",
                "",
                385,
                412,
                3,
                30,
                363,
                0.90845,
                3.60e-102,
                "",
                "",
                "",
                48.8,
                0.96429,
                1.23e-09,
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACA",
                "GTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTAC",
                "AATTTAGAAATAGGGGTCCCA...TCAAGATTCAGTGGAAGAGGG......TCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGT",
                "TTCGGCGGAGGGACCAATGTGGA",
                "CAAGACATT..................AACAATTAT",
                "GATGCA.....................TTG",
                "CAACAATATGACGAACTTCCCGTCACT",
                "AAACCTGTCATATCGG-1",
                "IGKC",
                9139,
                68,
                "IGKV1-8",
                "",
                "IGKJ4",
                "TGTCAACAATATGACGAACTTCCCGTCACTTTC",
                "CQQYDELPVTF",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCAGGCGAGTCAGGACATT..................AGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTACGATGCA.....................TCCAATTTGGAAACAGGGGTCCCA...TCAAGGTTCAGTGGAAGTGGA......TCTGGGACAGATTTTACTTTCACCATCAGCAGCCTGCAGCCTGAAGATATTGCAACATATTACTGTCAACAGTATGATAATCTCCCNNTCACTTTCGGCGGAGGGACCAAGGTGGA",
                "IGKV1-33*01,IGKV1D-33*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                420,
                556,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "DIQMTQSPSSLSASVGDRVTITCQAT",
                "VNWYQQKPGKAPKLLIY",
                "NLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYC",
                "FGGGTNV",
                "QDINNY",
                "DAL",
                "QQYDELPVT",
                "DIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELPVTFGGGTNV",
                "DIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELP",
                "",
                "TFGGGTNV",
                0.087096774,
                27,
            ],
            [
                "AAACCTGTCCGTTGTC-1_contig_2",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-69*01,IGHV1-69D*01",
                "IGHD3-22*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNGTATTACTATGATAGTAGTGGTTATTACNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "115S292=",
                "411S28=",
                "444S50=",
                "F",
                "T",
                "IGH",
                63,
                4,
                5,
                116,
                407,
                1,
                316,
                412,
                439,
                1,
                28,
                445,
                494,
                1,
                50,
                456,
                1,
                2.64e-130,
                54.5,
                1,
                1.35e-11,
                96.8,
                1,
                4.25e-24,
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGG",
                "AACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGAGGCACCTTC............AGCAGCTATGCT",
                "ATCATCCCTATC......TTTGGTACAGCA",
                "GCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATC",
                "AAACCTGTCCGTTGTC-1",
                "IGHM",
                4161,
                51,
                "IGHV1-69D",
                "IGHD3-22",
                "IGHJ3",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-69*01,IGHV1-69D*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                494,
                565,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                21,
                "QVQLVQSGAEVKKPGSSVKVSCKAS",
                "ISWVRQAPGQGLEWMGG",
                "NYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYC",
                "WGQGTMVTVSS",
                "GGTFSSYA",
                "IIPIFGTA",
                "ATTYYYDSSGYYQNDAFDI",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSS",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCA",
                "YYYDSSGYY",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-1_contig_1",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-8*01",
                "",
                "IGKJ1*01",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "93S286=",
                "",
                "379S2N36=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                379,
                1,
                334,
                "",
                "",
                "",
                "",
                380,
                415,
                3,
                38,
                447,
                1,
                1.68e-127,
                "",
                "",
                "",
                69.9,
                1,
                5.25e-16,
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "CAGGGTATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTACCCTCGGACG",
                "AAACCTGTCCGTTGTC-1",
                "IGKC",
                5679,
                43,
                "IGKV1-8",
                "",
                "IGKJ1",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "IGKV1-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "AIRMTQSPSSFSASTGDRVTITCRAS",
                "LAWYQQKPGKAPKLLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKVEIK",
                "QGISSY",
                "AAS",
                "QQYYSYPRT",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIK",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYP",
                "",
                "TFGQGTKVEIK",
                0,
                0,
            ],
            [
                "AAACCTGTCGAGAACG-1_contig_2",
                "GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-2*02",
                "IGHD3-16*01,IGHD3-16*02,IGHD4-17*01",
                "IGHJ3*02",
                "CAGGTGCAACTGGTGCAGTCTGGGGGT...GAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAAC......AGTGGTGACACCAACTATGCACAGAAGTTTCAG...GGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAAC......AGTGGTGGCACAAACTATGCACAGAAGTTTCAG...GGCAGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGANNNNNNGGGGGANNNNNNTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG",
                "CAREIEGDGVFEIW",
                "121S8=1X16=1X6=1X58=1X5=2X2=1X2=1X3=1X7=1X11=1X8=1X8=1X1=2X19=1X3=1X26=1X29=1X14=1X8=1X23=1X17=",
                "423S17N6=",
                "435S6N6=1X37=",
                "F",
                "T",
                "IGH",
                42,
                6,
                6,
                122,
                417,
                1,
                320,
                424,
                429,
                18,
                23,
                436,
                479,
                7,
                50,
                394,
                0.92568,
                1.48e-111,
                12.2,
                1,
                71,
                79.5,
                0.97727,
                6.69e-19,
                "CAGGTGCAACTGGTGCAGTCTGGGGGT...GAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTA",
                "AACTATGCACAGAAGTTTCAG...GGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGATACACCTTC............ACCGACTACTTC",
                "ATCAACCCTAAC......AGTGGTGACACC",
                "GCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATC",
                "AAACCTGTCGAGAACG-1",
                "IGHM",
                5080,
                47,
                "IGHV1-2",
                "",
                "IGHJ3",
                "TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG",
                "CAREIEGDGVFEIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAAC......AGTGGTGGCACAAACTATGCACAGAAGTTTCAG...GGCAGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGANNNNNNNNNNNNNNNNNNTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-2*02",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                479,
                550,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                14,
                "QVQLVQSGGEVKKPGASVKVSCKAS",
                "IQWLRHAPGQGLDWMGL",
                "NYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYC",
                "WGQGTMVTVSS",
                "GYTFTDYF",
                "INPNSGDT",
                "AREIEGDGVFEI",
                "QVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAREIEGDGVFEIWGQGTMVTVSS",
                "QVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAR",
                "G",
                "FEIWGQGTMVTVSS",
                0.064896755,
                22,
            ],
            [
                "AAACCTGTCGAGAACG-1_contig_1",
                "ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "F",
                "T",
                "IGLV5-45*02",
                "",
                "IGLJ3*02",
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTT.........GGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTACGCAGTGGCATCAATGTT.........GGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC",
                "CMIWHSSAWVV",
                "85S52=1X3=1X22=1X15=1X11=1X10=1X165=1X25=",
                "",
                "395S2N5=1X30=",
                "F",
                "T",
                "IGL",
                33,
                0,
                "",
                86,
                395,
                1,
                334,
                "",
                "",
                "",
                "",
                396,
                431,
                3,
                38,
                463,
                0.97742,
                4.02e-132,
                "",
                "",
                "",
                64.1,
                0.97222,
                3.34e-14,
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGC",
                "ATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGG",
                "CAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGT",
                "GTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "AGTGACATCAATGTT.........GGTACGTACAGG",
                "TACAAATCAGAC.........TCAGATAAG",
                "ATGATTTGGCACAGCAGCGCTTGGGTG",
                "AAACCTGTCGAGAACG-1",
                "IGLC3",
                13160,
                90,
                "IGLV5-45",
                "",
                "IGLJ3",
                "TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC",
                "CMIWHSSAWVV",
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTACGCAGTGGCATCAATGTT.........GGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "IGLV5-45*02",
                "sample1",
                "GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                431,
                642,
                392.61,
                100,
                1.00e-111,
                "IGLC3",
                11,
                "QAVLTQPSSLSASPGASGRLTCTLR",
                "IYWYQRKPGSPPQYLLR",
                "QQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYC",
                "VGGGTKLTVL",
                "SDINVGTYR",
                "YKSDSDK",
                "MIWHSSAWV",
                "QAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSAWVVGGGTKLTVL",
                "QAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSA",
                "",
                "VVGGGTKLTVL",
                0.023188406,
                8,
            ],
            [
                "AAACCTGTCTTGAGAC-1_contig_2",
                "GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV5-51*01",
                "IGHD1-26*01,IGHD1/OR15-1a*01,IGHD1/OR15-1b*01",
                "IGHJ3*02",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACANNNNNGTGGGANNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG",
                "CARHIRGNRFGNDAFDIW",
                "116S41=1X254=",
                "417S6N6=",
                "436S50=",
                "F",
                "T",
                "IGH",
                54,
                5,
                13,
                117,
                412,
                1,
                320,
                418,
                423,
                7,
                12,
                437,
                486,
                1,
                50,
                459,
                0.99662,
                3.00e-131,
                12.2,
                1,
                72,
                96.8,
                1,
                4.19e-24,
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCT",
                "ATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATC",
                "AGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGATACAGCTTT............ACCAGCTACTGG",
                "ATCTATCCTGGT......GACTCTGATACC",
                "GCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATC",
                "AAACCTGTCTTGAGAC-1",
                "IGHM",
                8292,
                80,
                "IGHV5-51",
                "",
                "IGHJ3",
                "TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG",
                "CARHIRGNRFGNDAFDIW",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACANNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV5-51*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                486,
                557,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                18,
                "EVQLVQSGAEVKKPGESLKISCKGS",
                "IGWVRQMPGKGLEWMGI",
                "RYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYC",
                "WGQGTMVTVSS",
                "GYSFTSYW",
                "IYPGDSDT",
                "ARHIRGNRFGNDAFDI",
                "EVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCARHIRGNRFGNDAFDIWGQGTMVTVSS",
                "EVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCAR",
                "G",
                "DAFDIWGQGTMVTVSS",
                0.002898551,
                1,
            ],
            [
                "AAACCTGTCTTGAGAC-1_contig_1",
                "AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1D-8*01",
                "",
                "IGKJ2*01",
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "TGTCAACAGTATTATAGTTTCCCGTACACTTTT",
                "CQQYYSFPYTF",
                "93S284=",
                "",
                "377S1N38=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                377,
                1,
                332,
                "",
                "",
                "",
                "",
                378,
                415,
                2,
                39,
                444,
                1,
                1.45e-126,
                "",
                "",
                "",
                73.7,
                1,
                3.65e-17,
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "CAGGGCATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTTCCCGTACACT",
                "AAACCTGTCTTGAGAC-1",
                "IGKC",
                2813,
                22,
                "IGKV1D-8",
                "",
                "IGKJ2",
                "TGTCAACAGTATTATAGTTTCCCGTACACTTTT",
                "CQQYYSFPYTF",
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "IGKV1D-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "VIWMTQSPSLLSASTGDRVTISCRMS",
                "LAWYQQKPGKAPELLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKLEIK",
                "QGISSY",
                "AAS",
                "QQYYSFPYT",
                "VIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFPYTFGQGTKLEIK",
                "VIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFP",
                "",
                "YTFGQGTKLEIK",
                0,
                0,
            ],
            [
                "AAACGGGAGCGACGTA-1_contig_1",
                "GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV4-4*07",
                "IGHD6-13*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGNNNNNNNNNNAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG",
                "CARVGYRAAAGTDAFDIW",
                "96S292=",
                "398S6N15=",
                "413S50=",
                "F",
                "T",
                "IGH",
                54,
                10,
                0,
                97,
                388,
                1,
                319,
                399,
                413,
                7,
                21,
                414,
                463,
                1,
                50,
                456,
                1,
                2.49e-130,
                29.5,
                1,
                0.000427,
                96.8,
                1,
                4.01e-24,
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCT",
                "TGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGT",
                "AACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGTGGCTCCATC............AGTAGTTACTAC",
                "ATCTATACCAGT.........GGGAGCACC",
                "GCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATC",
                "AAACGGGAGCGACGTA-1",
                "IGHM",
                1235,
                18,
                "IGHV4-59",
                "",
                "IGHJ3",
                "TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG",
                "CARVGYRAAAGTDAFDIW",
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV4-4*07",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                463,
                534,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                18,
                "QVQLQESGPGLVKPSETLSLTCTVS",
                "WSWIRQPAGKGLEWIGR",
                "NYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYC",
                "WGQGTMVTVSS",
                "GGSISSYY",
                "IYTSGST",
                "ARVGYRAAAGTDAFDI",
                "QVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCARVGYRAAAGTDAFDIWGQGTMVTVSS",
                "QVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCAR",
                "AAAGT",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
        ],
        columns=[
            "sequence_id",
            "sequence",
            "rev_comp",
            "productive",
            "v_call",
            "d_call",
            "j_call",
            "sequence_alignment",
            "germline_alignment",
            "junction",
            "junction_aa",
            "v_cigar",
            "d_cigar",
            "j_cigar",
            "stop_codon",
            "vj_in_frame",
            "locus",
            "junction_length",
            "np1_length",
            "np2_length",
            "v_sequence_start",
            "v_sequence_end",
            "v_germline_start",
            "v_germline_end",
            "d_sequence_start",
            "d_sequence_end",
            "d_germline_start",
            "d_germline_end",
            "j_sequence_start",
            "j_sequence_end",
            "j_germline_start",
            "j_germline_end",
            "v_score",
            "v_identity",
            "v_support",
            "d_score",
            "d_identity",
            "d_support",
            "j_score",
            "j_identity",
            "j_support",
            "fwr1",
            "fwr2",
            "fwr3",
            "fwr4",
            "cdr1",
            "cdr2",
            "cdr3",
            "cell_id",
            "c_call",
            "consensus_count",
            "umi_count",
            "v_call_10x",
            "d_call_10x",
            "j_call_10x",
            "junction_10x",
            "junction_10x_aa",
            "germline_alignment_d_mask",
            "v_call_genotyped",
            "sample_id",
            "c_sequence_alignment",
            "c_germline_alignment",
            "c_sequence_start",
            "c_sequence_end",
            "c_score",
            "c_identity",
            "c_support",
            "c_call_10x",
            "junction_aa_length",
            "fwr1_aa",
            "fwr2_aa",
            "fwr3_aa",
            "fwr4_aa",
            "cdr1_aa",
            "cdr2_aa",
            "cdr3_aa",
            "sequence_alignment_aa",
            "v_sequence_alignment_aa",
            "d_sequence_alignment_aa",
            "j_sequence_alignment_aa",
            "mu_freq",
            "mu_count",
        ],
    )
    return annotated


@pytest.fixture
def airr_reannotated2():
    """airr reannotated 2"""
    annotated = pd.DataFrame(
        [
            [
                "AAACCTGTCATATCGG-1_contig_1",
                "TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-33*01,IGKV1D-33*01",
                "",
                "IGKJ4*01",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATT..................AACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCA.....................TTGAATTTAGAAATAGGGGTCCCA...TCAAGATTCAGTGGAAGAGGG......TCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGA",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCAGGCGAGTCAGGACATT..................AGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTACGATGCA.....................TCCAATTTGGAAACAGGGGTCCCA...TCAAGGTTCAGTGGAAGTGGA......TCTGGGACAGATTTTACTTTCACCATCAGCAGCCTGCAGCCTGAAGATATTGCAACATATTACTGTCAACAGTATGATAATCTCCCNNTCACTTTCGGCGGAGGGACCAAGGTGGA",
                "TGTCAACAATATGACGAACTTCCCGTCACTTTC",
                "CQQYDELPVTF",
                "98S44=1X31=2X2=1X7=1X3=1X3=1X37=1X19=2X5=1X4=1X15=1X11=1X2=1X10=2X4=1X1=1X29=1X10=1X11=1X5=2X1=1X2=1X2=",
                "",
                "384S2N22=1X5=",
                "F",
                "T",
                "IGK",
                33,
                2,
                "",
                99,
                382,
                1,
                332,
                "",
                "",
                "",
                "",
                385,
                412,
                3,
                30,
                363,
                0.90845,
                3.60e-102,
                "",
                "",
                "",
                48.8,
                0.96429,
                1.23e-09,
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACA",
                "GTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTAC",
                "AATTTAGAAATAGGGGTCCCA...TCAAGATTCAGTGGAAGAGGG......TCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGT",
                "TTCGGCGGAGGGACCAATGTGGA",
                "CAAGACATT..................AACAATTAT",
                "GATGCA.....................TTG",
                "CAACAATATGACGAACTTCCCGTCACT",
                "AAACCTGTCATATCGG-1",
                "IGKC",
                9139,
                68,
                "IGKV1-8",
                "",
                "IGKJ4",
                "TGTCAACAATATGACGAACTTCCCGTCACTTTC",
                "CQQYDELPVTF",
                "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCAGGCGAGTCAGGACATT..................AGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTACGATGCA.....................TCCAATTTGGAAACAGGGGTCCCA...TCAAGGTTCAGTGGAAGTGGA......TCTGGGACAGATTTTACTTTCACCATCAGCAGCCTGCAGCCTGAAGATATTGCAACATATTACTGTCAACAGTATGATAATCTCCCNNTCACTTTCGGCGGAGGGACCAAGGTGGA",
                "IGKV1-33*01,IGKV1D-33*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                420,
                556,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "DIQMTQSPSSLSASVGDRVTITCQAT",
                "VNWYQQKPGKAPKLLIY",
                "NLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYC",
                "FGGGTNV",
                "QDINNY",
                "DAL",
                "QQYDELPVT",
                "DIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELPVTFGGGTNV",
                "DIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELP",
                "",
                "TFGGGTNV",
                0.087096774,
                27,
            ],
            [
                "AAACCTGTCCGTTGTC-1_contig_2",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-69*01,IGHV1-69D*01",
                "IGHD3-22*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNGTATTACTATGATAGTAGTGGTTATTACNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "115S292=",
                "411S28=",
                "444S50=",
                "F",
                "T",
                "IGH",
                63,
                4,
                5,
                116,
                407,
                1,
                316,
                412,
                439,
                1,
                28,
                445,
                494,
                1,
                50,
                456,
                1,
                2.64e-130,
                54.5,
                1,
                1.35e-11,
                96.8,
                1,
                4.25e-24,
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGG",
                "AACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGAGGCACCTTC............AGCAGCTATGCT",
                "ATCATCCCTATC......TTTGGTACAGCA",
                "GCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATC",
                "AAACCTGTCCGTTGTC-1",
                "IGHM",
                4161,
                51,
                "IGHV1-69D",
                "IGHD3-22",
                "IGHJ3",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-69*01,IGHV1-69D*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                494,
                565,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                21,
                "QVQLVQSGAEVKKPGSSVKVSCKAS",
                "ISWVRQAPGQGLEWMGG",
                "NYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYC",
                "WGQGTMVTVSS",
                "GGTFSSYA",
                "IIPIFGTA",
                "ATTYYYDSSGYYQNDAFDI",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSS",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCA",
                "YYYDSSGYY",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-1_contig_1",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-8*01",
                "",
                "IGKJ1*01",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "93S286=",
                "",
                "379S2N36=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                379,
                1,
                334,
                "",
                "",
                "",
                "",
                380,
                415,
                3,
                38,
                447,
                1,
                1.68e-127,
                "",
                "",
                "",
                69.9,
                1,
                5.25e-16,
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "CAGGGTATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTACCCTCGGACG",
                "AAACCTGTCCGTTGTC-1",
                "IGKC",
                5679,
                43,
                "IGKV1-8",
                "",
                "IGKJ1",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "IGKV1-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "AIRMTQSPSSFSASTGDRVTITCRAS",
                "LAWYQQKPGKAPKLLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKVEIK",
                "QGISSY",
                "AAS",
                "QQYYSYPRT",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIK",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYP",
                "",
                "TFGQGTKVEIK",
                0,
                0,
            ],
            [
                "AAACCTGTCGAGAACG-1_contig_2",
                "GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-2*02",
                "IGHD3-16*01,IGHD3-16*02,IGHD4-17*01",
                "IGHJ3*02",
                "CAGGTGCAACTGGTGCAGTCTGGGGGT...GAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAAC......AGTGGTGACACCAACTATGCACAGAAGTTTCAG...GGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAAC......AGTGGTGGCACAAACTATGCACAGAAGTTTCAG...GGCAGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGANNNNNNGGGGGANNNNNNTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG",
                "CAREIEGDGVFEIW",
                "121S8=1X16=1X6=1X58=1X5=2X2=1X2=1X3=1X7=1X11=1X8=1X8=1X1=2X19=1X3=1X26=1X29=1X14=1X8=1X23=1X17=",
                "423S17N6=",
                "435S6N6=1X37=",
                "F",
                "T",
                "IGH",
                42,
                6,
                6,
                122,
                417,
                1,
                320,
                424,
                429,
                18,
                23,
                436,
                479,
                7,
                50,
                394,
                0.92568,
                1.48e-111,
                12.2,
                1,
                71,
                79.5,
                0.97727,
                6.69e-19,
                "CAGGTGCAACTGGTGCAGTCTGGGGGT...GAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTA",
                "AACTATGCACAGAAGTTTCAG...GGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGATACACCTTC............ACCGACTACTTC",
                "ATCAACCCTAAC......AGTGGTGACACC",
                "GCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATC",
                "AAACCTGTCGAGAACG-1",
                "IGHM",
                5080,
                47,
                "IGHV1-2",
                "",
                "IGHJ3",
                "TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG",
                "CAREIEGDGVFEIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAAC......AGTGGTGGCACAAACTATGCACAGAAGTTTCAG...GGCAGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGANNNNNNNNNNNNNNNNNNTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-2*02",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                479,
                550,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                14,
                "QVQLVQSGGEVKKPGASVKVSCKAS",
                "IQWLRHAPGQGLDWMGL",
                "NYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYC",
                "WGQGTMVTVSS",
                "GYTFTDYF",
                "INPNSGDT",
                "AREIEGDGVFEI",
                "QVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAREIEGDGVFEIWGQGTMVTVSS",
                "QVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAR",
                "G",
                "FEIWGQGTMVTVSS",
                0.064896755,
                22,
            ],
            [
                "AAACCTGTCGAGAACG-1_contig_1",
                "ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "F",
                "T",
                "IGLV5-45*02",
                "",
                "IGLJ3*02",
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTT.........GGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTACGCAGTGGCATCAATGTT.........GGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC",
                "CMIWHSSAWVV",
                "85S52=1X3=1X22=1X15=1X11=1X10=1X165=1X25=",
                "",
                "395S2N5=1X30=",
                "F",
                "T",
                "IGL",
                33,
                0,
                "",
                86,
                395,
                1,
                334,
                "",
                "",
                "",
                "",
                396,
                431,
                3,
                38,
                463,
                0.97742,
                4.02e-132,
                "",
                "",
                "",
                64.1,
                0.97222,
                3.34e-14,
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGC",
                "ATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGG",
                "CAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGT",
                "GTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "AGTGACATCAATGTT.........GGTACGTACAGG",
                "TACAAATCAGAC.........TCAGATAAG",
                "ATGATTTGGCACAGCAGCGCTTGGGTG",
                "AAACCTGTCGAGAACG-1",
                "IGLC3",
                13160,
                90,
                "IGLV5-45",
                "",
                "IGLJ3",
                "TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC",
                "CMIWHSSAWVV",
                "CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTACGCAGTGGCATCAATGTT.........GGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "IGLV5-45*02",
                "sample1",
                "GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                431,
                642,
                392.61,
                100,
                1.00e-111,
                "IGLC3",
                11,
                "QAVLTQPSSLSASPGASGRLTCTLR",
                "IYWYQRKPGSPPQYLLR",
                "QQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYC",
                "VGGGTKLTVL",
                "SDINVGTYR",
                "YKSDSDK",
                "MIWHSSAWV",
                "QAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSAWVVGGGTKLTVL",
                "QAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSA",
                "",
                "VVGGGTKLTVL",
                0.023188406,
                8,
            ],
            [
                "AAACCTGTCTTGAGAC-1_contig_2",
                "GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV5-51*01",
                "IGHD1-26*01,IGHD1/OR15-1a*01,IGHD1/OR15-1b*01",
                "IGHJ3*02",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACANNNNNGTGGGANNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG",
                "CARHIRGNRFGNDAFDIW",
                "116S41=1X254=",
                "417S6N6=",
                "436S50=",
                "F",
                "T",
                "IGH",
                54,
                5,
                13,
                117,
                412,
                1,
                320,
                418,
                423,
                7,
                12,
                437,
                486,
                1,
                50,
                459,
                0.99662,
                3.00e-131,
                12.2,
                1,
                72,
                96.8,
                1,
                4.19e-24,
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCT",
                "ATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATC",
                "AGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGATACAGCTTT............ACCAGCTACTGG",
                "ATCTATCCTGGT......GACTCTGATACC",
                "GCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATC",
                "AAACCTGTCTTGAGAC-1",
                "IGHM",
                8292,
                80,
                "IGHV5-51",
                "",
                "IGHJ3",
                "TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG",
                "CARHIRGNRFGNDAFDIW",
                "GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACANNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV5-51*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                486,
                557,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                18,
                "EVQLVQSGAEVKKPGESLKISCKGS",
                "IGWVRQMPGKGLEWMGI",
                "RYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYC",
                "WGQGTMVTVSS",
                "GYSFTSYW",
                "IYPGDSDT",
                "ARHIRGNRFGNDAFDI",
                "EVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCARHIRGNRFGNDAFDIWGQGTMVTVSS",
                "EVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCAR",
                "G",
                "DAFDIWGQGTMVTVSS",
                0.002898551,
                1,
            ],
            [
                "AAACCTGTCTTGAGAC-1_contig_1",
                "AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1D-8*01",
                "",
                "IGKJ2*01",
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "TGTCAACAGTATTATAGTTTCCCGTACACTTTT",
                "CQQYYSFPYTF",
                "93S284=",
                "",
                "377S1N38=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                377,
                1,
                332,
                "",
                "",
                "",
                "",
                378,
                415,
                2,
                39,
                444,
                1,
                1.45e-126,
                "",
                "",
                "",
                73.7,
                1,
                3.65e-17,
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "CAGGGCATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTTCCCGTACACT",
                "AAACCTGTCTTGAGAC-1",
                "IGKC",
                2813,
                22,
                "IGKV1D-8",
                "",
                "IGKJ2",
                "TGTCAACAGTATTATAGTTTCCCGTACACTTTT",
                "CQQYYSFPYTF",
                "GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC",
                "IGKV1D-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "VIWMTQSPSLLSASTGDRVTISCRMS",
                "LAWYQQKPGKAPELLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKLEIK",
                "QGISSY",
                "AAS",
                "QQYYSFPYT",
                "VIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFPYTFGQGTKLEIK",
                "VIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFP",
                "",
                "YTFGQGTKLEIK",
                0,
                0,
            ],
            [
                "AAACGGGAGCGACGTA-1_contig_1",
                "GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV4-4*07",
                "IGHD6-13*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGNNNNNNNNNNAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG",
                "CARVGYRAAAGTDAFDIW",
                "96S292=",
                "398S6N15=",
                "413S50=",
                "F",
                "T",
                "IGH",
                54,
                10,
                0,
                97,
                388,
                1,
                319,
                399,
                413,
                7,
                21,
                414,
                463,
                1,
                50,
                456,
                1,
                2.49e-130,
                29.5,
                1,
                0.000427,
                96.8,
                1,
                4.01e-24,
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCT",
                "TGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGT",
                "AACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGTGGCTCCATC............AGTAGTTACTAC",
                "ATCTATACCAGT.........GGGAGCACC",
                "GCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATC",
                "AAACGGGAGCGACGTA-1",
                "IGHM",
                1235,
                18,
                "IGHV4-59",
                "",
                "IGHJ3",
                "TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG",
                "CARVGYRAAAGTDAFDIW",
                "CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV4-4*07",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                463,
                534,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                18,
                "QVQLQESGPGLVKPSETLSLTCTVS",
                "WSWIRQPAGKGLEWIGR",
                "NYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYC",
                "WGQGTMVTVSS",
                "GGSISSYY",
                "IYTSGST",
                "ARVGYRAAAGTDAFDI",
                "QVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCARVGYRAAAGTDAFDIWGQGTMVTVSS",
                "QVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCAR",
                "AAAGT",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
            [
                "AAACGGGAGCGACGTA-1_contig_2",
                "TCTGGCACCAGGGGTCCCTTCCAATATCAGCACCATGGCCTGGACTCCTCTCTTTCTGTTCCTCCTCACTTGCTGCCCAGGGTCCAATTCCCAGGCTGTGGTGACTCAGGAGCCCTCACTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACCAGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACAAGCAACAAACACTCCTGGACACCTGCCCGGTTCTCAGGCTCCCTCCTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "F",
                "F",
                "IGLV7-46*01",
                "",
                "IGLJ3*02",
                "CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACC.........AGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACA.....................AGCAACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACC.........AGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACA.....................AGCAACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTCGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "TGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTC",
                "CLLSYSGARGV",
                "91S291=1X2=",
                "",
                "385S3N35=",
                "F",
                "F",
                "IGL",
                34,
                0,
                "",
                92,
                385,
                1,
                336,
                "",
                "",
                "",
                "",
                386,
                420,
                4,
                38,
                456,
                0.9966,
                2.97e-130,
                "",
                "",
                "",
                68,
                1,
                2.29e-15,
                "CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGC",
                "CCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTAT",
                "AACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGC",
                "TTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "ACTGGAGCTGTCACC.........AGTGGTCATTAT",
                "GATACA.....................AGC",
                "TTGCTCTCCTATAGTGGTGCTAGGGGTG",
                "AAACGGGAGCGACGTA-1",
                "IGLC",
                739,
                8,
                "IGLV7-46",
                "",
                "IGLJ3",
                "",
                "",
                "CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACC.........AGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACA.....................AGCAACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTCGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG",
                "IGLV7-46*01",
                "sample1",
                "GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                "GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
                420,
                631,
                392.61,
                100,
                1.00e-111,
                "IGLC2",
                11,
                "QAVVTQEPSLTVSPGGTVTLTCGSS",
                "PYWFQQKPGQAPRTLIY",
                "NKHSWTPARFSGSLLGGKAALTLSGAQPEDEAEYYC",
                "RRRDQADRP",
                "TGAVTSGHY",
                "DTS",
                "LLSYSGARG",
                "QAVVTQEPSLTVSPGGTVTLTCGSSTGAVTSGHYPYWFQQKPGQAPRTLIYDTSNKHSWTPARFSGSLLGGKAALTLSGAQPEDEAEYYCLLSYSGARGVRRRDQADRP",
                "QAVVTQEPSLTVSPGGTVTLTCGSSTGAVTSGHYPYWFQQKPGQAPRTLIYDTSNKHSWTPARFSGSLLGGKAALTLSGAQPEDEAEYYCLLSYSGAR",
                "",
                "GVRRRDQADRP",
                0.003058104,
                1,
            ],
            [
                "AAACCTGTCCGTTGTC-2_contig_2",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-69*01,IGHV1-69D*01",
                "IGHD3-22*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNGTATTACTATGATAGTAGTGGTTATTACNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "115S292=",
                "411S28=",
                "444S50=",
                "F",
                "T",
                "IGH",
                63,
                4,
                5,
                116,
                407,
                1,
                316,
                412,
                439,
                1,
                28,
                445,
                494,
                1,
                50,
                456,
                1,
                2.64e-130,
                54.5,
                1,
                1.35e-11,
                96.8,
                1,
                4.25e-24,
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGG",
                "AACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGAGGCACCTTC............AGCAGCTATGCT",
                "ATCATCCCTATC......TTTGGTACAGCA",
                "GCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATC",
                "AAACCTGTCCGTTGTC-2",
                "IGHM",
                4161,
                51,
                "IGHV1-69D",
                "IGHD3-22",
                "IGHJ3",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-69*01,IGHV1-69D*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                494,
                565,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                21,
                "QVQLVQSGAEVKKPGSSVKVSCKAS",
                "ISWVRQAPGQGLEWMGG",
                "NYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYC",
                "WGQGTMVTVSS",
                "GGTFSSYA",
                "IIPIFGTA",
                "ATTYYYDSSGYYQNDAFDI",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSS",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCA",
                "YYYDSSGYY",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-2_contig_1",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-8*01",
                "",
                "IGKJ1*01",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "93S286=",
                "",
                "379S2N36=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                379,
                1,
                334,
                "",
                "",
                "",
                "",
                380,
                415,
                3,
                38,
                447,
                1,
                1.68e-127,
                "",
                "",
                "",
                69.9,
                1,
                5.25e-16,
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "CAGGGTATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTACCCTCGGACG",
                "AAACCTGTCCGTTGTC-2",
                "IGKC",
                5679,
                43,
                "IGKV1-8",
                "",
                "IGKJ1",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "IGKV1-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "AIRMTQSPSSFSASTGDRVTITCRAS",
                "LAWYQQKPGKAPKLLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKVEIK",
                "QGISSY",
                "AAS",
                "QQYYSYPRT",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIK",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYP",
                "",
                "TFGQGTKVEIK",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-3_contig_2",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-69*01,IGHV1-69D*01",
                "IGHD3-22*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNGTATTACTATGATAGTAGTGGTTATTACNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "115S292=",
                "411S28=",
                "444S50=",
                "F",
                "T",
                "IGH",
                63,
                4,
                5,
                116,
                407,
                1,
                316,
                412,
                439,
                1,
                28,
                445,
                494,
                1,
                50,
                456,
                1,
                2.64e-130,
                54.5,
                1,
                1.35e-11,
                96.8,
                1,
                4.25e-24,
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGG",
                "AACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGAGGCACCTTC............AGCAGCTATGCT",
                "ATCATCCCTATC......TTTGGTACAGCA",
                "GCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATC",
                "AAACCTGTCCGTTGTC-3",
                "IGHM",
                4161,
                51,
                "IGHV1-69D",
                "IGHD3-22",
                "IGHJ3",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-69*01,IGHV1-69D*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                494,
                565,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                21,
                "QVQLVQSGAEVKKPGSSVKVSCKAS",
                "ISWVRQAPGQGLEWMGG",
                "NYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYC",
                "WGQGTMVTVSS",
                "GGTFSSYA",
                "IIPIFGTA",
                "ATTYYYDSSGYYQNDAFDI",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSS",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCA",
                "YYYDSSGYY",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-3_contig_1",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-8*01",
                "",
                "IGKJ1*01",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "93S286=",
                "",
                "379S2N36=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                379,
                1,
                334,
                "",
                "",
                "",
                "",
                380,
                415,
                3,
                38,
                447,
                1,
                1.68e-127,
                "",
                "",
                "",
                69.9,
                1,
                5.25e-16,
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "CAGGGTATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTACCCTCGGACG",
                "AAACCTGTCCGTTGTC-3",
                "IGKC",
                5679,
                43,
                "IGKV1-8",
                "",
                "IGKJ1",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "IGKV1-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "AIRMTQSPSSFSASTGDRVTITCRAS",
                "LAWYQQKPGKAPKLLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKVEIK",
                "QGISSY",
                "AAS",
                "QQYYSYPRT",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIK",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYP",
                "",
                "TFGQGTKVEIK",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-4_contig_2",
                "ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "F",
                "T",
                "IGHV1-69*01,IGHV1-69D*01",
                "IGHD3-22*01",
                "IGHJ3*02",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNGTATTACTATGATAGTAGTGGTTATTACNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "115S292=",
                "411S28=",
                "444S50=",
                "F",
                "T",
                "IGH",
                63,
                4,
                5,
                116,
                407,
                1,
                316,
                412,
                439,
                1,
                28,
                445,
                494,
                1,
                50,
                456,
                1,
                2.64e-130,
                54.5,
                1,
                1.35e-11,
                96.8,
                1,
                4.25e-24,
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCT",
                "ATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGG",
                "AACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGT",
                "TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "GGAGGCACCTTC............AGCAGCTATGCT",
                "ATCATCCCTATC......TTTGGTACAGCA",
                "GCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATC",
                "AAACCTGTCCGTTGTC-4",
                "IGHM",
                4161,
                51,
                "IGHV1-69D",
                "IGHD3-22",
                "IGHJ3",
                "TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG",
                "CATTYYYDSSGYYQNDAFDIW",
                "CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG",
                "IGHV1-69*01,IGHV1-69D*01",
                "sample1",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                "GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
                494,
                565,
                134.079,
                100,
                7.00e-34,
                "IGHM",
                21,
                "QVQLVQSGAEVKKPGSSVKVSCKAS",
                "ISWVRQAPGQGLEWMGG",
                "NYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYC",
                "WGQGTMVTVSS",
                "GGTFSSYA",
                "IIPIFGTA",
                "ATTYYYDSSGYYQNDAFDI",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSS",
                "QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCA",
                "YYYDSSGYY",
                "DAFDIWGQGTMVTVSS",
                0,
                0,
            ],
            [
                "AAACCTGTCCGTTGTC-4_contig_1",
                "AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "F",
                "T",
                "IGKV1-8*01",
                "",
                "IGKJ1*01",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "93S286=",
                "",
                "379S2N36=",
                "F",
                "T",
                "IGK",
                33,
                0,
                "",
                94,
                379,
                1,
                334,
                "",
                "",
                "",
                "",
                380,
                415,
                3,
                38,
                447,
                1,
                1.68e-127,
                "",
                "",
                "",
                69.9,
                1,
                5.25e-16,
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGT",
                "TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT",
                "TTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "CAGGGTATT..................AGCAGTTAT",
                "GCTGCA.....................TCC",
                "CAACAGTATTATAGTTACCCTCGGACG",
                "AAACCTGTCCGTTGTC-4",
                "IGKC",
                5679,
                43,
                "IGKV1-8",
                "",
                "IGKJ1",
                "TGTCAACAGTATTATAGTTACCCTCGGACGTTC",
                "CQQYYSYPRTF",
                "GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC",
                "IGKV1-8*01",
                "sample1",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                "CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
                415,
                551,
                254.111,
                100,
                5.00e-70,
                "IGKC",
                11,
                "AIRMTQSPSSFSASTGDRVTITCRAS",
                "LAWYQQKPGKAPKLLIY",
                "TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC",
                "FGQGTKVEIK",
                "QGISSY",
                "AAS",
                "QQYYSYPRT",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIK",
                "AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYP",
                "",
                "TFGQGTKVEIK",
                0,
                0,
            ],
        ],
        columns=[
            "sequence_id",
            "sequence",
            "rev_comp",
            "productive",
            "v_call",
            "d_call",
            "j_call",
            "sequence_alignment",
            "germline_alignment",
            "junction",
            "junction_aa",
            "v_cigar",
            "d_cigar",
            "j_cigar",
            "stop_codon",
            "vj_in_frame",
            "locus",
            "junction_length",
            "np1_length",
            "np2_length",
            "v_sequence_start",
            "v_sequence_end",
            "v_germline_start",
            "v_germline_end",
            "d_sequence_start",
            "d_sequence_end",
            "d_germline_start",
            "d_germline_end",
            "j_sequence_start",
            "j_sequence_end",
            "j_germline_start",
            "j_germline_end",
            "v_score",
            "v_identity",
            "v_support",
            "d_score",
            "d_identity",
            "d_support",
            "j_score",
            "j_identity",
            "j_support",
            "fwr1",
            "fwr2",
            "fwr3",
            "fwr4",
            "cdr1",
            "cdr2",
            "cdr3",
            "cell_id",
            "c_call",
            "consensus_count",
            "umi_count",
            "v_call_10x",
            "d_call_10x",
            "j_call_10x",
            "junction_10x",
            "junction_10x_aa",
            "germline_alignment_d_mask",
            "v_call_genotyped",
            "sample_id",
            "c_sequence_alignment",
            "c_germline_alignment",
            "c_sequence_start",
            "c_sequence_end",
            "c_score",
            "c_identity",
            "c_support",
            "c_call_10x",
            "junction_aa_length",
            "fwr1_aa",
            "fwr2_aa",
            "fwr3_aa",
            "fwr4_aa",
            "cdr1_aa",
            "cdr2_aa",
            "cdr3_aa",
            "sequence_alignment_aa",
            "v_sequence_alignment_aa",
            "d_sequence_alignment_aa",
            "j_sequence_alignment_aa",
            "mu_freq",
            "mu_count",
        ],
    )
    return annotated


@pytest.fixture
def fasta_10x_tr1():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        "AACTGGTTCTTTAGTC-1_contig_2": "AGGCAGAAGGTGGTTGAGAGGCAGAGCTGCCCCTGAGTGAGCCATGCAGAGGATCTCCTCCCTCATCCATCTCTCTCTCTTCTGGGCAGGAGTCATGTCAGCCATTGAGTTGGTGCCTGAACACCAAACAGTGCCTGTGTCAATAGGGGTCCCTGCCACCCTCAGGTGCTCCATGAAAGGAGAAGCGATCGGTAACTACTATATCAACTGGTACAGGAAGACCCAAGGTAACACAATGACTTTCATATACCGAGAAAAGGACATCTATGGCCCTGGTTTCAAAGACAATTTCCAAGGTGACATTGATATTGCAAAGAACCTGGCTGTACTTAAGATACTTGCACCATCAGAGAGAGATGAAGGGTCTTACTACTGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA",
        "AAGACCTCACTACAGT-1_contig_1": "GACTGAACTTTGAGCTTCAGGCAGCACAACTCACATTTGTGCAAAGAGCTACATGCCACATGCTGTTCTCCAGCCTGCTGTGTGTATTTGTGGCCTTCAGCTACTCTGGATCAAGTGTGGCCCAGAAGGTTACTCAAGCCCAGTCATCAGTATCCATGCCAGTGAGGAAAGCAGTCACCCTGAACTGCCTGTATGAAACAAGTTGGTGGTCATATTATATTTTTTGGTACAAGCAACTTCCCAGCAAAGAGATGATTTTCCTTATTCGCCAGGGTTCTGATGAACAGAATGCAAAAAGTGGTCGCTATTCTGTCAACTTCAAGAAAGCAGCGAAATCCGTCGCCTTAACCATTTCAGCCTTACAGCTAGAAGATTCAGCAAAGTACTTTTGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA",
    }
    return seq


@pytest.fixture
def fasta_10x_tr2():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        "AAGACCTCACTACAGT-1_contig_1": "GACTGAACTTTGAGCTTCAGGCAGCACAACTCACATTTGTGCAAAGAGCTACATGCCACATGCTGTTCTCCAGCCTGCTGTGTGTATTTGTGGCCTTCAGCTACTCTGGATCAAGTGTGGCCCAGAAGGTTACTCAAGCCCAGTCATCAGTATCCATGCCAGTGAGGAAAGCAGTCACCCTGAACTGCCTGTATGAAACAAGTTGGTGGTCATATTATATTTTTTGGTACAAGCAACTTCCCAGCAAAGAGATGATTTTCCTTATTCGCCAGGGTTCTGATGAACAGAATGCAAAAAGTGGTCGCTATTCTGTCAACTTCAAGAAAGCAGCGAAATCCGTCGCCTTAACCATTTCAGCCTTACAGCTAGAAGATTCAGCAAAGTACTTTTGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA",
        "AACTGGTTCTTTAGTC-1_contig_2": "AGGCAGAAGGTGGTTGAGAGGCAGAGCTGCCCCTGAGTGAGCCATGCAGAGGATCTCCTCCCTCATCCATCTCTCTCTCTTCTGGGCAGGAGTCATGTCAGCCATTGAGTTGGTGCCTGAACACCAAACAGTGCCTGTGTCAATAGGGGTCCCTGCCACCCTCAGGTGCTCCATGAAAGGAGAAGCGATCGGTAACTACTATATCAACTGGTACAGGAAGACCCAAGGTAACACAATGACTTTCATATACCGAGAAAAGGACATCTATGGCCCTGGTTTCAAAGACAATTTCCAAGGTGACATTGATATTGCAAAGAACCTGGCTGTACTTAAGATACTTGCACCATCAGAGAGAGATGAAGGGTCTTACTACTGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA",
        "AAAGATGCACCCTATC-1_contig_2": "TGGGGGAACTTTGAGCTTCAGGCAGCACAACTCACATTTGTGCAAAGAGCTACATGCCACATGCTGTTCTCCAGCCTGCTGTGTGTATTTGTGGCCTTCAGCTACTCTGGATCAAGTGTGGCCCAGAAGGTTACTCAAGCCCAGTCATCAGTATCCATGCCAGTGAGGAAAGCAGTCACCCTGAACTGCCTGTATGAAACAAGTTGGTGGTCATATTATATTTTTTGGTACAAGCAACTTCCCAGCAAAGAGATGATTTTCCTTATTCGCCAGGGTTCTGATGAACAGAATGCAAAAAGTGGTCGCTATTCTGTCAACTTCAAGAAAGCAGCGAAATCCGTCGCCTTAACCATTTCAGCCTTACAGCTAGAAGATTCAGCAAAGTACTTTTGTGCTCTTGGGGACCCCAGGCCTTCCTACAGGTACTGGGGGATACAGGGCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA",
    }
    return seq


@pytest.fixture
def annotation_10x_tr1():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame(
        [
            [
                "AACTGGTTCTTTAGTC-1",
                "false",
                "AACTGGTTCTTTAGTC-1_contig_2",
                "true",
                522,
                "TRD",
                "TRDV2",
                "",
                "TRDJ1",
                "TRDC",
                "true",
                "false",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "CACDILGDKADKLIF",
                "TGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTT",
                "",
                "",
                1176,
                6,
                "",
                "",
                "",
            ],
            [
                "AAGACCTCACTACAGT-1",
                "false",
                "AAGACCTCACTACAGT-1_contig_1",
                "true",
                544,
                "TRD",
                "TRDV1",
                "",
                "TRDJ1",
                "TRDC",
                "true",
                "false",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "",
                "CALGEPFLPGWGDKLIF",
                "TGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTT",
                "",
                "",
                2932,
                11,
                "",
                "",
                "",
            ],
        ],
        columns=[
            "barcode",
            "is_cell",
            "contig_id",
            "high_confidence",
            "length",
            "chain",
            "v_gene",
            "d_gene",
            "j_gene",
            "c_gene",
            "full_length",
            "productive",
            "fwr1",
            "fwr1_nt",
            "cdr1",
            "cdr1_nt",
            "fwr2",
            "fwr2_nt",
            "cdr2",
            "cdr2_nt",
            "fwr3",
            "fwr3_nt",
            "cdr3",
            "cdr3_nt",
            "fwr4",
            "fwr4_nt",
            "reads",
            "umis",
            "raw_clonotype_id",
            "raw_consensus_id",
            "exact_subclonotype_id",
        ],
    )

    return annot


@pytest.fixture
def json_10x_cr6():
    """cr6 10x json"""
    all_json = [
        {
            "barcode": "AAACCTGAGACTGTAA-1",
            "contig_name": "AAACCTGAGACTGTAA-1_contig_1",
            "sequence": "AAGAACCTGCCCAGCCTGGGCCTCAGGAAGCAGCATCGGAGGTGCCTCAGCCATGGCATGGATCCCTCTCTTCCTCGGCGTCCTTGCTTACTGCACAGGATCCGTGGCCTCCTATGAGCTGACTCAGCCACCCTCAGTGTCCGTGTCCCCAGGACAGACAGCCAGCATCACCTGCTCTGGAGATAAATTGGGGGATAAATATGCTTCGTGGTATCAGCAGAAGCCAGGCCAGTCCCCTGTGCTGGTCATCTATCAAGATAGCAAGCGGCCCTCAGGGATCCCTGAGCGATTCTCTGGCTCCAACTCTGGGAACACAGCCACTCTGACCATCAGCGGGACCCAGGCTATGGATGAGGGTGACTATTACTGTCAGGCGTGGGACAGCAGCACTGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 464,
            "umi_count": 16,
            "start_codon_pos": 52,
            "stop_codon_pos": None,
            "aa_sequence": "MAWIPLFLGVLAYCTGSVASYELTQPPSVSVSPGQTASITCSGDKLGDKYASWYQQKPGQSPVLVIYQDSKRPSGIPERFSGSNSGNTATLTISGTQAMDEGDYYCQAWDSSTVVFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
            "frame": None,
            "cdr3": "CQAWDSSTVVF",
            "cdr3_seq": "TGTCAGGCGTGGGACAGCAGCACTGTGGTATTC",
            "cdr3_start": 367,
            "cdr3_stop": 400,
            "fwr1": {
                "start": 109,
                "stop": 175,
                "nt_seq": "TCCTATGAGCTGACTCAGCCACCCTCAGTGTCCGTGTCCCCAGGACAGACAGCCAGCATCACCTGC",
                "aa_seq": "SYELTQPPSVSVSPGQTASITC",
            },
            "cdr1": {
                "start": 175,
                "stop": 208,
                "nt_seq": "TCTGGAGATAAATTGGGGGATAAATATGCTTCG",
                "aa_seq": "SGDKLGDKYAS",
            },
            "fwr2": {
                "start": 208,
                "stop": 253,
                "nt_seq": "TGGTATCAGCAGAAGCCAGGCCAGTCCCCTGTGCTGGTCATCTAT",
                "aa_seq": "WYQQKPGQSPVLVIY",
            },
            "cdr2": {
                "start": 253,
                "stop": 274,
                "nt_seq": "CAAGATAGCAAGCGGCCCTCA",
                "aa_seq": "QDSKRPS",
            },
            "fwr3": {
                "start": 274,
                "stop": 367,
                "nt_seq": "GGGATCCCTGAGCGATTCTCTGGCTCCAACTCTGGGAACACAGCCACTCTGACCATCAGCGGGACCCAGGCTATGGATGAGGGTGACTATTAC",
                "aa_seq": "GIPERFSGSNSGNTATLTISGTQAMDEGDYY",
            },
            "fwr4": {
                "start": 400,
                "stop": 428,
                "nt_seq": "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                "aa_seq": "GGGTKLTVL",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 52,
                    "annotation_match_start": 0,
                    "annotation_match_end": 52,
                    "annotation_length": 52,
                    "cigar": "52M587S",
                    "score": 104,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV3-1",
                        "feature_id": 348,
                        "gene_name": "IGLV3-1",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 52,
                    "contig_match_end": 392,
                    "annotation_match_start": 0,
                    "annotation_match_end": 340,
                    "annotation_length": 346,
                    "cigar": "52S340M247S",
                    "score": 665,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV3-1",
                        "feature_id": 349,
                        "gene_name": "IGLV3-1",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 390,
                    "contig_match_end": 428,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "390S38M211S",
                    "score": 76,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLJ2",
                        "feature_id": 311,
                        "gene_name": "IGLJ2",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 428,
                    "contig_match_end": 639,
                    "annotation_match_start": 0,
                    "annotation_match_end": 211,
                    "annotation_length": 317,
                    "cigar": "428S211M",
                    "score": 422,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLC2",
                        "feature_id": 306,
                        "gene_name": "IGLC2",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1691",
                "raw_consensus_id": "clonotype1691_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGAGACTGTAA-1",
            "contig_name": "AAACCTGAGACTGTAA-1_contig_2",
            "sequence": "ATGCTTTCTGAGAGTCATGGATCTCATGTGCAAGAAAATGAAGCACCTGTGGTTCTTCCTCCTGCTGGTGGCGGCTCCCAGATGGGTCCTGTCCCAGCTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGTCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGCAGTAGTAGTTACTACTGGGGCTGGATCCGCCAGCCCCCAGGGAAGGGGCTGGAGTGGATTGGGAATATCTATTATAGTGGGAGCACCTACTACAACCCGTCCCTCAAGAGTCGAGTCACCATATCCGTAGACACGTCCAAGAACCAATTCTCCCTGAAGGTGAACTCTGTGACCGCCGCAGACACGGCTGTGTATTACTGTGCGGTCCAGTATTACTATGATAGTAGTGGTTACCGGAGAACTGAGGTGCCCTTTGACTACTGGGGCCAGGGAACGCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 488,
            "umi_count": 30,
            "start_codon_pos": 16,
            "stop_codon_pos": None,
            "aa_sequence": "MDLMCKKMKHLWFFLLLVAAPRWVLSQLQLQESGPGLVKSSETLSLTCTVSGGSISSSSYYWGWIRQPPGKGLEWIGNIYYSGSTYYNPSLKSRVTISVDTSKNQFSLKVNSVTAADTAVYYCAVQYYYDSSGYRRTEVPFDYWGQGTLVTVSSGSASAPTLFPLVSCENSPSDTSSV",
            "frame": None,
            "cdr3": "CAVQYYYDSSGYRRTEVPFDYW",
            "cdr3_seq": "TGTGCGGTCCAGTATTACTATGATAGTAGTGGTTACCGGAGAACTGAGGTGCCCTTTGACTACTGG",
            "cdr3_start": 382,
            "cdr3_stop": 448,
            "fwr1": {
                "start": 94,
                "stop": 169,
                "nt_seq": "CAGCTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGTCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCT",
                "aa_seq": "QLQLQESGPGLVKSSETLSLTCTVS",
            },
            "cdr1": {
                "start": 169,
                "stop": 196,
                "nt_seq": "GGTGGCTCCATCAGCAGTAGTAGTTAC",
                "aa_seq": "GGSISSSSY",
            },
            "fwr2": {
                "start": 196,
                "stop": 253,
                "nt_seq": "TACTGGGGCTGGATCCGCCAGCCCCCAGGGAAGGGGCTGGAGTGGATTGGGAATATC",
                "aa_seq": "YWGWIRQPPGKGLEWIGNI",
            },
            "cdr2": {
                "start": 253,
                "stop": 268,
                "nt_seq": "TATTATAGTGGGAGC",
                "aa_seq": "YYSGS",
            },
            "fwr3": {
                "start": 268,
                "stop": 382,
                "nt_seq": "ACCTACTACAACCCGTCCCTCAAGAGTCGAGTCACCATATCCGTAGACACGTCCAAGAACCAATTCTCCCTGAAGGTGAACTCTGTGACCGCCGCAGACACGGCTGTGTATTAC",
                "aa_seq": "TYYNPSLKSRVTISVDTSKNQFSLKVNSVTAADTAVYY",
            },
            "fwr4": {
                "start": 448,
                "stop": 479,
                "nt_seq": "GGCCAGGGAACGCTGGTCACCGTCTCCTCAG",
                "aa_seq": "GQGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 16,
                    "contig_match_end": 393,
                    "annotation_match_start": 0,
                    "annotation_match_end": 377,
                    "annotation_length": 377,
                    "cigar": "16S377M157S",
                    "score": 714,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV4-39",
                        "feature_id": 190,
                        "gene_name": "IGHV4-39",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 393,
                    "contig_match_end": 417,
                    "annotation_match_start": 0,
                    "annotation_match_end": 24,
                    "annotation_length": 31,
                    "cigar": "393S24M133S",
                    "score": 48,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHD3-22",
                        "feature_id": 20,
                        "gene_name": "IGHD3-22",
                        "region_type": "D-REGION",
                    },
                },
                {
                    "contig_match_start": 431,
                    "contig_match_end": 479,
                    "annotation_match_start": 0,
                    "annotation_match_end": 48,
                    "annotation_length": 48,
                    "cigar": "431S48M71S",
                    "score": 71,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 55,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 479,
                    "contig_match_end": 550,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "479S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1691",
                "raw_consensus_id": "clonotype1691_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGAGACTGTAA-1",
            "contig_name": "AAACCTGAGACTGTAA-1_contig_3",
            "sequence": "CTCAGTAAGAGCTGAGCTCTTCCTGTGCTGTGAAAACAGACAAACCAACCAAGTCTCACCTAAGAAAATAAACACATGCTATGTCAGAGAAGCCTCAGGGTTTCCACACCTGCTCGAAAAGGGAGTTGAGCTTCAGCAGCTGACCCAGGACTCTGTTCCCCTTTGGTGAGAAGGGTTTTTGTTCAGCAAGACAATGGAGAGCTCTCACTGTGGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>+>>>>>>IIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 120,
            "umi_count": 3,
            "start_codon_pos": None,
            "stop_codon_pos": None,
            "aa_sequence": None,
            "frame": None,
            "cdr3": None,
            "cdr3_seq": None,
            "cdr3_start": None,
            "cdr3_stop": None,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 212,
                    "contig_match_end": 250,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "212S38M136S",
                    "score": 76,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ1",
                        "feature_id": 213,
                        "gene_name": "IGKJ1",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 250,
                    "contig_match_end": 386,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "250S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": False,
        },
        {
            "barcode": "AAACCTGAGGAATTAC-1",
            "contig_name": "AAACCTGAGGAATTAC-1_contig_1",
            "sequence": "GGGAGCTCTGGGAGAGGAGCCCCAGGCCCGGGATTCCCAGGTGTTTCCATTCAGTGATCAGCACTGAAGACAGAAGACTCATCATGGAGTTCTGGCTGAGCTGGGTTCTCCTTGTTGCCATTTTAAAAGATGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCTGGATTCACCTTCAGTAGCTATGCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAACTATTAGTAGTAATGGGGGTATCACAGACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTTCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTATATTACTGTGTGAAAGATCGAGAATACGGGGAGGTTGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": ">>]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 1101,
            "umi_count": 63,
            "start_codon_pos": 83,
            "stop_codon_pos": None,
            "aa_sequence": "MEFWLSWVLLVAILKDVQCEVQLVESGGGLVQPGGSLRLSCSASGFTFSSYAMHWVRQAPGKGLEYVSTISSNGGITDYADSVKGRFTIFRDNSKNTLYLQMSSLRAEDTAVYYCVKDREYGEVDYWGQGTLVTVSSGSASAPTLFPLVSCENSPSDTSSV",
            "frame": None,
            "cdr3": "CVKDREYGEVDYW",
            "cdr3_seq": "TGTGTGAAAGATCGAGAATACGGGGAGGTTGACTACTGG",
            "cdr3_start": 425,
            "cdr3_stop": 464,
            "fwr1": {
                "start": 140,
                "stop": 215,
                "nt_seq": "GAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCT",
                "aa_seq": "EVQLVESGGGLVQPGGSLRLSCSAS",
            },
            "cdr1": {
                "start": 215,
                "stop": 236,
                "nt_seq": "GGATTCACCTTCAGTAGCTAT",
                "aa_seq": "GFTFSSY",
            },
            "fwr2": {
                "start": 236,
                "stop": 293,
                "nt_seq": "GCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAACTATT",
                "aa_seq": "AMHWVRQAPGKGLEYVSTI",
            },
            "cdr2": {
                "start": 293,
                "stop": 311,
                "nt_seq": "AGTAGTAATGGGGGTATC",
                "aa_seq": "SSNGGI",
            },
            "fwr3": {
                "start": 311,
                "stop": 425,
                "nt_seq": "ACAGACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTTCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTATATTAC",
                "aa_seq": "TDYADSVKGRFTIFRDNSKNTLYLQMSSLRAEDTAVYY",
            },
            "fwr4": {
                "start": 464,
                "stop": 495,
                "nt_seq": "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                "aa_seq": "GQGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 83,
                    "contig_match_end": 436,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "83S353M130S",
                    "score": 681,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-64D",
                        "feature_id": 151,
                        "gene_name": "IGHV3-64D",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 447,
                    "contig_match_end": 495,
                    "annotation_match_start": 0,
                    "annotation_match_end": 48,
                    "annotation_length": 48,
                    "cigar": "447S48M71S",
                    "score": 71,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 55,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 495,
                    "contig_match_end": 566,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "495S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype686",
                "raw_consensus_id": "clonotype686_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGAGGAATTAC-1",
            "contig_name": "AAACCTGAGGAATTAC-1_contig_2",
            "sequence": "GGAGTCAGTCTCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTACTCTGGCTCCGAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCGGGCAAGTCAGAGCATTAGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCGTCCAGTTTGCAAAGTGGGGTCCCATCGAGGTTCAGTGGCAGTATATCTGGGACAGATTTCACTCTCACCATCAGCAGTCTGCAACCTGAAGATTTTGCAACTTACTACTGTCAACAGAGTTACAGTACCCCTCGAACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 2418,
            "umi_count": 105,
            "start_codon_pos": 26,
            "stop_codon_pos": None,
            "aa_sequence": "MDMRVPAQLLGLLLLWLRGARCDIQMTQSPSSLSASVGDRVTITCRASQSISNYLNWYQQKPGKAPKLLIYAASSLQSGVPSRFSGSISGTDFTLTISSLQPEDFATYYCQQSYSTPRTFGQGTKVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
            "frame": None,
            "cdr3": "CQQSYSTPRTF",
            "cdr3_seq": "TGTCAACAGAGTTACAGTACCCCTCGAACGTTC",
            "cdr3_start": 353,
            "cdr3_stop": 386,
            "fwr1": {
                "start": 92,
                "stop": 161,
                "nt_seq": "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGC",
                "aa_seq": "DIQMTQSPSSLSASVGDRVTITC",
            },
            "cdr1": {
                "start": 161,
                "stop": 194,
                "nt_seq": "CGGGCAAGTCAGAGCATTAGCAACTATTTAAAT",
                "aa_seq": "RASQSISNYLN",
            },
            "fwr2": {
                "start": 194,
                "stop": 239,
                "nt_seq": "TGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT",
                "aa_seq": "WYQQKPGKAPKLLIY",
            },
            "cdr2": {
                "start": 239,
                "stop": 260,
                "nt_seq": "GCTGCGTCCAGTTTGCAAAGT",
                "aa_seq": "AASSLQS",
            },
            "fwr3": {
                "start": 260,
                "stop": 353,
                "nt_seq": "GGGGTCCCATCGAGGTTCAGTGGCAGTATATCTGGGACAGATTTCACTCTCACCATCAGCAGTCTGCAACCTGAAGATTTTGCAACTTACTAC",
                "aa_seq": "GVPSRFSGSISGTDFTLTISSLQPEDFATYY",
            },
            "fwr4": {
                "start": 386,
                "stop": 414,
                "nt_seq": "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                "aa_seq": "GQGTKVEIK",
            },
            "annotations": [
                {
                    "contig_match_start": 26,
                    "contig_match_end": 379,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "26S353M171S",
                    "score": 676,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV1D-39",
                        "feature_id": 253,
                        "gene_name": "IGKV1D-39",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 376,
                    "contig_match_end": 414,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "376S38M136S",
                    "score": 61,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ1",
                        "feature_id": 213,
                        "gene_name": "IGKJ1",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 414,
                    "contig_match_end": 550,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "414S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype686",
                "raw_consensus_id": "clonotype686_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGAGGAATTAC-1",
            "contig_name": "AAACCTGAGGAATTAC-1_contig_3",
            "sequence": "GGGACTCCTGGGGGTGACTTAGCAGCCGTCTGCTTGCAGTTGGACTTCCCAGGCCGACAGTGGTCTGGCTTCTGAGGGGTCAGGCCAGAATGTGGGGTACGTGGGAGGCCAGCAGAGGGTTCCATGAGAAGGGCAGGACAGGGCCACGGACAGTCAGCTTCCATGTGACGCCCGGAGACAGAAGGTCTCTGGGTGGCTGGGTTTTTGTGGGGTGAGGATGGACATTCTGCCATTGTGATTACTACTACTACTACGGTATGGACGTCTGGGGCCAAGGGACCACGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": "IIIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 463,
            "umi_count": 11,
            "start_codon_pos": None,
            "stop_codon_pos": None,
            "aa_sequence": None,
            "frame": None,
            "cdr3": None,
            "cdr3_seq": None,
            "cdr3_start": None,
            "cdr3_stop": None,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 237,
                    "contig_match_end": 300,
                    "annotation_match_start": 0,
                    "annotation_match_end": 63,
                    "annotation_length": 63,
                    "cigar": "237S63M71S",
                    "score": 126,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ6",
                        "feature_id": 737,
                        "gene_name": "IGHJ6",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 300,
                    "contig_match_end": 371,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "300S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": False,
        },
        {
            "barcode": "AAACCTGAGTCAATAG-1",
            "contig_name": "AAACCTGAGTCAATAG-1_contig_1",
            "sequence": "TCAGCTTCAGCTGTGGTAGAGAAGACAGGATTCAGGACAATCTCCAGCATGGCCGGCTTCCCTCTCCTCCTCACCCTCCTCACTCACTGTGCAGGGTCCTGGGCCCAGTCTGTGCTGACTCAGCCACCCTCAGCGTCTGGGACCCCCGGGCAGAGGGTCACCATCTCTTGTTCTGGAAGCAGCTCCAACATCGGAAGTAATTATGTATACTGGTACCAGCAGCTCCCAGGAACGGCCCCCAAACTCCTCATCTATAGGAATAATCAGCGGCCCTCAGGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCAGTGGGCTCCGGTCCGAGGATGAGGCTGATTATTACTGTGCAGCATGGGATGACAGCCTGAGTGGTGTGCTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 1877,
            "umi_count": 118,
            "start_codon_pos": 48,
            "stop_codon_pos": None,
            "aa_sequence": "MAGFPLLLTLLTHCAGSWAQSVLTQPPSASGTPGQRVTISCSGSSSNIGSNYVYWYQQLPGTAPKLLIYRNNQRPSGVPDRFSGSKSGTSASLAISGLRSEDEADYYCAAWDDSLSGVLFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
            "frame": None,
            "cdr3": "CAAWDDSLSGVLF",
            "cdr3_seq": "TGTGCAGCATGGGATGACAGCCTGAGTGGTGTGCTATTC",
            "cdr3_start": 369,
            "cdr3_stop": 408,
            "fwr1": {
                "start": 105,
                "stop": 171,
                "nt_seq": "CAGTCTGTGCTGACTCAGCCACCCTCAGCGTCTGGGACCCCCGGGCAGAGGGTCACCATCTCTTGT",
                "aa_seq": "QSVLTQPPSASGTPGQRVTISC",
            },
            "cdr1": {
                "start": 171,
                "stop": 210,
                "nt_seq": "TCTGGAAGCAGCTCCAACATCGGAAGTAATTATGTATAC",
                "aa_seq": "SGSSSNIGSNYVY",
            },
            "fwr2": {
                "start": 210,
                "stop": 255,
                "nt_seq": "TGGTACCAGCAGCTCCCAGGAACGGCCCCCAAACTCCTCATCTAT",
                "aa_seq": "WYQQLPGTAPKLLIY",
            },
            "cdr2": {
                "start": 255,
                "stop": 276,
                "nt_seq": "AGGAATAATCAGCGGCCCTCA",
                "aa_seq": "RNNQRPS",
            },
            "fwr3": {
                "start": 276,
                "stop": 369,
                "nt_seq": "GGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCAGTGGGCTCCGGTCCGAGGATGAGGCTGATTATTAC",
                "aa_seq": "GVPDRFSGSKSGTSASLAISGLRSEDEADYY",
            },
            "fwr4": {
                "start": 408,
                "stop": 436,
                "nt_seq": "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                "aa_seq": "GGGTKLTVL",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 48,
                    "annotation_match_start": 8,
                    "annotation_match_end": 56,
                    "annotation_length": 56,
                    "cigar": "48M599S",
                    "score": 96,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV1-47",
                        "feature_id": 326,
                        "gene_name": "IGLV1-47",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 48,
                    "contig_match_end": 399,
                    "annotation_match_start": 0,
                    "annotation_match_end": 351,
                    "annotation_length": 351,
                    "cigar": "48S351M248S",
                    "score": 697,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV1-47",
                        "feature_id": 327,
                        "gene_name": "IGLV1-47",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 398,
                    "contig_match_end": 436,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "398S38M211S",
                    "score": 71,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLJ2",
                        "feature_id": 311,
                        "gene_name": "IGLJ2",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 436,
                    "contig_match_end": 647,
                    "annotation_match_start": 0,
                    "annotation_match_end": 211,
                    "annotation_length": 317,
                    "cigar": "436S211M",
                    "score": 422,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLC2",
                        "feature_id": 306,
                        "gene_name": "IGLC2",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1317",
                "raw_consensus_id": "clonotype1317_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGAGTCAATAG-1",
            "contig_name": "AAACCTGAGTCAATAG-1_contig_2",
            "sequence": "GAGCTCTGGGAGAGGAGCCCAGCACTGGAAGTCGCCGGTGTTTCCATTCGGTGATCAGCACTGAACACAGAGGACTCACCATGGAGTTTGGGCTGAGCTGGGTTTTCCTCGTCGCTCTTTTAAGAGGTGTCCAGTGTCAGGTGCAGCTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCTGGATTCACCTTCAGTAGCTATGCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATATCGTATGATGGAAGTCATAAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATATGCAGATGAACAGCCTGAGAGCTGAGGACACGGCTGTGTATTTCTGTGCGACAGATCGAGGGACTGGTTACTGTGATAGTAGTGGTTATTACCCTGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCCTCCAAGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
            "quals": ">>>]]]]]]]]]]]]]]]]]]]]]]]]]]\\]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 375,
            "umi_count": 23,
            "start_codon_pos": 80,
            "stop_codon_pos": None,
            "aa_sequence": "MEFGLSWVFLVALLRGVQCQVQLVESGGGVVQPGRSLRLSCAASGFTFSSYAMHWVRQAPGKGLEWVAVISYDGSHKYYADSVKGRFTISRDNSKNTLYMQMNSLRAEDTAVYFCATDRGTGYCDSSGYYPDYWGQGTLVTVSSASTKGPSVFPLAPSSKSTSGGTAALGCLVKDYFPEPVTVSWNSGALTSGVHTFPAVLQSSG",
            "frame": None,
            "cdr3": "CATDRGTGYCDSSGYYPDYW",
            "cdr3_seq": "TGTGCGACAGATCGAGGGACTGGTTACTGTGATAGTAGTGGTTATTACCCTGACTACTGG",
            "cdr3_start": 422,
            "cdr3_stop": 482,
            "fwr1": {
                "start": 137,
                "stop": 212,
                "nt_seq": "CAGGTGCAGCTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCT",
                "aa_seq": "QVQLVESGGGVVQPGRSLRLSCAAS",
            },
            "cdr1": {
                "start": 212,
                "stop": 233,
                "nt_seq": "GGATTCACCTTCAGTAGCTAT",
                "aa_seq": "GFTFSSY",
            },
            "fwr2": {
                "start": 233,
                "stop": 290,
                "nt_seq": "GCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATA",
                "aa_seq": "AMHWVRQAPGKGLEWVAVI",
            },
            "cdr2": {
                "start": 290,
                "stop": 308,
                "nt_seq": "TCGTATGATGGAAGTCAT",
                "aa_seq": "SYDGSH",
            },
            "fwr3": {
                "start": 308,
                "stop": 422,
                "nt_seq": "AAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATATGCAGATGAACAGCCTGAGAGCTGAGGACACGGCTGTGTATTTC",
                "aa_seq": "KYYADSVKGRFTISRDNSKNTLYMQMNSLRAEDTAVYF",
            },
            "fwr4": {
                "start": 482,
                "stop": 513,
                "nt_seq": "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                "aa_seq": "GQGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 80,
                    "contig_match_end": 433,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "80S353M262S",
                    "score": 661,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-33",
                        "feature_id": 128,
                        "gene_name": "IGHV3-33",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 442,
                    "contig_match_end": 473,
                    "annotation_match_start": 0,
                    "annotation_match_end": 31,
                    "annotation_length": 31,
                    "cigar": "442S31M222S",
                    "score": 27,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHD3-22",
                        "feature_id": 20,
                        "gene_name": "IGHD3-22",
                        "region_type": "D-REGION",
                    },
                },
                {
                    "contig_match_start": 465,
                    "contig_match_end": 513,
                    "annotation_match_start": 0,
                    "annotation_match_end": 48,
                    "annotation_length": 48,
                    "cigar": "465S48M182S",
                    "score": 81,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 55,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 513,
                    "contig_match_end": 695,
                    "annotation_match_start": 0,
                    "annotation_match_end": 182,
                    "annotation_length": 884,
                    "cigar": "513S182M",
                    "score": 359,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHG1",
                        "feature_id": 40,
                        "gene_name": "IGHG1",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1317",
                "raw_consensus_id": "clonotype1317_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGAGTCAATAG-1",
            "contig_name": "AAACCTGAGTCAATAG-1_contig_3",
            "sequence": "GAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAACAGTGGTGGCACAAACTCTGCACAGAAGTTTCAGGGCTGGGTCACCATGACCAGGGACACGTCCATCAGCACATCGGGAGGGACTACGGTGACAATCTTACTACTACTACGGTATGGACGTCTGGGGCCAAGGGACCACGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCCTCCAAGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
            "quals": ">]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 211,
            "umi_count": 20,
            "start_codon_pos": 62,
            "stop_codon_pos": None,
            "aa_sequence": "MDWTWRILFLVAAATGAHSQVQLVQSGAEVKKPGASVKVSCKASGYTFTGYYMHWVRQAPGQGLEWMGWINPNSGGTNSAQKFQGWVTMTRDTSISTSGGTTVTILLLLRYGRLGPRDHGHRLLSLHQGPIGLPPGTLLQEHLWGHSGPGLPGQGLLPRTGDGVVELRRPDQRRAHLPGCPTVLR",
            "frame": None,
            "cdr3": "CLVKDYFPEPVTVSWNSGALTS",
            "cdr3_seq": "TGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGC",
            "cdr3_start": 513,
            "cdr3_stop": 579,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 62,
                    "annotation_match_start": 2,
                    "annotation_match_end": 64,
                    "annotation_length": 64,
                    "cigar": "62M556S",
                    "score": 124,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-2",
                        "feature_id": 66,
                        "gene_name": "IGHV1-2",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 62,
                    "contig_match_end": 353,
                    "annotation_match_start": 0,
                    "annotation_match_end": 291,
                    "annotation_length": 353,
                    "cigar": "62S291M265S",
                    "score": 577,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-2",
                        "feature_id": 67,
                        "gene_name": "IGHV1-2",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 373,
                    "contig_match_end": 436,
                    "annotation_match_start": 0,
                    "annotation_match_end": 63,
                    "annotation_length": 63,
                    "cigar": "373S63M182S",
                    "score": 111,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ6",
                        "feature_id": 737,
                        "gene_name": "IGHJ6",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 436,
                    "contig_match_end": 618,
                    "annotation_match_start": 0,
                    "annotation_match_end": 182,
                    "annotation_length": 884,
                    "cigar": "436S182M",
                    "score": 359,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHG1",
                        "feature_id": 40,
                        "gene_name": "IGHG1",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGCATTTCACT-1",
            "contig_name": "AAACCTGCATTTCACT-1_contig_1",
            "sequence": "AGAGCTCTGGAGAAGAGCTGCTCAGTTAGGACCCAGAGGGAACCATGGAAACCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACCGGAGAAATTGTGTTGACGCAGTCTCCAGGCACTGTGTCTTTGTCTCCAGGGGATAGAGCCTCCCTCTACTGCAGGGCCACTGAGAGTGTTGACGTCAGCAACTTAGCCTGGTACCAGCTGAAAGTTGGCCAGGCTCCCAGGCTCCTCATCTATGGTGCATCCAAGAGGGCCACTGGGGTCCCAGACAGGTTCGCTGGCAGTGGGTCTGGGACAGAATTCACTCTCACCATCAGCAGACTGGAGCCTGAAGACTTTGCAGTTTATTACTGTCAACACTATGGGCGCTCACCCTGGACGTTCGGCCAAGGGACCAGGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": "IIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 1306,
            "umi_count": 41,
            "start_codon_pos": 44,
            "stop_codon_pos": None,
            "aa_sequence": "METPAQLLFLLLLWLPDTTGEIVLTQSPGTVSLSPGDRASLYCRATESVDVSNLAWYQLKVGQAPRLLIYGASKRATGVPDRFAGSGSGTEFTLTISRLEPEDFAVYYCQHYGRSPWTFGQGTRVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
            "frame": None,
            "cdr3": "CQHYGRSPWTF",
            "cdr3_seq": "TGTCAACACTATGGGCGCTCACCCTGGACGTTC",
            "cdr3_start": 368,
            "cdr3_stop": 401,
            "fwr1": {
                "start": 104,
                "stop": 173,
                "nt_seq": "GAAATTGTGTTGACGCAGTCTCCAGGCACTGTGTCTTTGTCTCCAGGGGATAGAGCCTCCCTCTACTGC",
                "aa_seq": "EIVLTQSPGTVSLSPGDRASLYC",
            },
            "cdr1": {
                "start": 173,
                "stop": 209,
                "nt_seq": "AGGGCCACTGAGAGTGTTGACGTCAGCAACTTAGCC",
                "aa_seq": "RATESVDVSNLA",
            },
            "fwr2": {
                "start": 209,
                "stop": 254,
                "nt_seq": "TGGTACCAGCTGAAAGTTGGCCAGGCTCCCAGGCTCCTCATCTAT",
                "aa_seq": "WYQLKVGQAPRLLIY",
            },
            "cdr2": {
                "start": 254,
                "stop": 275,
                "nt_seq": "GGTGCATCCAAGAGGGCCACT",
                "aa_seq": "GASKRAT",
            },
            "fwr3": {
                "start": 275,
                "stop": 368,
                "nt_seq": "GGGGTCCCAGACAGGTTCGCTGGCAGTGGGTCTGGGACAGAATTCACTCTCACCATCAGCAGACTGGAGCCTGAAGACTTTGCAGTTTATTAC",
                "aa_seq": "GVPDRFAGSGSGTEFTLTISRLEPEDFAVYY",
            },
            "fwr4": {
                "start": 401,
                "stop": 429,
                "nt_seq": "GGCCAAGGGACCAGGGTGGAAATCAAAC",
                "aa_seq": "GQGTRVEIK",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 44,
                    "annotation_match_start": 8,
                    "annotation_match_end": 52,
                    "annotation_length": 52,
                    "cigar": "44M521S",
                    "score": 88,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-20",
                        "feature_id": 282,
                        "gene_name": "IGKV3-20",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 44,
                    "contig_match_end": 392,
                    "annotation_match_start": 0,
                    "annotation_match_end": 348,
                    "annotation_length": 348,
                    "cigar": "44S348M173S",
                    "score": 551,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-20",
                        "feature_id": 283,
                        "gene_name": "IGKV3-20",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 391,
                    "contig_match_end": 429,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "391S38M136S",
                    "score": 66,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ1",
                        "feature_id": 213,
                        "gene_name": "IGKJ1",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 429,
                    "contig_match_end": 565,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "429S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1692",
                "raw_consensus_id": "clonotype1692_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGCATTTCACT-1",
            "contig_name": "AAACCTGCATTTCACT-1_contig_2",
            "sequence": "GCTCTGCTTCAGCTGTGGGCACAAGAGGCAGCACTCAGGACAATCTCCAGCATGGCCTGGTCTCCTCTCCTCCTCACTCTCCTCGCTCACTGCACAGGGTCCTGGGCCCAGTCTGTGCTGACGCAGCCGCCCTCAGTGTCTGGGGCCCCAGGGCAGAGGGTCACCATCTCCTGCACTGGGAGCAGCTCCAACATCGGGGCAGGTTATGATGTACACTGGTACCAGCAGCTTCCAGGAACAGCCCCCAAACTCCTCATCTATGGTAACAGCAATCGGCCCTCAGGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCACTGGGCTCCAGGCTGAGGATGAGGCTGATTATTACTGCCAGTCCTATGACAGCACCGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 190,
            "umi_count": 11,
            "start_codon_pos": 51,
            "stop_codon_pos": None,
            "aa_sequence": "MAWSPLLLTLLAHCTGSWAQSVLTQPPSVSGAPGQRVTISCTGSSSNIGAGYDVHWYQQLPGTAPKLLIYGNSNRPSGVPDRFSGSKSGTSASLAITGLQAEDEADYYCQSYDSTVFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
            "frame": None,
            "cdr3": "CQSYDSTVF",
            "cdr3_seq": "TGCCAGTCCTATGACAGCACCGTGTTC",
            "cdr3_start": 375,
            "cdr3_stop": 402,
            "fwr1": {
                "start": 108,
                "stop": 174,
                "nt_seq": "CAGTCTGTGCTGACGCAGCCGCCCTCAGTGTCTGGGGCCCCAGGGCAGAGGGTCACCATCTCCTGC",
                "aa_seq": "QSVLTQPPSVSGAPGQRVTISC",
            },
            "cdr1": {
                "start": 174,
                "stop": 216,
                "nt_seq": "ACTGGGAGCAGCTCCAACATCGGGGCAGGTTATGATGTACAC",
                "aa_seq": "TGSSSNIGAGYDVH",
            },
            "fwr2": {
                "start": 216,
                "stop": 261,
                "nt_seq": "TGGTACCAGCAGCTTCCAGGAACAGCCCCCAAACTCCTCATCTAT",
                "aa_seq": "WYQQLPGTAPKLLIY",
            },
            "cdr2": {
                "start": 261,
                "stop": 282,
                "nt_seq": "GGTAACAGCAATCGGCCCTCA",
                "aa_seq": "GNSNRPS",
            },
            "fwr3": {
                "start": 282,
                "stop": 375,
                "nt_seq": "GGGGTCCCTGACCGATTCTCTGGCTCCAAGTCTGGCACCTCAGCCTCCCTGGCCATCACTGGGCTCCAGGCTGAGGATGAGGCTGATTATTAC",
                "aa_seq": "GVPDRFSGSKSGTSASLAITGLQAEDEADYY",
            },
            "fwr4": {
                "start": 402,
                "stop": 430,
                "nt_seq": "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                "aa_seq": "GGGTKLTVL",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 51,
                    "annotation_match_start": 0,
                    "annotation_match_end": 51,
                    "annotation_length": 51,
                    "cigar": "51M590S",
                    "score": 102,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV1-40",
                        "feature_id": 320,
                        "gene_name": "IGLV1-40",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 51,
                    "contig_match_end": 394,
                    "annotation_match_start": 0,
                    "annotation_match_end": 343,
                    "annotation_length": 356,
                    "cigar": "51S343M247S",
                    "score": 686,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV1-40",
                        "feature_id": 321,
                        "gene_name": "IGLV1-40",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 392,
                    "contig_match_end": 430,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "392S38M211S",
                    "score": 56,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLJ3",
                        "feature_id": 313,
                        "gene_name": "IGLJ3",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 430,
                    "contig_match_end": 641,
                    "annotation_match_start": 0,
                    "annotation_match_end": 211,
                    "annotation_length": 317,
                    "cigar": "430S211M",
                    "score": 422,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLC2",
                        "feature_id": 306,
                        "gene_name": "IGLC2",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1692",
                "raw_consensus_id": "clonotype1692_consensus_3",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGCATTTCACT-1",
            "contig_name": "AAACCTGCATTTCACT-1_contig_3",
            "sequence": "CCACATCCCTCCTCAGAAGCCCCCAGAGCACAACGCCTCACCATGGACTGGACCTGGAGGATCCTCTTTTTGGTGGCAGCAGCCACAGGTGCCCACTCCCAGGTCCAGCTTGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTTTCCTGCAAGGCTTCTGGATACACCTTCACTAGCTATGCTATGCATTGGGTGCGCCAGGCCCCCGGACAAAGGCTTGAGTGGATGGGATGGATCAACGCTGGCAATGGTAACACAAAATATTCACAGAAGTTCCAGGGCAGAGTCACCATTACCAGGGACACATCCGCGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAAGACACGGCTGTGTATTACTGTGCGAGAGATTATCGCTCTTCCAAGCGTTATCAGCCGCATATTGTAGTAGTACCAGCTGCTATGGGTAGCGGGGCTCTTGGGTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 363,
            "umi_count": 17,
            "start_codon_pos": 42,
            "stop_codon_pos": None,
            "aa_sequence": "MDWTWRILFLVAAATGAHSQVQLVQSGAEVKKPGASVKVSCKASGYTFTSYAMHWVRQAPGQRLEWMGWINAGNGNTKYSQKFQGRVTITRDTSASTAYMELSSLRSEDTAVYYCARDYRSSKRYQPHIVVVPAAMGSGALGYWGQGTLVTVSSGSASAPTLFPLVSCENSPSDTSSV",
            "frame": None,
            "cdr3": "CARDYRSSKRYQPHIVVVPAAMGSGALGYW",
            "cdr3_seq": "TGTGCGAGAGATTATCGCTCTTCCAAGCGTTATCAGCCGCATATTGTAGTAGTACCAGCTGCTATGGGTAGCGGGGCTCTTGGGTACTGG",
            "cdr3_start": 384,
            "cdr3_stop": 474,
            "fwr1": {
                "start": 99,
                "stop": 174,
                "nt_seq": "CAGGTCCAGCTTGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTTTCCTGCAAGGCTTCT",
                "aa_seq": "QVQLVQSGAEVKKPGASVKVSCKAS",
            },
            "cdr1": {
                "start": 174,
                "stop": 195,
                "nt_seq": "GGATACACCTTCACTAGCTAT",
                "aa_seq": "GYTFTSY",
            },
            "fwr2": {
                "start": 195,
                "stop": 252,
                "nt_seq": "GCTATGCATTGGGTGCGCCAGGCCCCCGGACAAAGGCTTGAGTGGATGGGATGGATC",
                "aa_seq": "AMHWVRQAPGQRLEWMGWI",
            },
            "cdr2": {
                "start": 252,
                "stop": 270,
                "nt_seq": "AACGCTGGCAATGGTAAC",
                "aa_seq": "NAGNGN",
            },
            "fwr3": {
                "start": 270,
                "stop": 384,
                "nt_seq": "ACAAAATATTCACAGAAGTTCCAGGGCAGAGTCACCATTACCAGGGACACATCCGCGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAAGACACGGCTGTGTATTAC",
                "aa_seq": "TKYSQKFQGRVTITRDTSASTAYMELSSLRSEDTAVYY",
            },
            "fwr4": {
                "start": 474,
                "stop": 505,
                "nt_seq": "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                "aa_seq": "GQGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 42,
                    "annotation_match_start": 0,
                    "annotation_match_end": 42,
                    "annotation_length": 42,
                    "cigar": "42M534S",
                    "score": 84,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-3",
                        "feature_id": 72,
                        "gene_name": "IGHV1-3",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 42,
                    "contig_match_end": 395,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "42S353M181S",
                    "score": 706,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-3",
                        "feature_id": 73,
                        "gene_name": "IGHV1-3",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 421,
                    "contig_match_end": 452,
                    "annotation_match_start": 0,
                    "annotation_match_end": 31,
                    "annotation_length": 31,
                    "cigar": "421S31M124S",
                    "score": 42,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHD2-2",
                        "feature_id": 14,
                        "gene_name": "IGHD2-2",
                        "region_type": "D-REGION",
                    },
                },
                {
                    "contig_match_start": 468,
                    "contig_match_end": 505,
                    "annotation_match_start": 9,
                    "annotation_match_end": 46,
                    "annotation_length": 46,
                    "cigar": "468S37M71S",
                    "score": 74,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 54,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 505,
                    "contig_match_end": 576,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "505S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1692",
                "raw_consensus_id": "clonotype1692_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGGTAACGTTC-1",
            "contig_name": "AAACCTGGTAACGTTC-1_contig_1",
            "sequence": "GCCTGGGTCAGAGCTCTGGGGAGGAACTGCTCAGTTAGGACCCAGACGGAACCATGGAAGCCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACTGGAGAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGCAGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCCTGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTATGGTGCATCCACCAGGGCCACTGGCATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTACTGTCAGCAGTATAATAACTGGCCTCAATTCACTTTCGGCCCTGGGACCAAAGTGGATATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": ">>>>>>>>>]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 640,
            "umi_count": 27,
            "start_codon_pos": 53,
            "stop_codon_pos": None,
            "aa_sequence": "MEAPAQLLFLLLLWLPDTTGEIVMTQSPATLSVSPGERATLSCRASQSVSSNLAWYQQKPGQAPRLLIYGASTRATGIPARFSGSGSGTEFTLTISSLQSEDFAVYYCQQYNNWPQFTFGPGTKVDIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
            "frame": None,
            "cdr3": "CQQYNNWPQFTF",
            "cdr3_seq": "TGTCAGCAGTATAATAACTGGCCTCAATTCACTTTC",
            "cdr3_start": 374,
            "cdr3_stop": 410,
            "fwr1": {
                "start": 113,
                "stop": 182,
                "nt_seq": "GAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGC",
                "aa_seq": "EIVMTQSPATLSVSPGERATLSC",
            },
            "cdr1": {
                "start": 182,
                "stop": 215,
                "nt_seq": "AGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCC",
                "aa_seq": "RASQSVSSNLA",
            },
            "fwr2": {
                "start": 215,
                "stop": 260,
                "nt_seq": "TGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTAT",
                "aa_seq": "WYQQKPGQAPRLLIY",
            },
            "cdr2": {
                "start": 260,
                "stop": 281,
                "nt_seq": "GGTGCATCCACCAGGGCCACT",
                "aa_seq": "GASTRAT",
            },
            "fwr3": {
                "start": 281,
                "stop": 374,
                "nt_seq": "GGCATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTAC",
                "aa_seq": "GIPARFSGSGSGTEFTLTISSLQSEDFAVYY",
            },
            "fwr4": {
                "start": 410,
                "stop": 438,
                "nt_seq": "GGCCCTGGGACCAAAGTGGATATCAAAC",
                "aa_seq": "GPGTKVDIK",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 53,
                    "annotation_match_start": 44,
                    "annotation_match_end": 97,
                    "annotation_length": 97,
                    "cigar": "53M521S",
                    "score": 106,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-15",
                        "feature_id": 280,
                        "gene_name": "IGKV3-15",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 53,
                    "contig_match_end": 398,
                    "annotation_match_start": 0,
                    "annotation_match_end": 345,
                    "annotation_length": 345,
                    "cigar": "53S345M176S",
                    "score": 685,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-15",
                        "feature_id": 281,
                        "gene_name": "IGKV3-15",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 400,
                    "contig_match_end": 438,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "400S38M136S",
                    "score": 76,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ3",
                        "feature_id": 215,
                        "gene_name": "IGKJ3",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 438,
                    "contig_match_end": 574,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "438S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1016",
                "raw_consensus_id": "clonotype1016_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGGTAACGTTC-1",
            "contig_name": "AAACCTGGTAACGTTC-1_contig_2",
            "sequence": "GGAGTCTCCCTCACTGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACGCCAGTGGGGCAGCAGCTGGGTCGGGTGGTTCGACCCCTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 536,
            "umi_count": 30,
            "start_codon_pos": 59,
            "stop_codon_pos": None,
            "aa_sequence": "MGSTAILALLLAVLQGVCAEVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCARRQWGSSWVGWFDPWGQGTLVTVSSGSASAPTLFPLVSCENSPSDTSSV",
            "frame": None,
            "cdr3": "CARRQWGSSWVGWFDPW",
            "cdr3_seq": "TGTGCGAGACGCCAGTGGGGCAGCAGCTGGGTCGGGTGGTTCGACCCCTGG",
            "cdr3_start": 401,
            "cdr3_stop": 452,
            "fwr1": {
                "start": 116,
                "stop": 191,
                "nt_seq": "GAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCT",
                "aa_seq": "EVQLVQSGAEVKKPGESLKISCKGS",
            },
            "cdr1": {
                "start": 191,
                "stop": 212,
                "nt_seq": "GGATACAGCTTTACCAGCTAC",
                "aa_seq": "GYSFTSY",
            },
            "fwr2": {
                "start": 212,
                "stop": 269,
                "nt_seq": "TGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATC",
                "aa_seq": "WIGWVRQMPGKGLEWMGII",
            },
            "cdr2": {
                "start": 269,
                "stop": 287,
                "nt_seq": "TATCCTGGTGACTCTGAT",
                "aa_seq": "YPGDSD",
            },
            "fwr3": {
                "start": 287,
                "stop": 401,
                "nt_seq": "ACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTAC",
                "aa_seq": "TRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYY",
            },
            "fwr4": {
                "start": 452,
                "stop": 483,
                "nt_seq": "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                "aa_seq": "GQGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 59,
                    "annotation_match_start": 0,
                    "annotation_match_end": 59,
                    "annotation_length": 59,
                    "cigar": "59M495S",
                    "score": 113,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV5-51",
                        "feature_id": 202,
                        "gene_name": "IGHV5-51",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 59,
                    "contig_match_end": 412,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "59S353M142S",
                    "score": 701,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV5-51",
                        "feature_id": 203,
                        "gene_name": "IGHV5-51",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 432,
                    "contig_match_end": 483,
                    "annotation_match_start": 0,
                    "annotation_match_end": 51,
                    "annotation_length": 51,
                    "cigar": "432S51M71S",
                    "score": 82,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ5",
                        "feature_id": 57,
                        "gene_name": "IGHJ5",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 483,
                    "contig_match_end": 554,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "483S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1016",
                "raw_consensus_id": "clonotype1016_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGTCCAAGTAC-1",
            "contig_name": "AAACCTGTCCAAGTAC-1_contig_1",
            "sequence": "GCTGGGGTCTCAGGAGGCAGCGCTCTCAGGACATCTCCACCATGGCCTGGGCTCTGCTGCTCCTCACCCTCCTCACTCAGGGCACAGGGTCCGGGGCCCAGTCTGCCCTGACTCAGCCTGCCTCCGTGTCGGGGTCTCCGGGACAGTCGATCACCATTCCCTGCACTGGAACCAGCAGGGACATTGGTTATTACAATTATGTCTCCTGGTACCAACACCACCCCGGCAAAGCCCCCCAACTCATTATTTATGAGGTCAGTAATCGGCCCTCAGGGGTTTCTAATCGCTTCTCTGGCTCCAAGTCTGGCAACACGGCCTCCCTGACCATCTCTGGGCTCCAGGCTGAGGACGAGGCTCATTATTACTGCAACTCATATACAGGCAGCAGAACCCGATATGTGTTGTTCGGCGGAGGGACCAAACTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 4247,
            "umi_count": 293,
            "start_codon_pos": 41,
            "stop_codon_pos": None,
            "aa_sequence": "MAWALLLLTLLTQGTGSGAQSALTQPASVSGSPGQSITIPCTGTSRDIGYYNYVSWYQHHPGKAPQLIIYEVSNRPSGVSNRFSGSKSGNTASLTISGLQAEDEAHYYCNSYTGSRTRYVLFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
            "frame": None,
            "cdr3": "CNSYTGSRTRYVLF",
            "cdr3_seq": "TGCAACTCATATACAGGCAGCAGAACCCGATATGTGTTGTTC",
            "cdr3_start": 365,
            "cdr3_stop": 407,
            "fwr1": {
                "start": 98,
                "stop": 164,
                "nt_seq": "CAGTCTGCCCTGACTCAGCCTGCCTCCGTGTCGGGGTCTCCGGGACAGTCGATCACCATTCCCTGC",
                "aa_seq": "QSALTQPASVSGSPGQSITIPC",
            },
            "cdr1": {
                "start": 164,
                "stop": 206,
                "nt_seq": "ACTGGAACCAGCAGGGACATTGGTTATTACAATTATGTCTCC",
                "aa_seq": "TGTSRDIGYYNYVS",
            },
            "fwr2": {
                "start": 206,
                "stop": 251,
                "nt_seq": "TGGTACCAACACCACCCCGGCAAAGCCCCCCAACTCATTATTTAT",
                "aa_seq": "WYQHHPGKAPQLIIY",
            },
            "cdr2": {
                "start": 251,
                "stop": 272,
                "nt_seq": "GAGGTCAGTAATCGGCCCTCA",
                "aa_seq": "EVSNRPS",
            },
            "fwr3": {
                "start": 272,
                "stop": 365,
                "nt_seq": "GGGGTTTCTAATCGCTTCTCTGGCTCCAAGTCTGGCAACACGGCCTCCCTGACCATCTCTGGGCTCCAGGCTGAGGACGAGGCTCATTATTAC",
                "aa_seq": "GVSNRFSGSKSGNTASLTISGLQAEDEAHYY",
            },
            "fwr4": {
                "start": 407,
                "stop": 435,
                "nt_seq": "GGCGGAGGGACCAAACTGACCGTCCTAG",
                "aa_seq": "GGGTKLTVL",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 41,
                    "annotation_match_start": 0,
                    "annotation_match_end": 41,
                    "annotation_length": 41,
                    "cigar": "41M605S",
                    "score": 82,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV2-14",
                        "feature_id": 338,
                        "gene_name": "IGLV2-14",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 41,
                    "contig_match_end": 388,
                    "annotation_match_start": 0,
                    "annotation_match_end": 347,
                    "annotation_length": 361,
                    "cigar": "41S347M258S",
                    "score": 604,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV2-14",
                        "feature_id": 339,
                        "gene_name": "IGLV2-14",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 397,
                    "contig_match_end": 435,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "397S38M211S",
                    "score": 61,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLJ2",
                        "feature_id": 311,
                        "gene_name": "IGLJ2",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 435,
                    "contig_match_end": 646,
                    "annotation_match_start": 0,
                    "annotation_match_end": 211,
                    "annotation_length": 317,
                    "cigar": "435S211M",
                    "score": 422,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLC2",
                        "feature_id": 306,
                        "gene_name": "IGLC2",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1",
                "raw_consensus_id": "clonotype1_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGTCCAAGTAC-1",
            "contig_name": "AAACCTGTCCAAGTAC-1_contig_2",
            "sequence": "GAGCTCTGGGAGAGGAGCCCAGCACTGGAAGTCGCCGGTGTTTCCATTCGGTGATCAGCACTGAACACAGAGGACTCACCATGGAGTTTGGGCTGAGCTGGGTTTTCCTCGTTGCTCTTTTAAGAGGTGTCCAGTGTCAGGTGCAACTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCTGGATTCACCTTCAGTAGCCATGCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATATCATATGATGGAAGTAATGAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGGACACCCTGTATCTGCAAGTGGACAGCCTGAGACCTGAGGACACGGCTGTGTATTATTGTGCGAGAGATCGAATTGATAGCAGCAGCTGGCCATATTACTACTTCTACTACGGAATGGACGTCTGGGGCCAAGGGACCACGCTCACCGTCTCCTCAGCATCCCCGACCAGCCCCAAGGTCTTCCCGCTGAGCCTCTGCAGCACCCAGCCAGATGGGAACGTGGTCATCGCCTGCCTGGTCCAGGGCTTCTTCCCCCAGGAGCCACTCAGTGTGACCTGGAGCGAAAGCGGACAGGGCGTGACCGCCAGAAACTTCCC",
            "quals": "II]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 282,
            "umi_count": 57,
            "start_codon_pos": 80,
            "stop_codon_pos": None,
            "aa_sequence": "MEFGLSWVFLVALLRGVQCQVQLVESGGGVVQPGRSLRLSCAASGFTFSSHAMHWVRQAPGKGLEWVAVISYDGSNEYYADSVKGRFTISRDNSKDTLYLQVDSLRPEDTAVYYCARDRIDSSSWPYYYFYYGMDVWGQGTTLTVSSASPTSPKVFPLSLCSTQPDGNVVIACLVQGFFPQEPLSVTWSESGQGVTARNF",
            "frame": None,
            "cdr3": "CARDRIDSSSWPYYYFYYGMDVW",
            "cdr3_seq": "TGTGCGAGAGATCGAATTGATAGCAGCAGCTGGCCATATTACTACTTCTACTACGGAATGGACGTCTGG",
            "cdr3_start": 422,
            "cdr3_stop": 491,
            "fwr1": {
                "start": 137,
                "stop": 212,
                "nt_seq": "CAGGTGCAACTGGTGGAGTCTGGGGGAGGCGTGGTCCAGCCTGGGAGGTCCCTGAGACTCTCCTGTGCAGCCTCT",
                "aa_seq": "QVQLVESGGGVVQPGRSLRLSCAAS",
            },
            "cdr1": {
                "start": 212,
                "stop": 233,
                "nt_seq": "GGATTCACCTTCAGTAGCCAT",
                "aa_seq": "GFTFSSH",
            },
            "fwr2": {
                "start": 233,
                "stop": 290,
                "nt_seq": "GCTATGCACTGGGTCCGCCAGGCTCCAGGCAAGGGGCTAGAGTGGGTGGCAGTTATA",
                "aa_seq": "AMHWVRQAPGKGLEWVAVI",
            },
            "cdr2": {
                "start": 290,
                "stop": 308,
                "nt_seq": "TCATATGATGGAAGTAAT",
                "aa_seq": "SYDGSN",
            },
            "fwr3": {
                "start": 308,
                "stop": 422,
                "nt_seq": "GAATACTACGCAGACTCCGTGAAGGGCCGATTCACCATCTCCAGAGACAATTCCAAGGACACCCTGTATCTGCAAGTGGACAGCCTGAGACCTGAGGACACGGCTGTGTATTAT",
                "aa_seq": "EYYADSVKGRFTISRDNSKDTLYLQVDSLRPEDTAVYY",
            },
            "fwr4": {
                "start": 491,
                "stop": 522,
                "nt_seq": "GGCCAAGGGACCACGCTCACCGTCTCCTCAG",
                "aa_seq": "GQGTTLTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 80,
                    "contig_match_end": 433,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "80S353M249S",
                    "score": 651,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-33",
                        "feature_id": 128,
                        "gene_name": "IGHV3-33",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 459,
                    "contig_match_end": 522,
                    "annotation_match_start": 0,
                    "annotation_match_end": 63,
                    "annotation_length": 63,
                    "cigar": "459S63M160S",
                    "score": 111,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ6",
                        "feature_id": 737,
                        "gene_name": "IGHJ6",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 522,
                    "contig_match_end": 682,
                    "annotation_match_start": 0,
                    "annotation_match_end": 160,
                    "annotation_length": 1058,
                    "cigar": "522S160M",
                    "score": 320,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHA1",
                        "feature_id": 1,
                        "gene_name": "IGHA1",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype1",
                "raw_consensus_id": "clonotype1_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACCTGTCCAAGTAC-1",
            "contig_name": "AAACCTGTCCAAGTAC-1_contig_3",
            "sequence": "TTCACAGAAGTTCCAGGGCAGAGTCACCATTACCAGGGACACATCCGCGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAAGACACGGCTGTGTATTACTGTGCGAGAGATGTGGGGATACCCGGGGACACCGTATGGTTCGGGGAGTCCATACCTAATCGCCGTAACTGGTTCGACCCCTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCCTCCAAGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
            "quals": "IIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]X]]]]]]]]]XXX]IIIIIIII",
            "read_count": 92,
            "umi_count": 3,
            "start_codon_pos": None,
            "stop_codon_pos": None,
            "aa_sequence": None,
            "frame": None,
            "cdr3": "CARDVGIPGDTVWFGESIPNRRNWFDPW",
            "cdr3_seq": "TGTGCGAGAGATGTGGGGATACCCGGGGACACCGTATGGTTCGGGGAGTCCATACCTAATCGCCGTAACTGGTTCGACCCCTGG",
            "cdr3_start": 106,
            "cdr3_stop": 190,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 117,
                    "annotation_match_start": 236,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "117M286S",
                    "score": 234,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-3",
                        "feature_id": 73,
                        "gene_name": "IGHV1-3",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 170,
                    "contig_match_end": 221,
                    "annotation_match_start": 0,
                    "annotation_match_end": 51,
                    "annotation_length": 51,
                    "cigar": "170S51M182S",
                    "score": 92,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ5",
                        "feature_id": 57,
                        "gene_name": "IGHJ5",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 221,
                    "contig_match_end": 403,
                    "annotation_match_start": 0,
                    "annotation_match_end": 182,
                    "annotation_length": 884,
                    "cigar": "221S182M",
                    "score": 359,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHG1",
                        "feature_id": 40,
                        "gene_name": "IGHG1",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": False,
        },
        {
            "barcode": "AAACGGGCACAACGCC-1",
            "contig_name": "AAACGGGCACAACGCC-1_contig_1",
            "sequence": "AGAGCTCTGGGGAGGAACTGCTCAGTTAGGACCCAGACGGAACCATGGAAGCCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACTGGAGAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGCAGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCCTGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTATGGTGCATCCACCAGGGCCACTGGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTACTGTCAGCAGTATAATAACTGGCCTCCGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 1619,
            "umi_count": 66,
            "start_codon_pos": 44,
            "stop_codon_pos": None,
            "aa_sequence": "MEAPAQLLFLLLLWLPDTTGEIVMTQSPATLSVSPGERATLSCRASQSVSSNLAWYQQKPGQAPRLLIYGASTRATGIPARFSGSGSGTEFTLTISSLQSEDFAVYYCQQYNNWPPWTFGQGTKVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
            "frame": None,
            "cdr3": "CQQYNNWPPWTF",
            "cdr3_seq": "TGTCAGCAGTATAATAACTGGCCTCCGTGGACGTTC",
            "cdr3_start": 365,
            "cdr3_stop": 401,
            "fwr1": {
                "start": 104,
                "stop": 173,
                "nt_seq": "GAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGC",
                "aa_seq": "EIVMTQSPATLSVSPGERATLSC",
            },
            "cdr1": {
                "start": 173,
                "stop": 206,
                "nt_seq": "AGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCC",
                "aa_seq": "RASQSVSSNLA",
            },
            "fwr2": {
                "start": 206,
                "stop": 251,
                "nt_seq": "TGGTACCAGCAGAAACCTGGCCAGGCTCCCAGGCTCCTCATCTAT",
                "aa_seq": "WYQQKPGQAPRLLIY",
            },
            "cdr2": {
                "start": 251,
                "stop": 272,
                "nt_seq": "GGTGCATCCACCAGGGCCACT",
                "aa_seq": "GASTRAT",
            },
            "fwr3": {
                "start": 272,
                "stop": 365,
                "nt_seq": "GGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTAC",
                "aa_seq": "GIPARFSGSGSGTEFTLTISSLQSEDFAVYY",
            },
            "fwr4": {
                "start": 401,
                "stop": 429,
                "nt_seq": "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                "aa_seq": "GQGTKVEIK",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 44,
                    "annotation_match_start": 53,
                    "annotation_match_end": 97,
                    "annotation_length": 97,
                    "cigar": "44M521S",
                    "score": 88,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-15",
                        "feature_id": 280,
                        "gene_name": "IGKV3-15",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 44,
                    "contig_match_end": 389,
                    "annotation_match_start": 0,
                    "annotation_match_end": 345,
                    "annotation_length": 345,
                    "cigar": "44S345M176S",
                    "score": 690,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-15",
                        "feature_id": 281,
                        "gene_name": "IGKV3-15",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 391,
                    "contig_match_end": 429,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "391S38M136S",
                    "score": 76,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ1",
                        "feature_id": 213,
                        "gene_name": "IGKJ1",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 429,
                    "contig_match_end": 565,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "429S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype610",
                "raw_consensus_id": "clonotype610_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAACGGGCACAACGCC-1",
            "contig_name": "AAACGGGCACAACGCC-1_contig_2",
            "sequence": "AGCTCTGGGAGAGGAGCCCCAGGCCCGGGATTCCCAGGTGTTTCCATTCAGTGATCAGCACTGAAGACAGAAGACTCATCATGGAGTTCTGGCTGAGCTGGGTTCTCCTTGTTGCCATTTTAAAAGATGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCTGGATTCACCTTCAGTAGCTATGCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAGCTATTAGTAGTAATGGGGGTAGCACATACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTGTATTACTGTGTGAAAGATATAGTGGCTACGCCCAACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 500,
            "umi_count": 34,
            "start_codon_pos": 80,
            "stop_codon_pos": None,
            "aa_sequence": "MEFWLSWVLLVAILKDVQCEVQLVESGGGLVQPGGSLRLSCSASGFTFSSYAMHWVRQAPGKGLEYVSAISSNGGSTYYADSVKGRFTISRDNSKNTLYLQMSSLRAEDTAVYYCVKDIVATPNWGQGTLVTVSSGSASAPTLFPLVSCENSPSDTSSV",
            "frame": None,
            "cdr3": "CVKDIVATPNW",
            "cdr3_seq": "TGTGTGAAAGATATAGTGGCTACGCCCAACTGG",
            "cdr3_start": 422,
            "cdr3_stop": 455,
            "fwr1": {
                "start": 137,
                "stop": 212,
                "nt_seq": "GAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTCCAGCCTGGGGGGTCCCTGAGACTCTCCTGTTCAGCCTCT",
                "aa_seq": "EVQLVESGGGLVQPGGSLRLSCSAS",
            },
            "cdr1": {
                "start": 212,
                "stop": 233,
                "nt_seq": "GGATTCACCTTCAGTAGCTAT",
                "aa_seq": "GFTFSSY",
            },
            "fwr2": {
                "start": 233,
                "stop": 290,
                "nt_seq": "GCTATGCACTGGGTCCGCCAGGCTCCAGGGAAGGGACTGGAATATGTTTCAGCTATT",
                "aa_seq": "AMHWVRQAPGKGLEYVSAI",
            },
            "cdr2": {
                "start": 290,
                "stop": 308,
                "nt_seq": "AGTAGTAATGGGGGTAGC",
                "aa_seq": "SSNGGS",
            },
            "fwr3": {
                "start": 308,
                "stop": 422,
                "nt_seq": "ACATACTACGCAGACTCCGTGAAGGGCAGATTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATCTTCAAATGAGCAGTCTGAGAGCTGAGGACACGGCTGTGTATTAC",
                "aa_seq": "TYYADSVKGRFTISRDNSKNTLYLQMSSLRAEDTAVYY",
            },
            "fwr4": {
                "start": 455,
                "stop": 486,
                "nt_seq": "GGCCAGGGAACCCTGGTCACCGTCTCCTCAG",
                "aa_seq": "GQGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 80,
                    "contig_match_end": 433,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "80S353M124S",
                    "score": 706,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-64D",
                        "feature_id": 151,
                        "gene_name": "IGHV3-64D",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 450,
                    "contig_match_end": 486,
                    "annotation_match_start": 10,
                    "annotation_match_end": 46,
                    "annotation_length": 46,
                    "cigar": "450S36M71S",
                    "score": 72,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 54,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 486,
                    "contig_match_end": 557,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "486S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype610",
                "raw_consensus_id": "clonotype610_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAAGATGAGAGATGAG-1",
            "contig_name": "AAAGATGAGAGATGAG-1_contig_1",
            "sequence": "GGGGAGAAGTCTCTCTCAGTCAGGACACAGCATGGACATGAGGGTCCCTGCTCAGCTCCTGGGACTCCTGCTGCTCTGGCTCCCAGATACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCGGGCGAGTCAGGGCATTAGCAATTATTTAGCCTGGTATCAGCAGAAACCAGGGAAAGTTCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAATCAGGGGTCCCATCTCGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACTTATTACTGTCAAAAGTATAACAGTGCGCCGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": ">]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 1913,
            "umi_count": 81,
            "start_codon_pos": 31,
            "stop_codon_pos": None,
            "aa_sequence": "MDMRVPAQLLGLLLLWLPDTRCDIQMTQSPSSLSASVGDRVTITCRASQGISNYLAWYQQKPGKVPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISSLQPEDVATYYCQKYNSAPWTFGQGTKVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
            "frame": None,
            "cdr3": "CQKYNSAPWTF",
            "cdr3_seq": "TGTCAAAAGTATAACAGTGCGCCGTGGACGTTC",
            "cdr3_start": 358,
            "cdr3_stop": 391,
            "fwr1": {
                "start": 97,
                "stop": 166,
                "nt_seq": "GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGC",
                "aa_seq": "DIQMTQSPSSLSASVGDRVTITC",
            },
            "cdr1": {
                "start": 166,
                "stop": 199,
                "nt_seq": "CGGGCGAGTCAGGGCATTAGCAATTATTTAGCC",
                "aa_seq": "RASQGISNYLA",
            },
            "fwr2": {
                "start": 199,
                "stop": 244,
                "nt_seq": "TGGTATCAGCAGAAACCAGGGAAAGTTCCTAAGCTCCTGATCTAT",
                "aa_seq": "WYQQKPGKVPKLLIY",
            },
            "cdr2": {
                "start": 244,
                "stop": 265,
                "nt_seq": "GCTGCATCCACTTTGCAATCA",
                "aa_seq": "AASTLQS",
            },
            "fwr3": {
                "start": 265,
                "stop": 358,
                "nt_seq": "GGGGTCCCATCTCGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACTTATTAC",
                "aa_seq": "GVPSRFSGSGSGTDFTLTISSLQPEDVATYY",
            },
            "fwr4": {
                "start": 391,
                "stop": 419,
                "nt_seq": "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                "aa_seq": "GQGTKVEIK",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 31,
                    "annotation_match_start": 0,
                    "annotation_match_end": 31,
                    "annotation_length": 31,
                    "cigar": "31M524S",
                    "score": 52,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV1-27",
                        "feature_id": 224,
                        "gene_name": "IGKV1-27",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 31,
                    "contig_match_end": 382,
                    "annotation_match_start": 0,
                    "annotation_match_end": 351,
                    "annotation_length": 351,
                    "cigar": "31S351M173S",
                    "score": 692,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV1-27",
                        "feature_id": 225,
                        "gene_name": "IGKV1-27",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 381,
                    "contig_match_end": 419,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "381S38M136S",
                    "score": 76,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ1",
                        "feature_id": 213,
                        "gene_name": "IGKJ1",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 419,
                    "contig_match_end": 555,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "419S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype372",
                "raw_consensus_id": "clonotype372_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAAGATGAGAGATGAG-1",
            "contig_name": "AAAGATGAGAGATGAG-1_contig_2",
            "sequence": "CAGGCCCGTCTGGGCAGGTCAGCCTCAATGGGGACGGTGCTCGGGTCCTGCCTGGCTCCTCCTTGGGGTAGAGGGGACCTATCGCCTGGCCCTGGACCCCCTCAGGCCGCCCCTATTCTTGAAGAAACGCAGGACGGTGGAGGGTGCCCTGGAGGGGAGACGTCTGCTGCCCTGGGTGTCAGGTGCGGCCTCAGATCCTAAACGTGTCCTCCCTGCTCTCTCTGCTTCCGCTGACTTGGGACGGGGTTTCTGGCCTGGCGAGTGTCACTGTGTGCCCTGCCGGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCTTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCGCCCTGCTCCAGGAGCACCTCTGGGGGCACAGCGGCCCTGGGCTGCCTGGTCAAGGACTACTTCCCCGAACCGGTGACGGTGTCGTGGAACTCAGGCGCCCTGACCAGCGGCGTGCACACCTTCCCGGCTGTCCTACAGTCCTCAGGA",
            "quals": "IIIIIIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]\\]]]]]]]]]]]]M]M]]]]]>>]]]]]]]]]\\]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 293,
            "umi_count": 13,
            "start_codon_pos": None,
            "stop_codon_pos": None,
            "aa_sequence": None,
            "frame": None,
            "cdr3": None,
            "cdr3_seq": None,
            "cdr3_start": None,
            "cdr3_stop": None,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 282,
                    "contig_match_end": 322,
                    "annotation_match_start": 6,
                    "annotation_match_end": 46,
                    "annotation_length": 46,
                    "cigar": "282S40M182S",
                    "score": 80,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 54,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 322,
                    "contig_match_end": 504,
                    "annotation_match_start": 0,
                    "annotation_match_end": 182,
                    "annotation_length": 1130,
                    "cigar": "322S182M",
                    "score": 359,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHG3",
                        "feature_id": 45,
                        "gene_name": "IGHG3",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": False,
        },
        {
            "barcode": "AAAGATGAGGGATGGG-1",
            "contig_name": "AAAGATGAGGGATGGG-1_contig_1",
            "sequence": "AGAGCTCTGGGGAGGAACTGCTCAGTTAGGACCCAGACGGAACCATGGAAGCCCCAGCGCAGCTTCTCTTCCTCCTGCTACTCTGGCTCCCAGATACCACTGGAGAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGCAGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCCTGGTACCAGCAGAAACCTCGACAGGCTCCCAGGCTCCTCATCTATGGTGCATCCACCAGGGCCACTGGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTACTGTCAGCAGTATAATAACTGGCCTAGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 2955,
            "umi_count": 124,
            "start_codon_pos": 44,
            "stop_codon_pos": None,
            "aa_sequence": "MEAPAQLLFLLLLWLPDTTGEIVMTQSPATLSVSPGERATLSCRASQSVSSNLAWYQQKPRQAPRLLIYGASTRATGIPARFSGSGSGTEFTLTISSLQSEDFAVYYCQQYNNWPRTFGQGTKVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN",
            "frame": None,
            "cdr3": "CQQYNNWPRTF",
            "cdr3_seq": "TGTCAGCAGTATAATAACTGGCCTAGGACGTTC",
            "cdr3_start": 365,
            "cdr3_stop": 398,
            "fwr1": {
                "start": 104,
                "stop": 173,
                "nt_seq": "GAAATAGTGATGACGCAGTCTCCAGCCACCCTGTCTGTGTCTCCAGGGGAAAGAGCCACCCTCTCCTGC",
                "aa_seq": "EIVMTQSPATLSVSPGERATLSC",
            },
            "cdr1": {
                "start": 173,
                "stop": 206,
                "nt_seq": "AGGGCCAGTCAGAGTGTTAGCAGCAACTTAGCC",
                "aa_seq": "RASQSVSSNLA",
            },
            "fwr2": {
                "start": 206,
                "stop": 251,
                "nt_seq": "TGGTACCAGCAGAAACCTCGACAGGCTCCCAGGCTCCTCATCTAT",
                "aa_seq": "WYQQKPRQAPRLLIY",
            },
            "cdr2": {
                "start": 251,
                "stop": 272,
                "nt_seq": "GGTGCATCCACCAGGGCCACT",
                "aa_seq": "GASTRAT",
            },
            "fwr3": {
                "start": 272,
                "stop": 365,
                "nt_seq": "GGTATCCCAGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGAGTTCACTCTCACCATCAGCAGCCTGCAGTCTGAAGATTTTGCAGTTTATTAC",
                "aa_seq": "GIPARFSGSGSGTEFTLTISSLQSEDFAVYY",
            },
            "fwr4": {
                "start": 398,
                "stop": 426,
                "nt_seq": "GGCCAAGGGACCAAGGTGGAAATCAAAC",
                "aa_seq": "GQGTKVEIK",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 44,
                    "annotation_match_start": 53,
                    "annotation_match_end": 97,
                    "annotation_length": 97,
                    "cigar": "44M518S",
                    "score": 88,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-15",
                        "feature_id": 280,
                        "gene_name": "IGKV3-15",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 44,
                    "contig_match_end": 389,
                    "annotation_match_start": 0,
                    "annotation_match_end": 345,
                    "annotation_length": 345,
                    "cigar": "44S345M173S",
                    "score": 680,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKV3-15",
                        "feature_id": 281,
                        "gene_name": "IGKV3-15",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 388,
                    "contig_match_end": 426,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "388S38M136S",
                    "score": 66,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKJ1",
                        "feature_id": 213,
                        "gene_name": "IGKJ1",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 426,
                    "contig_match_end": 562,
                    "annotation_match_start": 0,
                    "annotation_match_end": 136,
                    "annotation_length": 320,
                    "cigar": "426S136M",
                    "score": 272,
                    "feature": {
                        "chain": "IGK",
                        "display_name": "IGKC",
                        "feature_id": 212,
                        "gene_name": "IGKC",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype127",
                "raw_consensus_id": "clonotype127_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAAGATGAGGGATGGG-1",
            "contig_name": "AAAGATGAGGGATGGG-1_contig_2",
            "sequence": "AAGGGGCTGGAGTGGGTCTCAGGTATTAGTGGTAGTGGTGGTGGCACATACTACGCAGACTCCGTGAAGGGCCGGTTCACCATCTCCAGAGACAATTCCAAGAACACGCTGTATCTGCAAATGACCAGCCTGGGAGCCGACGACACGGCCGTATATTACTGTGCGAAAGATTTTGCGAATACCTACCACTATGATAGTAGTGGCCTAGGTGGTCCTTTTGACTACTGGGGCCAGGGAACCCTGGTCACCGTCTCCTCAGCCTCCACCAAGGGCCCATCGGTCTTCCCCCTGGCACCCTCC",
            "quals": ">>>>>>>>]\\]]\\]]]M]]]]]\\]]\\]]]]]]]]]]]]]]]]\\]]]]]\\\\]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]X]X]]]]]]]]]]]X]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]\\\\M]]]]\\]]>]]]\\]]\\]]]]]]]M]]]]]]\\>+>+>>>]]]]\\]M]]]M]]\\\\]]]]]]]]\\]]GGIGIIIHHHIIGHHHIIHHI",
            "read_count": 18,
            "umi_count": 8,
            "start_codon_pos": None,
            "stop_codon_pos": None,
            "aa_sequence": None,
            "frame": None,
            "cdr3": "CAKDFANTYHYDSSGLGGPFDYW",
            "cdr3_seq": "TGTGCGAAAGATTTTGCGAATACCTACCACTATGATAGTAGTGGCCTAGGTGGTCCTTTTGACTACTGG",
            "cdr3_start": 159,
            "cdr3_stop": 228,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 170,
                    "annotation_match_start": 183,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "170M130S",
                    "score": 315,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-23",
                        "feature_id": 122,
                        "gene_name": "IGHV3-23",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 211,
                    "contig_match_end": 259,
                    "annotation_match_start": 0,
                    "annotation_match_end": 48,
                    "annotation_length": 48,
                    "cigar": "211S48M41S",
                    "score": 71,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ4",
                        "feature_id": 55,
                        "gene_name": "IGHJ4",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 259,
                    "contig_match_end": 300,
                    "annotation_match_start": 0,
                    "annotation_match_end": 41,
                    "annotation_length": 884,
                    "cigar": "259S41M",
                    "score": 82,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHG1",
                        "feature_id": 40,
                        "gene_name": "IGHG1",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": False,
        },
        {
            "barcode": "AAAGATGCAACACCTA-1",
            "contig_name": "AAAGATGCAACACCTA-1_contig_1",
            "sequence": "AGCTCTCAGAGAGGTGCCTTAGCCCTGGATTCCAAGGCATTTCCACTTGGTGATCAGCACTGAACACAGAGGACTCACCATGGAGTTGGGGCTGTGCTGGGTTTTCCTTGTTGCTATTTTAGAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTACAGCCTGGAGGGTCCCTGAGACTCTCCTGTGCAGCCTCTGGATTCACCTTCAGTAGTTATGAAATGAACTGGGTCCGCCAGGCTCCAGGGAAGGGGCTGGAGTGGGTTTCATACATTAGTAGTAGTGGTAGTACCATATACTACGCAGACTCTGTGAAGGGCCGATTCACCATCTCCAGAGACAACGCCAAGAACTCACTGTATCTGCAAATGAACAGCCTGAGAGCCGAGGACACGGCTGTTTATTACTGTGCGAGACCAATGAAACATCCACTCTGGGGCCGTGGCACCCTGGTCACTGTCTCCTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": "]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 659,
            "umi_count": 34,
            "start_codon_pos": 79,
            "stop_codon_pos": None,
            "aa_sequence": "MELGLCWVFLVAILEGVQCEVQLVESGGGLVQPGGSLRLSCAASGFTFSSYEMNWVRQAPGKGLEWVSYISSSGSTIYYADSVKGRFTISRDNAKNSLYLQMNSLRAEDTAVYYCARPMKHPLWGRGTLVTVSSGSASAPTLFPLVSCENSPSDTSSV",
            "frame": None,
            "cdr3": "CARPMKHPLW",
            "cdr3_seq": "TGTGCGAGACCAATGAAACATCCACTCTGG",
            "cdr3_start": 421,
            "cdr3_stop": 451,
            "fwr1": {
                "start": 136,
                "stop": 211,
                "nt_seq": "GAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTGGTACAGCCTGGAGGGTCCCTGAGACTCTCCTGTGCAGCCTCT",
                "aa_seq": "EVQLVESGGGLVQPGGSLRLSCAAS",
            },
            "cdr1": {
                "start": 211,
                "stop": 232,
                "nt_seq": "GGATTCACCTTCAGTAGTTAT",
                "aa_seq": "GFTFSSY",
            },
            "fwr2": {
                "start": 232,
                "stop": 289,
                "nt_seq": "GAAATGAACTGGGTCCGCCAGGCTCCAGGGAAGGGGCTGGAGTGGGTTTCATACATT",
                "aa_seq": "EMNWVRQAPGKGLEWVSYI",
            },
            "cdr2": {
                "start": 289,
                "stop": 307,
                "nt_seq": "AGTAGTAGTGGTAGTACC",
                "aa_seq": "SSSGST",
            },
            "fwr3": {
                "start": 307,
                "stop": 421,
                "nt_seq": "ATATACTACGCAGACTCTGTGAAGGGCCGATTCACCATCTCCAGAGACAACGCCAAGAACTCACTGTATCTGCAAATGAACAGCCTGAGAGCCGAGGACACGGCTGTTTATTAC",
                "aa_seq": "IYYADSVKGRFTISRDNAKNSLYLQMNSLRAEDTAVYY",
            },
            "fwr4": {
                "start": 451,
                "stop": 482,
                "nt_seq": "GGCCGTGGCACCCTGGTCACTGTCTCCTCAG",
                "aa_seq": "GRGTLVTVSS",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 79,
                    "annotation_match_start": 0,
                    "annotation_match_end": 79,
                    "annotation_length": 79,
                    "cigar": "79M474S",
                    "score": 158,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-48",
                        "feature_id": 141,
                        "gene_name": "IGHV3-48",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 79,
                    "contig_match_end": 432,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "79S353M121S",
                    "score": 696,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV3-48",
                        "feature_id": 142,
                        "gene_name": "IGHV3-48",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 445,
                    "contig_match_end": 482,
                    "annotation_match_start": 15,
                    "annotation_match_end": 52,
                    "annotation_length": 52,
                    "cigar": "445S37M71S",
                    "score": 74,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ2",
                        "feature_id": 50,
                        "gene_name": "IGHJ2",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 482,
                    "contig_match_end": 553,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "482S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype599",
                "raw_consensus_id": "clonotype599_consensus_1",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAAGATGCAACACCTA-1",
            "contig_name": "AAAGATGCAACACCTA-1_contig_2",
            "sequence": "AGGGCTGGTGGGATCAGTCCTGGTGGTAGCTCAGGAAGCAGAGCCTGGAGCATCTCCACTATGGCCTGGGCTCCACTACTTCTCACCCTCCTCGCTCACTGCACAGGTTCTTGGGCCAATTTTATGCTGACTCAGCCCCACTCTGTGTCGGAGTCTCCGGGGAAGACGGTAACCATCTCCTGCACCGGCAGCAGTGGCAGCATTGCCAGCAACTATGTGCAGTGGTACCAGCAGCGCCCGGGCAGTGCCCCCACCACTGTGATCTATGAGGATAACCAAAGACCCTCTGGGGTCCCTGATCGGTTCTCTGGCTCCATCGACAGCTCCTCCAACTCTGCCTCCCTCACCATCTCTGGACTGAAGACTGAGGACGAGGCTGACTACTACTGTCAGTCTTATGATAGCAGCAATCATTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA",
            "quals": "IIIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 657,
            "umi_count": 33,
            "start_codon_pos": 60,
            "stop_codon_pos": None,
            "aa_sequence": "MAWAPLLLTLLAHCTGSWANFMLTQPHSVSESPGKTVTISCTGSSGSIASNYVQWYQQRPGSAPTTVIYEDNQRPSGVPDRFSGSIDSSSNSASLTISGLKTEDEADYYCQSYDSSNHWVFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS",
            "frame": None,
            "cdr3": "CQSYDSSNHWVF",
            "cdr3_seq": "TGTCAGTCTTATGATAGCAGCAATCATTGGGTGTTC",
            "cdr3_start": 387,
            "cdr3_stop": 423,
            "fwr1": {
                "start": 117,
                "stop": 183,
                "nt_seq": "AATTTTATGCTGACTCAGCCCCACTCTGTGTCGGAGTCTCCGGGGAAGACGGTAACCATCTCCTGC",
                "aa_seq": "NFMLTQPHSVSESPGKTVTISC",
            },
            "cdr1": {
                "start": 183,
                "stop": 222,
                "nt_seq": "ACCGGCAGCAGTGGCAGCATTGCCAGCAACTATGTGCAG",
                "aa_seq": "TGSSGSIASNYVQ",
            },
            "fwr2": {
                "start": 222,
                "stop": 267,
                "nt_seq": "TGGTACCAGCAGCGCCCGGGCAGTGCCCCCACCACTGTGATCTAT",
                "aa_seq": "WYQQRPGSAPTTVIY",
            },
            "cdr2": {
                "start": 267,
                "stop": 288,
                "nt_seq": "GAGGATAACCAAAGACCCTCT",
                "aa_seq": "EDNQRPS",
            },
            "fwr3": {
                "start": 288,
                "stop": 387,
                "nt_seq": "GGGGTCCCTGATCGGTTCTCTGGCTCCATCGACAGCTCCTCCAACTCTGCCTCCCTCACCATCTCTGGACTGAAGACTGAGGACGAGGCTGACTACTAC",
                "aa_seq": "GVPDRFSGSIDSSSNSASLTISGLKTEDEADYY",
            },
            "fwr4": {
                "start": 423,
                "stop": 451,
                "nt_seq": "GGCGGAGGGACCAAGCTGACCGTCCTAG",
                "aa_seq": "GGGTKLTVL",
            },
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 60,
                    "annotation_match_start": 26,
                    "annotation_match_end": 86,
                    "annotation_length": 86,
                    "cigar": "60M602S",
                    "score": 120,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV6-57",
                        "feature_id": 732,
                        "gene_name": "IGLV6-57",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 60,
                    "contig_match_end": 413,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "60S353M249S",
                    "score": 706,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLV6-57",
                        "feature_id": 384,
                        "gene_name": "IGLV6-57",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 413,
                    "contig_match_end": 451,
                    "annotation_match_start": 0,
                    "annotation_match_end": 38,
                    "annotation_length": 38,
                    "cigar": "413S38M211S",
                    "score": 76,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLJ3",
                        "feature_id": 313,
                        "gene_name": "IGLJ3",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 451,
                    "contig_match_end": 662,
                    "annotation_match_start": 0,
                    "annotation_match_end": 211,
                    "annotation_length": 317,
                    "cigar": "451S211M",
                    "score": 422,
                    "feature": {
                        "chain": "IGL",
                        "display_name": "IGLC3",
                        "feature_id": 307,
                        "gene_name": "IGLC3",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": "clonotype599",
                "raw_consensus_id": "clonotype599_consensus_2",
                "exact_subclonotype_id": "1",
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": True,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
        {
            "barcode": "AAAGATGCAACACCTA-1",
            "contig_name": "AAAGATGCAACACCTA-1_contig_3",
            "sequence": "CCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGAGTAGACTGTTTATAAGGGGTCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG",
            "quals": ">>>>>>>>+>>>>HIHIIIIIIIHIIIIIIIIIIIIIIIIIIIIIIIIIIII]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]II]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]>>>>>>]]]]\\]]]]]]]]]]]]]]]]M]\\M]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]",
            "read_count": 100,
            "umi_count": 4,
            "start_codon_pos": 23,
            "stop_codon_pos": 467,
            "aa_sequence": "MDWTWRFLFVVAAATGVQSQVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCASRLFIRGLGPRDNGHRLFRECIRPNPFPPRLL*EFPVGYEQR",
            "frame": None,
            "cdr3": "CASRLFIR",
            "cdr3_seq": "TGTGCGAGTAGACTGTTTATAAGG",
            "cdr3_start": 365,
            "cdr3_stop": 389,
            "fwr1": None,
            "cdr1": None,
            "fwr2": None,
            "cdr2": None,
            "fwr3": None,
            "fwr4": None,
            "annotations": [
                {
                    "contig_match_start": 0,
                    "contig_match_end": 23,
                    "annotation_match_start": 35,
                    "annotation_match_end": 58,
                    "annotation_length": 58,
                    "cigar": "23M475S",
                    "score": 46,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-69D",
                        "feature_id": 87,
                        "gene_name": "IGHV1-69D",
                        "region_type": "5'UTR",
                    },
                },
                {
                    "contig_match_start": 23,
                    "contig_match_end": 376,
                    "annotation_match_start": 0,
                    "annotation_match_end": 353,
                    "annotation_length": 353,
                    "cigar": "23S353M122S",
                    "score": 691,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHV1-69D",
                        "feature_id": 88,
                        "gene_name": "IGHV1-69D",
                        "region_type": "L-REGION+V-REGION",
                    },
                },
                {
                    "contig_match_start": 391,
                    "contig_match_end": 427,
                    "annotation_match_start": 13,
                    "annotation_match_end": 49,
                    "annotation_length": 49,
                    "cigar": "391S36M71S",
                    "score": 72,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHJ3",
                        "feature_id": 52,
                        "gene_name": "IGHJ3",
                        "region_type": "J-REGION",
                    },
                },
                {
                    "contig_match_start": 427,
                    "contig_match_end": 498,
                    "annotation_match_start": 0,
                    "annotation_match_end": 71,
                    "annotation_length": 1358,
                    "cigar": "427S71M",
                    "score": 142,
                    "feature": {
                        "chain": "IGH",
                        "display_name": "IGHM",
                        "feature_id": 60,
                        "gene_name": "IGHM",
                        "region_type": "C-REGION",
                    },
                },
            ],
            "clonotype": None,
            "info": {
                "raw_clonotype_id": None,
                "raw_consensus_id": None,
                "exact_subclonotype_id": None,
            },
            "high_confidence": True,
            "is_cell": True,
            "productive": False,
            "filtered": True,
            "is_gex_cell": True,
            "is_asm_cell": True,
            "full_length": True,
        },
    ]
    return all_json


@pytest.fixture
def annotation_10x_tr2():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame(
        [
            [
                "AAGACCTCACTACAGT-1",
                "False",
                "AAGACCTCACTACAGT-1_contig_1",
                "True",
                544,
                "TRD",
                "TRDV1",
                "None",
                "TRDJ1",
                "TRDC",
                "True",
                "False",
                "CALGEPFLPGWGDKLIF",
                "TGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTT",
                2932,
                11,
                "None",
                "None",
            ],
            [
                "AACTGGTTCTTTAGTC-1",
                "False",
                "AACTGGTTCTTTAGTC-1_contig_2",
                "True",
                522,
                "TRD",
                "TRDV2",
                "None",
                "TRDJ1",
                "TRDC",
                "True",
                "False",
                "CACDILGDKADKLIF",
                "TGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTT",
                1176,
                6,
                "None",
                "None",
            ],
            [
                "AAAGATGCACCCTATC-1",
                "False",
                "AAAGATGCACCCTATC-1_contig_2",
                "True",
                560,
                "TRD",
                "TRDV1",
                "None",
                "TRDJ1",
                "TRDC",
                "True",
                "False",
                "CALGDPRPSYRYWGIQGDKLIF",
                "TGTGCTCTTGGGGACCCCAGGCCTTCCTACAGGTACTGGGGGATACAGGGCGATAAACTCATCTTT",
                1128,
                3,
                "None",
                "None",
            ],
        ],
        columns=[
            "barcode",
            "is_cell",
            "contig_id",
            "high_confidence",
            "length",
            "chain",
            "v_gene",
            "d_gene",
            "j_gene",
            "c_gene",
            "full_length",
            "productive",
            "cdr3",
            "cdr3_nt",
            "reads",
            "umis",
            "raw_clonotype_id",
            "raw_consensus_id",
        ],
    )

    return annot


@pytest.fixture
def airr_travdv():
    """travdv airr"""
    airr = pd.DataFrame(
        [
            [
                "ACCAGTAAGCGTCAAG-1_contig_2",
                "TCCTTCACCAGACAAGCGACATTTGTTCCATTTTTCATGACAAAAACGGATGGTTTGGTATGAGGCTGACTTCTTGGTTCCACAGTCACACGGGTTCCTTTTCCAAAGATGAGTTTATCGGTGTCACAGCACTTGAGGACGTTCCAAAAACCTCAGCGCTGCTCCCACCTACCTTCTTCCCGTCAGCTTTAGGACTGGAGGATCTTTAAGGCTCAGCTCCATGAAAGATGGCACTTGTTGCTTACCCAGCCGGTTCATAGATGGTTAATCATCTGCAGGTAGGGACAGGTTTGCTTTCGGAAGGCTGTTGTCTGAGGAGTATGGAGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAGCTGGCACTGTCAAAGGGGTGGGTCC",
                "T",
                "T",
                "TRAV36/DV7*01",
                "TRDD2*01",
                "TRDJ2*01",
                "....................................................................................................................................................................................................................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAG",
                "GAAGACAAGGTGGTACAAAGCCCTCTATCTCTGGTTGTCCACGAGGGAGACACCGTAACTCTCAATTGCAGTTATGAAGTGACTAAC..................TTTCGAAGCCTACTATGGTACAAGCAGGAAAAGAAAGCTCCC...ACATTTCTATTTATGCTAACTTCAAGT.........GGAATTGAAAAGAAGTCA...............GGAAGACTAAGTAGCATATTAGATAAGAAAGAACTTTCCAGCANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCTTCCTANNNNNNNNNNNNNNNNNNNNNNTTGGAAAG",
                "CATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTT",
                "HCSHFLDPYSALCV",
                "324S183N2=1X1=1X6=3X2=1X1=3X2=1X10=",
                "416S1N7=",
                "445S21N8=",
                "F",
                "T",
                "TRA",
                42,
                58,
                22,
                325,
                358,
                1,
                262,
                417,
                423,
                2,
                8,
                446,
                453,
                22,
                29,
                23.7,
                0.70588,
                2.5,
                14.1,
                1,
                0.83,
                16.1,
                1,
                31,
                "..............................................................................",
                "...................................................",
                ".................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACAT",
                "GTTGGAAAG",
                "....................................",
                "..............................",
                "TGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGT",
                "ACCAGTAAGCGTCAAG-1",
                "TRDC",
                6662,
                40,
                "",
                "",
                "TRDJ1",
                "",
                "",
            ],
            [
                "CAACCTCCATGTTGAC-1_contig_2",
                "TCCTTCACCAGACAAGCGACATTTGTTCCATTTTTCATGACAAAAACGGATGGTTTGGTATGAGGCTGACTTCTTGGTTCCACAGTCACACGGGTTCCTTTTCCAAAGATGAGTTTATCGGTGTCACAGCACTTGAGGACGTTCCAAAAACCTCAGCGCTGCTCCCACCTACCTTCTTCCCGTCAGCTTTAGGACTGGAGGATCTTTAAGGCTCAGCTCCATGAAAGATGGCACTTGTTGCTTACCCAGCCGGTTCATAGATGGTTAATCATCTGCAGGTAGGGACAGGTTTGCTTTCGGAAGGCTGTTGTCTGAGGAGTATGGAGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAGCTGGCACTGTCAAAGGGGTGGGTCTTGGCTCCATGACTACTCCAGGGTTAAGGCC",
                "T",
                "T",
                "TRAV36/DV7*01",
                "TRDD2*01",
                "TRDJ2*01",
                "....................................................................................................................................................................................................................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAG",
                "GAAGACAAGGTGGTACAAAGCCCTCTATCTCTGGTTGTCCACGAGGGAGACACCGTAACTCTCAATTGCAGTTATGAAGTGACTAAC..................TTTCGAAGCCTACTATGGTACAAGCAGGAAAAGAAAGCTCCC...ACATTTCTATTTATGCTAACTTCAAGT.........GGAATTGAAAAGAAGTCA...............GGAAGACTAAGTAGCATATTAGATAAGAAAGAACTTTCCAGCANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCTTCCTANNNNNNNNNNNNNNNNNNNNNNTTGGAAAG",
                "CATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTT",
                "HCSHFLDPYSALCV",
                "324S183N2=1X1=1X6=3X2=1X1=3X2=1X10=",
                "416S1N7=",
                "445S21N8=",
                "F",
                "T",
                "TRA",
                42,
                58,
                22,
                325,
                358,
                1,
                262,
                417,
                423,
                2,
                8,
                446,
                453,
                22,
                29,
                23.7,
                0.70588,
                2.6,
                14.1,
                1,
                0.72,
                16.1,
                1,
                33,
                "..............................................................................",
                "...................................................",
                ".................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACAT",
                "GTTGGAAAG",
                "....................................",
                "..............................",
                "TGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGT",
                "CAACCTCCATGTTGAC-1",
                "TRDC",
                1232,
                7,
                "",
                "",
                "TRDJ1",
                "",
                "",
            ],
            [
                "CACCTTGCAGGTGCCT-1_contig_1",
                "TCCTTCACCAGACAAGCGACATTTGTTCCATTTTTCATGACAAAAACGGATGGTTTGGTATGAGGCTGACTTCTTGGTTCCACAGTCACACGGGTTCCTTTTCCAAAGATGAGTTTATCGGTGTCACAGCACTTGAGGACGTTCCAAAAACCTCAGCGCTGCTCCCACCTACCTTCTTCCCGTCAGCTTTAGGACTGGAGGATCTTTAAGGCTCAGCTCCATGAAAGATGGCACTTGTTGCTTACCCAGCCGGTTCATAGATGGTTAATCATCTGCAGGTAGGGACAGGTTTGCTTTCGGAAGGCTGTTGTCTGAGGAGTATGGAGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAGCTGGCACTGTCAAAGGGGTGGGTCTTGGCTCCATGACTACTCCAGGGTTAAGGCCTTTACAGACACCAGTGCTGCTGCTTTAAAGGCAGATAACGTGTGGCACAGCAAGAGGAGAAAGGGTGGCATGAAGCACTGGGTGATTTACAGAGCAGTCAGCTTTAGAATGAGCAACTGTGGCCGGTGATTGCCTACAAAAAAATCAATCTCTC",
                "T",
                "T",
                "TRAV36/DV7*01",
                "TRDD2*01",
                "TRDJ2*01",
                "....................................................................................................................................................................................................................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAG",
                "GAAGACAAGGTGGTACAAAGCCCTCTATCTCTGGTTGTCCACGAGGGAGACACCGTAACTCTCAATTGCAGTTATGAAGTGACTAAC..................TTTCGAAGCCTACTATGGTACAAGCAGGAAAAGAAAGCTCCC...ACATTTCTATTTATGCTAACTTCAAGT.........GGAATTGAAAAGAAGTCA...............GGAAGACTAAGTAGCATATTAGATAAGAAAGAACTTTCCAGCANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCTTCCTANNNNNNNNNNNNNNNNNNNNNNTTGGAAAG",
                "CATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTT",
                "HCSHFLDPYSALCV",
                "324S183N2=1X1=1X6=3X2=1X1=3X2=1X10=",
                "416S1N7=",
                "445S21N8=",
                "F",
                "T",
                "TRA",
                42,
                58,
                22,
                325,
                358,
                1,
                262,
                417,
                423,
                2,
                8,
                446,
                453,
                22,
                29,
                23.7,
                0.70588,
                3.5,
                14.1,
                1,
                0.94,
                16.1,
                1,
                43,
                "..............................................................................",
                "...................................................",
                ".................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACAT",
                "GTTGGAAAG",
                "....................................",
                "..............................",
                "TGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGT",
                "CACCTTGCAGGTGCCT-1",
                "TRDC",
                388,
                3,
                "",
                "",
                "TRDJ1",
                "",
                "",
            ],
            [
                "CATGCCTGTACAGTGG-1_contig_1",
                "TCCTTCACCAGACAAGCGACATTTGTTCCATTTTTCATGACAAAAACGGATGGTTTGGTATGAGGCTGACTTCTTGGTTCCACAGTCACACGGGTTCCTTTTCCAAAGATGAGTTTATCGGTGTCACAGCACTTGAGGACGTTCCAAAAACCTCAGCGCTGCTCCCACCTACCTTCTTCCCGTCAGCTTTAGGACTGGAGGATCTTTAAGGCTCAGCTCCATGAAAGATGGCACTTGTTGCTTACCCAGCCGGTTCATAGATGGTTAATCATCTGCAGGTAGGGACAGGTTTGCTTTCGGAAGGCTGTTGTCTGAGGAGTATGGAGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAGCTGGCACTGTC",
                "T",
                "T",
                "TRAV36/DV7*01",
                "TRDD2*01",
                "TRDJ2*01",
                "....................................................................................................................................................................................................................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAG",
                "GAAGACAAGGTGGTACAAAGCCCTCTATCTCTGGTTGTCCACGAGGGAGACACCGTAACTCTCAATTGCAGTTATGAAGTGACTAAC..................TTTCGAAGCCTACTATGGTACAAGCAGGAAAAGAAAGCTCCC...ACATTTCTATTTATGCTAACTTCAAGT.........GGAATTGAAAAGAAGTCA...............GGAAGACTAAGTAGCATATTAGATAAGAAAGAACTTTCCAGCANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCTTCCTANNNNNNNNNNNNNNNNNNNNNNTTGGAAAG",
                "CATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTT",
                "HCSHFLDPYSALCV",
                "324S183N2=1X1=1X6=3X2=1X1=3X2=1X10=",
                "416S1N7=",
                "445S21N8=",
                "F",
                "T",
                "TRA",
                42,
                58,
                22,
                325,
                358,
                1,
                262,
                417,
                423,
                2,
                8,
                446,
                453,
                22,
                29,
                23.7,
                0.70588,
                2.4,
                14.1,
                1,
                0.81,
                16.1,
                1,
                30,
                "..............................................................................",
                "...................................................",
                ".................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACAT",
                "GTTGGAAAG",
                "....................................",
                "..............................",
                "TGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGT",
                "CATGCCTGTACAGTGG-1",
                "TRDC",
                534,
                4,
                "",
                "",
                "TRDJ1",
                "",
                "",
            ],
            [
                "CCATGTCGTGATGATA-1_contig_1",
                "TCCTTCACCAGACAAGCGACATTTGTTCCATTTTTCATGACAAAAACGGATGGTTTGGTATGAGGCTGACTTCTTGGTTCCACAGTCACACGGGTTCCTTTTCCAAAGATGAGTTTATCGGTGTCACAGCACTTGAGGACGTTCCAAAAACCTCAGCGCTGCTCCCACCTACCTTCTTCCCGTCAGCTTTAGGACTGGAGGATCTTTAAGGCTCAGCTCCATGAAAGATGGCACTTGTTGCTTACCCAGCCGGTTCATAGATGGTTAATCATCTGCAGGTAGGGACAGGTTTGCTTTCGGAAGGCTGTTGTCTGAGGAGTATGGAGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAGCTGGCACTGTCAAAGGGGTGGGTCTTGGCTCCATGACTACTCCAGGGTTC",
                "T",
                "T",
                "TRAV36/DV7*01",
                "TRDD2*01",
                "TRDJ2*01",
                "....................................................................................................................................................................................................................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAG",
                "GAAGACAAGGTGGTACAAAGCCCTCTATCTCTGGTTGTCCACGAGGGAGACACCGTAACTCTCAATTGCAGTTATGAAGTGACTAAC..................TTTCGAAGCCTACTATGGTACAAGCAGGAAAAGAAAGCTCCC...ACATTTCTATTTATGCTAACTTCAAGT.........GGAATTGAAAAGAAGTCA...............GGAAGACTAAGTAGCATATTAGATAAGAAAGAACTTTCCAGCANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCTTCCTANNNNNNNNNNNNNNNNNNNNNNTTGGAAAG",
                "CATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTT",
                "HCSHFLDPYSALCV",
                "324S183N2=1X1=1X6=3X2=1X1=3X2=1X10=",
                "416S1N7=",
                "445S21N8=",
                "F",
                "T",
                "TRA",
                42,
                58,
                22,
                325,
                358,
                1,
                262,
                417,
                423,
                2,
                8,
                446,
                453,
                22,
                29,
                23.7,
                0.70588,
                2.6,
                14.1,
                1,
                0.88,
                16.1,
                1,
                32,
                "..............................................................................",
                "...................................................",
                ".................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACAT",
                "GTTGGAAAG",
                "....................................",
                "..............................",
                "TGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGT",
                "CCATGTCGTGATGATA-1",
                "TRDC",
                90,
                7,
                "",
                "",
                "TRDJ1",
                "",
                "",
            ],
            [
                "TCTTCGGCAGACAGGT-1_contig_1",
                "TCCTTCACCAGACAAGCGACATTTGTTCCATTTTTCATGACAAAAACGGATGGTTTGGTATGAGGCTGACTTCTTGGTTCCACAGTCACACGGGTTCCTTTTCCAAAGATGAGTTTATCGGTGTCACAGCACTTGAGGACGTTCCAAAAACCTCAGCGCTGCTCCCACCTACCTTCTTCCCGTCAGCTTTAGGACTGGAGGATCTTTAAGGCTCAGCTCCATGAAAGATGGCACTTGTTGCTTACCCAGCCGGTTCATAGATGGTTAATCATCTGCAGGTAGGGACAGGTTTGCTTTCGGAAGGCTGTTGTCTGAGGAGTATGGAGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAGCTGGCACTGTCAAAGGGGGTGGGTCTTGGCTCCATGACTACTCCAGGGTTAAGG",
                "T",
                "T",
                "TRAV36/DV7*01",
                "TRDD2*01",
                "TRDJ2*01",
                "....................................................................................................................................................................................................................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTTGGAAAG",
                "GAAGACAAGGTGGTACAAAGCCCTCTATCTCTGGTTGTCCACGAGGGAGACACCGTAACTCTCAATTGCAGTTATGAAGTGACTAAC..................TTTCGAAGCCTACTATGGTACAAGCAGGAAAAGAAAGCTCCC...ACATTTCTATTTATGCTAACTTCAAGT.........GGAATTGAAAAGAAGTCA...............GGAAGACTAAGTAGCATATTAGATAAGAAAGAACTTTCCAGCANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCTTCCTANNNNNNNNNNNNNNNNNNNNNNTTGGAAAG",
                "CATTGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGTGTT",
                "HCSHFLDPYSALCV",
                "324S183N2=1X1=1X6=3X2=1X1=3X2=1X10=",
                "416S1N7=",
                "445S21N8=",
                "F",
                "T",
                "TRA",
                42,
                58,
                22,
                325,
                358,
                1,
                262,
                417,
                423,
                2,
                8,
                446,
                453,
                22,
                29,
                23.7,
                0.70588,
                2.6,
                14.1,
                1,
                0.72,
                16.1,
                1,
                32,
                "..............................................................................",
                "...................................................",
                ".................................AGGACCATATTTCCTATGGCGGACCTTTCCAGCACAGTACACTTGATGCAGTGCTCTGAACTGTTGCTTTCTATTGTTTTACAT",
                "GTTGGAAAG",
                "....................................",
                "..............................",
                "TGCTCTCACTTCCTAGACCCATACTCAGCTCTCTGT",
                "TCTTCGGCAGACAGGT-1",
                "TRDC",
                804,
                5,
                "",
                "",
                "TRDJ1",
                "",
                "",
            ],
        ],
        columns=[
            "sequence_id",
            "sequence",
            "rev_comp",
            "productive",
            "v_call",
            "d_call",
            "j_call",
            "sequence_alignment",
            "germline_alignment",
            "junction",
            "junction_aa",
            "v_cigar",
            "d_cigar",
            "j_cigar",
            "stop_codon",
            "vj_in_frame",
            "locus",
            "junction_length",
            "np1_length",
            "np2_length",
            "v_sequence_start",
            "v_sequence_end",
            "v_germline_start",
            "v_germline_end",
            "d_sequence_start",
            "d_sequence_end",
            "d_germline_start",
            "d_germline_end",
            "j_sequence_start",
            "j_sequence_end",
            "j_germline_start",
            "j_germline_end",
            "v_score",
            "v_identity",
            "v_support",
            "d_score",
            "d_identity",
            "d_support",
            "j_score",
            "j_identity",
            "j_support",
            "fwr1",
            "fwr2",
            "fwr3",
            "fwr4",
            "cdr1",
            "cdr2",
            "cdr3",
            "cell_id",
            "c_call",
            "consensus_count",
            "umi_count",
            "v_call_10x",
            "d_call_10x",
            "j_call_10x",
            "junction_10x",
            "junction_10x_aa",
        ],
    )
    return airr


@pytest.fixture
def fasta_10x_travdv():
    """travdv 10x fasta"""
    seq = {
        "ACCAGTAAGCGTCAAG-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGCGTGGGGATACTCCTGGGACACCCGACAGATGTTTTTCGGAACTGGCATCAAACTCTTCGTGGAGCCCCGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA",
        "AACGTTGGTCCGTCAG-1_contig_1": "AGCTGTGAGGTCTGGTTCCCCGACGTGCTGCAGCAAGTGCCTTTGCCCTGCCTGTGGGCTCCCTCCATGGCCAACTCTGCTATGGACACCAGACTACTCTGCTGTGCGGTCATCTGTCTTCTGGGGGCAGGTCTCTCAAATGCCGGCGTCATGCAGAACCCAAGACACCTGGTCAGGAGGAGGGGACAGGAGGCAAGACTGAGATGCAGCCCAATGAAAGGACACAGTCATGTTTACTGGTATCGGCAGCTCCCAGAGGAAGGTCTGAAATTCATGGTTTATCTCCAGAAAGAAAATATCATAGATGAGTCAGGAATGCCAAAGGAACGATTTTCTGCTGAATTTCCCAAAGAGGGCCCCAGCATCCTGAGGATCCAGCAGGTAGTGCGAGGAGATTCGGCAGCTTATTTCTGTGCCAGCTCGGGACAGGGGAACACTGAAGCTTTCTTTGGACAAGGCACCAGACTCACAGTTGTAGAGGACCTGAACAAGGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "AACGTTGGTCCGTCAG-1_contig_2": "GCACTCCTTGCAAGTCTTTACTAGTAGCTGTTGCTTCTGTTCTCTGGAAATTCTTTAAACCTAGAATCAGACACAAAAACTGAACTCTGGGTCCACAATCCTCATTTGTCCTTGAAGTATGAGGCTGGTGGCAAGAGTAACTGTGTTTCTGACCTTTGGAACTATAATTGATGCTAAGACCACCCAGCCCCCCTCCATGGATTGCGCTGAAGGAAGAGCTGCAAACCTGCCTTGTAATCACTCTACCATCAGTGGAAATGAGTATGTGTATTGGTATCGACAGATTCACTCCCAGGGGCCACAGTATATCATTCATGGTCTAAAAAACAATGAAACCAATGAAATGGCCTCTCTGATCATCACAGAAGACAGAAAGTCCAGCACCTTGATCCTGCCCCACGCTACGCTGAGAGACACTGCTGTGTACTATTGCATCGTCAGAGTCGCTACTGGAGCCAATAGTAAGCTGACATTTGGAAAAGGAATAACTCTGAGTGTTAGACCAGATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "AACGTTGGTCCGTCAG-1_contig_4": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "AATCGGTTCCGTCATC-1_contig_1": "AATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGCCGACAAGCTCATCTTTGGGACTGGGACCAGATTACAAGTCTTTCCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "ACGTCAAGTGATAAGT-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGTATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "ACGTCAAGTGATAAGT-1_contig_2": "GCCCTGCCTGTGGGCTCCCTCCATGGCCAACTCTGCTATGGACACCAGACTACTCTGCTGTGCGGTCATCTGTCTTCTGGGGGCAGGTCTCTCAAATGCCGGCGTCATGCAGAACCCAAGACACCTGGTCAGGAGGAGGGGACAGGAGGCAAGACTGAGATGCAGCCCAATGAAAGGACACAGTCATGTTTACTGGTATCGGCAGCTCCCAGAGGAAGGTCTGAAATTCATGGTTTATCTCCAGAAAGAAAATATCATAGATGAGTCAGGAATGCCAAAGGAACGATTTTCTGCTGAATTTCCCAAAGAGGGCCCCAGCATCCTGAGGATCCAGCAGGTAGTGCGAGGAGATTCGGCAGCTTATTTCTGTGCCAGCTCACGGGACAGGGGAAACATTCAGTACTTCGGCGCCGGGACCCGGCTCTCAGTGCTGGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "ACGTCAAGTGATAAGT-1_contig_3": "GATGGGAGTGATCACGTCACAAGGAGGTTGCTGACGGGGGCTAGGAGAGTGGGTGAAGAAGCCTTACAGAAAAAGCTACCACTACGATTTCTTTCTGAGCCAACCATGTGCCTCAGACTTCTCTGCTGTGTGGCCATTTCTTTCTGGGGAGCCAGGCTCCACGGACACCAAGGTCACCCAGAGACCTAGACTTCTGGCCAAAGCAAGTGAACAGAAAGCAAAGATGGATTGTGTTCCTATAAAAGCACATAGTTATGTTTACTGGTATCGTAAGAAGCTGGAAGAAGAGCTCAAGTTTTTGGTTTACTTTCAGAACGAAGAACTTATTCAGAAAGCAGAAATAATCAATGAGCGATTTTTAGCCCAATGCTCCAAAAACTCATCCTGTACCTTGGAGATCCAGTCCACGGAGTCAGGGGACACAGCACTGTATTTCTGTGCCAGCAGCAAAGGGGGCACTGAAGCTTTCTTTGGACAAGGCACCAGACTCACAGTTGTAGAGGACCTGAACAAGGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "AGACGTTGTCTAGGTT-1_contig_1": "AGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGCCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "AGCATACCATCACAAC-1_contig_1": "TATGGGGATGTTCACAGAGGGCCTGGTCTGGAATATTCCACATCTGCTCTCACTCTGCCATGGACTCCTGGACCCTCTGCTGTGTGTCCCTTTGCATCCTGGTAGCAAAGCACACAGATGCTGGAGTTATCCAGTCACCCCGGCACGAGGTGACAGAGATGGGACAAGAAGTGACTCTGAGATGTAAACCAATTTCAGGACACGACTACCTTTTCTGGTACAGACAGACCATGATGCGGGGACTGGAGTTGCTCATTTACTTTAACAACAACGTTCCGATAGATGATTCAGGGATGCCCGAGGATCGATTCTCAGCTAAGATGCCTAATGCATCATTCTCCACTCTGAAGATCCAGCCCTCAGAACCCAGGGACTCAGCTGTGTACTTCTGTGCCAGCAGTTTGGGGGGCAATGAGCAGTTCTTCGGGCCAGGGACACGGCTCACCGTGCTAGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "AGCATACCATCACAAC-1_contig_2": "GCCAGGTTCACTTCACAGTACAGAGTCCTGAAAATAAAGAAGAAAATTTTTTTTTATCTAGAAAAAGAACCAAGCATGTCACTTTCTAGCCTGCTGAAGGTGGTCACAGCTTCACTGTGGCTAGGACCTGGCATTGCCCAGAAGATAACTCAAACCCAACCAGGAATGTTCGTGCAGGAAAAGGAGGCTGTGACTCTGGACTGCACATATGACACCAGTGATCCAAGTTATGGTCTATTCTGGTACAAGCAGCCCAGCAGTGGGGAAATGATTTTTCTTATTTATCAGGGGTCTTATGACCAGCAAAATGCAACAGAAGGTCGCTACTCATTGAATTTCCAGAAGGCAAGAAAATCCGCCAACCTTGTCATCTCCGCTTCACAACTGGGGGACTCAGCAATGTATTTCTGTGCAATGGGATTTCTAACGACTACAAGCTCAGCTTTGGAGCCGGAACCACAGTAACTGTAAGAGCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "AGTGGGATCGAGAACG-1_contig_1": "AGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGTATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "CGACTTCGTCTAGGTT-1_contig_1": "GCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "GCATACAAGGGTCGAT-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGTATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "GGGAGATTCACCGGGT-1_contig_1": "GCCCACAAGGGCCTCATCTAGGTGAAGGCTCCACCTGCCCCACCCTGCCATGGCCACCAGGCTCCTCTGCTGTGTGGTTCTTTGTCTCCTGGGAGAAGACAGACCATGATGCAAGGACTGGAGTTGCTGGCTTACTTCCGCAACCGGGCTCCTCTAGATGATTCGGGGATGCCGAAGGATCGATTCTCAGCAGAGATGCCTGATGCAACTTTAGCCACTCTGAAGATCCAGCCCTCAGAACCCAGGGACTCAGCTGTGTATTTTTGTGCTAGTGGTTTGAACAGCTACGAGCAGTACTTCGGGCCGGGCACCAGGCTCACGGTCACAGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "TAAGCGTAGCAGGTCA-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TAAGCGTAGCAGGTCA-1_contig_2": "AGAGGCCCCATCTCAGACCCGAGGCTAGCATGGGCTGCAGGCTGCTCTGCTGTGCGGTTCTCTGTCTCCTGGGAGCGGTCCCCATGGAAACGGGAGTTACGCAGACACCAAGACACCTGGTCATGGGAATGACAAATAAGAAGTCTTTGAAATGTGAACAACATCTGGGGCATAACGCTATGTATTGGTACAAGCAAAGTGCTAAGAAGCCACTGGAGCTCATGTTTGTCTACAACTTTAAAGAACAGACTGAAAACAACAGTGTGCCAAGTCGCTTCTCACCTGAATGCCCCAACAGCTCTCACTTATTCCTTCACCTACACACCCTGCAGCCAGAAGACTCGGCCCTGTATCTCTGTGCCAGCAGCCAAGATACAGGGTACGAGCAGTACTTCGGGCCGGGCACCAGGCTCACGGTCACAGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "TACGGATAGATATGGT-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TAGACCAAGCGATAGC-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTTGGGACTGGAACAAGACTCACCATCATACCCAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TAGTTGGGTCTGGAGA-1_contig_1": "AGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGCCGACAAGCTCATCTTTGGGACTGGGACCAGATTACAAGTCTTTCCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TCAGGTATCATTGCGA-1_contig_1": "ATATGTAAAATGAAGGGTCTGTGGAAGGACATGAATAAAGCACAGGAGGTTGAAGTCAGATTTGCAGCTTTCTAGGCAGGAGACAAGACAATCTGCATCTTCACAGGAGGGATGGCCATGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGCGCCGACAAGCTCATCTTTGGGACTGGGACCAGATTACAAGTCTTTCCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TCTCATATCCTCATTA-1_contig_1": "AAAAAAAAAAAAAACCCATTCAGGAAATAATTCTTTGCTGATAAGGATGCTCCTTGAACATTTATTAATAATCTTGTGGATGCAGCTGACATGGGTCAGTGGTCAACAGCTGAATCAGAGTCCTCAATCTATGTTTATCCAGGAAGGAGAAGATGTCTCCATGAACTGCACTTCTTCAAGCATATTTAACACCTGGCTATGGTACAAGCAGGACCCTGGGGAAGGTCCTGTCCTCTTGATAGCCTTATATAAGGCTGGTGAATTGACCTCAAATGGAAGACTGACTGCTCAGTTTGGTATAACCAGAAAGGACAGCTTCCTGAATATCTCAGCATCCATACCTAGTGATGTAGGCATCTACTTCTGTGCTGGGCGGGCCAACCTCTTCTTTGGGACTGGAACGAGACTCACCGTTATTCCCTATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TCTCATATCCTCATTA-1_contig_2": "AGGTAGAGACTCCAGGATCATCCACTGAGCACTGGACATAAGGAAGGCTGCATGGGGAGGACTCAGGACAGTGACATCACAGGATACCCCTCCTATTAGGAAAATCAAGGCCCAGAATTCACTCGGCTCTTCCCCAGGAGGACCAAGCCCTGAATCAGGTGCAGTGCTGCCTGCCCCACTGTGCCATGGGCCCTGGGCTCCTCTGCTGGGCGCTGCTTTGTCTTCTGGGAGCAGGCTCAGTGGAGACTGGAGTCACCCAAAGTCCCACACACCTGATCAAAACGAGAGGACAGCAAGTGACTCTGAGATGCTCTTCTCAGTCTGGGCACAACACTGTGTCCTGGTACCAACAGGCCCTGGGTCAGGGGCCCCAGTTTATCTTTCAGTATTATAGGGAGGAAGAGAATGGCAGAGGAAACTTCCCTCCTAGATTCTCAGGTCTCCAGTTCCCTAATTATAGCTCTGAGCTGAATGTGAACGCCTTGGAGCTGGACGACTCGGCCCTGTATCTCTGTGCCAGCAGCTTGCTGACAAAAAACTATGGCTACACCTTCGGTTCGGGGACCAGGTTAACCGTTGTAGAGGACCTGAACAAGGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "TCTCATATCCTCATTA-1_contig_3": "TTATCTGGGGCTCCTGGGGGCATCAGTGCTGATTCTGTGGCTTCAGCCAGACTGGGTAAACAGTCAACAGAAGAATGATGACCAGCAAGTTAAGCAAAATTCACCATCCCTGAGCGTCCAGGAAGGAAGAATTTCTATTCTGAACTGTGACTATACTAACAGCATGTTTGATTATTTCCTATGGTACAAAAAATACCCTGCTGAAGGTCCTACATTCCTGATATCTATAAGTTCCATTAAGGATAAAAATGAAGATGGAAGATTCACTGTCTTCTTAAACAAAAGTGCCAAGCACCTCTCTCTGCACATTGTGCCCTCCCAGCCTGGAGACTCTGCAGTGTACTTCTGTGCAGCAAGGAGGGGTATAGTGGAGGTAGCAACTATAAACTGACATTTGGAAAAGGAACTCTCTTAACCGTGAATCCAAGCAGGAGGTCGGAAAGAATAAACAATGAGAGTCACATTAAAAACACAAAATCCTACGGAAATACTGAAGAATGAGTCTCAGCACTAAGGAAAAGCCTCCAGCAGCTCCTGCTTTCTGAGGGTGAAGGATAGACGCTGTGGCTCTGCATGACTCACTAGCACTCTATCACGGCCATATTCTGGCAGGATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "AACGTTGGTCCGTCAG-1_contig_3": "TAACGTTGGTCCGTTAGTGTTGACGTATTTATTATTTGGGGGAGAGGGGTGGGTACTGGAGAAGACCAGCCCCTTCGCCAAACAGCCTTACAAAGACATCCAGCTCTAAGGAGCTCAAAACATCCTGAGGACAGTGCCTGGAGGTGAGAAGGAAGCCCCCGGCCTGGTCCATACCCCACCACCAACTTGCATAATGGGGGGTGATGTCACCCACCCTCCACTCCCCTCAAAGGAGCAGCTGCTCTGGTGGTCTCTCCCAGGCTCTGGGGGCGGACCCATGGGAGGGGCTGTTTTTGTACAAAGCTGTAACATTGTGGGGACAGGGGCCAAAAACATTCAGTACTTCGGCGCCGGGACCCGGCTCTCAGTGCTGGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "CGTTGGGAGCGTTTAC-1_contig_1": "TATTTGGGGAGGAGGAATCAGTCCCTCCAGCTGCACATGGTCACAGCTGCTCTAAGCCCTGGAGCTCCTTCCCAGTAATACCCTTGCTCCTGTGCCTGCACAGGGAAAGGAGATGCCTGCCTGCCTGCCTGCCTACCGGTCTGCCTTTTGGTCCGCCGCTTGGTGGGGCCCCAGGTGTAACCTCTGGCTGTCTCTTCTGCCTGGTTTTTGTTGAGCTTCCTATCACAGTGGAACACCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "GATCGTACAAGCTGTT-1_contig_1": "TGGGGAGGAATCAGTCCCTCCAGCTGCACATGGTCACAGCTGCTCTAAGCCCTGGAGCTCCTTCCCAGTAATACCCTTGCTCCTGTGCCTGCACAGGGAAAGGAGATGCCTGCCTGCCTGCCTGCCTACCGGTCTGCCTTTTGGTCCGCCGCTTGGTGGGGCCCCAGGTGTAACCTCTGGCTGTCTCTTCTGCCTGGTTTTTGTTGAGCTTCCTATCACAGTGGAACACCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "GGCGTGTTCTGTGCAA-1_contig_1": "AGATGGGGCCCTCCAGCTGCACATGGTCACAGCTGCTCTAAGCCCTGGAGCTCCTTCCCAGTAATACCCTTGCTCCTGTGCCTGCACAGGGAAAGGAGATGCCTGCCTGCCTGCCTGCCTACCGGTCTGCCTTTTGGTCCGCCGCTTGGTGGGGCCCCAGGTGTAACCTCTGGCTGTCTCTTCTGCCTGGTTTTTGTTGAGCTTCCTATCACAGTGGAACACCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAAGCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "GGTATTGTCATTGCGA-1_contig_1": "ATCTGGGGCTGCAATATCGCCCAGCTCCAGCCGTTCCAGTACCACCAATGCCCCTTCAGATTTCAAATCCACTGTGTTGTCCCCCAGCCAAGTGGATTCTCCTCTGCAAATTGGTGGTGGCCTCATGCAAGATCCAGGTTACCGTGTCCAGCTAACTCGAGACAGGAAAAGATAGGCTCAGGAAAGAGAGGAAGGGTGTGCCCTCTGTCTGTGCTAAGGGAGGTGGGGAAGGAGAAGGAATTCTGGGCAGCCCCTTCCCACTGTGCTCCTACAATGAGCAGTTCTTCGGGCCAGGGACACGGCTCACCGTGCTAGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "GGTATTGTCATTGCGA-1_contig_2": "GGTATTGTCATTGCGAGGCCTTACACTTTCGTCTCTGGGGACTCTGAGAGGGGCTGTGCTGAGAGGCGCTGCTGGGCGTCTGGGCGGAGGACTCCTGGTTCTGGGTGCTGGGAGAGCGATGGGGCTCTCAGCGGTGGGAAGGACCCGAGCTGAGTCTGGGACAGCAGAGCGGGCAGCACCGGTTTTTGTCCTGGGCCTCCAGGCTGTGAGCACAGATACGCAGTATTTTGGCCCAGGCACCCGGCTGACAGTGCTCGAGGACCTGAAAAACGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "GGTATTGTCATTGCGA-1_contig_3": "TGGGGCTCTCAGGAGGAATCAGTCCCTCCAGCTGCACATGGTCACAGCTGCTCTAAGCCCTGGAGCTCCTTCCCAGTAATACCCTTGCTCCTGTGCCTGCACAGGGAAAGGAGATGCCTGCCTGCCTGCCTGCCTACCGGTCTGCCTTTTGGTCCGCCGCTTGGTGGGGCCCCAGGTGTAACCTCTGGCTGTCTCTTCTGCCTGGTTTTTGTTGAGCTTCCTATCACAGTGGAACACCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TAAGCGTAGCAGGTCA-1_contig_3": "TTAGCTGGGGATCCTGAGGACAGTGCCTGGAGGTGAGAAGGAAGCCCCCGGCCTGGTCCATACCCCACCACCAACTTGCATAATGGGGGGTGATGTCACCCACCCTCCACTCCCCTCAAAGGAGCAGCTGCTCTGGTGGTCTCTCCCAGGCTCTGGGGGCGGACCCATGGGAGGGGCTGTTTTTGTACAAAGCTGTAACATTGTGGGGACAGGGCACTGAAGCTTTCTTTGGACAAGGCACCAGACTCACAGTTGTAGAGGACCTGAACAAGGTGTTCCCACCCGAGGTCGCTGTGTTTGAGCCATCAGA",
        "TGAGCATTCCACGAAT-1_contig_1": "TGGGGAGTGTGTGTCTACCCTTCTTGCCTTTGTTTTCAGACCCTTCCTCTCTAGATCCTATTCCTTCCATCTCCAACTCCTGTATGTAAATCCCTGCTCTCTCCTTTGAATTCCTGACTGCTATGCCCATGCCCAGCACCGGCACCCCCACGCACACTGCTGCCTGCCTCTGTAACCATCAGCTGTATTTCTGAGGACTTTCCTAGCTCCTTGAGTGATCCTCAAATGGCTCAGGGCCCTGAGGGAATCTTTGCTGAGGGTTGGCCCCAGCAGAGCCGCTGCAGAGGTAGAGGTGGCCACCAAGGGACACAGACTGCCTGCATGAAGGCTGGAGCTGGGCCCAGGATGAGGAAAAGCCTCAGGAAGGAAGGGCTGACACGAAATAAGGAATACCATGGCATTCATGAGATGTGCGTCTGAATCCTCTCTCTTGCCTGAGAAGCTTTAGCTTCCACCTTGAGACACAAAACATGTGGTTATGAAGAGATGACAAGGTTTTTGTAAAAGAATGAGCCATTGTGGATAGGCTTTGGGAATGTGCTGCATTGCGGGTCCGGCACTCAAGTGATTGTTTTACCACATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
        "TGGCGCACATTGTGCA-1_contig_1": "ACTCATTCTTATTTCTGAGATGGGGCCTTTTAATCCCCTCTCAGGAGGAATCAGTCCCTCCAGCCGCACATGGTCACAGCTGCTCTAAGCCCTGGAGCTCCTTCCCAGTAATACCCTTGCTCCTGTGCCTGCACAGGGAAAGGAGATGCCTGCCTGCCTGCCTGCCTACCGGTCTGCCTTTTGGTCCGCCGCTTGGTGGGGCCCCAGGTGTAACCTCTGGCTGTCTCTTCTGCCTGGTTTTTGTTGAGCTTCCTATCACAGTGGAACACCGGTAACCAGTTCTATTTTGGGACAGGGACAAGTTTGACGGTCATTCCAAATATCCAGAACCCTGACCCTGCCGTGTACCAGCTGAGAGACT",
    }
    return seq


@pytest.fixture
def annotation_10x_travdv():
    """travdv 10x annot"""
    annot = pd.DataFrame(
        [
            [
                "ACCAGTAAGCGTCAAG-1",
                "FALSE",
                "ACCAGTAAGCGTCAAG-1_contig_1",
                "TRUE",
                611,
                "Multi",
                "TRAV29/DV5",
                "None",
                "TRDJ3",
                "TRDC",
                "TRUE",
                "FALSE",
                "CAASAWGYSWDTRQMFF",
                "TGTGCAGCAAGCGCGTGGGGATACTCCTGGGACACCCGACAGATGTTTTTC",
                817,
                18,
                "None",
                "None",
            ],
            [
                "AACGTTGGTCCGTCAG-1",
                "TRUE",
                "AACGTTGGTCCGTCAG-1_contig_1",
                "TRUE",
                530,
                "TRB",
                "TRBV18",
                "None",
                "TRBJ1-1",
                "TRBC1",
                "TRUE",
                "TRUE",
                "CASSGQGNTEAFF",
                "TGTGCCAGCTCGGGACAGGGGAACACTGAAGCTTTCTTT",
                46048,
                24,
                "clonotype3",
                "clonotype3_consensus_2",
            ],
            [
                "AACGTTGGTCCGTCAG-1",
                "TRUE",
                "AACGTTGGTCCGTCAG-1_contig_2",
                "TRUE",
                548,
                "TRA",
                "TRAV26-1",
                "None",
                "TRAJ56",
                "TRAC",
                "TRUE",
                "TRUE",
                "CIVRVATGANSKLTF",
                "TGCATCGTCAGAGTCGCTACTGGAGCCAATAGTAAGCTGACATTT",
                15466,
                7,
                "clonotype3",
                "clonotype3_consensus_1",
            ],
            [
                "AACGTTGGTCCGTCAG-1",
                "TRUE",
                "AACGTTGGTCCGTCAG-1_contig_3",
                "TRUE",
                426,
                "TRB",
                "None",
                "None",
                "TRBJ2-4",
                "TRBC2",
                "FALSE",
                "FALSE",
                "None",
                "None",
                14010,
                7,
                "clonotype3",
                "None",
            ],
            [
                "AACGTTGGTCCGTCAG-1",
                "TRUE",
                "AACGTTGGTCCGTCAG-1_contig_4",
                "TRUE",
                561,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ49",
                "TRAC",
                "TRUE",
                "FALSE",
                "CSKRGNQFYF",
                "TGCAGCAAGCGCGGTAACCAGTTCTATTTT",
                2234,
                1,
                "clonotype3",
                "None",
            ],
            [
                "ACGTCAAGTGATAAGT-1",
                "TRUE",
                "ACGTCAAGTGATAAGT-1_contig_1",
                "TRUE",
                574,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASVSNFGNEKLTF",
                "TGTGCAGCAAGCGTATCTAACTTTGGAAATGAGAAATTAACCTTT",
                20644,
                2,
                "clonotype9",
                "clonotype9_consensus_1",
            ],
            [
                "ACGTCAAGTGATAAGT-1",
                "TRUE",
                "ACGTCAAGTGATAAGT-1_contig_2",
                "TRUE",
                486,
                "TRB",
                "TRBV18",
                "None",
                "TRBJ2-4",
                "TRBC2",
                "TRUE",
                "TRUE",
                "CASSRDRGNIQYF",
                "TGTGCCAGCTCACGGGACAGGGGAAACATTCAGTACTTC",
                12452,
                1,
                "clonotype9",
                "clonotype9_consensus_2",
            ],
            [
                "ACGTCAAGTGATAAGT-1",
                "TRUE",
                "ACGTCAAGTGATAAGT-1_contig_3",
                "TRUE",
                552,
                "TRB",
                "TRBV21-1",
                "None",
                "TRBJ1-1",
                "TRBC1",
                "TRUE",
                "FALSE",
                "CASSKGGTEAFF",
                "TGTGCCAGCAGCAAAGGGGGCACTGAAGCTTTCTTT",
                10176,
                1,
                "clonotype9",
                "None",
            ],
            [
                "AGCATACCATCACAAC-1",
                "TRUE",
                "AGCATACCATCACAAC-1_contig_1",
                "TRUE",
                505,
                "TRB",
                "TRBV12-4",
                "None",
                "TRBJ2-1",
                "TRBC2",
                "TRUE",
                "TRUE",
                "CASSLGGNEQFF",
                "TGTGCCAGCAGTTTGGGGGGCAATGAGCAGTTCTTC",
                58814,
                14,
                "clonotype13",
                "clonotype13_consensus_1",
            ],
            [
                "AGCATACCATCACAAC-1",
                "TRUE",
                "AGCATACCATCACAAC-1_contig_2",
                "TRUE",
                519,
                "TRA",
                "TRAV14/DV4",
                "None",
                "TRAJ20",
                "TRAC",
                "TRUE",
                "FALSE",
                "CNGISNDYKLSF",
                "TGCAATGGGATTTCTAACGACTACAAGCTCAGCTTT",
                16032,
                4,
                "clonotype13",
                "None",
            ],
            [
                "TAAGCGTAGCAGGTCA-1",
                "TRUE",
                "TAAGCGTAGCAGGTCA-1_contig_1",
                "TRUE",
                574,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASISNFGNEKLTF",
                "TGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTT",
                13694,
                2,
                "clonotype35",
                "clonotype35_consensus_2",
            ],
            [
                "TAAGCGTAGCAGGTCA-1",
                "TRUE",
                "TAAGCGTAGCAGGTCA-1_contig_2",
                "TRUE",
                475,
                "TRB",
                "TRBV4-2",
                "None",
                "TRBJ2-7",
                "TRBC2",
                "TRUE",
                "TRUE",
                "CASSQDTGYEQYF",
                "TGTGCCAGCAGCCAAGATACAGGGTACGAGCAGTACTTC",
                53230,
                9,
                "clonotype35",
                "clonotype35_consensus_1",
            ],
            [
                "TAAGCGTAGCAGGTCA-1",
                "TRUE",
                "TAAGCGTAGCAGGTCA-1_contig_3",
                "TRUE",
                310,
                "TRB",
                "None",
                "None",
                "TRBJ1-1",
                "TRBC1",
                "FALSE",
                "FALSE",
                "None",
                "None",
                9886,
                1,
                "clonotype35",
                "None",
            ],
            [
                "TCTCATATCCTCATTA-1",
                "TRUE",
                "TCTCATATCCTCATTA-1_contig_1",
                "TRUE",
                464,
                "TRA",
                "TRAV35",
                "None",
                "TRAJ36",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAGRANLFF",
                "TGTGCTGGGCGGGCCAACCTCTTCTTT",
                32204,
                4,
                "clonotype40",
                "clonotype40_consensus_2",
            ],
            [
                "TCTCATATCCTCATTA-1",
                "TRUE",
                "TCTCATATCCTCATTA-1_contig_2",
                "TRUE",
                634,
                "TRB",
                "TRBV5-4",
                "None",
                "TRBJ1-2",
                "TRBC1",
                "TRUE",
                "TRUE",
                "CASSLLTKNYGYTF",
                "TGTGCCAGCAGCTTGCTGACAAAAAACTATGGCTACACCTTC",
                36796,
                21,
                "clonotype40",
                "clonotype40_consensus_1",
            ],
            [
                "TCTCATATCCTCATTA-1",
                "TRUE",
                "TCTCATATCCTCATTA-1_contig_3",
                "TRUE",
                655,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ53",
                "TRAC",
                "TRUE",
                "FALSE",
                "CSKEGYSGGSNYKLTF",
                "TGCAGCAAGGAGGGGTATAGTGGAGGTAGCAACTATAAACTGACATTT",
                2266,
                1,
                "clonotype40",
                "None",
            ],
            [
                "AATCGGTTCCGTCATC-1",
                "FALSE",
                "AATCGGTTCCGTCATC-1_contig_1",
                "FALSE",
                470,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ34",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASADKLIF",
                "TGTGCAGCAAGCGCCGACAAGCTCATCTTT",
                14850,
                1,
                "None",
                "None",
            ],
            [
                "AGACGTTGTCTAGGTT-1",
                "FALSE",
                "AGACGTTGTCTAGGTT-1_contig_1",
                "FALSE",
                505,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ49",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASAGNQFYF",
                "TGTGCAGCAAGCGCCGGTAACCAGTTCTATTTT",
                13938,
                1,
                "None",
                "None",
            ],
            [
                "AGTGGGATCGAGAACG-1",
                "FALSE",
                "AGTGGGATCGAGAACG-1_contig_1",
                "FALSE",
                517,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASVSNFGNEKLTF",
                "TGTGCAGCAAGCGTATCTAACTTTGGAAATGAGAAATTAACCTTT",
                13988,
                1,
                "None",
                "None",
            ],
            [
                "CGACTTCGTCTAGGTT-1",
                "FALSE",
                "CGACTTCGTCTAGGTT-1_contig_1",
                "FALSE",
                511,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASISNFGNEKLTF",
                "TGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTT",
                11724,
                1,
                "None",
                "None",
            ],
            [
                "CGTTGGGAGCGTTTAC-1",
                "FALSE",
                "CGTTGGGAGCGTTTAC-1_contig_1",
                "TRUE",
                328,
                "TRA",
                "None",
                "None",
                "TRAJ49",
                "TRAC",
                "FALSE",
                "FALSE",
                "None",
                "None",
                47120,
                8,
                "None",
                "None",
            ],
            [
                "GATCGTACAAGCTGTT-1",
                "FALSE",
                "GATCGTACAAGCTGTT-1_contig_1",
                "TRUE",
                321,
                "TRA",
                "None",
                "None",
                "TRAJ49",
                "TRAC",
                "FALSE",
                "FALSE",
                "None",
                "None",
                41770,
                7,
                "None",
                "None",
            ],
            [
                "GCATACAAGGGTCGAT-1",
                "FALSE",
                "GCATACAAGGGTCGAT-1_contig_1",
                "FALSE",
                574,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASVSNFGNEKLTF",
                "TGTGCAGCAAGCGTATCTAACTTTGGAAATGAGAAATTAACCTTT",
                11026,
                1,
                "None",
                "None",
            ],
            [
                "GGCGTGTTCTGTGCAA-1",
                "FALSE",
                "GGCGTGTTCTGTGCAA-1_contig_1",
                "TRUE",
                314,
                "TRA",
                "None",
                "None",
                "TRAJ49",
                "TRAC",
                "FALSE",
                "FALSE",
                "None",
                "None",
                48368,
                10,
                "None",
                "None",
            ],
            [
                "GGGAGATTCACCGGGT-1",
                "FALSE",
                "GGGAGATTCACCGGGT-1_contig_1",
                "FALSE",
                380,
                "TRB",
                "TRBV12-5",
                "None",
                "TRBJ2-7",
                "TRBC2",
                "TRUE",
                "FALSE",
                "CASGLNSYEQYF",
                "TGTGCTAGTGGTTTGAACAGCTACGAGCAGTACTTC",
                13528,
                1,
                "None",
                "None",
            ],
            [
                "GGTATTGTCATTGCGA-1",
                "FALSE",
                "GGTATTGTCATTGCGA-1_contig_1",
                "TRUE",
                367,
                "TRB",
                "None",
                "None",
                "TRBJ2-1",
                "TRBC2",
                "FALSE",
                "FALSE",
                "None",
                "None",
                10558,
                1,
                "None",
                "None",
            ],
            [
                "GGTATTGTCATTGCGA-1",
                "FALSE",
                "GGTATTGTCATTGCGA-1_contig_2",
                "TRUE",
                309,
                "TRB",
                "None",
                "None",
                "TRBJ2-2P",
                "TRBC2",
                "FALSE",
                "FALSE",
                "None",
                "None",
                16258,
                2,
                "None",
                "None",
            ],
            [
                "GGTATTGTCATTGCGA-1",
                "FALSE",
                "GGTATTGTCATTGCGA-1_contig_3",
                "TRUE",
                329,
                "TRA",
                "None",
                "None",
                "TRAJ49",
                "TRAC",
                "FALSE",
                "FALSE",
                "None",
                "None",
                42944,
                2,
                "None",
                "None",
            ],
            [
                "TACGGATAGATATGGT-1",
                "FALSE",
                "TACGGATAGATATGGT-1_contig_1",
                "FALSE",
                574,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASISNFGNEKLTF",
                "TGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTT",
                9654,
                1,
                "None",
                "None",
            ],
            [
                "TAGACCAAGCGATAGC-1",
                "FALSE",
                "TAGACCAAGCGATAGC-1_contig_1",
                "FALSE",
                574,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ48",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASISNFGNEKLTF",
                "TGTGCAGCAAGCATATCTAACTTTGGAAATGAGAAATTAACCTTT",
                8998,
                1,
                "None",
                "None",
            ],
            [
                "TAGTTGGGTCTGGAGA-1",
                "FALSE",
                "TAGTTGGGTCTGGAGA-1_contig_1",
                "FALSE",
                502,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ34",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASADKLIF",
                "TGTGCAGCAAGCGCCGACAAGCTCATCTTT",
                12702,
                1,
                "None",
                "None",
            ],
            [
                "TCAGGTATCATTGCGA-1",
                "FALSE",
                "TCAGGTATCATTGCGA-1_contig_1",
                "FALSE",
                559,
                "TRA",
                "TRAV29/DV5",
                "None",
                "TRAJ34",
                "TRAC",
                "TRUE",
                "TRUE",
                "CAASADKLIF",
                "TGTGCAGCAAGCGCCGACAAGCTCATCTTT",
                5846,
                1,
                "None",
                "None",
            ],
            [
                "TGAGCATTCCACGAAT-1",
                "FALSE",
                "TGAGCATTCCACGAAT-1_contig_1",
                "TRUE",
                622,
                "TRA",
                "None",
                "None",
                "TRAJ35",
                "TRAC",
                "FALSE",
                "FALSE",
                "None",
                "None",
                44070,
                4,
                "None",
                "None",
            ],
            [
                "TGGCGCACATTGTGCA-1",
                "FALSE",
                "TGGCGCACATTGTGCA-1_contig_1",
                "TRUE",
                361,
                "TRA",
                "None",
                "None",
                "TRAJ49",
                "TRAC",
                "FALSE",
                "FALSE",
                "None",
                "None",
                46356,
                3,
                "None",
                "None",
            ],
        ],
        columns=[
            "barcode",
            "is_cell",
            "contig_id",
            "high_confidence",
            "length",
            "chain",
            "v_gene",
            "d_gene",
            "j_gene",
            "c_gene",
            "full_length",
            "productive",
            "cdr3",
            "cdr3_nt",
            "reads",
            "umis",
            "raw_clonotype_id",
            "raw_consensus_id",
        ],
    )
    return annot


@pytest.fixture
def dummy_adata_travdv():
    """Dummy anndata for tests."""
    barcodes = [
        "ACCAGTAAGCGTCAAG-1",
        "CAACCTCCATGTTGAC-1",
        "CACCTTGCAGGTGCCT-1",
        "CATGCCTGTACAGTGG-1",
        "CCATGTCGTGATGATA-1",
        "TCTTCGGCAGACAGGT-1",
        "GGGAGATTCACCGGGT-1",
        "TGAGCATTCCACGAAT-1",
        "AGCATACCATCACAAC-1",
        "CGACTTCGTCTAGGTT-1",
        "CGTTGGGAGCGTTTAC-1",
        "GGTATTGTCATTGCGA-1",
        "TACGGATAGATATGGT-1",
        "GCATACAAGGGTCGAT-1",
        "TCTCATATCCTCATTA-1",
        "AACGTTGGTCCGTCAG-1",
        "TAGTTGGGTCTGGAGA-1",
        "TAGACCAAGCGATAGC-1",
        "AATCGGTTCCGTCATC-1",
        "GATCGTACAAGCTGTT-1",
        "ACGTCAAGTGATAAGT-1",
        "GGCGTGTTCTGTGCAA-1",
        "TGGCGCACATTGTGCA-1",
        "AGACGTTGTCTAGGTT-1",
        "TAAGCGTAGCAGGTCA-1",
        "TCAGGTATCATTGCGA-1",
        "AGTGGGATCGAGAACG-1",
    ]
    obs = pd.DataFrame(index=barcodes)
    obs["sample_id"] = "sample_test"
    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format="csr"), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep="X", n_neighbors=3)

    return adata
