import os
import pathlib
import configparser
import urllib
import requests
from requests.exceptions import RequestException
import rich
from rich.console import Console
from task_forest_cli import __version__, PROG_NAME
import click

CONFIG_DIR = "task-forest"
CONFIG_FILENAME = "task-forest-config.ini"

console = Console()


class TaskForestCliError(Exception):
    pass


# This function from https://github.com/tensorflow/tensorboard/blob/master/tensorboard/uploader/util.py
# Apache 2 licensed
def get_user_config_directory():
    """Returns a platform-specific root directory for user config settings."""
    # On Windows, prefer %LOCALAPPDATA%, then %APPDATA%, since we can expect the
    # AppData directories to be ACLed to be visible only to the user and admin
    # users (https://stackoverflow.com/a/7617601/1179226). If neither is set,
    # return None instead of falling back to something that may be world-readable.
    if os.name == "nt":
        appdata = os.getenv("LOCALAPPDATA")
        if appdata:
            return appdata
        appdata = os.getenv("APPDATA")
        if appdata:
            return appdata
        return None
    # On non-windows, use XDG_CONFIG_HOME if set, else default to ~/.config.
    xdg_config_home = os.getenv("XDG_CONFIG_HOME")
    if xdg_config_home:
        return xdg_config_home
    return os.path.join(os.path.expanduser("~"), ".config")


def get_task_forest_cli_dir() -> str:
    dir = pathlib.Path(get_user_config_directory(), CONFIG_DIR)
    # Create the directory if it doesn't exist yet.
    dir.mkdir(parents=True, exist_ok=True)
    return str(dir)


def get_task_forest_cli_config_location() -> str:
    return str(pathlib.Path(get_task_forest_cli_dir(), CONFIG_FILENAME))


def get_task_forest_config():
    try:
        config = configparser.ConfigParser()
        config.sections()
        config.read(get_task_forest_cli_config_location())
        return config
    except configparser.Error:
        print_error("Unable to read config file at " + get_task_forest_cli_config_location())
        raise TaskForestCliError


def is_task_forest_server_setup() -> bool:
    try:
        config = get_task_forest_config()
        if config["default"]["api_key"] is not None and config["default"]["task_forest_server"] is not None:
            return True
        return False
    except TaskForestCliError:
        return False
    except KeyError:
        return False


def write_config_value(key, value):
    try:
        config = get_task_forest_config()
    except TaskForestCliError:
        pass

    if "default" not in config:
        config["default"] = {}
    config["default"][key] = value

    with open(get_task_forest_cli_config_location(), "w") as configfile:
        config.write(configfile)


def get_config_value(key):
    try:
        config = get_task_forest_config()
        if "default" not in config:
            return None
        if key not in config["default"]:
            return None
        return config["default"][key]
    except TaskForestCliError:
        return None


def api_req(api_url, method="GET", data={}, params={}):
    url = urllib.parse.urljoin(get_config_value("task_forest_server"), api_url)
    headers = {"X-API-KEY": get_config_value("api_key")}

    try:
        with working():
            r = requests.request(method, url, headers=headers, timeout=5, json=data, params=params)

        if r.ok:
            if len(r.text) == 0:
                # Handles cases where there's a 204 or something with non content, but success.
                return {}
            res = r.json()
            return res
        elif r.status_code == 400:
            # This is similar to a 422, but we generate this, not FastAPI.
            print_error("The input given was incorrect, please check all your parameters.", fatal=False)
            print_error("    Details: [bold]%s[/bold]" % r.json()["detail"])
        elif r.status_code == 401:
            print_error("Your credentials are not correct, please check your API key configuration.", fatal=False)
            print_error("    Details: [bold]%s[/bold]" % r.json()["detail"])
            return None
        elif r.status_code == 403:
            print_error("You do not appear to have permissions to perform that action.", fatal=False)
            print_error("    Details: [bold]%s[/bold]" % r.json()["detail"])
            return None
        elif r.status_code == 404:
            print_error("Not found.", fatal=False)
            print_error("    Details: [bold]%s[/bold]" % r.json()["detail"])
            return None
        elif r.status_code == 422:
            print_error("The input given was incorrect, please check all your parameters.")
            # The reason we don't give details here is that a 422 is normally automatically generated by the server,
            # not something we can control and to which we can add details.
            return None
        else:
            print_error("The server has returned an unknown error. Please see the server logs for more information.")
            return None
    except RequestException:
        print_error("The Task Forest server did not respond. Is the URL correct and the server at "
                    "[bold]%s[/bold] running?"
                    % get_config_value("task_forest_server"))
        return None


def _print_line(header, msg):
    rich.print(header, msg)


def print_info(msg):
    _print_line("[[bold blue]INFO[/bold blue]]", msg)


def print_success(msg):
    _print_line("[[bold green]SUCCESS[/bold green]]", msg)


def print_warning(msg):
    _print_line("[[bold yellow]WARN[/bold yellow]]", msg)


def print_error(msg, fatal=True):
    _print_line("[[bold red]ERROR[/bold red]]", msg)

    if fatal:
        quit(-1)


def print_version():
    rich.print("[bold]%s[/bold] v%s" % (PROG_NAME, __version__))


def working():
    return console.status("Working...")


# From https://stackoverflow.com/a/56043912
def default_from_context(default_name):
    class OptionDefaultFromContext(click.Option):
        def get_default(self, ctx):
            self.default = ctx.obj[default_name]
            return super(OptionDefaultFromContext, self).get_default(ctx, call=True)

    return OptionDefaultFromContext


def priority_str_to_int(priority_str: str) -> int:
    priority_str = priority_str.lower()

    if priority_str == "high":
        return 1
    elif priority_str == "medium":
        return 2
    elif priority_str == "low":
        return 3
    else:
        return 99


def priority_int_to_str(priority_int: int) -> str:
    if priority_int == 1:
        return "[red]High[/red]"
    elif priority_int == 2:
        return "[yellow]Medium[/yellow]"
    elif priority_int == 3:
        return "[blue]Low[/blue]"
    else:
        return "None"
