##############################################################################
#
# Copyright (c) 2007 Zope Foundation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################
"""Z3c development recipes

$Id: settings.py 5080 2021-01-21 14:44:34Z roger.ineichen $
"""

import re
import os

import zc.buildout
import zc.recipe.egg

from p01.recipe.setup import CHMODMixin
from p01.recipe.setup import CHOWNMixin
from p01.recipe.setup import CreatePathMixin


initialization_template = """import os
sys.argv[0] = os.path.abspath(sys.argv[0])
"""

env_template = """os.environ['%s'] = %r
"""


class SettingsRecipe(CHMODMixin, CHOWNMixin, CreatePathMixin):
    """Script installation recipe."""

    def __init__(self, buildout, name, options):
        self.egg = None
        self.buildout = buildout
        self.name = name
        self.options = options
        options['script'] = os.path.join(buildout['buildout']['bin-directory'],
                                         options.get('script', self.name),
                                         )
        if not options.get('working-directory', ''):
            options['location'] = os.path.join(
                buildout['buildout']['parts-directory'], name)

        if options.get('eggs'):
            self.egg = zc.recipe.egg.Egg(buildout, name, options)

        self.source = options.get("source")
        fName =  options.get('settings', 'settings.py')
        self.target = os.path.join(
                buildout['buildout']['parts-directory'], name, fName)
        # mode
        self.mode = options.get('mode', '0644')
        if self.mode is not None:
            try:
                self.mode = int(self.mode, 8)
            except ValueError:
                raise zc.buildout.UserError(
                    "'mode' must be an octal number: " % self.mode)
        # owner
        self.owner = options.get('owner')

    def install(self):
        dest = []
        options = self.options
        module = options['module']
        method = options.get('method', 'py')
        executable = self.buildout['buildout']['executable']

        # setup additional egg path
        if self.egg:
            extra_paths = self.egg.extra_paths
            eggs, ws = self.egg.working_set()
        else:
            extra_paths = []
            ws = []

        # replace variable with options
        template = re.sub(r"\$\{([^:]+?)\}", r"${%s:\1}" % self.name, self.source)
        self.result = self.options._sub(template, [])
        self.result = self.result.replace('\\', '/')

        # create settings.py
        self.doCreatePaths(os.path.dirname(self.target))
        target = open(self.target, "wt")
        target.write('# generated by p01.recipe.setup:settings recipe\n')
        target.write(self.result)
        target.close()
        dest.append(self.target)

        # set mode if given
        self.doChmod(self.target, self.mode)
        # set owner if given
        self.doChown(self.target, self.owner)

        # add settings location to extra_paths
        extra_paths.append(os.path.dirname(self.target))

        # setup script default vars
        arguments = options.get('arguments', '').strip()
        arguments = arguments.split(',')
        # cleanup bad win paths
        arguments = ','.join(['%s' % str(a).replace('\\', '/') for a in arguments if a])

        # setup environment
        initialization = initialization_template
        env_section = self.options.get('environment', '').strip()
        if env_section:
            env = self.buildout[env_section]
            for key, value in list(env.items()):
                initialization += env_template % (key, value)

        if options.get('initialization'):
            # append additional initialization after sys path loading if given
            initialization += options.get('initialization')

        wd = options.get('working-directory', options['location'])

        dest.extend(zc.buildout.easy_install.scripts(
            [(options['script'], module, method)],
            ws, executable, self.buildout['buildout']['bin-directory'],
            extra_paths = extra_paths,
            arguments = arguments,
            initialization = initialization,
            ))

        return dest

    update = install
