import zipfile
from os.path import expanduser
import os
import requests
import sys
import re
import csv
import bisect


UCD_URL = "https://unicode.org/Public/UCD/latest/ucd/UCD.zip"

def ucd_dir():
  ucddir = os.path.expanduser("~/.youseedee")
  if not os.path.isdir(ucddir):
    os.mkdir(ucddir)
  return ucddir

def ensure_files():
  if os.path.isfile(os.path.join(ucd_dir(), "UnicodeData.txt")):
    return

  zip_path = os.path.join(ucd_dir(), "UCD.zip")
  if not os.path.isfile(zip_path):
    with open(zip_path, "wb") as f:
      print("Downloading Unicode Character Database...")
      response = requests.get(UCD_URL, stream=True)
      total_length = response.headers.get('content-length')

      if total_length is None: # no content length header
        f.write(response.content)
      else:
        dl = 0
        total_length = int(total_length)
        for data in response.iter_content(chunk_size=4096):
          dl += len(data)
          f.write(data)
          done = int(50 * dl / total_length)
          sys.stdout.write("\r[%s%s]" % ('=' * done, ' ' * (50-done)) )
          sys.stdout.flush()

  with zipfile.ZipFile(zip_path, 'r') as zip_ref:
    zip_ref.extractall(ucd_dir())

def parse_file_ranges(filename):
  ensure_files()
  ranges = []
  with open(os.path.join(ucd_dir(), filename), "r") as f:
    for line in f:
      if re.match("^\s*#", line): continue
      if re.match("^\s*$", line): continue
      line = re.sub("#.*","",line)
      matches = re.match("^([0-9A-F]{4,})(?:\.\.([0-9A-F]{4,}))?\s*;\s*([^;]+?)\s*$", line)
      start, end, content = matches.groups()
      if end is None: end = start
      ranges.append(( int(start,16), int(end,16),content))
  return ranges

def parse_file_semicolonsep(filename):
  ensure_files()
  data = {}
  with open(os.path.join(ucd_dir(), filename), "r", newline='') as f:
    reader = csv.reader(f, delimiter=';',skipinitialspace=True)
    for row in reader:
      if len(row) < 2: continue
      if re.match("^#",row[0]): continue
      row[-1] = re.sub("\s*#.*","",row[-1])
      row[0] = int(row[0],16)
      data[row[0]] = row[1:]
  return data

def dictget(filename, codepoint):
  fileentry = database[filename]
  if not "data" in fileentry:
    fileentry["data"] = fileentry["reader"](filename)
  if not codepoint in fileentry["data"]:
    return {}
  d = fileentry["data"][codepoint]
  r = {}
  for ix, p in enumerate(database[filename]["properties"]):
    if p == "IGNORE": continue
    r[p] = d[ix]
  return r

def rangereader(filename, codepoint):
  fileentry = database[filename]
  if not "data" in fileentry:
    fileentry["data"] = list(sorted(fileentry["reader"](filename), key=lambda x:x[0]))
  range_index = bisect.bisect_right(fileentry["data"], codepoint, key=lambda x:x[0])
  rangerow = fileentry["data"][range_index-1]
  start, end = rangerow[0],rangerow[1]
  if codepoint >= start and codepoint <= end:
    data = rangerow[2:]
    r = {}
    for ix, p in enumerate(database[filename]["properties"]):
      if p == "IGNORE": continue
      r[p] = data[ix]
    return r
  return {}

database = {
  "ArabicShaping.txt": {
    "properties": [ "IGNORE", "Joining_Type", "Joining_Group" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "BidiBrackets.txt": {
    "properties": [ "Bidi_Paired_Bracket", "Bidi_Paired_Bracket_Type" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "BidiMirroring.txt": {
    "properties": [ "Bidi_Mirroring_Glyph" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "Blocks.txt": {
    "properties": [ "Block" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "CaseFolding.txt": {
    "properties": [ "Case_Folding_Status", "Case_Folding_Mapping" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "DerivedAge.txt": {
    "properties": [ "Age" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "EastAsianWidth.txt": {
    "properties": [ "East_Asian_Width" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "HangulSyllableType.txt": {
    "properties": [ "Hangul_Syllable_Type" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "IndicPositionalCategory.txt": {
    "properties": [ "Indic_Positional_Category" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "IndicSyllabicCategory.txt": {
    "properties": [ "Indic_Syllabic_Category" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "Jamo.txt": {
    "properties": [ "Jamo_Short_Name" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "LineBreak.txt": {
    "properties": [ "Line_Break" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "NameAliases.txt": {
    "properties": [ "Name_Alias" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "Scripts.txt": {
    "properties": [ "Script" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "ScriptExtensions.txt": {
    "properties": [ "Script_Extensions" ],
    "reader": parse_file_ranges, "datareader": rangereader
  },
  "SpecialCasing.txt": {
    "properties": [ "Uppercase_Mapping", "Lowercase_Mapping", "Titlecase_Mapping" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "UnicodeData.txt": {
    "properties": [ "Name", "General_Category", "Canonical_Combining_Class" ],
    "reader": parse_file_semicolonsep, "datareader": dictget
  },
  "USECategory.txt": {
     "properties": [ "USE_Category" ],
     "datareader": rangereader
   },
}

def ucd_data(codepoint):
  out = {}
  for file, props in database.items():
    out.update(props["datareader"](file, codepoint))
  return out

database["USECategory.txt"]["data"] = (
 (0x002d,0x002d, "GB"), (0x0030,0x0039, "B"),
 (0x00a0,0x00a0, "GB"), (0x00b2,0x00b3, "FMPst"),
 (0x00d7,0x00d7, "GB"), (0x0640,0x07ea, "B"),
 (0x07eb,0x07f3, "VMAbv"), (0x07fa,0x07fa, "B"),
 (0x07fd,0x07fd, "VMAbv"), (0x0840,0x0858, "B"),
 (0x0859,0x085b, "CMBlw"), (0x0900,0x0902, "VMAbv"),
 (0x0903,0x0903, "VMPst"), (0x0904,0x0939, "B"),
 (0x093a,0x093a, "VAbv"), (0x093b,0x093b, "VPst"),
 (0x093c,0x093c, "CMBlw"), (0x093d,0x093d, "B"),
 (0x093e,0x093e, "VPst"), (0x093f,0x093f, "VPre"),
 (0x0940,0x0940, "VPst"), (0x0941,0x0944, "VBlw"),
 (0x0945,0x0948, "VAbv"), (0x0949,0x094c, "VPst"),
 (0x094d,0x094d, "H"), (0x094e,0x094e, "VPre"),
 (0x094f,0x094f, "VPst"), (0x0951,0x0951, "VMAbv"),
 (0x0952,0x0952, "VMBlw"), (0x0953,0x0954, "O"),
 (0x0955,0x0955, "VAbv"), (0x0956,0x0957, "VBlw"),
 (0x0958,0x0961, "B"), (0x0962,0x0963, "VBlw"),
 (0x0966,0x097f, "B"), (0x0980,0x0980, "GB"),
 (0x0981,0x0981, "VMAbv"), (0x0982,0x0983, "VMPst"),
 (0x0985,0x09b9, "B"), (0x09bc,0x09bc, "CMBlw"),
 (0x09bd,0x09bd, "B"), (0x09be,0x09be, "VPst"),
 (0x09bf,0x09bf, "VPre"), (0x09c0,0x09c0, "VPst"),
 (0x09c1,0x09c4, "VBlw"), (0x09c7,0x09cc, "VPre"),
 (0x09cd,0x09cd, "H"), (0x09ce,0x09ce, "IND"),
 (0x09d7,0x09d7, "VPst"), (0x09dc,0x09e1, "B"),
 (0x09e2,0x09e3, "VBlw"), (0x09e6,0x09fc, "B"),
 (0x09fe,0x09fe, "FMAbv"), (0x0a01,0x0a02, "VMAbv"),
 (0x0a03,0x0a03, "VMPst"), (0x0a05,0x0a39, "B"),
 (0x0a3c,0x0a3c, "CMBlw"), (0x0a3e,0x0a3e, "VPst"),
 (0x0a3f,0x0a3f, "VPre"), (0x0a40,0x0a40, "VPst"),
 (0x0a41,0x0a42, "VBlw"), (0x0a47,0x0a4c, "VAbv"),
 (0x0a4d,0x0a4d, "H"), (0x0a51,0x0a51, "VMBlw"),
 (0x0a59,0x0a6f, "B"), (0x0a70,0x0a70, "VMAbv"),
 (0x0a71,0x0a71, "CMAbv"), (0x0a72,0x0a73, "GB"),
 (0x0a75,0x0a75, "MBlw"), (0x0a81,0x0a82, "VMAbv"),
 (0x0a83,0x0a83, "VMPst"), (0x0a85,0x0ab9, "B"),
 (0x0abc,0x0abc, "CMBlw"), (0x0abd,0x0abd, "B"),
 (0x0abe,0x0abe, "VPst"), (0x0abf,0x0abf, "VPre"),
 (0x0ac0,0x0ac0, "VPst"), (0x0ac1,0x0ac4, "VBlw"),
 (0x0ac5,0x0ac9, "VAbv"), (0x0acb,0x0acc, "VPst"),
 (0x0acd,0x0acd, "H"), (0x0ae0,0x0ae1, "B"),
 (0x0ae2,0x0ae3, "VBlw"), (0x0ae6,0x0af9, "B"),
 (0x0afa,0x0afc, "VMAbv"), (0x0afd,0x0aff, "CMAbv"),
 (0x0b01,0x0b01, "VMAbv"), (0x0b02,0x0b03, "VMPst"),
 (0x0b05,0x0b39, "B"), (0x0b3c,0x0b3c, "CMBlw"),
 (0x0b3d,0x0b3d, "B"), (0x0b3e,0x0b3e, "VPst"),
 (0x0b3f,0x0b3f, "VAbv"), (0x0b40,0x0b40, "VPst"),
 (0x0b41,0x0b44, "VBlw"), (0x0b47,0x0b4c, "VPre"),
 (0x0b4d,0x0b4d, "H"), (0x0b55,0x0b57, "VAbv"),
 (0x0b5c,0x0b61, "B"), (0x0b62,0x0b63, "VBlw"),
 (0x0b66,0x0b71, "B"), (0x0b82,0x0b82, "VMAbv"),
 (0x0b83,0x0b83, "IND"), (0x0b85,0x0bb9, "B"),
 (0x0bbe,0x0bbf, "VPst"), (0x0bc0,0x0bc0, "VAbv"),
 (0x0bc1,0x0bc2, "VPst"), (0x0bc6,0x0bcc, "VPre"),
 (0x0bcd,0x0bcd, "H"), (0x0bd7,0x0bd7, "VPst"),
 (0x0be6,0x0bef, "B"), (0x0c00,0x0c00, "VMAbv"),
 (0x0c01,0x0c03, "VMPst"), (0x0c04,0x0c04, "VMAbv"),
 (0x0c05,0x0c3d, "B"), (0x0c3e,0x0c40, "VAbv"),
 (0x0c41,0x0c44, "VPst"), (0x0c46,0x0c4c, "VAbv"),
 (0x0c4d,0x0c4d, "H"), (0x0c55,0x0c55, "VAbv"),
 (0x0c56,0x0c56, "VBlw"), (0x0c58,0x0c61, "B"),
 (0x0c62,0x0c63, "VBlw"), (0x0c66,0x0c80, "B"),
 (0x0c81,0x0c81, "VMAbv"), (0x0c82,0x0c83, "VMPst"),
 (0x0c85,0x0cb9, "B"), (0x0cbc,0x0cbc, "CMBlw"),
 (0x0cbd,0x0cbd, "B"), (0x0cbe,0x0cbe, "VPst"),
 (0x0cbf,0x0cc0, "VAbv"), (0x0cc1,0x0cc4, "VPst"),
 (0x0cc6,0x0ccc, "VAbv"), (0x0ccd,0x0ccd, "H"),
 (0x0cd5,0x0cd6, "VPst"), (0x0cde,0x0ce1, "B"),
 (0x0ce2,0x0ce3, "VBlw"), (0x0ce6,0x0cef, "B"),
 (0x0cf1,0x0cf2, "CS"), (0x0d00,0x0d01, "VMAbv"),
 (0x0d02,0x0d03, "VMPst"), (0x0d04,0x0d3a, "B"),
 (0x0d3b,0x0d3c, "VAbv"), (0x0d3d,0x0d3d, "B"),
 (0x0d3e,0x0d42, "VPst"), (0x0d43,0x0d44, "VBlw"),
 (0x0d46,0x0d4c, "VPre"), (0x0d4d,0x0d4d, "H"),
 (0x0d4e,0x0d4e, "R"), (0x0d54,0x0d56, "IND"),
 (0x0d57,0x0d57, "VPst"), (0x0d5f,0x0d61, "B"),
 (0x0d62,0x0d63, "VBlw"), (0x0d66,0x0d6f, "B"),
 (0x0d7a,0x0d7f, "IND"), (0x0d81,0x0d81, "VMAbv"),
 (0x0d82,0x0d83, "VMPst"), (0x0d85,0x0dc6, "B"),
 (0x0dca,0x0dca, "H"), (0x0dcf,0x0dd1, "VPst"),
 (0x0dd2,0x0dd3, "VAbv"), (0x0dd4,0x0dd6, "VBlw"),
 (0x0dd8,0x0dd8, "VPst"), (0x0dd9,0x0dde, "VPre"),
 (0x0ddf,0x0ddf, "VPst"), (0x0de6,0x0def, "B"),
 (0x0df2,0x0df3, "VPst"), (0x0f00,0x0f06, "B"),
 (0x0f18,0x0f19, "VBlw"), (0x0f20,0x0f33, "B"),
 (0x0f35,0x0f37, "FBlw"), (0x0f39,0x0f39, "CMAbv"),
 (0x0f3e,0x0f3e, "VPst"), (0x0f3f,0x0f3f, "VPre"),
 (0x0f40,0x0f6c, "B"), (0x0f71,0x0f71, "CMBlw"),
 (0x0f72,0x0f72, "VBlw"), (0x0f73,0x0f74, "VAbv"),
 (0x0f75,0x0f75, "VBlw"), (0x0f76,0x0f79, "VAbv"),
 (0x0f7a,0x0f7d, "VBlw"), (0x0f7e,0x0f7e, "VMAbv"),
 (0x0f7f,0x0f7f, "IND"), (0x0f80,0x0f80, "VBlw"),
 (0x0f81,0x0f81, "VAbv"), (0x0f82,0x0f83, "VMAbv"),
 (0x0f84,0x0f84, "VBlw"), (0x0f85,0x0f85, "IND"),
 (0x0f86,0x0f87, "VMAbv"), (0x0f88,0x0f8c, "B"),
 (0x0f8d,0x0fbc, "SUB"), (0x0fc6,0x0fc6, "FBlw"),
 (0x1000,0x102a, "B"), (0x102b,0x102c, "VPst"),
 (0x102d,0x102e, "VAbv"), (0x102f,0x1030, "VBlw"),
 (0x1031,0x1031, "VPre"), (0x1032,0x1035, "VAbv"),
 (0x1036,0x1036, "VMAbv"), (0x1037,0x1037, "VMBlw"),
 (0x1038,0x1038, "VMPst"), (0x1039,0x1039, "H"),
 (0x103a,0x103a, "VAbv"), (0x103b,0x103b, "MPst"),
 (0x103c,0x103c, "MPre"), (0x103d,0x103e, "MBlw"),
 (0x103f,0x1049, "B"), (0x104b,0x104e, "GB"),
 (0x1050,0x1055, "B"), (0x1056,0x1057, "VPst"),
 (0x1058,0x1059, "VBlw"), (0x105a,0x105d, "B"),
 (0x105e,0x1060, "MBlw"), (0x1061,0x1061, "B"),
 (0x1062,0x1062, "VPst"), (0x1063,0x1064, "VMPst"),
 (0x1065,0x1066, "B"), (0x1067,0x1068, "VPst"),
 (0x1069,0x106d, "VMPst"), (0x106e,0x1070, "B"),
 (0x1071,0x1074, "VAbv"), (0x1075,0x1081, "B"),
 (0x1082,0x1082, "MBlw"), (0x1083,0x1083, "VPst"),
 (0x1084,0x1084, "VPre"), (0x1085,0x1086, "VAbv"),
 (0x1087,0x108c, "VMPst"), (0x108d,0x108d, "VMBlw"),
 (0x108e,0x108e, "B"), (0x108f,0x108f, "VMPst"),
 (0x1090,0x1099, "B"), (0x109a,0x109b, "VMPst"),
 (0x109c,0x109c, "VPst"), (0x109d,0x109d, "VAbv"),
 (0x1700,0x1711, "B"), (0x1712,0x1712, "VAbv"),
 (0x1713,0x1714, "VBlw"), (0x1720,0x1731, "B"),
 (0x1732,0x1732, "VAbv"), (0x1733,0x1734, "VBlw"),
 (0x1740,0x1751, "B"), (0x1752,0x1752, "VAbv"),
 (0x1753,0x1753, "VBlw"), (0x1760,0x1770, "B"),
 (0x1772,0x1772, "VAbv"), (0x1773,0x1773, "VBlw"),
 (0x1780,0x17b3, "B"), (0x17b6,0x17b6, "VPst"),
 (0x17b7,0x17ba, "VAbv"), (0x17bb,0x17bd, "VBlw"),
 (0x17be,0x17c5, "VPre"), (0x17c6,0x17c6, "VMAbv"),
 (0x17c7,0x17c7, "VMPst"), (0x17c8,0x17c8, "VPst"),
 (0x17c9,0x17ca, "VMAbv"), (0x17cb,0x17cb, "FMAbv"),
 (0x17cc,0x17cc, "FAbv"), (0x17cd,0x17cd, "CMAbv"),
 (0x17ce,0x17ce, "FMAbv"), (0x17cf,0x17cf, "VMAbv"),
 (0x17d0,0x17d0, "FMAbv"), (0x17d1,0x17d1, "VAbv"),
 (0x17d2,0x17d2, "H"), (0x17d3,0x17d3, "FMAbv"),
 (0x17dc,0x17dc, "B"), (0x17dd,0x17dd, "FMAbv"),
 (0x17e0,0x180a, "B"), (0x180b,0x180d, "O"),
 (0x1820,0x1878, "B"), (0x1880,0x1884, "GB"),
 (0x1885,0x1886, "CMAbv"), (0x1887,0x18a8, "B"),
 (0x18a9,0x18a9, "CMBlw"), (0x18aa,0x18aa, "B"),
 (0x1900,0x1900, "GB"), (0x1901,0x191e, "B"),
 (0x1920,0x1921, "VAbv"), (0x1922,0x1922, "VBlw"),
 (0x1923,0x1924, "VPst"), (0x1925,0x1928, "VAbv"),
 (0x1929,0x192b, "SUB"), (0x1930,0x1931, "FPst"),
 (0x1932,0x1932, "VMBlw"), (0x1933,0x1938, "FPst"),
 (0x1939,0x1939, "FBlw"), (0x193a,0x193a, "VMAbv"),
 (0x193b,0x193b, "FMBlw"), (0x1946,0x19c7, "B"),
 (0x19c8,0x19c9, "VMPst"), (0x19d0,0x1a16, "B"),
 (0x1a17,0x1a18, "VAbv"), (0x1a19,0x1a19, "VPre"),
 (0x1a1a,0x1a1a, "VPst"), (0x1a1b,0x1a1b, "VAbv"),
 (0x1a20,0x1a54, "B"), (0x1a55,0x1a55, "MPre"),
 (0x1a56,0x1a56, "MBlw"), (0x1a57,0x1a57, "SUB"),
 (0x1a58,0x1a59, "FAbv"), (0x1a5a,0x1a5a, "MAbv"),
 (0x1a5b,0x1a5e, "SUB"), (0x1a60,0x1a60, "Sk"),
 (0x1a61,0x1a61, "VPst"), (0x1a62,0x1a62, "VAbv"),
 (0x1a63,0x1a64, "VPst"), (0x1a65,0x1a68, "VAbv"),
 (0x1a69,0x1a6a, "VBlw"), (0x1a6b,0x1a6b, "VAbv"),
 (0x1a6c,0x1a6c, "VBlw"), (0x1a6d,0x1a6d, "VPst"),
 (0x1a6e,0x1a72, "VPre"), (0x1a73,0x1a73, "VAbv"),
 (0x1a74,0x1a79, "VMAbv"), (0x1a7a,0x1a7a, "VAbv"),
 (0x1a7b,0x1a7c, "VMAbv"), (0x1a7f,0x1a7f, "VMBlw"),
 (0x1a80,0x1a99, "B"), (0x1b00,0x1b02, "VMAbv"),
 (0x1b03,0x1b03, "FAbv"), (0x1b04,0x1b04, "VMPst"),
 (0x1b05,0x1b33, "B"), (0x1b34,0x1b34, "CMAbv"),
 (0x1b35,0x1b35, "VPst"), (0x1b36,0x1b37, "VAbv"),
 (0x1b38,0x1b3b, "VBlw"), (0x1b3c,0x1b3d, "VAbv"),
 (0x1b3e,0x1b41, "VPre"), (0x1b42,0x1b43, "VAbv"),
 (0x1b44,0x1b44, "H"), (0x1b45,0x1b59, "B"),
 (0x1b5b,0x1b5f, "GB"), (0x1b61,0x1b61, "S"),
 (0x1b62,0x1b62, "GB"), (0x1b63,0x1b67, "S"),
 (0x1b68,0x1b68, "GB"), (0x1b69,0x1b6a, "S"),
 (0x1b6b,0x1b6b, "SMAbv"), (0x1b6c,0x1b6c, "SMBlw"),
 (0x1b6d,0x1b73, "SMAbv"), (0x1b80,0x1b80, "VMAbv"),
 (0x1b81,0x1b81, "FAbv"), (0x1b82,0x1b82, "VMPst"),
 (0x1b83,0x1ba0, "B"), (0x1ba1,0x1ba3, "SUB"),
 (0x1ba4,0x1ba4, "VAbv"), (0x1ba5,0x1ba5, "VBlw"),
 (0x1ba6,0x1ba6, "VPre"), (0x1ba7,0x1ba7, "VPst"),
 (0x1ba8,0x1ba9, "VAbv"), (0x1baa,0x1baa, "VPst"),
 (0x1bab,0x1bab, "H"), (0x1bac,0x1bad, "SUB"),
 (0x1bae,0x1be5, "B"), (0x1be6,0x1be6, "CMAbv"),
 (0x1be7,0x1be7, "VPst"), (0x1be8,0x1be9, "VAbv"),
 (0x1bea,0x1bec, "VPst"), (0x1bed,0x1bed, "VAbv"),
 (0x1bee,0x1bee, "VPst"), (0x1bef,0x1bef, "VAbv"),
 (0x1bf0,0x1bf1, "FAbv"), (0x1bf2,0x1bf3, "CMBlw"),
 (0x1c00,0x1c23, "B"), (0x1c24,0x1c25, "SUB"),
 (0x1c26,0x1c26, "VPst"), (0x1c27,0x1c29, "VPre"),
 (0x1c2a,0x1c2b, "VPst"), (0x1c2c,0x1c2c, "VBlw"),
 (0x1c2d,0x1c33, "FAbv"), (0x1c34,0x1c35, "VMPre"),
 (0x1c36,0x1c36, "FMAbv"), (0x1c37,0x1c37, "CMBlw"),
 (0x1c40,0x1c4f, "B"), (0x1cd0,0x1cd2, "VMAbv"),
 (0x1cd4,0x1cd9, "VMBlw"), (0x1cda,0x1cdb, "VMAbv"),
 (0x1cdc,0x1cdf, "VMBlw"), (0x1ce0,0x1ce0, "VMAbv"),
 (0x1ce1,0x1ce1, "VMPst"), (0x1ce2,0x1ced, "VMBlw"),
 (0x1cf2,0x1cf3, "IND"), (0x1cf4,0x1cf4, "VMAbv"),
 (0x1cf5,0x1cf6, "CS"), (0x1cf7,0x1cf7, "VMPst"),
 (0x1cf8,0x1cf9, "VMAbv"), (0x1cfa,0x1cfa, "GB"),
 (0x1dfb,0x1dfb, "FMAbv"), (0x200c,0x200c, "ZWNJ"),
 (0x200d,0x200d, "ZWJ"), (0x2010,0x2014, "GB"),
 (0x2060,0x2060, "WJ"), (0x2074,0x2084, "FMPst"),
 (0x20f0,0x20f0, "VMAbv"), (0x25cc,0x2d6f, "B"),
 (0x2d7f,0x2d7f, "H"), (0xa800,0xa801, "B"),
 (0xa802,0xa802, "VAbv"), (0xa803,0xa805, "B"),
 (0xa806,0xa806, "H"), (0xa807,0xa80a, "B"),
 (0xa80b,0xa80b, "VMAbv"), (0xa80c,0xa822, "B"),
 (0xa823,0xa824, "VPst"), (0xa825,0xa825, "VBlw"),
 (0xa826,0xa826, "VAbv"), (0xa827,0xa827, "VPst"),
 (0xa82c,0xa82c, "VBlw"), (0xa840,0xa873, "B"),
 (0xa880,0xa881, "VMPst"), (0xa882,0xa8b3, "B"),
 (0xa8b4,0xa8b4, "MPst"), (0xa8b5,0xa8c3, "VPst"),
 (0xa8c4,0xa8c4, "H"), (0xa8c5,0xa8c5, "VMAbv"),
 (0xa8d0,0xa8d9, "B"), (0xa8e0,0xa8f1, "VMAbv"),
 (0xa8f2,0xa8fe, "B"), (0xa8ff,0xa8ff, "VAbv"),
 (0xa900,0xa925, "B"), (0xa926,0xa92a, "VAbv"),
 (0xa92b,0xa92d, "VMBlw"), (0xa930,0xa946, "B"),
 (0xa947,0xa949, "VBlw"), (0xa94a,0xa94a, "VAbv"),
 (0xa94b,0xa94e, "VBlw"), (0xa94f,0xa951, "FAbv"),
 (0xa952,0xa952, "FPst"), (0xa953,0xa953, "VPst"),
 (0xa980,0xa981, "VMAbv"), (0xa982,0xa982, "FAbv"),
 (0xa983,0xa983, "VMPst"), (0xa984,0xa9b2, "B"),
 (0xa9b3,0xa9b3, "CMAbv"), (0xa9b4,0xa9b5, "VPst"),
 (0xa9b6,0xa9b7, "VAbv"), (0xa9b8,0xa9b9, "VBlw"),
 (0xa9ba,0xa9bb, "VPre"), (0xa9bc,0xa9bc, "VAbv"),
 (0xa9bd,0xa9bd, "MBlw"), (0xa9be,0xa9be, "MPst"),
 (0xa9bf,0xa9bf, "MBlw"), (0xa9c0,0xa9c0, "H"),
 (0xa9d0,0xa9e4, "B"), (0xa9e5,0xa9e5, "VAbv"),
 (0xa9e7,0xaa28, "B"), (0xaa29,0xaa29, "VMAbv"),
 (0xaa2a,0xaa2c, "VAbv"), (0xaa2d,0xaa2d, "VBlw"),
 (0xaa2e,0xaa2e, "VAbv"), (0xaa2f,0xaa30, "VPre"),
 (0xaa31,0xaa31, "VAbv"), (0xaa32,0xaa32, "VBlw"),
 (0xaa33,0xaa33, "MPst"), (0xaa34,0xaa34, "MPre"),
 (0xaa35,0xaa35, "MAbv"), (0xaa36,0xaa36, "MBlw"),
 (0xaa40,0xaa42, "B"), (0xaa43,0xaa43, "FAbv"),
 (0xaa44,0xaa4b, "B"), (0xaa4c,0xaa4c, "FAbv"),
 (0xaa4d,0xaa4d, "FPst"), (0xaa50,0xaa73, "B"),
 (0xaa74,0xaa76, "GB"), (0xaa7a,0xaa7a, "B"),
 (0xaa7b,0xaa7b, "VMPst"), (0xaa7c,0xaa7c, "VMAbv"),
 (0xaa7d,0xaa7d, "VMPst"), (0xaa7e,0xaaaf, "B"),
 (0xaab0,0xaab0, "VAbv"), (0xaab1,0xaab1, "B"),
 (0xaab2,0xaab3, "VAbv"), (0xaab4,0xaab4, "VBlw"),
 (0xaab5,0xaab6, "B"), (0xaab7,0xaab8, "VAbv"),
 (0xaab9,0xaabd, "B"), (0xaabe,0xaabe, "VAbv"),
 (0xaabf,0xaabf, "VMAbv"), (0xaac0,0xaac0, "B"),
 (0xaac1,0xaac1, "VMAbv"), (0xaac2,0xaaea, "B"),
 (0xaaeb,0xaaeb, "VPre"), (0xaaec,0xaaec, "VBlw"),
 (0xaaed,0xaaed, "VAbv"), (0xaaee,0xaaee, "VPre"),
 (0xaaef,0xaaef, "VPst"), (0xaaf5,0xaaf5, "VMPst"),
 (0xaaf6,0xaaf6, "H"), (0xabc0,0xabe2, "B"),
 (0xabe3,0xabe4, "VPst"), (0xabe5,0xabe5, "VAbv"),
 (0xabe6,0xabe7, "VPst"), (0xabe8,0xabe8, "VBlw"),
 (0xabe9,0xabea, "VPst"), (0xabec,0xabec, "VMPst"),
 (0xabed,0xabed, "VBlw"), (0xabf0,0x10a00, "B"),
 (0x10a01,0x10a03, "VBlw"), (0x10a05,0x10a05, "VAbv"),
 (0x10a06,0x10a06, "VBlw"), (0x10a0c,0x10a0c, "VPst"),
 (0x10a0d,0x10a0e, "VMBlw"), (0x10a0f,0x10a0f, "VMAbv"),
 (0x10a10,0x10a35, "B"), (0x10a38,0x10a38, "CMAbv"),
 (0x10a39,0x10a3a, "CMBlw"), (0x10a3f,0x10a3f, "H"),
 (0x10a40,0x10ae4, "B"), (0x10ae5,0x10ae6, "CMBlw"),
 (0x10b80,0x10d23, "B"), (0x10d24,0x10d26, "VMAbv"),
 (0x10d27,0x10d27, "CMAbv"), (0x10d30,0x10ea9, "B"),
 (0x10eab,0x10eac, "VAbv"), (0x10eb0,0x10f45, "B"),
 (0x10f46,0x10f50, "VMBlw"), (0x10f51,0x10fcb, "B"),
 (0x11000,0x11000, "VMPst"), (0x11001,0x11001, "VMAbv"),
 (0x11002,0x11002, "VMPst"), (0x11003,0x11004, "CS"),
 (0x11005,0x11037, "B"), (0x11038,0x1103b, "VAbv"),
 (0x1103c,0x11041, "VBlw"), (0x11042,0x11045, "VAbv"),
 (0x11046,0x11046, "HVM"), (0x11052,0x11065, "N"),
 (0x11066,0x1106f, "B"), (0x1107f,0x1107f, "HN"),
 (0x11080,0x11081, "VMAbv"), (0x11082,0x11082, "VMPst"),
 (0x11083,0x110af, "B"), (0x110b0,0x110b0, "VPst"),
 (0x110b1,0x110b1, "VPre"), (0x110b2,0x110b2, "VPst"),
 (0x110b3,0x110b4, "VBlw"), (0x110b5,0x110b6, "VAbv"),
 (0x110b7,0x110b8, "VPst"), (0x110b9,0x110b9, "H"),
 (0x110ba,0x110ba, "CMBlw"), (0x11100,0x11102, "VMAbv"),
 (0x11103,0x11126, "B"), (0x11127,0x11129, "VBlw"),
 (0x1112a,0x1112b, "VAbv"), (0x1112c,0x1112c, "VPre"),
 (0x1112d,0x1112d, "VBlw"), (0x1112e,0x1112f, "VAbv"),
 (0x11130,0x11130, "VBlw"), (0x11131,0x11132, "VAbv"),
 (0x11133,0x11133, "H"), (0x11134,0x11134, "CMAbv"),
 (0x11136,0x11144, "B"), (0x11145,0x11146, "VPst"),
 (0x11147,0x11172, "B"), (0x11173,0x11173, "CMBlw"),
 (0x11180,0x11181, "VMAbv"), (0x11182,0x11182, "VMPst"),
 (0x11183,0x111b2, "B"), (0x111b3,0x111b3, "VPst"),
 (0x111b4,0x111b4, "VPre"), (0x111b5,0x111b5, "VPst"),
 (0x111b6,0x111bb, "VBlw"), (0x111bc,0x111bf, "VAbv"),
 (0x111c0,0x111c0, "H"), (0x111c1,0x111c1, "B"),
 (0x111c2,0x111c3, "R"), (0x111c8,0x111c8, "GB"),
 (0x111c9,0x111c9, "FMBlw"), (0x111ca,0x111ca, "CMBlw"),
 (0x111cb,0x111cb, "VAbv"), (0x111cc,0x111cc, "VBlw"),
 (0x111ce,0x111ce, "VPre"), (0x111cf,0x111cf, "VMAbv"),
 (0x111d0,0x1122b, "B"), (0x1122c,0x1122e, "VPst"),
 (0x1122f,0x1122f, "VBlw"), (0x11230,0x11233, "VAbv"),
 (0x11234,0x11234, "VMAbv"), (0x11235,0x11235, "H"),
 (0x11236,0x11237, "CMAbv"), (0x1123e,0x1123e, "VMAbv"),
 (0x11280,0x112de, "B"), (0x112df,0x112df, "VMAbv"),
 (0x112e0,0x112e0, "VPst"), (0x112e1,0x112e1, "VPre"),
 (0x112e2,0x112e2, "VPst"), (0x112e3,0x112e4, "VBlw"),
 (0x112e5,0x112e8, "VAbv"), (0x112e9,0x112e9, "CMBlw"),
 (0x112ea,0x112ea, "VBlw"), (0x112f0,0x112f9, "B"),
 (0x11300,0x11303, "VMAbv"), (0x11305,0x11339, "B"),
 (0x1133b,0x1133c, "CMBlw"), (0x1133d,0x1133d, "B"),
 (0x1133e,0x1133f, "VPst"), (0x11340,0x11340, "VAbv"),
 (0x11341,0x11344, "VPst"), (0x11347,0x1134c, "VPre"),
 (0x1134d,0x1134d, "HVM"), (0x11357,0x11357, "VPst"),
 (0x1135e,0x11361, "B"), (0x11362,0x11363, "VPst"),
 (0x11366,0x11374, "VMAbv"), (0x11400,0x11434, "B"),
 (0x11435,0x11435, "VPst"), (0x11436,0x11436, "VPre"),
 (0x11437,0x11437, "VPst"), (0x11438,0x1143d, "VBlw"),
 (0x1143e,0x1143f, "VAbv"), (0x11440,0x11441, "VPst"),
 (0x11442,0x11442, "H"), (0x11443,0x11444, "VMAbv"),
 (0x11445,0x11445, "VMPst"), (0x11446,0x11446, "CMBlw"),
 (0x11447,0x11459, "B"), (0x1145e,0x1145e, "FMAbv"),
 (0x1145f,0x1145f, "B"), (0x11460,0x11461, "CS"),
 (0x11481,0x114af, "B"), (0x114b0,0x114b0, "VPst"),
 (0x114b1,0x114b1, "VPre"), (0x114b2,0x114b2, "VPst"),
 (0x114b3,0x114b8, "VBlw"), (0x114b9,0x114b9, "VPre"),
 (0x114ba,0x114ba, "VAbv"), (0x114bb,0x114bc, "VPre"),
 (0x114bd,0x114bd, "VPst"), (0x114be,0x114be, "VPre"),
 (0x114bf,0x114c1, "VMAbv"), (0x114c2,0x114c2, "H"),
 (0x114c3,0x114c3, "CMBlw"), (0x114c4,0x115ae, "B"),
 (0x115af,0x115af, "VPst"), (0x115b0,0x115b0, "VPre"),
 (0x115b1,0x115b1, "VPst"), (0x115b2,0x115b5, "VBlw"),
 (0x115b8,0x115bb, "VPre"), (0x115bc,0x115bd, "VMAbv"),
 (0x115be,0x115be, "VMPst"), (0x115bf,0x115bf, "H"),
 (0x115c0,0x115c0, "CMBlw"), (0x115d8,0x115db, "B"),
 (0x115dc,0x115dd, "VBlw"), (0x11600,0x1162f, "B"),
 (0x11630,0x11632, "VPst"), (0x11633,0x11638, "VBlw"),
 (0x11639,0x1163a, "VAbv"), (0x1163b,0x1163c, "VPst"),
 (0x1163d,0x1163d, "VMAbv"), (0x1163e,0x1163e, "VMPst"),
 (0x1163f,0x1163f, "H"), (0x11640,0x11640, "VAbv"),
 (0x11650,0x116aa, "B"), (0x116ab,0x116ab, "VMAbv"),
 (0x116ac,0x116ac, "VMPst"), (0x116ad,0x116ad, "VAbv"),
 (0x116ae,0x116ae, "VPre"), (0x116af,0x116af, "VPst"),
 (0x116b0,0x116b1, "VBlw"), (0x116b2,0x116b5, "VAbv"),
 (0x116b6,0x116b6, "H"), (0x116b7,0x116b7, "CMBlw"),
 (0x116b8,0x1171a, "B"), (0x1171d,0x1171d, "MBlw"),
 (0x1171e,0x1171e, "MPre"), (0x1171f,0x1171f, "MAbv"),
 (0x11720,0x11721, "VPst"), (0x11722,0x11723, "VAbv"),
 (0x11724,0x11725, "VBlw"), (0x11726,0x11726, "VPre"),
 (0x11727,0x11727, "VAbv"), (0x11728,0x11728, "VBlw"),
 (0x11729,0x1172a, "VAbv"), (0x1172b,0x1172b, "VMAbv"),
 (0x11730,0x1182b, "B"), (0x1182c,0x1182c, "VPst"),
 (0x1182d,0x1182d, "VPre"), (0x1182e,0x1182e, "VPst"),
 (0x1182f,0x11832, "VBlw"), (0x11833,0x11836, "VAbv"),
 (0x11837,0x11837, "VMAbv"), (0x11838,0x11838, "VMPst"),
 (0x11839,0x11839, "H"), (0x1183a,0x1183a, "CMBlw"),
 (0x11900,0x1192f, "B"), (0x11930,0x11934, "VPst"),
 (0x11935,0x11938, "VPre"), (0x1193b,0x1193c, "VMAbv"),
 (0x1193d,0x1193d, "VPst"), (0x1193e,0x1193e, "H"),
 (0x1193f,0x1193f, "R"), (0x11940,0x11940, "MPst"),
 (0x11941,0x11941, "R"), (0x11942,0x11942, "MPst"),
 (0x11943,0x11943, "CMBlw"), (0x11950,0x119d0, "B"),
 (0x119d1,0x119d1, "VPst"), (0x119d2,0x119d2, "VPre"),
 (0x119d3,0x119d3, "VPst"), (0x119d4,0x119d7, "VBlw"),
 (0x119da,0x119db, "VAbv"), (0x119dc,0x119dd, "VPst"),
 (0x119de,0x119df, "VMPst"), (0x119e0,0x119e0, "H"),
 (0x119e1,0x119e1, "B"), (0x119e4,0x119e4, "VPre"),
 (0x11a00,0x11a00, "B"), (0x11a01,0x11a01, "VAbv"),
 (0x11a02,0x11a03, "VBlw"), (0x11a04,0x11a09, "VAbv"),
 (0x11a0a,0x11a0a, "VBlw"), (0x11a0b,0x11a32, "B"),
 (0x11a33,0x11a33, "FMBlw"), (0x11a34,0x11a34, "VBlw"),
 (0x11a35,0x11a38, "VMAbv"), (0x11a39,0x11a39, "VMPst"),
 (0x11a3a,0x11a3a, "R"), (0x11a3b,0x11a3e, "MBlw"),
 (0x11a3f,0x11a45, "GB"), (0x11a47,0x11a47, "H"),
 (0x11a50,0x11a50, "B"), (0x11a51,0x11a51, "VAbv"),
 (0x11a52,0x11a53, "VBlw"), (0x11a54,0x11a56, "VAbv"),
 (0x11a57,0x11a58, "VPst"), (0x11a59,0x11a5b, "VBlw"),
 (0x11a5c,0x11a83, "B"), (0x11a84,0x11a89, "R"),
 (0x11a8a,0x11a95, "FBlw"), (0x11a96,0x11a96, "VMAbv"),
 (0x11a97,0x11a97, "VMPst"), (0x11a98,0x11a98, "CMAbv"),
 (0x11a99,0x11a99, "H"), (0x11a9d,0x11c2e, "B"),
 (0x11c2f,0x11c2f, "VPst"), (0x11c30,0x11c31, "VAbv"),
 (0x11c32,0x11c36, "VBlw"), (0x11c38,0x11c3b, "VAbv"),
 (0x11c3c,0x11c3d, "VMAbv"), (0x11c3e,0x11c3e, "VMPst"),
 (0x11c3f,0x11c3f, "H"), (0x11c40,0x11c40, "B"),
 (0x11c44,0x11c45, "GB"), (0x11c50,0x11c8f, "B"),
 (0x11c92,0x11caf, "SUB"), (0x11cb0,0x11cb0, "VBlw"),
 (0x11cb1,0x11cb1, "VPre"), (0x11cb2,0x11cb2, "VBlw"),
 (0x11cb3,0x11cb3, "VAbv"), (0x11cb4,0x11cb4, "VPst"),
 (0x11cb5,0x11cb6, "VMAbv"), (0x11d00,0x11d30, "B"),
 (0x11d31,0x11d35, "VAbv"), (0x11d36,0x11d36, "VBlw"),
 (0x11d3a,0x11d3f, "VAbv"), (0x11d40,0x11d41, "VMAbv"),
 (0x11d42,0x11d42, "CMBlw"), (0x11d43,0x11d43, "VAbv"),
 (0x11d44,0x11d44, "VBlw"), (0x11d45,0x11d45, "H"),
 (0x11d46,0x11d46, "R"), (0x11d47,0x11d47, "MBlw"),
 (0x11d50,0x11d89, "B"), (0x11d8a,0x11d8e, "VPst"),
 (0x11d90,0x11d91, "VAbv"), (0x11d93,0x11d94, "VPst"),
 (0x11d95,0x11d95, "VMAbv"), (0x11d96,0x11d96, "VMPst"),
 (0x11d97,0x11d97, "H"), (0x11da0,0x11ef1, "B"),
 (0x11ef2,0x11ef2, "GB"), (0x11ef3,0x11ef3, "VAbv"),
 (0x11ef4,0x11ef4, "VBlw"), (0x11ef5,0x11ef5, "VPre"),
 (0x11ef6,0x11ef6, "VPst"), (0x13430,0x13436, "H"),
 (0x16b00,0x16b2f, "B"), (0x16b30,0x16b36, "VMAbv"),
 (0x16f00,0x16f4a, "B"), (0x16f4f,0x16f4f, "CMBlw"),
 (0x16f50,0x16f50, "IND"), (0x16f51,0x16f87, "VBlw"),
 (0x16f8f,0x16f92, "VMBlw"), (0x16fe4,0x1bc99, "B"),
 (0x1bc9d,0x1bc9e, "CMBlw"), (0x1e100,0x1e12c, "B"),
 (0x1e130,0x1e136, "VMAbv"), (0x1e137,0x1e2eb, "B"),
 (0x1e2ec,0x1e2ef, "VMAbv"), (0x1e2f0,0x1e943, "B"),
 (0x1e944,0x1e94a, "CMAbv"), (0x1e94b,0x1e959, "B"),
)
