"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractQualifierFromArn = exports.Version = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
/**
 * Tag the current state of a Function with a Version number
 *
 * Avoid using this resource directly. If you need a Version object, use
 * `function.currentVersion` instead. That will add a Version object to your
 * template, and make sure the Version is invalidated whenever the Function
 * object changes. If you use the `Version` resource directly, you are
 * responsible for making sure it is invalidated (by changing its
 * logical ID) whenever necessary.
 *
 * Version resources can then be used in `Alias` resources to refer to a
 * particular deployment of a Lambda.
 *
 * If you want to ensure that you're associating the right version with
 * the right deployment, specify the `codeSha256` property while
 * creating the `Version.
 */
class Version extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.canCreatePermissions = true;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_VersionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.lambda = props.lambda;
        this.architecture = props.lambda.architecture;
        const version = new lambda_generated_1.CfnVersion(this, 'Resource', {
            codeSha256: props.codeSha256,
            description: props.description,
            functionName: props.lambda.functionName,
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        if (props.removalPolicy) {
            version.applyRemovalPolicy(props.removalPolicy, {
                default: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.version = version.attrVersion;
        this.functionArn = version.ref;
        this.functionName = `${this.lambda.functionName}:${this.version}`;
        this.qualifier = version.attrVersion;
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
    }
    /**
     * Construct a Version object from a Version ARN.
     *
     * @param scope The cdk scope creating this resource
     * @param id The cdk id of this resource
     * @param versionArn The version ARN to create this version from
     */
    static fromVersionArn(scope, id, versionArn) {
        const version = extractQualifierFromArn(versionArn);
        const lambda = function_1.Function.fromFunctionArn(scope, `${id}Function`, versionArn);
        class Import extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.version = version;
                this.lambda = lambda;
                this.functionName = `${lambda.functionName}:${version}`;
                this.functionArn = versionArn;
                this.grantPrincipal = lambda.grantPrincipal;
                this.role = lambda.role;
                this.architecture = lambda.architecture;
                this.qualifier = version;
                this.canCreatePermissions = this._isStackAccount();
            }
            addAlias(name, opts = {}) {
                return util_1.addAlias(this, this, name, opts);
            }
            get edgeArn() {
                if (version === '$LATEST') {
                    throw new Error('$LATEST function version cannot be used for Lambda@Edge');
                }
                return this.functionArn;
            }
        }
        return new Import(scope, id);
    }
    static fromVersionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_VersionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVersionAttributes);
            }
            throw error;
        }
        class Import extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.version = attrs.version;
                this.lambda = attrs.lambda;
                this.functionName = `${attrs.lambda.functionName}:${attrs.version}`;
                this.functionArn = `${attrs.lambda.functionArn}:${attrs.version}`;
                this.grantPrincipal = attrs.lambda.grantPrincipal;
                this.role = attrs.lambda.role;
                this.architecture = attrs.lambda.architecture;
                this.qualifier = attrs.version;
                this.canCreatePermissions = this._isStackAccount();
            }
            addAlias(name, opts = {}) {
                return util_1.addAlias(this, this, name, opts);
            }
            get edgeArn() {
                if (attrs.version === '$LATEST') {
                    throw new Error('$LATEST function version cannot be used for Lambda@Edge');
                }
                return this.functionArn;
            }
        }
        return new Import(scope, id);
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get role() {
        return this.lambda.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differs from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the ARN from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionArn}:${this.version}`,
            },
            ...props,
        });
    }
    /**
     * Defines an alias for this version.
     * @param aliasName The name of the alias (e.g. "live")
     * @param options Alias options
     * @deprecated Calling `addAlias` on a `Version` object will cause the Alias to be replaced on every function update. Call `function.addAlias()` or `new Alias()` instead.
     */
    addAlias(aliasName, options = {}) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Version#addAlias", "Calling `addAlias` on a `Version` object will cause the Alias to be replaced on every function update. Call `function.addAlias()` or `new Alias()` instead.");
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_AliasOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlias);
            }
            throw error;
        }
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        // Validate first that this version can be used for Lambda@Edge
        if (this.version === '$LATEST') {
            throw new Error('$LATEST function version cannot be used for Lambda@Edge');
        }
        // Check compatibility at synthesis. It could be that the version was associated
        // with a CloudFront distribution first and made incompatible afterwards.
        return core_1.Lazy.string({
            produce: () => {
                // Validate that the underlying function can be used for Lambda@Edge
                if (this.lambda instanceof function_1.Function) {
                    this.lambda._checkEdgeCompatibility();
                }
                return this.functionArn;
            },
        });
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Version = Version;
_a = JSII_RTTI_SYMBOL_1;
Version[_a] = { fqn: "@aws-cdk/aws-lambda.Version", version: "1.155.0" };
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the
 * qualifier (= version or alias) from the ARN.
 *
 * Version ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name:qualifier
 *
 * ..which means that in order to extract the `qualifier` component from the ARN, we can
 * split the ARN using ":" and select the component in index 7.
 *
 * @returns `FnSelect(7, FnSplit(':', arn))`
 */
function extractQualifierFromArn(arn) {
    return core_1.Fn.select(7, core_1.Fn.split(':', arn));
}
exports.extractQualifierFromArn = extractQualifierFromArn;
//# sourceMappingURL=data:application/json;base64,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