"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const architecture_1 = require("./architecture");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const handler_1 = require("./handler");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const layers_1 = require("./layers");
const runtime_1 = require("./runtime");
const util_1 = require("./util");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_FunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.functionName && !core_1.Token.isUnresolved(props.functionName)) {
            if (props.functionName.length > 64) {
                throw new Error(`Function name can not be longer than 64 characters but has ${props.functionName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-_]+$/.test(props.functionName)) {
                throw new Error(`Function name ${props.functionName} can contain only letters, numbers, hyphens, or underscores with no spaces.`);
            }
        }
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additional managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _b;
                    (_b = this.role) === null || _b === void 0 ? void 0 : _b.addToPrincipalPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPrincipalPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfiling(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfiling(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        // DLQ can be either sns.ITopic or sqs.IQueue
        const dlqTopicOrQueue = this.buildDeadLetterQueue(props);
        if (dlqTopicOrQueue !== undefined) {
            if (this.isQueue(dlqTopicOrQueue)) {
                this.deadLetterQueue = dlqTopicOrQueue;
            }
            else {
                this.deadLetterTopic = dlqTopicOrQueue;
            }
        }
        let fileSystemConfigs = undefined;
        if (props.filesystem) {
            fileSystemConfigs = [{
                    arn: props.filesystem.config.arn,
                    localMountPath: props.filesystem.config.localMountPath,
                }];
        }
        if (props.architecture && props.architectures !== undefined) {
            throw new Error('Either architecture or architectures must be specified but not both.');
        }
        if (props.architectures && props.architectures.length > 1) {
            throw new Error('Only one architecture must be specified.');
        }
        this._architecture = (_b = props.architecture) !== null && _b !== void 0 ? _b : (props.architectures && props.architectures[0]);
        if (props.ephemeralStorageSize && !props.ephemeralStorageSize.isUnresolved()
            && (props.ephemeralStorageSize.toMebibytes() < 512 || props.ephemeralStorageSize.toMebibytes() > 10240)) {
            throw new Error(`Ephemeral storage size must be between 512 and 10240 MB, received ${props.ephemeralStorageSize}.`);
        }
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
                imageUri: (_c = code.image) === null || _c === void 0 ? void 0 : _c.imageUri,
            },
            layers: core_1.Lazy.list({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler === handler_1.Handler.FROM_IMAGE ? undefined : props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            packageType: props.runtime === runtime_1.Runtime.FROM_IMAGE ? 'Image' : undefined,
            runtime: props.runtime === runtime_1.Runtime.FROM_IMAGE ? undefined : props.runtime.name,
            role: this.role.roleArn,
            // Uncached because calling '_checkEdgeCompatibility', which gets called in the resolve of another
            // Token, actually *modifies* the 'environment' map.
            environment: core_1.Lazy.uncachedAny({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            ephemeralStorage: props.ephemeralStorageSize ? {
                size: props.ephemeralStorageSize.toMebibytes(),
            } : undefined,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(dlqTopicOrQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
            imageConfig: undefinedIfNoKeys({
                command: (_d = code.image) === null || _d === void 0 ? void 0 : _d.cmd,
                entryPoint: (_e = code.image) === null || _e === void 0 ? void 0 : _e.entrypoint,
                workingDirectory: (_f = code.image) === null || _f === void 0 ? void 0 : _f.workingDirectory,
            }),
            kmsKeyArn: (_g = props.environmentEncryption) === null || _g === void 0 ? void 0 : _g.keyArn,
            fileSystemConfigs,
            codeSigningConfigArn: (_h = props.codeSigningConfig) === null || _h === void 0 ? void 0 : _h.codeSigningConfigArn,
            architectures: this._architecture ? [this._architecture.name] : undefined,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.runtime = props.runtime;
        this.timeout = props.timeout;
        this.architecture = (_j = props.architecture) !== null && _j !== void 0 ? _j : architecture_1.Architecture.X86_64;
        if (props.layers) {
            if (props.runtime === runtime_1.Runtime.FROM_IMAGE) {
                throw new Error('Layers are not supported for container image functions');
            }
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            if (!props.vpc) {
                throw new Error('Cannot configure \'filesystem\' without configuring a VPC.');
            }
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            // There could be a race if the Lambda is used in a CustomResource. It is possible for the Lambda to
            // fail to attach to a given FileSystem if we do not have a dependency on the SecurityGroup ingress/egress
            // rules that were created between this Lambda's SG & the Filesystem SG.
            this.connections.securityGroups.forEach(sg => {
                sg.node.findAll().forEach(child => {
                    if (child instanceof core_1.CfnResource && child.cfnResourceType === 'AWS::EC2::SecurityGroupEgress') {
                        resource.node.addDependency(child);
                    }
                });
            });
            (_k = config.connections) === null || _k === void 0 ? void 0 : _k.securityGroups.forEach(sg => {
                sg.node.findAll().forEach(child => {
                    if (child instanceof core_1.CfnResource && child.cfnResourceType === 'AWS::EC2::SecurityGroupIngress') {
                        resource.node.addDependency(child);
                    }
                });
            });
        }
        // Configure Lambda insights
        this.configureLambdaInsights(props);
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        if (this._warnIfCurrentVersionCalled) {
            this.warnInvokeFunctionPermissions(this);
        }
        ;
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: () => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    get resourceArnsForGrantInvoke() {
        return [this.functionArn, `${this.functionArn}:*`];
    }
    /**
     * Record whether specific properties in the `AWS::Lambda::Function` resource should
     * also be associated to the Version resource.
     * See 'currentVersion' section in the module README for more details.
     * @param propertyName The property to classify
     * @param locked whether the property should be associated to the version or not.
     */
    static classifyVersionProperty(propertyName, locked) {
        this._VER_PROPS[propertyName] = locked;
    }
    /**
     * Import a lambda function into the CDK using its name
     */
    static fromFunctionName(scope, id, functionName) {
        return Function.fromFunctionAttributes(scope, id, {
            functionArn: core_1.Stack.of(scope).formatArn({
                service: 'lambda',
                resource: 'function',
                resourceName: functionName,
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            }),
        });
    }
    /**
     * Import a lambda function into the CDK using its ARN
     */
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_FunctionAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromFunctionAttributes);
            }
            throw error;
        }
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                var _b, _c, _d;
                super(s, i, {
                    environmentFromArn: functionArn,
                });
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.architecture = (_b = attrs.architecture) !== null && _b !== void 0 ? _b : architecture_1.Architecture.X86_64;
                this.resourceArnsForGrantInvoke = [this.functionArn, `${this.functionArn}:*`];
                this.canCreatePermissions = (_c = attrs.sameEnvironment) !== null && _c !== void 0 ? _c : this._isStackAccount();
                this._skipPermissions = (_d = attrs.skipPermissions) !== null && _d !== void 0 ? _d : false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     */
    addEnvironment(key, value, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_EnvironmentOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEnvironment);
            }
            throw error;
        }
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_ILayerVersion(layers);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLayers);
            }
            throw error;
        }
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            // Currently no validations for compatible architectures since Lambda service
            // allows layers configured with one architecture to be used with a Lambda function
            // from another architecture.
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Function#addVersion", "This method will create an AWS::Lambda::Version resource which\nsnapshots the AWS Lambda function *at the time of its creation* and it\nwon't get updated when the function changes. Instead, use\n`this.currentVersion` to obtain a reference to a version resource that gets\nautomatically recreated when the function configuration (or code) changes.");
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventInvokeConfigOptions(asyncInvokeConfig);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addVersion);
            }
            throw error;
        }
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * Defines an alias for this function.
     *
     * The alias will automatically be updated to point to the latest version of
     * the function as it is being updated during a deployment.
     *
     * ```ts
     * declare const fn: lambda.Function;
     *
     * fn.addAlias('Live');
     *
     * // Is equivalent to
     *
     * new lambda.Alias(this, 'AliasLive', {
     *   aliasName: 'Live',
     *   version: fn.currentVersion,
     * });
     *
     * @param aliasName The name of the alias
     * @param options Alias options
     */
    addAlias(aliasName, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_AliasOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlias);
            }
            throw error;
        }
        return util_1.addAlias(this, this.currentVersion, aliasName, options);
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                core_1.Annotations.of(this).addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    /**
     * Configured lambda insights on the function if specified. This is acheived by adding an imported layer which is added to the
     * list of lambda layers on synthesis.
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
     */
    configureLambdaInsights(props) {
        var _b;
        if (props.insightsVersion === undefined) {
            return;
        }
        if (props.runtime !== runtime_1.Runtime.FROM_IMAGE) {
            // Layers cannot be added to Lambda container images. The image should have the insights agent installed.
            // See https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
            this.addLayers(layers_1.LayerVersion.fromLayerVersionArn(this, 'LambdaInsightsLayer', props.insightsVersion._bind(this, this).arn));
        }
        (_b = this.role) === null || _b === void 0 ? void 0 : _b.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        var _b;
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = (_b = props.allowPublicSubnet) !== null && _b !== void 0 ? _b : false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function int a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    isQueue(deadLetterQueue) {
        return deadLetterQueue.queueArn !== undefined;
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled && !props.deadLetterTopic) {
            return undefined;
        }
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (props.deadLetterTopic && (props.deadLetterQueue || props.deadLetterQueueEnabled !== undefined)) {
            throw new Error('deadLetterQueue and deadLetterTopic cannot be specified together at the same time');
        }
        let deadLetterQueue;
        if (props.deadLetterTopic) {
            deadLetterQueue = props.deadLetterTopic;
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [deadLetterQueue.topicArn],
            }));
        }
        else {
            deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
                retentionPeriod: core_1.Duration.days(14),
            });
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['sqs:SendMessage'],
                resources: [deadLetterQueue.queueArn],
            }));
        }
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: this.isQueue(deadLetterQueue) ? deadLetterQueue.queueArn : deadLetterQueue.topicArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfiling(props) {
        if (!props.runtime.supportsCodeGuruProfiling) {
            throw new Error(`CodeGuru profiling is not supported by runtime ${props.runtime.name}`);
        }
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
_a = JSII_RTTI_SYMBOL_1;
Function[_a] = { fqn: "@aws-cdk/aws-lambda.Function", version: "1.155.0" };
/** @internal */
Function._VER_PROPS = {};
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, props) {
    // mutually exclusive
    const codeType = [code.inlineCode, code.s3Location, code.image];
    if (codeType.filter(x => !!x).length !== 1) {
        throw new Error('lambda.Code must specify exactly one of: "inlineCode", "s3Location", or "image"');
    }
    if (!!code.image === (props.handler !== handler_1.Handler.FROM_IMAGE)) {
        throw new Error('handler must be `Handler.FROM_IMAGE` when using image asset for Lambda function');
    }
    if (!!code.image === (props.runtime !== runtime_1.Runtime.FROM_IMAGE)) {
        throw new Error('runtime must be `Runtime.FROM_IMAGE` when using image asset for Lambda function');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !props.runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${props.runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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