"use strict";
const assert_1 = require("@aws-cdk/assert");
const assert_2 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/cdk");
const fc = require("fast-check");
const util_1 = require("./util");
const util_2 = require("./util");
/**
 * Synthesize the given step scaling setup to a template
 */
function setupStepScaling(intervals) {
    const stack = new cdk.Stack();
    const target = util_2.createScalableTarget(stack);
    target.scaleOnMetric('ScaleInterval', {
        metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Success' }),
        scalingSteps: intervals
    });
    return new ScalingStackTemplate(assert_2.SynthUtils.synthesize(stack).template);
}
class ScalingStackTemplate {
    constructor(template) {
        this.template = template;
        this.lowerPolicy = 'TargetScaleIntervalLowerPolicy6F26D597';
        this.lowerAlarm = 'TargetScaleIntervalLowerAlarm4B5CE869';
        this.upperPolicy = 'TargetScaleIntervalUpperPolicy7C751132';
        this.upperAlarm = 'TargetScaleIntervalUpperAlarm69FD1BBB';
    }
    get lowerThreshold() {
        return this.threshold(this.lowerAlarm);
    }
    get upperThreshold() {
        return this.threshold(this.upperAlarm);
    }
    get lowerSteps() {
        return this.steps(this.lowerPolicy);
    }
    get upperSteps() {
        return this.steps(this.upperPolicy);
    }
    allStepsAbsolute() {
        const ret = new Array();
        const lowerThreshold = this.lowerThreshold;
        if (lowerThreshold !== undefined) {
            ret.push(...this.lowerSteps.map(x => makeAbsolute(lowerThreshold, x)));
        }
        const upperThreshold = this.upperThreshold;
        if (upperThreshold !== undefined) {
            ret.push(...this.upperSteps.map(x => makeAbsolute(upperThreshold, x)));
        }
        return ret;
    }
    resource(id) {
        return this.template.Resources[id];
    }
    threshold(id) {
        return apply(this.resource(id), x => x.Properties.Threshold);
    }
    steps(id) {
        return apply(this.resource(id), x => x.Properties.StepScalingPolicyConfiguration.StepAdjustments);
    }
}
function makeAbsolute(threshold, step) {
    return concrete({
        MetricIntervalLowerBound: apply(step.MetricIntervalLowerBound, x => x + threshold),
        MetricIntervalUpperBound: apply(step.MetricIntervalUpperBound, x => x + threshold),
        ScalingAdjustment: step.ScalingAdjustment
    });
}
function overlaps(a, b) {
    return (a.MetricIntervalLowerBound < b.MetricIntervalUpperBound
        && a.MetricIntervalUpperBound > b.MetricIntervalLowerBound);
}
function concrete(step) {
    return {
        MetricIntervalLowerBound: ifUndefined(step.MetricIntervalLowerBound, -Infinity),
        MetricIntervalUpperBound: ifUndefined(step.MetricIntervalUpperBound, Infinity),
        ScalingAdjustment: step.ScalingAdjustment
    };
}
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
function apply(x, f) {
    if (x === undefined) {
        return undefined;
    }
    return f(x);
}
/**
 * Helper function to print variables in case of a failing property check
 */
function reportFalse(cond, ...repr) {
    if (!cond) {
        // tslint:disable-next-line:no-console
        console.error('PROPERTY FAILS ON:', ...repr);
    }
    return cond;
}
module.exports = {
    'alarm thresholds are valid numbers'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const lowerThreshold = template.lowerThreshold;
            const upperThreshold = template.upperThreshold;
            return reportFalse((lowerThreshold === undefined || (lowerThreshold > 0 && lowerThreshold !== Infinity))
                && (upperThreshold === undefined || (upperThreshold > 0 && upperThreshold !== Infinity)), lowerThreshold, upperThreshold);
        }));
        test.done();
    },
    'generated step intervals are valid intervals'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            return reportFalse(steps.every(step => {
                return step.MetricIntervalLowerBound < step.MetricIntervalUpperBound;
            }), steps, 'template', JSON.stringify(template, undefined, 2));
        }));
        test.done();
    },
    'generated step intervals are nonoverlapping'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            for (let i = 0; i < steps.length; i++) {
                const compareTo = steps.slice(i + 1);
                if (compareTo.some(x => overlaps(steps[i], x))) {
                    return reportFalse(false, steps);
                }
            }
            return true;
        }), { verbose: true });
        test.done();
    },
    'all template intervals occur in input array'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const steps = template.allStepsAbsolute();
            return steps.every(step => {
                return reportFalse(intervals.find(interval => {
                    const acceptableLowerBounds = step.MetricIntervalLowerBound === -Infinity ? [undefined, 0] : [undefined, step.MetricIntervalLowerBound];
                    // tslint:disable-next-line:max-line-length
                    const acceptableUpperBounds = step.MetricIntervalUpperBound === Infinity ? [undefined, Infinity] : [undefined, step.MetricIntervalUpperBound];
                    return (acceptableLowerBounds.includes(interval.lower) && acceptableUpperBounds.includes(interval.upper));
                }) !== undefined, step, intervals);
            });
        }));
        test.done();
    },
    'lower alarm uses lower policy'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const alarm = template.resource(template.lowerAlarm);
            fc.pre(alarm !== undefined);
            return reportFalse(alarm.Properties.AlarmActions[0].Ref === template.lowerPolicy, alarm);
        }));
        test.done();
    },
    'upper alarm uses upper policy'(test) {
        fc.assert(fc.property(util_1.arbitrary_input_intervals(), (intervals) => {
            const template = setupStepScaling(intervals);
            const alarm = template.resource(template.upperAlarm);
            fc.pre(alarm !== undefined);
            return reportFalse(alarm.Properties.AlarmActions[0].Ref === template.upperPolicy, alarm);
        }));
        test.done();
    },
    'test step scaling on metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const target = util_2.createScalableTarget(stack);
        // WHEN
        target.scaleOnMetric('Tracking', {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            scalingSteps: [
                { upper: 0, change: -1 },
                { lower: 100, change: +1 },
                { lower: 500, change: +5 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "StepScaling",
            ScalingTargetId: {
                Ref: "Target3191CF44"
            },
            StepScalingPolicyConfiguration: {
                AdjustmentType: "ChangeInCapacity",
                MetricAggregationType: "Average",
                StepAdjustments: [
                    {
                        MetricIntervalUpperBound: 0,
                        ScalingAdjustment: -1
                    }
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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